/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a flow log.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FlowLog implements SdkPojo, Serializable, ToCopyableBuilder<FlowLog.Builder, FlowLog> {
    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime")
            .getter(getter(FlowLog::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime")
                    .unmarshallLocationName("creationTime").build()).build();

    private static final SdkField<String> DELIVER_LOGS_ERROR_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeliverLogsErrorMessage")
            .getter(getter(FlowLog::deliverLogsErrorMessage))
            .setter(setter(Builder::deliverLogsErrorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliverLogsErrorMessage")
                    .unmarshallLocationName("deliverLogsErrorMessage").build()).build();

    private static final SdkField<String> DELIVER_LOGS_PERMISSION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeliverLogsPermissionArn")
            .getter(getter(FlowLog::deliverLogsPermissionArn))
            .setter(setter(Builder::deliverLogsPermissionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliverLogsPermissionArn")
                    .unmarshallLocationName("deliverLogsPermissionArn").build()).build();

    private static final SdkField<String> DELIVER_CROSS_ACCOUNT_ROLE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeliverCrossAccountRole")
            .getter(getter(FlowLog::deliverCrossAccountRole))
            .setter(setter(Builder::deliverCrossAccountRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliverCrossAccountRole")
                    .unmarshallLocationName("deliverCrossAccountRole").build()).build();

    private static final SdkField<String> DELIVER_LOGS_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeliverLogsStatus")
            .getter(getter(FlowLog::deliverLogsStatus))
            .setter(setter(Builder::deliverLogsStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeliverLogsStatus")
                    .unmarshallLocationName("deliverLogsStatus").build()).build();

    private static final SdkField<String> FLOW_LOG_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FlowLogId")
            .getter(getter(FlowLog::flowLogId))
            .setter(setter(Builder::flowLogId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FlowLogId")
                    .unmarshallLocationName("flowLogId").build()).build();

    private static final SdkField<String> FLOW_LOG_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FlowLogStatus")
            .getter(getter(FlowLog::flowLogStatus))
            .setter(setter(Builder::flowLogStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FlowLogStatus")
                    .unmarshallLocationName("flowLogStatus").build()).build();

    private static final SdkField<String> LOG_GROUP_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogGroupName")
            .getter(getter(FlowLog::logGroupName))
            .setter(setter(Builder::logGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogGroupName")
                    .unmarshallLocationName("logGroupName").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceId")
            .getter(getter(FlowLog::resourceId))
            .setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("resourceId").build()).build();

    private static final SdkField<String> TRAFFIC_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TrafficType")
            .getter(getter(FlowLog::trafficTypeAsString))
            .setter(setter(Builder::trafficType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrafficType")
                    .unmarshallLocationName("trafficType").build()).build();

    private static final SdkField<String> LOG_DESTINATION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogDestinationType")
            .getter(getter(FlowLog::logDestinationTypeAsString))
            .setter(setter(Builder::logDestinationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestinationType")
                    .unmarshallLocationName("logDestinationType").build()).build();

    private static final SdkField<String> LOG_DESTINATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogDestination")
            .getter(getter(FlowLog::logDestination))
            .setter(setter(Builder::logDestination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogDestination")
                    .unmarshallLocationName("logDestination").build()).build();

    private static final SdkField<String> LOG_FORMAT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LogFormat")
            .getter(getter(FlowLog::logFormat))
            .setter(setter(Builder::logFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogFormat")
                    .unmarshallLocationName("logFormat").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(FlowLog::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> MAX_AGGREGATION_INTERVAL_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxAggregationInterval")
            .getter(getter(FlowLog::maxAggregationInterval))
            .setter(setter(Builder::maxAggregationInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxAggregationInterval")
                    .unmarshallLocationName("maxAggregationInterval").build()).build();

    private static final SdkField<DestinationOptionsResponse> DESTINATION_OPTIONS_FIELD = SdkField
            .<DestinationOptionsResponse> builder(MarshallingType.SDK_POJO)
            .memberName("DestinationOptions")
            .getter(getter(FlowLog::destinationOptions))
            .setter(setter(Builder::destinationOptions))
            .constructor(DestinationOptionsResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationOptions")
                    .unmarshallLocationName("destinationOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATION_TIME_FIELD,
            DELIVER_LOGS_ERROR_MESSAGE_FIELD, DELIVER_LOGS_PERMISSION_ARN_FIELD, DELIVER_CROSS_ACCOUNT_ROLE_FIELD,
            DELIVER_LOGS_STATUS_FIELD, FLOW_LOG_ID_FIELD, FLOW_LOG_STATUS_FIELD, LOG_GROUP_NAME_FIELD, RESOURCE_ID_FIELD,
            TRAFFIC_TYPE_FIELD, LOG_DESTINATION_TYPE_FIELD, LOG_DESTINATION_FIELD, LOG_FORMAT_FIELD, TAGS_FIELD,
            MAX_AGGREGATION_INTERVAL_FIELD, DESTINATION_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant creationTime;

    private final String deliverLogsErrorMessage;

    private final String deliverLogsPermissionArn;

    private final String deliverCrossAccountRole;

    private final String deliverLogsStatus;

    private final String flowLogId;

    private final String flowLogStatus;

    private final String logGroupName;

    private final String resourceId;

    private final String trafficType;

    private final String logDestinationType;

    private final String logDestination;

    private final String logFormat;

    private final List<Tag> tags;

    private final Integer maxAggregationInterval;

    private final DestinationOptionsResponse destinationOptions;

    private FlowLog(BuilderImpl builder) {
        this.creationTime = builder.creationTime;
        this.deliverLogsErrorMessage = builder.deliverLogsErrorMessage;
        this.deliverLogsPermissionArn = builder.deliverLogsPermissionArn;
        this.deliverCrossAccountRole = builder.deliverCrossAccountRole;
        this.deliverLogsStatus = builder.deliverLogsStatus;
        this.flowLogId = builder.flowLogId;
        this.flowLogStatus = builder.flowLogStatus;
        this.logGroupName = builder.logGroupName;
        this.resourceId = builder.resourceId;
        this.trafficType = builder.trafficType;
        this.logDestinationType = builder.logDestinationType;
        this.logDestination = builder.logDestination;
        this.logFormat = builder.logFormat;
        this.tags = builder.tags;
        this.maxAggregationInterval = builder.maxAggregationInterval;
        this.destinationOptions = builder.destinationOptions;
    }

    /**
     * <p>
     * The date and time the flow log was created.
     * </p>
     * 
     * @return The date and time the flow log was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * Information about the error that occurred. <code>Rate limited</code> indicates that CloudWatch Logs throttling
     * has been applied for one or more network interfaces, or that you've reached the limit on the number of log groups
     * that you can create. <code>Access error</code> indicates that the IAM role associated with the flow log does not
     * have sufficient permissions to publish to CloudWatch Logs. <code>Unknown error</code> indicates an internal
     * error.
     * </p>
     * 
     * @return Information about the error that occurred. <code>Rate limited</code> indicates that CloudWatch Logs
     *         throttling has been applied for one or more network interfaces, or that you've reached the limit on the
     *         number of log groups that you can create. <code>Access error</code> indicates that the IAM role
     *         associated with the flow log does not have sufficient permissions to publish to CloudWatch Logs.
     *         <code>Unknown error</code> indicates an internal error.
     */
    public final String deliverLogsErrorMessage() {
        return deliverLogsErrorMessage;
    }

    /**
     * <p>
     * The ARN of the IAM role allows the service to publish logs to CloudWatch Logs.
     * </p>
     * 
     * @return The ARN of the IAM role allows the service to publish logs to CloudWatch Logs.
     */
    public final String deliverLogsPermissionArn() {
        return deliverLogsPermissionArn;
    }

    /**
     * <p>
     * The ARN of the IAM role that allows the service to publish flow logs across accounts.
     * </p>
     * 
     * @return The ARN of the IAM role that allows the service to publish flow logs across accounts.
     */
    public final String deliverCrossAccountRole() {
        return deliverCrossAccountRole;
    }

    /**
     * <p>
     * The status of the logs delivery (<code>SUCCESS</code> | <code>FAILED</code>).
     * </p>
     * 
     * @return The status of the logs delivery (<code>SUCCESS</code> | <code>FAILED</code>).
     */
    public final String deliverLogsStatus() {
        return deliverLogsStatus;
    }

    /**
     * <p>
     * The ID of the flow log.
     * </p>
     * 
     * @return The ID of the flow log.
     */
    public final String flowLogId() {
        return flowLogId;
    }

    /**
     * <p>
     * The status of the flow log (<code>ACTIVE</code>).
     * </p>
     * 
     * @return The status of the flow log (<code>ACTIVE</code>).
     */
    public final String flowLogStatus() {
        return flowLogStatus;
    }

    /**
     * <p>
     * The name of the flow log group.
     * </p>
     * 
     * @return The name of the flow log group.
     */
    public final String logGroupName() {
        return logGroupName;
    }

    /**
     * <p>
     * The ID of the resource being monitored.
     * </p>
     * 
     * @return The ID of the resource being monitored.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The type of traffic captured for the flow log.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trafficType} will
     * return {@link TrafficType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trafficTypeAsString}.
     * </p>
     * 
     * @return The type of traffic captured for the flow log.
     * @see TrafficType
     */
    public final TrafficType trafficType() {
        return TrafficType.fromValue(trafficType);
    }

    /**
     * <p>
     * The type of traffic captured for the flow log.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trafficType} will
     * return {@link TrafficType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #trafficTypeAsString}.
     * </p>
     * 
     * @return The type of traffic captured for the flow log.
     * @see TrafficType
     */
    public final String trafficTypeAsString() {
        return trafficType;
    }

    /**
     * <p>
     * The type of destination for the flow log data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #logDestinationType} will return {@link LogDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #logDestinationTypeAsString}.
     * </p>
     * 
     * @return The type of destination for the flow log data.
     * @see LogDestinationType
     */
    public final LogDestinationType logDestinationType() {
        return LogDestinationType.fromValue(logDestinationType);
    }

    /**
     * <p>
     * The type of destination for the flow log data.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #logDestinationType} will return {@link LogDestinationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #logDestinationTypeAsString}.
     * </p>
     * 
     * @return The type of destination for the flow log data.
     * @see LogDestinationType
     */
    public final String logDestinationTypeAsString() {
        return logDestinationType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the destination for the flow log data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the destination for the flow log data.
     */
    public final String logDestination() {
        return logDestination;
    }

    /**
     * <p>
     * The format of the flow log record.
     * </p>
     * 
     * @return The format of the flow log record.
     */
    public final String logFormat() {
        return logFormat;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags for the flow log.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags for the flow log.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The maximum interval of time, in seconds, during which a flow of packets is captured and aggregated into a flow
     * log record.
     * </p>
     * <p>
     * When a network interface is attached to a <a
     * href="https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html">Nitro-based instance</a>,
     * the aggregation interval is always 60 seconds (1 minute) or less, regardless of the specified value.
     * </p>
     * <p>
     * Valid Values: <code>60</code> | <code>600</code>
     * </p>
     * 
     * @return The maximum interval of time, in seconds, during which a flow of packets is captured and aggregated into
     *         a flow log record.</p>
     *         <p>
     *         When a network interface is attached to a <a
     *         href="https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html">Nitro-based
     *         instance</a>, the aggregation interval is always 60 seconds (1 minute) or less, regardless of the
     *         specified value.
     *         </p>
     *         <p>
     *         Valid Values: <code>60</code> | <code>600</code>
     */
    public final Integer maxAggregationInterval() {
        return maxAggregationInterval;
    }

    /**
     * <p>
     * The destination options.
     * </p>
     * 
     * @return The destination options.
     */
    public final DestinationOptionsResponse destinationOptions() {
        return destinationOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(deliverLogsErrorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(deliverLogsPermissionArn());
        hashCode = 31 * hashCode + Objects.hashCode(deliverCrossAccountRole());
        hashCode = 31 * hashCode + Objects.hashCode(deliverLogsStatus());
        hashCode = 31 * hashCode + Objects.hashCode(flowLogId());
        hashCode = 31 * hashCode + Objects.hashCode(flowLogStatus());
        hashCode = 31 * hashCode + Objects.hashCode(logGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(trafficTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logDestinationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logDestination());
        hashCode = 31 * hashCode + Objects.hashCode(logFormat());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxAggregationInterval());
        hashCode = 31 * hashCode + Objects.hashCode(destinationOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FlowLog)) {
            return false;
        }
        FlowLog other = (FlowLog) obj;
        return Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(deliverLogsErrorMessage(), other.deliverLogsErrorMessage())
                && Objects.equals(deliverLogsPermissionArn(), other.deliverLogsPermissionArn())
                && Objects.equals(deliverCrossAccountRole(), other.deliverCrossAccountRole())
                && Objects.equals(deliverLogsStatus(), other.deliverLogsStatus())
                && Objects.equals(flowLogId(), other.flowLogId()) && Objects.equals(flowLogStatus(), other.flowLogStatus())
                && Objects.equals(logGroupName(), other.logGroupName()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(trafficTypeAsString(), other.trafficTypeAsString())
                && Objects.equals(logDestinationTypeAsString(), other.logDestinationTypeAsString())
                && Objects.equals(logDestination(), other.logDestination()) && Objects.equals(logFormat(), other.logFormat())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(maxAggregationInterval(), other.maxAggregationInterval())
                && Objects.equals(destinationOptions(), other.destinationOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FlowLog").add("CreationTime", creationTime())
                .add("DeliverLogsErrorMessage", deliverLogsErrorMessage())
                .add("DeliverLogsPermissionArn", deliverLogsPermissionArn())
                .add("DeliverCrossAccountRole", deliverCrossAccountRole()).add("DeliverLogsStatus", deliverLogsStatus())
                .add("FlowLogId", flowLogId()).add("FlowLogStatus", flowLogStatus()).add("LogGroupName", logGroupName())
                .add("ResourceId", resourceId()).add("TrafficType", trafficTypeAsString())
                .add("LogDestinationType", logDestinationTypeAsString()).add("LogDestination", logDestination())
                .add("LogFormat", logFormat()).add("Tags", hasTags() ? tags() : null)
                .add("MaxAggregationInterval", maxAggregationInterval()).add("DestinationOptions", destinationOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "DeliverLogsErrorMessage":
            return Optional.ofNullable(clazz.cast(deliverLogsErrorMessage()));
        case "DeliverLogsPermissionArn":
            return Optional.ofNullable(clazz.cast(deliverLogsPermissionArn()));
        case "DeliverCrossAccountRole":
            return Optional.ofNullable(clazz.cast(deliverCrossAccountRole()));
        case "DeliverLogsStatus":
            return Optional.ofNullable(clazz.cast(deliverLogsStatus()));
        case "FlowLogId":
            return Optional.ofNullable(clazz.cast(flowLogId()));
        case "FlowLogStatus":
            return Optional.ofNullable(clazz.cast(flowLogStatus()));
        case "LogGroupName":
            return Optional.ofNullable(clazz.cast(logGroupName()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "TrafficType":
            return Optional.ofNullable(clazz.cast(trafficTypeAsString()));
        case "LogDestinationType":
            return Optional.ofNullable(clazz.cast(logDestinationTypeAsString()));
        case "LogDestination":
            return Optional.ofNullable(clazz.cast(logDestination()));
        case "LogFormat":
            return Optional.ofNullable(clazz.cast(logFormat()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "MaxAggregationInterval":
            return Optional.ofNullable(clazz.cast(maxAggregationInterval()));
        case "DestinationOptions":
            return Optional.ofNullable(clazz.cast(destinationOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("DeliverLogsErrorMessage", DELIVER_LOGS_ERROR_MESSAGE_FIELD);
        map.put("DeliverLogsPermissionArn", DELIVER_LOGS_PERMISSION_ARN_FIELD);
        map.put("DeliverCrossAccountRole", DELIVER_CROSS_ACCOUNT_ROLE_FIELD);
        map.put("DeliverLogsStatus", DELIVER_LOGS_STATUS_FIELD);
        map.put("FlowLogId", FLOW_LOG_ID_FIELD);
        map.put("FlowLogStatus", FLOW_LOG_STATUS_FIELD);
        map.put("LogGroupName", LOG_GROUP_NAME_FIELD);
        map.put("ResourceId", RESOURCE_ID_FIELD);
        map.put("TrafficType", TRAFFIC_TYPE_FIELD);
        map.put("LogDestinationType", LOG_DESTINATION_TYPE_FIELD);
        map.put("LogDestination", LOG_DESTINATION_FIELD);
        map.put("LogFormat", LOG_FORMAT_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("MaxAggregationInterval", MAX_AGGREGATION_INTERVAL_FIELD);
        map.put("DestinationOptions", DESTINATION_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FlowLog, T> g) {
        return obj -> g.apply((FlowLog) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FlowLog> {
        /**
         * <p>
         * The date and time the flow log was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time the flow log was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * Information about the error that occurred. <code>Rate limited</code> indicates that CloudWatch Logs
         * throttling has been applied for one or more network interfaces, or that you've reached the limit on the
         * number of log groups that you can create. <code>Access error</code> indicates that the IAM role associated
         * with the flow log does not have sufficient permissions to publish to CloudWatch Logs.
         * <code>Unknown error</code> indicates an internal error.
         * </p>
         * 
         * @param deliverLogsErrorMessage
         *        Information about the error that occurred. <code>Rate limited</code> indicates that CloudWatch Logs
         *        throttling has been applied for one or more network interfaces, or that you've reached the limit on
         *        the number of log groups that you can create. <code>Access error</code> indicates that the IAM role
         *        associated with the flow log does not have sufficient permissions to publish to CloudWatch Logs.
         *        <code>Unknown error</code> indicates an internal error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverLogsErrorMessage(String deliverLogsErrorMessage);

        /**
         * <p>
         * The ARN of the IAM role allows the service to publish logs to CloudWatch Logs.
         * </p>
         * 
         * @param deliverLogsPermissionArn
         *        The ARN of the IAM role allows the service to publish logs to CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverLogsPermissionArn(String deliverLogsPermissionArn);

        /**
         * <p>
         * The ARN of the IAM role that allows the service to publish flow logs across accounts.
         * </p>
         * 
         * @param deliverCrossAccountRole
         *        The ARN of the IAM role that allows the service to publish flow logs across accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverCrossAccountRole(String deliverCrossAccountRole);

        /**
         * <p>
         * The status of the logs delivery (<code>SUCCESS</code> | <code>FAILED</code>).
         * </p>
         * 
         * @param deliverLogsStatus
         *        The status of the logs delivery (<code>SUCCESS</code> | <code>FAILED</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deliverLogsStatus(String deliverLogsStatus);

        /**
         * <p>
         * The ID of the flow log.
         * </p>
         * 
         * @param flowLogId
         *        The ID of the flow log.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder flowLogId(String flowLogId);

        /**
         * <p>
         * The status of the flow log (<code>ACTIVE</code>).
         * </p>
         * 
         * @param flowLogStatus
         *        The status of the flow log (<code>ACTIVE</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder flowLogStatus(String flowLogStatus);

        /**
         * <p>
         * The name of the flow log group.
         * </p>
         * 
         * @param logGroupName
         *        The name of the flow log group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logGroupName(String logGroupName);

        /**
         * <p>
         * The ID of the resource being monitored.
         * </p>
         * 
         * @param resourceId
         *        The ID of the resource being monitored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The type of traffic captured for the flow log.
         * </p>
         * 
         * @param trafficType
         *        The type of traffic captured for the flow log.
         * @see TrafficType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrafficType
         */
        Builder trafficType(String trafficType);

        /**
         * <p>
         * The type of traffic captured for the flow log.
         * </p>
         * 
         * @param trafficType
         *        The type of traffic captured for the flow log.
         * @see TrafficType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrafficType
         */
        Builder trafficType(TrafficType trafficType);

        /**
         * <p>
         * The type of destination for the flow log data.
         * </p>
         * 
         * @param logDestinationType
         *        The type of destination for the flow log data.
         * @see LogDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDestinationType
         */
        Builder logDestinationType(String logDestinationType);

        /**
         * <p>
         * The type of destination for the flow log data.
         * </p>
         * 
         * @param logDestinationType
         *        The type of destination for the flow log data.
         * @see LogDestinationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDestinationType
         */
        Builder logDestinationType(LogDestinationType logDestinationType);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the destination for the flow log data.
         * </p>
         * 
         * @param logDestination
         *        The Amazon Resource Name (ARN) of the destination for the flow log data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logDestination(String logDestination);

        /**
         * <p>
         * The format of the flow log record.
         * </p>
         * 
         * @param logFormat
         *        The format of the flow log record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logFormat(String logFormat);

        /**
         * <p>
         * The tags for the flow log.
         * </p>
         * 
         * @param tags
         *        The tags for the flow log.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags for the flow log.
         * </p>
         * 
         * @param tags
         *        The tags for the flow log.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags for the flow log.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The maximum interval of time, in seconds, during which a flow of packets is captured and aggregated into a
         * flow log record.
         * </p>
         * <p>
         * When a network interface is attached to a <a
         * href="https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html">Nitro-based
         * instance</a>, the aggregation interval is always 60 seconds (1 minute) or less, regardless of the specified
         * value.
         * </p>
         * <p>
         * Valid Values: <code>60</code> | <code>600</code>
         * </p>
         * 
         * @param maxAggregationInterval
         *        The maximum interval of time, in seconds, during which a flow of packets is captured and aggregated
         *        into a flow log record.</p>
         *        <p>
         *        When a network interface is attached to a <a
         *        href="https://docs.aws.amazon.com/ec2/latest/instancetypes/ec2-nitro-instances.html">Nitro-based
         *        instance</a>, the aggregation interval is always 60 seconds (1 minute) or less, regardless of the
         *        specified value.
         *        </p>
         *        <p>
         *        Valid Values: <code>60</code> | <code>600</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxAggregationInterval(Integer maxAggregationInterval);

        /**
         * <p>
         * The destination options.
         * </p>
         * 
         * @param destinationOptions
         *        The destination options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationOptions(DestinationOptionsResponse destinationOptions);

        /**
         * <p>
         * The destination options.
         * </p>
         * This is a convenience method that creates an instance of the {@link DestinationOptionsResponse.Builder}
         * avoiding the need to create one manually via {@link DestinationOptionsResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DestinationOptionsResponse.Builder#build()} is called immediately
         * and its result is passed to {@link #destinationOptions(DestinationOptionsResponse)}.
         * 
         * @param destinationOptions
         *        a consumer that will call methods on {@link DestinationOptionsResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #destinationOptions(DestinationOptionsResponse)
         */
        default Builder destinationOptions(Consumer<DestinationOptionsResponse.Builder> destinationOptions) {
            return destinationOptions(DestinationOptionsResponse.builder().applyMutation(destinationOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant creationTime;

        private String deliverLogsErrorMessage;

        private String deliverLogsPermissionArn;

        private String deliverCrossAccountRole;

        private String deliverLogsStatus;

        private String flowLogId;

        private String flowLogStatus;

        private String logGroupName;

        private String resourceId;

        private String trafficType;

        private String logDestinationType;

        private String logDestination;

        private String logFormat;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Integer maxAggregationInterval;

        private DestinationOptionsResponse destinationOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(FlowLog model) {
            creationTime(model.creationTime);
            deliverLogsErrorMessage(model.deliverLogsErrorMessage);
            deliverLogsPermissionArn(model.deliverLogsPermissionArn);
            deliverCrossAccountRole(model.deliverCrossAccountRole);
            deliverLogsStatus(model.deliverLogsStatus);
            flowLogId(model.flowLogId);
            flowLogStatus(model.flowLogStatus);
            logGroupName(model.logGroupName);
            resourceId(model.resourceId);
            trafficType(model.trafficType);
            logDestinationType(model.logDestinationType);
            logDestination(model.logDestination);
            logFormat(model.logFormat);
            tags(model.tags);
            maxAggregationInterval(model.maxAggregationInterval);
            destinationOptions(model.destinationOptions);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getDeliverLogsErrorMessage() {
            return deliverLogsErrorMessage;
        }

        public final void setDeliverLogsErrorMessage(String deliverLogsErrorMessage) {
            this.deliverLogsErrorMessage = deliverLogsErrorMessage;
        }

        @Override
        public final Builder deliverLogsErrorMessage(String deliverLogsErrorMessage) {
            this.deliverLogsErrorMessage = deliverLogsErrorMessage;
            return this;
        }

        public final String getDeliverLogsPermissionArn() {
            return deliverLogsPermissionArn;
        }

        public final void setDeliverLogsPermissionArn(String deliverLogsPermissionArn) {
            this.deliverLogsPermissionArn = deliverLogsPermissionArn;
        }

        @Override
        public final Builder deliverLogsPermissionArn(String deliverLogsPermissionArn) {
            this.deliverLogsPermissionArn = deliverLogsPermissionArn;
            return this;
        }

        public final String getDeliverCrossAccountRole() {
            return deliverCrossAccountRole;
        }

        public final void setDeliverCrossAccountRole(String deliverCrossAccountRole) {
            this.deliverCrossAccountRole = deliverCrossAccountRole;
        }

        @Override
        public final Builder deliverCrossAccountRole(String deliverCrossAccountRole) {
            this.deliverCrossAccountRole = deliverCrossAccountRole;
            return this;
        }

        public final String getDeliverLogsStatus() {
            return deliverLogsStatus;
        }

        public final void setDeliverLogsStatus(String deliverLogsStatus) {
            this.deliverLogsStatus = deliverLogsStatus;
        }

        @Override
        public final Builder deliverLogsStatus(String deliverLogsStatus) {
            this.deliverLogsStatus = deliverLogsStatus;
            return this;
        }

        public final String getFlowLogId() {
            return flowLogId;
        }

        public final void setFlowLogId(String flowLogId) {
            this.flowLogId = flowLogId;
        }

        @Override
        public final Builder flowLogId(String flowLogId) {
            this.flowLogId = flowLogId;
            return this;
        }

        public final String getFlowLogStatus() {
            return flowLogStatus;
        }

        public final void setFlowLogStatus(String flowLogStatus) {
            this.flowLogStatus = flowLogStatus;
        }

        @Override
        public final Builder flowLogStatus(String flowLogStatus) {
            this.flowLogStatus = flowLogStatus;
            return this;
        }

        public final String getLogGroupName() {
            return logGroupName;
        }

        public final void setLogGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
        }

        @Override
        public final Builder logGroupName(String logGroupName) {
            this.logGroupName = logGroupName;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getTrafficType() {
            return trafficType;
        }

        public final void setTrafficType(String trafficType) {
            this.trafficType = trafficType;
        }

        @Override
        public final Builder trafficType(String trafficType) {
            this.trafficType = trafficType;
            return this;
        }

        @Override
        public final Builder trafficType(TrafficType trafficType) {
            this.trafficType(trafficType == null ? null : trafficType.toString());
            return this;
        }

        public final String getLogDestinationType() {
            return logDestinationType;
        }

        public final void setLogDestinationType(String logDestinationType) {
            this.logDestinationType = logDestinationType;
        }

        @Override
        public final Builder logDestinationType(String logDestinationType) {
            this.logDestinationType = logDestinationType;
            return this;
        }

        @Override
        public final Builder logDestinationType(LogDestinationType logDestinationType) {
            this.logDestinationType(logDestinationType == null ? null : logDestinationType.toString());
            return this;
        }

        public final String getLogDestination() {
            return logDestination;
        }

        public final void setLogDestination(String logDestination) {
            this.logDestination = logDestination;
        }

        @Override
        public final Builder logDestination(String logDestination) {
            this.logDestination = logDestination;
            return this;
        }

        public final String getLogFormat() {
            return logFormat;
        }

        public final void setLogFormat(String logFormat) {
            this.logFormat = logFormat;
        }

        @Override
        public final Builder logFormat(String logFormat) {
            this.logFormat = logFormat;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxAggregationInterval() {
            return maxAggregationInterval;
        }

        public final void setMaxAggregationInterval(Integer maxAggregationInterval) {
            this.maxAggregationInterval = maxAggregationInterval;
        }

        @Override
        public final Builder maxAggregationInterval(Integer maxAggregationInterval) {
            this.maxAggregationInterval = maxAggregationInterval;
            return this;
        }

        public final DestinationOptionsResponse.Builder getDestinationOptions() {
            return destinationOptions != null ? destinationOptions.toBuilder() : null;
        }

        public final void setDestinationOptions(DestinationOptionsResponse.BuilderImpl destinationOptions) {
            this.destinationOptions = destinationOptions != null ? destinationOptions.build() : null;
        }

        @Override
        public final Builder destinationOptions(DestinationOptionsResponse destinationOptions) {
            this.destinationOptions = destinationOptions;
            return this;
        }

        @Override
        public FlowLog build() {
            return new FlowLog(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
