/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the directory owner account details shared with the directory consumer account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OwnerDirectoryDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<OwnerDirectoryDescription.Builder, OwnerDirectoryDescription> {
    private static final SdkField<String> DIRECTORY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryId").getter(getter(OwnerDirectoryDescription::directoryId))
            .setter(setter(Builder::directoryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryId").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(OwnerDirectoryDescription::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<List<String>> DNS_IP_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpAddrs")
            .getter(getter(OwnerDirectoryDescription::dnsIpAddrs))
            .setter(setter(Builder::dnsIpAddrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpAddrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DNS_IPV6_ADDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsIpv6Addrs")
            .getter(getter(OwnerDirectoryDescription::dnsIpv6Addrs))
            .setter(setter(Builder::dnsIpv6Addrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsIpv6Addrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DirectoryVpcSettingsDescription> VPC_SETTINGS_FIELD = SdkField
            .<DirectoryVpcSettingsDescription> builder(MarshallingType.SDK_POJO).memberName("VpcSettings")
            .getter(getter(OwnerDirectoryDescription::vpcSettings)).setter(setter(Builder::vpcSettings))
            .constructor(DirectoryVpcSettingsDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSettings").build()).build();

    private static final SdkField<RadiusSettings> RADIUS_SETTINGS_FIELD = SdkField
            .<RadiusSettings> builder(MarshallingType.SDK_POJO).memberName("RadiusSettings")
            .getter(getter(OwnerDirectoryDescription::radiusSettings)).setter(setter(Builder::radiusSettings))
            .constructor(RadiusSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RadiusSettings").build()).build();

    private static final SdkField<String> RADIUS_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RadiusStatus").getter(getter(OwnerDirectoryDescription::radiusStatusAsString))
            .setter(setter(Builder::radiusStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RadiusStatus").build()).build();

    private static final SdkField<String> NETWORK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NetworkType").getter(getter(OwnerDirectoryDescription::networkTypeAsString))
            .setter(setter(Builder::networkType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ID_FIELD,
            ACCOUNT_ID_FIELD, DNS_IP_ADDRS_FIELD, DNS_IPV6_ADDRS_FIELD, VPC_SETTINGS_FIELD, RADIUS_SETTINGS_FIELD,
            RADIUS_STATUS_FIELD, NETWORK_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String directoryId;

    private final String accountId;

    private final List<String> dnsIpAddrs;

    private final List<String> dnsIpv6Addrs;

    private final DirectoryVpcSettingsDescription vpcSettings;

    private final RadiusSettings radiusSettings;

    private final String radiusStatus;

    private final String networkType;

    private OwnerDirectoryDescription(BuilderImpl builder) {
        this.directoryId = builder.directoryId;
        this.accountId = builder.accountId;
        this.dnsIpAddrs = builder.dnsIpAddrs;
        this.dnsIpv6Addrs = builder.dnsIpv6Addrs;
        this.vpcSettings = builder.vpcSettings;
        this.radiusSettings = builder.radiusSettings;
        this.radiusStatus = builder.radiusStatus;
        this.networkType = builder.networkType;
    }

    /**
     * <p>
     * Identifier of the Managed Microsoft AD directory in the directory owner account.
     * </p>
     * 
     * @return Identifier of the Managed Microsoft AD directory in the directory owner account.
     */
    public final String directoryId() {
        return directoryId;
    }

    /**
     * <p>
     * Identifier of the directory owner account.
     * </p>
     * 
     * @return Identifier of the directory owner account.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIpAddrs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsIpAddrs() {
        return dnsIpAddrs != null && !(dnsIpAddrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * IP address of the directory’s domain controllers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIpAddrs} method.
     * </p>
     * 
     * @return IP address of the directory’s domain controllers.
     */
    public final List<String> dnsIpAddrs() {
        return dnsIpAddrs;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsIpv6Addrs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDnsIpv6Addrs() {
        return dnsIpv6Addrs != null && !(dnsIpv6Addrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * IPv6 addresses of the directory’s domain controllers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsIpv6Addrs} method.
     * </p>
     * 
     * @return IPv6 addresses of the directory’s domain controllers.
     */
    public final List<String> dnsIpv6Addrs() {
        return dnsIpv6Addrs;
    }

    /**
     * <p>
     * Information about the VPC settings for the directory.
     * </p>
     * 
     * @return Information about the VPC settings for the directory.
     */
    public final DirectoryVpcSettingsDescription vpcSettings() {
        return vpcSettings;
    }

    /**
     * <p>
     * Information about the <a>RadiusSettings</a> object server configuration.
     * </p>
     * 
     * @return Information about the <a>RadiusSettings</a> object server configuration.
     */
    public final RadiusSettings radiusSettings() {
        return radiusSettings;
    }

    /**
     * <p>
     * The status of the RADIUS server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #radiusStatus} will
     * return {@link RadiusStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #radiusStatusAsString}.
     * </p>
     * 
     * @return The status of the RADIUS server.
     * @see RadiusStatus
     */
    public final RadiusStatus radiusStatus() {
        return RadiusStatus.fromValue(radiusStatus);
    }

    /**
     * <p>
     * The status of the RADIUS server.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #radiusStatus} will
     * return {@link RadiusStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #radiusStatusAsString}.
     * </p>
     * 
     * @return The status of the RADIUS server.
     * @see RadiusStatus
     */
    public final String radiusStatusAsString() {
        return radiusStatus;
    }

    /**
     * <p>
     * Network type of the directory in the directory owner account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkType} will
     * return {@link NetworkType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkTypeAsString}.
     * </p>
     * 
     * @return Network type of the directory in the directory owner account.
     * @see NetworkType
     */
    public final NetworkType networkType() {
        return NetworkType.fromValue(networkType);
    }

    /**
     * <p>
     * Network type of the directory in the directory owner account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkType} will
     * return {@link NetworkType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkTypeAsString}.
     * </p>
     * 
     * @return Network type of the directory in the directory owner account.
     * @see NetworkType
     */
    public final String networkTypeAsString() {
        return networkType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(directoryId());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpAddrs() ? dnsIpAddrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsIpv6Addrs() ? dnsIpv6Addrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcSettings());
        hashCode = 31 * hashCode + Objects.hashCode(radiusSettings());
        hashCode = 31 * hashCode + Objects.hashCode(radiusStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(networkTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OwnerDirectoryDescription)) {
            return false;
        }
        OwnerDirectoryDescription other = (OwnerDirectoryDescription) obj;
        return Objects.equals(directoryId(), other.directoryId()) && Objects.equals(accountId(), other.accountId())
                && hasDnsIpAddrs() == other.hasDnsIpAddrs() && Objects.equals(dnsIpAddrs(), other.dnsIpAddrs())
                && hasDnsIpv6Addrs() == other.hasDnsIpv6Addrs() && Objects.equals(dnsIpv6Addrs(), other.dnsIpv6Addrs())
                && Objects.equals(vpcSettings(), other.vpcSettings()) && Objects.equals(radiusSettings(), other.radiusSettings())
                && Objects.equals(radiusStatusAsString(), other.radiusStatusAsString())
                && Objects.equals(networkTypeAsString(), other.networkTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OwnerDirectoryDescription").add("DirectoryId", directoryId()).add("AccountId", accountId())
                .add("DnsIpAddrs", hasDnsIpAddrs() ? dnsIpAddrs() : null)
                .add("DnsIpv6Addrs", hasDnsIpv6Addrs() ? dnsIpv6Addrs() : null).add("VpcSettings", vpcSettings())
                .add("RadiusSettings", radiusSettings()).add("RadiusStatus", radiusStatusAsString())
                .add("NetworkType", networkTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryId":
            return Optional.ofNullable(clazz.cast(directoryId()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "DnsIpAddrs":
            return Optional.ofNullable(clazz.cast(dnsIpAddrs()));
        case "DnsIpv6Addrs":
            return Optional.ofNullable(clazz.cast(dnsIpv6Addrs()));
        case "VpcSettings":
            return Optional.ofNullable(clazz.cast(vpcSettings()));
        case "RadiusSettings":
            return Optional.ofNullable(clazz.cast(radiusSettings()));
        case "RadiusStatus":
            return Optional.ofNullable(clazz.cast(radiusStatusAsString()));
        case "NetworkType":
            return Optional.ofNullable(clazz.cast(networkTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DirectoryId", DIRECTORY_ID_FIELD);
        map.put("AccountId", ACCOUNT_ID_FIELD);
        map.put("DnsIpAddrs", DNS_IP_ADDRS_FIELD);
        map.put("DnsIpv6Addrs", DNS_IPV6_ADDRS_FIELD);
        map.put("VpcSettings", VPC_SETTINGS_FIELD);
        map.put("RadiusSettings", RADIUS_SETTINGS_FIELD);
        map.put("RadiusStatus", RADIUS_STATUS_FIELD);
        map.put("NetworkType", NETWORK_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OwnerDirectoryDescription, T> g) {
        return obj -> g.apply((OwnerDirectoryDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OwnerDirectoryDescription> {
        /**
         * <p>
         * Identifier of the Managed Microsoft AD directory in the directory owner account.
         * </p>
         * 
         * @param directoryId
         *        Identifier of the Managed Microsoft AD directory in the directory owner account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryId(String directoryId);

        /**
         * <p>
         * Identifier of the directory owner account.
         * </p>
         * 
         * @param accountId
         *        Identifier of the directory owner account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * IP address of the directory’s domain controllers.
         * </p>
         * 
         * @param dnsIpAddrs
         *        IP address of the directory’s domain controllers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(Collection<String> dnsIpAddrs);

        /**
         * <p>
         * IP address of the directory’s domain controllers.
         * </p>
         * 
         * @param dnsIpAddrs
         *        IP address of the directory’s domain controllers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpAddrs(String... dnsIpAddrs);

        /**
         * <p>
         * IPv6 addresses of the directory’s domain controllers.
         * </p>
         * 
         * @param dnsIpv6Addrs
         *        IPv6 addresses of the directory’s domain controllers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpv6Addrs(Collection<String> dnsIpv6Addrs);

        /**
         * <p>
         * IPv6 addresses of the directory’s domain controllers.
         * </p>
         * 
         * @param dnsIpv6Addrs
         *        IPv6 addresses of the directory’s domain controllers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsIpv6Addrs(String... dnsIpv6Addrs);

        /**
         * <p>
         * Information about the VPC settings for the directory.
         * </p>
         * 
         * @param vpcSettings
         *        Information about the VPC settings for the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSettings(DirectoryVpcSettingsDescription vpcSettings);

        /**
         * <p>
         * Information about the VPC settings for the directory.
         * </p>
         * This is a convenience method that creates an instance of the {@link DirectoryVpcSettingsDescription.Builder}
         * avoiding the need to create one manually via {@link DirectoryVpcSettingsDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DirectoryVpcSettingsDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #vpcSettings(DirectoryVpcSettingsDescription)}.
         * 
         * @param vpcSettings
         *        a consumer that will call methods on {@link DirectoryVpcSettingsDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcSettings(DirectoryVpcSettingsDescription)
         */
        default Builder vpcSettings(Consumer<DirectoryVpcSettingsDescription.Builder> vpcSettings) {
            return vpcSettings(DirectoryVpcSettingsDescription.builder().applyMutation(vpcSettings).build());
        }

        /**
         * <p>
         * Information about the <a>RadiusSettings</a> object server configuration.
         * </p>
         * 
         * @param radiusSettings
         *        Information about the <a>RadiusSettings</a> object server configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder radiusSettings(RadiusSettings radiusSettings);

        /**
         * <p>
         * Information about the <a>RadiusSettings</a> object server configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link RadiusSettings.Builder} avoiding the need
         * to create one manually via {@link RadiusSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RadiusSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #radiusSettings(RadiusSettings)}.
         * 
         * @param radiusSettings
         *        a consumer that will call methods on {@link RadiusSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #radiusSettings(RadiusSettings)
         */
        default Builder radiusSettings(Consumer<RadiusSettings.Builder> radiusSettings) {
            return radiusSettings(RadiusSettings.builder().applyMutation(radiusSettings).build());
        }

        /**
         * <p>
         * The status of the RADIUS server.
         * </p>
         * 
         * @param radiusStatus
         *        The status of the RADIUS server.
         * @see RadiusStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RadiusStatus
         */
        Builder radiusStatus(String radiusStatus);

        /**
         * <p>
         * The status of the RADIUS server.
         * </p>
         * 
         * @param radiusStatus
         *        The status of the RADIUS server.
         * @see RadiusStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RadiusStatus
         */
        Builder radiusStatus(RadiusStatus radiusStatus);

        /**
         * <p>
         * Network type of the directory in the directory owner account.
         * </p>
         * 
         * @param networkType
         *        Network type of the directory in the directory owner account.
         * @see NetworkType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkType
         */
        Builder networkType(String networkType);

        /**
         * <p>
         * Network type of the directory in the directory owner account.
         * </p>
         * 
         * @param networkType
         *        Network type of the directory in the directory owner account.
         * @see NetworkType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkType
         */
        Builder networkType(NetworkType networkType);
    }

    static final class BuilderImpl implements Builder {
        private String directoryId;

        private String accountId;

        private List<String> dnsIpAddrs = DefaultSdkAutoConstructList.getInstance();

        private List<String> dnsIpv6Addrs = DefaultSdkAutoConstructList.getInstance();

        private DirectoryVpcSettingsDescription vpcSettings;

        private RadiusSettings radiusSettings;

        private String radiusStatus;

        private String networkType;

        private BuilderImpl() {
        }

        private BuilderImpl(OwnerDirectoryDescription model) {
            directoryId(model.directoryId);
            accountId(model.accountId);
            dnsIpAddrs(model.dnsIpAddrs);
            dnsIpv6Addrs(model.dnsIpv6Addrs);
            vpcSettings(model.vpcSettings);
            radiusSettings(model.radiusSettings);
            radiusStatus(model.radiusStatus);
            networkType(model.networkType);
        }

        public final String getDirectoryId() {
            return directoryId;
        }

        public final void setDirectoryId(String directoryId) {
            this.directoryId = directoryId;
        }

        @Override
        public final Builder directoryId(String directoryId) {
            this.directoryId = directoryId;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Collection<String> getDnsIpAddrs() {
            if (dnsIpAddrs instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpAddrs;
        }

        public final void setDnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
        }

        @Override
        public final Builder dnsIpAddrs(Collection<String> dnsIpAddrs) {
            this.dnsIpAddrs = DnsIpAddrsCopier.copy(dnsIpAddrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpAddrs(String... dnsIpAddrs) {
            dnsIpAddrs(Arrays.asList(dnsIpAddrs));
            return this;
        }

        public final Collection<String> getDnsIpv6Addrs() {
            if (dnsIpv6Addrs instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsIpv6Addrs;
        }

        public final void setDnsIpv6Addrs(Collection<String> dnsIpv6Addrs) {
            this.dnsIpv6Addrs = DnsIpv6AddrsCopier.copy(dnsIpv6Addrs);
        }

        @Override
        public final Builder dnsIpv6Addrs(Collection<String> dnsIpv6Addrs) {
            this.dnsIpv6Addrs = DnsIpv6AddrsCopier.copy(dnsIpv6Addrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsIpv6Addrs(String... dnsIpv6Addrs) {
            dnsIpv6Addrs(Arrays.asList(dnsIpv6Addrs));
            return this;
        }

        public final DirectoryVpcSettingsDescription.Builder getVpcSettings() {
            return vpcSettings != null ? vpcSettings.toBuilder() : null;
        }

        public final void setVpcSettings(DirectoryVpcSettingsDescription.BuilderImpl vpcSettings) {
            this.vpcSettings = vpcSettings != null ? vpcSettings.build() : null;
        }

        @Override
        public final Builder vpcSettings(DirectoryVpcSettingsDescription vpcSettings) {
            this.vpcSettings = vpcSettings;
            return this;
        }

        public final RadiusSettings.Builder getRadiusSettings() {
            return radiusSettings != null ? radiusSettings.toBuilder() : null;
        }

        public final void setRadiusSettings(RadiusSettings.BuilderImpl radiusSettings) {
            this.radiusSettings = radiusSettings != null ? radiusSettings.build() : null;
        }

        @Override
        public final Builder radiusSettings(RadiusSettings radiusSettings) {
            this.radiusSettings = radiusSettings;
            return this;
        }

        public final String getRadiusStatus() {
            return radiusStatus;
        }

        public final void setRadiusStatus(String radiusStatus) {
            this.radiusStatus = radiusStatus;
        }

        @Override
        public final Builder radiusStatus(String radiusStatus) {
            this.radiusStatus = radiusStatus;
            return this;
        }

        @Override
        public final Builder radiusStatus(RadiusStatus radiusStatus) {
            this.radiusStatus(radiusStatus == null ? null : radiusStatus.toString());
            return this;
        }

        public final String getNetworkType() {
            return networkType;
        }

        public final void setNetworkType(String networkType) {
            this.networkType = networkType;
        }

        @Override
        public final Builder networkType(String networkType) {
            this.networkType = networkType;
            return this;
        }

        @Override
        public final Builder networkType(NetworkType networkType) {
            this.networkType(networkType == null ? null : networkType.toString());
            return this;
        }

        @Override
        public OwnerDirectoryDescription build() {
            return new OwnerDirectoryDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
