/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies how your DataSync agent connects to Amazon Web Services using a <a href=
 * "https://docs.aws.amazon.com/datasync/latest/userguide/choose-service-endpoint.html#choose-service-endpoint-vpc"
 * >virtual private cloud (VPC) service endpoint</a>. An agent that uses a VPC endpoint isn't accessible over the public
 * internet.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PrivateLinkConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<PrivateLinkConfig.Builder, PrivateLinkConfig> {
    private static final SdkField<String> VPC_ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointId").getter(getter(PrivateLinkConfig::vpcEndpointId)).setter(setter(Builder::vpcEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointId").build()).build();

    private static final SdkField<String> PRIVATE_LINK_ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PrivateLinkEndpoint").getter(getter(PrivateLinkConfig::privateLinkEndpoint))
            .setter(setter(Builder::privateLinkEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateLinkEndpoint").build())
            .build();

    private static final SdkField<List<String>> SUBNET_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetArns")
            .getter(getter(PrivateLinkConfig::subnetArns))
            .setter(setter(Builder::subnetArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupArns")
            .getter(getter(PrivateLinkConfig::securityGroupArns))
            .setter(setter(Builder::securityGroupArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VPC_ENDPOINT_ID_FIELD,
            PRIVATE_LINK_ENDPOINT_FIELD, SUBNET_ARNS_FIELD, SECURITY_GROUP_ARNS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String vpcEndpointId;

    private final String privateLinkEndpoint;

    private final List<String> subnetArns;

    private final List<String> securityGroupArns;

    private PrivateLinkConfig(BuilderImpl builder) {
        this.vpcEndpointId = builder.vpcEndpointId;
        this.privateLinkEndpoint = builder.privateLinkEndpoint;
        this.subnetArns = builder.subnetArns;
        this.securityGroupArns = builder.securityGroupArns;
    }

    /**
     * <p>
     * Specifies the ID of the VPC endpoint that your agent connects to.
     * </p>
     * 
     * @return Specifies the ID of the VPC endpoint that your agent connects to.
     */
    public final String vpcEndpointId() {
        return vpcEndpointId;
    }

    /**
     * <p>
     * Specifies the VPC endpoint provided by <a
     * href="https://docs.aws.amazon.com/vpc/latest/privatelink/privatelink-share-your-services.html">Amazon Web
     * Services PrivateLink</a> that your agent connects to.
     * </p>
     * 
     * @return Specifies the VPC endpoint provided by <a
     *         href="https://docs.aws.amazon.com/vpc/latest/privatelink/privatelink-share-your-services.html">Amazon Web
     *         Services PrivateLink</a> that your agent connects to.
     */
    public final String privateLinkEndpoint() {
        return privateLinkEndpoint;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubnetArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubnetArns() {
        return subnetArns != null && !(subnetArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the ARN of the subnet where your VPC endpoint is located. You can only specify one ARN.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubnetArns} method.
     * </p>
     * 
     * @return Specifies the ARN of the subnet where your VPC endpoint is located. You can only specify one ARN.
     */
    public final List<String> subnetArns() {
        return subnetArns;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupArns() {
        return securityGroupArns != null && !(securityGroupArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the Amazon Resource Names (ARN) of the security group that provides DataSync access to your VPC
     * endpoint. You can only specify one ARN.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupArns} method.
     * </p>
     * 
     * @return Specifies the Amazon Resource Names (ARN) of the security group that provides DataSync access to your VPC
     *         endpoint. You can only specify one ARN.
     */
    public final List<String> securityGroupArns() {
        return securityGroupArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(privateLinkEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetArns() ? subnetArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupArns() ? securityGroupArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PrivateLinkConfig)) {
            return false;
        }
        PrivateLinkConfig other = (PrivateLinkConfig) obj;
        return Objects.equals(vpcEndpointId(), other.vpcEndpointId())
                && Objects.equals(privateLinkEndpoint(), other.privateLinkEndpoint()) && hasSubnetArns() == other.hasSubnetArns()
                && Objects.equals(subnetArns(), other.subnetArns()) && hasSecurityGroupArns() == other.hasSecurityGroupArns()
                && Objects.equals(securityGroupArns(), other.securityGroupArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PrivateLinkConfig").add("VpcEndpointId", vpcEndpointId())
                .add("PrivateLinkEndpoint", privateLinkEndpoint()).add("SubnetArns", hasSubnetArns() ? subnetArns() : null)
                .add("SecurityGroupArns", hasSecurityGroupArns() ? securityGroupArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VpcEndpointId":
            return Optional.ofNullable(clazz.cast(vpcEndpointId()));
        case "PrivateLinkEndpoint":
            return Optional.ofNullable(clazz.cast(privateLinkEndpoint()));
        case "SubnetArns":
            return Optional.ofNullable(clazz.cast(subnetArns()));
        case "SecurityGroupArns":
            return Optional.ofNullable(clazz.cast(securityGroupArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("VpcEndpointId", VPC_ENDPOINT_ID_FIELD);
        map.put("PrivateLinkEndpoint", PRIVATE_LINK_ENDPOINT_FIELD);
        map.put("SubnetArns", SUBNET_ARNS_FIELD);
        map.put("SecurityGroupArns", SECURITY_GROUP_ARNS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PrivateLinkConfig, T> g) {
        return obj -> g.apply((PrivateLinkConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PrivateLinkConfig> {
        /**
         * <p>
         * Specifies the ID of the VPC endpoint that your agent connects to.
         * </p>
         * 
         * @param vpcEndpointId
         *        Specifies the ID of the VPC endpoint that your agent connects to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpointId(String vpcEndpointId);

        /**
         * <p>
         * Specifies the VPC endpoint provided by <a
         * href="https://docs.aws.amazon.com/vpc/latest/privatelink/privatelink-share-your-services.html">Amazon Web
         * Services PrivateLink</a> that your agent connects to.
         * </p>
         * 
         * @param privateLinkEndpoint
         *        Specifies the VPC endpoint provided by <a
         *        href="https://docs.aws.amazon.com/vpc/latest/privatelink/privatelink-share-your-services.html">Amazon
         *        Web Services PrivateLink</a> that your agent connects to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateLinkEndpoint(String privateLinkEndpoint);

        /**
         * <p>
         * Specifies the ARN of the subnet where your VPC endpoint is located. You can only specify one ARN.
         * </p>
         * 
         * @param subnetArns
         *        Specifies the ARN of the subnet where your VPC endpoint is located. You can only specify one ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetArns(Collection<String> subnetArns);

        /**
         * <p>
         * Specifies the ARN of the subnet where your VPC endpoint is located. You can only specify one ARN.
         * </p>
         * 
         * @param subnetArns
         *        Specifies the ARN of the subnet where your VPC endpoint is located. You can only specify one ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetArns(String... subnetArns);

        /**
         * <p>
         * Specifies the Amazon Resource Names (ARN) of the security group that provides DataSync access to your VPC
         * endpoint. You can only specify one ARN.
         * </p>
         * 
         * @param securityGroupArns
         *        Specifies the Amazon Resource Names (ARN) of the security group that provides DataSync access to your
         *        VPC endpoint. You can only specify one ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(Collection<String> securityGroupArns);

        /**
         * <p>
         * Specifies the Amazon Resource Names (ARN) of the security group that provides DataSync access to your VPC
         * endpoint. You can only specify one ARN.
         * </p>
         * 
         * @param securityGroupArns
         *        Specifies the Amazon Resource Names (ARN) of the security group that provides DataSync access to your
         *        VPC endpoint. You can only specify one ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(String... securityGroupArns);
    }

    static final class BuilderImpl implements Builder {
        private String vpcEndpointId;

        private String privateLinkEndpoint;

        private List<String> subnetArns = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PrivateLinkConfig model) {
            vpcEndpointId(model.vpcEndpointId);
            privateLinkEndpoint(model.privateLinkEndpoint);
            subnetArns(model.subnetArns);
            securityGroupArns(model.securityGroupArns);
        }

        public final String getVpcEndpointId() {
            return vpcEndpointId;
        }

        public final void setVpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
        }

        @Override
        public final Builder vpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
            return this;
        }

        public final String getPrivateLinkEndpoint() {
            return privateLinkEndpoint;
        }

        public final void setPrivateLinkEndpoint(String privateLinkEndpoint) {
            this.privateLinkEndpoint = privateLinkEndpoint;
        }

        @Override
        public final Builder privateLinkEndpoint(String privateLinkEndpoint) {
            this.privateLinkEndpoint = privateLinkEndpoint;
            return this;
        }

        public final Collection<String> getSubnetArns() {
            if (subnetArns instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetArns;
        }

        public final void setSubnetArns(Collection<String> subnetArns) {
            this.subnetArns = PLSubnetArnListCopier.copy(subnetArns);
        }

        @Override
        public final Builder subnetArns(Collection<String> subnetArns) {
            this.subnetArns = PLSubnetArnListCopier.copy(subnetArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetArns(String... subnetArns) {
            subnetArns(Arrays.asList(subnetArns));
            return this;
        }

        public final Collection<String> getSecurityGroupArns() {
            if (securityGroupArns instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupArns;
        }

        public final void setSecurityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = PLSecurityGroupArnListCopier.copy(securityGroupArns);
        }

        @Override
        public final Builder securityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = PLSecurityGroupArnListCopier.copy(securityGroupArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupArns(String... securityGroupArns) {
            securityGroupArns(Arrays.asList(securityGroupArns));
            return this;
        }

        @Override
        public PrivateLinkConfig build() {
            return new PrivateLinkConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
