/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information that defines a MySQL endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MySQLSettings implements SdkPojo, Serializable, ToCopyableBuilder<MySQLSettings.Builder, MySQLSettings> {
    private static final SdkField<String> AFTER_CONNECT_SCRIPT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AfterConnectScript").getter(getter(MySQLSettings::afterConnectScript))
            .setter(setter(Builder::afterConnectScript))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AfterConnectScript").build())
            .build();

    private static final SdkField<Boolean> CLEAN_SOURCE_METADATA_ON_MISMATCH_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CleanSourceMetadataOnMismatch")
            .getter(getter(MySQLSettings::cleanSourceMetadataOnMismatch))
            .setter(setter(Builder::cleanSourceMetadataOnMismatch))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CleanSourceMetadataOnMismatch")
                    .build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(MySQLSettings::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<Integer> EVENTS_POLL_INTERVAL_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("EventsPollInterval").getter(getter(MySQLSettings::eventsPollInterval))
            .setter(setter(Builder::eventsPollInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventsPollInterval").build())
            .build();

    private static final SdkField<String> TARGET_DB_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetDbType").getter(getter(MySQLSettings::targetDbTypeAsString)).setter(setter(Builder::targetDbType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetDbType").build()).build();

    private static final SdkField<Integer> MAX_FILE_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxFileSize").getter(getter(MySQLSettings::maxFileSize)).setter(setter(Builder::maxFileSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxFileSize").build()).build();

    private static final SdkField<Integer> PARALLEL_LOAD_THREADS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ParallelLoadThreads").getter(getter(MySQLSettings::parallelLoadThreads))
            .setter(setter(Builder::parallelLoadThreads))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParallelLoadThreads").build())
            .build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(MySQLSettings::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(MySQLSettings::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> SERVER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerName").getter(getter(MySQLSettings::serverName)).setter(setter(Builder::serverName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerName").build()).build();

    private static final SdkField<String> SERVER_TIMEZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerTimezone").getter(getter(MySQLSettings::serverTimezone)).setter(setter(Builder::serverTimezone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerTimezone").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(MySQLSettings::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<String> SECRETS_MANAGER_ACCESS_ROLE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SecretsManagerAccessRoleArn")
            .getter(getter(MySQLSettings::secretsManagerAccessRoleArn))
            .setter(setter(Builder::secretsManagerAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretsManagerAccessRoleArn")
                    .build()).build();

    private static final SdkField<String> SECRETS_MANAGER_SECRET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretsManagerSecretId").getter(getter(MySQLSettings::secretsManagerSecretId))
            .setter(setter(Builder::secretsManagerSecretId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretsManagerSecretId").build())
            .build();

    private static final SdkField<Integer> EXECUTE_TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExecuteTimeout").getter(getter(MySQLSettings::executeTimeout)).setter(setter(Builder::executeTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecuteTimeout").build()).build();

    private static final SdkField<String> SERVICE_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceAccessRoleArn").getter(getter(MySQLSettings::serviceAccessRoleArn))
            .setter(setter(Builder::serviceAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceAccessRoleArn").build())
            .build();

    private static final SdkField<String> AUTHENTICATION_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthenticationMethod").getter(getter(MySQLSettings::authenticationMethodAsString))
            .setter(setter(Builder::authenticationMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationMethod").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AFTER_CONNECT_SCRIPT_FIELD,
            CLEAN_SOURCE_METADATA_ON_MISMATCH_FIELD, DATABASE_NAME_FIELD, EVENTS_POLL_INTERVAL_FIELD, TARGET_DB_TYPE_FIELD,
            MAX_FILE_SIZE_FIELD, PARALLEL_LOAD_THREADS_FIELD, PASSWORD_FIELD, PORT_FIELD, SERVER_NAME_FIELD,
            SERVER_TIMEZONE_FIELD, USERNAME_FIELD, SECRETS_MANAGER_ACCESS_ROLE_ARN_FIELD, SECRETS_MANAGER_SECRET_ID_FIELD,
            EXECUTE_TIMEOUT_FIELD, SERVICE_ACCESS_ROLE_ARN_FIELD, AUTHENTICATION_METHOD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String afterConnectScript;

    private final Boolean cleanSourceMetadataOnMismatch;

    private final String databaseName;

    private final Integer eventsPollInterval;

    private final String targetDbType;

    private final Integer maxFileSize;

    private final Integer parallelLoadThreads;

    private final String password;

    private final Integer port;

    private final String serverName;

    private final String serverTimezone;

    private final String username;

    private final String secretsManagerAccessRoleArn;

    private final String secretsManagerSecretId;

    private final Integer executeTimeout;

    private final String serviceAccessRoleArn;

    private final String authenticationMethod;

    private MySQLSettings(BuilderImpl builder) {
        this.afterConnectScript = builder.afterConnectScript;
        this.cleanSourceMetadataOnMismatch = builder.cleanSourceMetadataOnMismatch;
        this.databaseName = builder.databaseName;
        this.eventsPollInterval = builder.eventsPollInterval;
        this.targetDbType = builder.targetDbType;
        this.maxFileSize = builder.maxFileSize;
        this.parallelLoadThreads = builder.parallelLoadThreads;
        this.password = builder.password;
        this.port = builder.port;
        this.serverName = builder.serverName;
        this.serverTimezone = builder.serverTimezone;
        this.username = builder.username;
        this.secretsManagerAccessRoleArn = builder.secretsManagerAccessRoleArn;
        this.secretsManagerSecretId = builder.secretsManagerSecretId;
        this.executeTimeout = builder.executeTimeout;
        this.serviceAccessRoleArn = builder.serviceAccessRoleArn;
        this.authenticationMethod = builder.authenticationMethod;
    }

    /**
     * <p>
     * Specifies a script to run immediately after DMS connects to the endpoint. The migration task continues running
     * regardless if the SQL statement succeeds or fails.
     * </p>
     * <p>
     * For this parameter, provide the code of the script itself, not the name of a file containing the script.
     * </p>
     * 
     * @return Specifies a script to run immediately after DMS connects to the endpoint. The migration task continues
     *         running regardless if the SQL statement succeeds or fails.</p>
     *         <p>
     *         For this parameter, provide the code of the script itself, not the name of a file containing the script.
     */
    public final String afterConnectScript() {
        return afterConnectScript;
    }

    /**
     * <p>
     * Cleans and recreates table metadata information on the replication instance when a mismatch occurs. For example,
     * in a situation where running an alter DDL on the table could result in different information about the table
     * cached in the replication instance.
     * </p>
     * 
     * @return Cleans and recreates table metadata information on the replication instance when a mismatch occurs. For
     *         example, in a situation where running an alter DDL on the table could result in different information
     *         about the table cached in the replication instance.
     */
    public final Boolean cleanSourceMetadataOnMismatch() {
        return cleanSourceMetadataOnMismatch;
    }

    /**
     * <p>
     * Database name for the endpoint. For a MySQL source or target endpoint, don't explicitly specify the database
     * using the <code>DatabaseName</code> request parameter on either the <code>CreateEndpoint</code> or
     * <code>ModifyEndpoint</code> API call. Specifying <code>DatabaseName</code> when you create or modify a MySQL
     * endpoint replicates all the task tables to this single database. For MySQL endpoints, you specify the database
     * only when you specify the schema in the table-mapping rules of the DMS task.
     * </p>
     * 
     * @return Database name for the endpoint. For a MySQL source or target endpoint, don't explicitly specify the
     *         database using the <code>DatabaseName</code> request parameter on either the <code>CreateEndpoint</code>
     *         or <code>ModifyEndpoint</code> API call. Specifying <code>DatabaseName</code> when you create or modify a
     *         MySQL endpoint replicates all the task tables to this single database. For MySQL endpoints, you specify
     *         the database only when you specify the schema in the table-mapping rules of the DMS task.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * Specifies how often to check the binary log for new changes/events when the database is idle. The default is five
     * seconds.
     * </p>
     * <p>
     * Example: <code>eventsPollInterval=5;</code>
     * </p>
     * <p>
     * In the example, DMS checks for changes in the binary logs every five seconds.
     * </p>
     * 
     * @return Specifies how often to check the binary log for new changes/events when the database is idle. The default
     *         is five seconds.</p>
     *         <p>
     *         Example: <code>eventsPollInterval=5;</code>
     *         </p>
     *         <p>
     *         In the example, DMS checks for changes in the binary logs every five seconds.
     */
    public final Integer eventsPollInterval() {
        return eventsPollInterval;
    }

    /**
     * <p>
     * Specifies where to migrate source tables on the target, either to a single database or multiple databases. If you
     * specify <code>SPECIFIC_DATABASE</code>, specify the database name using the <code>DatabaseName</code> parameter
     * of the <code>Endpoint</code> object.
     * </p>
     * <p>
     * Example: <code>targetDbType=MULTIPLE_DATABASES</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetDbType} will
     * return {@link TargetDbType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetDbTypeAsString}.
     * </p>
     * 
     * @return Specifies where to migrate source tables on the target, either to a single database or multiple
     *         databases. If you specify <code>SPECIFIC_DATABASE</code>, specify the database name using the
     *         <code>DatabaseName</code> parameter of the <code>Endpoint</code> object.</p>
     *         <p>
     *         Example: <code>targetDbType=MULTIPLE_DATABASES</code>
     * @see TargetDbType
     */
    public final TargetDbType targetDbType() {
        return TargetDbType.fromValue(targetDbType);
    }

    /**
     * <p>
     * Specifies where to migrate source tables on the target, either to a single database or multiple databases. If you
     * specify <code>SPECIFIC_DATABASE</code>, specify the database name using the <code>DatabaseName</code> parameter
     * of the <code>Endpoint</code> object.
     * </p>
     * <p>
     * Example: <code>targetDbType=MULTIPLE_DATABASES</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetDbType} will
     * return {@link TargetDbType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetDbTypeAsString}.
     * </p>
     * 
     * @return Specifies where to migrate source tables on the target, either to a single database or multiple
     *         databases. If you specify <code>SPECIFIC_DATABASE</code>, specify the database name using the
     *         <code>DatabaseName</code> parameter of the <code>Endpoint</code> object.</p>
     *         <p>
     *         Example: <code>targetDbType=MULTIPLE_DATABASES</code>
     * @see TargetDbType
     */
    public final String targetDbTypeAsString() {
        return targetDbType;
    }

    /**
     * <p>
     * Specifies the maximum size (in KB) of any .csv file used to transfer data to a MySQL-compatible database.
     * </p>
     * <p>
     * Example: <code>maxFileSize=512</code>
     * </p>
     * 
     * @return Specifies the maximum size (in KB) of any .csv file used to transfer data to a MySQL-compatible
     *         database.</p>
     *         <p>
     *         Example: <code>maxFileSize=512</code>
     */
    public final Integer maxFileSize() {
        return maxFileSize;
    }

    /**
     * <p>
     * Improves performance when loading data into the MySQL-compatible target database. Specifies how many threads to
     * use to load the data into the MySQL-compatible target database. Setting a large number of threads can have an
     * adverse effect on database performance, because a separate connection is required for each thread. The default is
     * one.
     * </p>
     * <p>
     * Example: <code>parallelLoadThreads=1</code>
     * </p>
     * 
     * @return Improves performance when loading data into the MySQL-compatible target database. Specifies how many
     *         threads to use to load the data into the MySQL-compatible target database. Setting a large number of
     *         threads can have an adverse effect on database performance, because a separate connection is required for
     *         each thread. The default is one.</p>
     *         <p>
     *         Example: <code>parallelLoadThreads=1</code>
     */
    public final Integer parallelLoadThreads() {
        return parallelLoadThreads;
    }

    /**
     * <p>
     * Endpoint connection password.
     * </p>
     * 
     * @return Endpoint connection password.
     */
    public final String password() {
        return password;
    }

    /**
     * <p>
     * Endpoint TCP port.
     * </p>
     * 
     * @return Endpoint TCP port.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The host name of the endpoint database.
     * </p>
     * <p>
     * For an Amazon RDS MySQL instance, this is the output of <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
     * >DescribeDBInstances</a>, in the
     * <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
     * field.
     * </p>
     * <p>
     * For an Aurora MySQL instance, this is the output of <a
     * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
     * >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
     * </p>
     * 
     * @return The host name of the endpoint database. </p>
     *         <p>
     *         For an Amazon RDS MySQL instance, this is the output of <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
     *         >DescribeDBInstances</a>, in the
     *         <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
     *         field.
     *         </p>
     *         <p>
     *         For an Aurora MySQL instance, this is the output of <a
     *         href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
     *         >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
     */
    public final String serverName() {
        return serverName;
    }

    /**
     * <p>
     * Specifies the time zone for the source MySQL database.
     * </p>
     * <p>
     * Example: <code>serverTimezone=US/Pacific;</code>
     * </p>
     * <p>
     * Note: Do not enclose time zones in single quotes.
     * </p>
     * 
     * @return Specifies the time zone for the source MySQL database.</p>
     *         <p>
     *         Example: <code>serverTimezone=US/Pacific;</code>
     *         </p>
     *         <p>
     *         Note: Do not enclose time zones in single quotes.
     */
    public final String serverTimezone() {
        return serverTimezone;
    }

    /**
     * <p>
     * Endpoint connection user name.
     * </p>
     * 
     * @return Endpoint connection user name.
     */
    public final String username() {
        return username;
    }

    /**
     * <p>
     * The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and grants the
     * required permissions to access the value in <code>SecretsManagerSecret</code>. The role must allow the
     * <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of the Amazon Web Services
     * Secrets Manager secret that allows access to the MySQL endpoint.
     * </p>
     * <note>
     * <p>
     * You can specify one of two sets of values for these permissions. You can specify the values for this setting and
     * <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for <code>UserName</code>,
     * <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You can't specify both. For more
     * information on creating this <code>SecretsManagerSecret</code> and the <code>SecretsManagerAccessRoleArn</code>
     * and <code>SecretsManagerSecretId</code> required to access it, see <a
     * href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager">Using
     * secrets to access Database Migration Service resources</a> in the <i>Database Migration Service User Guide</i>.
     * </p>
     * </note>
     * 
     * @return The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and grants
     *         the required permissions to access the value in <code>SecretsManagerSecret</code>. The role must allow
     *         the <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of the Amazon Web
     *         Services Secrets Manager secret that allows access to the MySQL endpoint.</p> <note>
     *         <p>
     *         You can specify one of two sets of values for these permissions. You can specify the values for this
     *         setting and <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for
     *         <code>UserName</code>, <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You can't
     *         specify both. For more information on creating this <code>SecretsManagerSecret</code> and the
     *         <code>SecretsManagerAccessRoleArn</code> and <code>SecretsManagerSecretId</code> required to access it,
     *         see <a
     *         href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager"
     *         >Using secrets to access Database Migration Service resources</a> in the <i>Database Migration Service
     *         User Guide</i>.
     *         </p>
     */
    public final String secretsManagerAccessRoleArn() {
        return secretsManagerAccessRoleArn;
    }

    /**
     * <p>
     * The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the MySQL
     * endpoint connection details.
     * </p>
     * 
     * @return The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the
     *         MySQL endpoint connection details.
     */
    public final String secretsManagerSecretId() {
        return secretsManagerSecretId;
    }

    /**
     * <p>
     * Sets the client statement timeout (in seconds) for a MySQL source endpoint.
     * </p>
     * 
     * @return Sets the client statement timeout (in seconds) for a MySQL source endpoint.
     */
    public final Integer executeTimeout() {
        return executeTimeout;
    }

    /**
     * <p>
     * The IAM role you can use to authenticate when connecting to your endpoint. Ensure to include
     * <code>iam:PassRole</code> and <code>rds-db:connect</code> actions in permission policy.
     * </p>
     * 
     * @return The IAM role you can use to authenticate when connecting to your endpoint. Ensure to include
     *         <code>iam:PassRole</code> and <code>rds-db:connect</code> actions in permission policy.
     */
    public final String serviceAccessRoleArn() {
        return serviceAccessRoleArn;
    }

    /**
     * <p>
     * This attribute allows you to specify the authentication method as "iam auth".
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationMethod} will return {@link MySQLAuthenticationMethod#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationMethodAsString}.
     * </p>
     * 
     * @return This attribute allows you to specify the authentication method as "iam auth".
     * @see MySQLAuthenticationMethod
     */
    public final MySQLAuthenticationMethod authenticationMethod() {
        return MySQLAuthenticationMethod.fromValue(authenticationMethod);
    }

    /**
     * <p>
     * This attribute allows you to specify the authentication method as "iam auth".
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationMethod} will return {@link MySQLAuthenticationMethod#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationMethodAsString}.
     * </p>
     * 
     * @return This attribute allows you to specify the authentication method as "iam auth".
     * @see MySQLAuthenticationMethod
     */
    public final String authenticationMethodAsString() {
        return authenticationMethod;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(afterConnectScript());
        hashCode = 31 * hashCode + Objects.hashCode(cleanSourceMetadataOnMismatch());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(eventsPollInterval());
        hashCode = 31 * hashCode + Objects.hashCode(targetDbTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxFileSize());
        hashCode = 31 * hashCode + Objects.hashCode(parallelLoadThreads());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(serverName());
        hashCode = 31 * hashCode + Objects.hashCode(serverTimezone());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(secretsManagerAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(secretsManagerSecretId());
        hashCode = 31 * hashCode + Objects.hashCode(executeTimeout());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationMethodAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MySQLSettings)) {
            return false;
        }
        MySQLSettings other = (MySQLSettings) obj;
        return Objects.equals(afterConnectScript(), other.afterConnectScript())
                && Objects.equals(cleanSourceMetadataOnMismatch(), other.cleanSourceMetadataOnMismatch())
                && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(eventsPollInterval(), other.eventsPollInterval())
                && Objects.equals(targetDbTypeAsString(), other.targetDbTypeAsString())
                && Objects.equals(maxFileSize(), other.maxFileSize())
                && Objects.equals(parallelLoadThreads(), other.parallelLoadThreads())
                && Objects.equals(password(), other.password()) && Objects.equals(port(), other.port())
                && Objects.equals(serverName(), other.serverName()) && Objects.equals(serverTimezone(), other.serverTimezone())
                && Objects.equals(username(), other.username())
                && Objects.equals(secretsManagerAccessRoleArn(), other.secretsManagerAccessRoleArn())
                && Objects.equals(secretsManagerSecretId(), other.secretsManagerSecretId())
                && Objects.equals(executeTimeout(), other.executeTimeout())
                && Objects.equals(serviceAccessRoleArn(), other.serviceAccessRoleArn())
                && Objects.equals(authenticationMethodAsString(), other.authenticationMethodAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MySQLSettings").add("AfterConnectScript", afterConnectScript())
                .add("CleanSourceMetadataOnMismatch", cleanSourceMetadataOnMismatch()).add("DatabaseName", databaseName())
                .add("EventsPollInterval", eventsPollInterval()).add("TargetDbType", targetDbTypeAsString())
                .add("MaxFileSize", maxFileSize()).add("ParallelLoadThreads", parallelLoadThreads())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***").add("Port", port())
                .add("ServerName", serverName()).add("ServerTimezone", serverTimezone()).add("Username", username())
                .add("SecretsManagerAccessRoleArn", secretsManagerAccessRoleArn())
                .add("SecretsManagerSecretId", secretsManagerSecretId()).add("ExecuteTimeout", executeTimeout())
                .add("ServiceAccessRoleArn", serviceAccessRoleArn()).add("AuthenticationMethod", authenticationMethodAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AfterConnectScript":
            return Optional.ofNullable(clazz.cast(afterConnectScript()));
        case "CleanSourceMetadataOnMismatch":
            return Optional.ofNullable(clazz.cast(cleanSourceMetadataOnMismatch()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "EventsPollInterval":
            return Optional.ofNullable(clazz.cast(eventsPollInterval()));
        case "TargetDbType":
            return Optional.ofNullable(clazz.cast(targetDbTypeAsString()));
        case "MaxFileSize":
            return Optional.ofNullable(clazz.cast(maxFileSize()));
        case "ParallelLoadThreads":
            return Optional.ofNullable(clazz.cast(parallelLoadThreads()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "ServerName":
            return Optional.ofNullable(clazz.cast(serverName()));
        case "ServerTimezone":
            return Optional.ofNullable(clazz.cast(serverTimezone()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "SecretsManagerAccessRoleArn":
            return Optional.ofNullable(clazz.cast(secretsManagerAccessRoleArn()));
        case "SecretsManagerSecretId":
            return Optional.ofNullable(clazz.cast(secretsManagerSecretId()));
        case "ExecuteTimeout":
            return Optional.ofNullable(clazz.cast(executeTimeout()));
        case "ServiceAccessRoleArn":
            return Optional.ofNullable(clazz.cast(serviceAccessRoleArn()));
        case "AuthenticationMethod":
            return Optional.ofNullable(clazz.cast(authenticationMethodAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AfterConnectScript", AFTER_CONNECT_SCRIPT_FIELD);
        map.put("CleanSourceMetadataOnMismatch", CLEAN_SOURCE_METADATA_ON_MISMATCH_FIELD);
        map.put("DatabaseName", DATABASE_NAME_FIELD);
        map.put("EventsPollInterval", EVENTS_POLL_INTERVAL_FIELD);
        map.put("TargetDbType", TARGET_DB_TYPE_FIELD);
        map.put("MaxFileSize", MAX_FILE_SIZE_FIELD);
        map.put("ParallelLoadThreads", PARALLEL_LOAD_THREADS_FIELD);
        map.put("Password", PASSWORD_FIELD);
        map.put("Port", PORT_FIELD);
        map.put("ServerName", SERVER_NAME_FIELD);
        map.put("ServerTimezone", SERVER_TIMEZONE_FIELD);
        map.put("Username", USERNAME_FIELD);
        map.put("SecretsManagerAccessRoleArn", SECRETS_MANAGER_ACCESS_ROLE_ARN_FIELD);
        map.put("SecretsManagerSecretId", SECRETS_MANAGER_SECRET_ID_FIELD);
        map.put("ExecuteTimeout", EXECUTE_TIMEOUT_FIELD);
        map.put("ServiceAccessRoleArn", SERVICE_ACCESS_ROLE_ARN_FIELD);
        map.put("AuthenticationMethod", AUTHENTICATION_METHOD_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MySQLSettings, T> g) {
        return obj -> g.apply((MySQLSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MySQLSettings> {
        /**
         * <p>
         * Specifies a script to run immediately after DMS connects to the endpoint. The migration task continues
         * running regardless if the SQL statement succeeds or fails.
         * </p>
         * <p>
         * For this parameter, provide the code of the script itself, not the name of a file containing the script.
         * </p>
         * 
         * @param afterConnectScript
         *        Specifies a script to run immediately after DMS connects to the endpoint. The migration task continues
         *        running regardless if the SQL statement succeeds or fails.</p>
         *        <p>
         *        For this parameter, provide the code of the script itself, not the name of a file containing the
         *        script.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder afterConnectScript(String afterConnectScript);

        /**
         * <p>
         * Cleans and recreates table metadata information on the replication instance when a mismatch occurs. For
         * example, in a situation where running an alter DDL on the table could result in different information about
         * the table cached in the replication instance.
         * </p>
         * 
         * @param cleanSourceMetadataOnMismatch
         *        Cleans and recreates table metadata information on the replication instance when a mismatch occurs.
         *        For example, in a situation where running an alter DDL on the table could result in different
         *        information about the table cached in the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cleanSourceMetadataOnMismatch(Boolean cleanSourceMetadataOnMismatch);

        /**
         * <p>
         * Database name for the endpoint. For a MySQL source or target endpoint, don't explicitly specify the database
         * using the <code>DatabaseName</code> request parameter on either the <code>CreateEndpoint</code> or
         * <code>ModifyEndpoint</code> API call. Specifying <code>DatabaseName</code> when you create or modify a MySQL
         * endpoint replicates all the task tables to this single database. For MySQL endpoints, you specify the
         * database only when you specify the schema in the table-mapping rules of the DMS task.
         * </p>
         * 
         * @param databaseName
         *        Database name for the endpoint. For a MySQL source or target endpoint, don't explicitly specify the
         *        database using the <code>DatabaseName</code> request parameter on either the
         *        <code>CreateEndpoint</code> or <code>ModifyEndpoint</code> API call. Specifying
         *        <code>DatabaseName</code> when you create or modify a MySQL endpoint replicates all the task tables to
         *        this single database. For MySQL endpoints, you specify the database only when you specify the schema
         *        in the table-mapping rules of the DMS task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * Specifies how often to check the binary log for new changes/events when the database is idle. The default is
         * five seconds.
         * </p>
         * <p>
         * Example: <code>eventsPollInterval=5;</code>
         * </p>
         * <p>
         * In the example, DMS checks for changes in the binary logs every five seconds.
         * </p>
         * 
         * @param eventsPollInterval
         *        Specifies how often to check the binary log for new changes/events when the database is idle. The
         *        default is five seconds.</p>
         *        <p>
         *        Example: <code>eventsPollInterval=5;</code>
         *        </p>
         *        <p>
         *        In the example, DMS checks for changes in the binary logs every five seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventsPollInterval(Integer eventsPollInterval);

        /**
         * <p>
         * Specifies where to migrate source tables on the target, either to a single database or multiple databases. If
         * you specify <code>SPECIFIC_DATABASE</code>, specify the database name using the <code>DatabaseName</code>
         * parameter of the <code>Endpoint</code> object.
         * </p>
         * <p>
         * Example: <code>targetDbType=MULTIPLE_DATABASES</code>
         * </p>
         * 
         * @param targetDbType
         *        Specifies where to migrate source tables on the target, either to a single database or multiple
         *        databases. If you specify <code>SPECIFIC_DATABASE</code>, specify the database name using the
         *        <code>DatabaseName</code> parameter of the <code>Endpoint</code> object.</p>
         *        <p>
         *        Example: <code>targetDbType=MULTIPLE_DATABASES</code>
         * @see TargetDbType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetDbType
         */
        Builder targetDbType(String targetDbType);

        /**
         * <p>
         * Specifies where to migrate source tables on the target, either to a single database or multiple databases. If
         * you specify <code>SPECIFIC_DATABASE</code>, specify the database name using the <code>DatabaseName</code>
         * parameter of the <code>Endpoint</code> object.
         * </p>
         * <p>
         * Example: <code>targetDbType=MULTIPLE_DATABASES</code>
         * </p>
         * 
         * @param targetDbType
         *        Specifies where to migrate source tables on the target, either to a single database or multiple
         *        databases. If you specify <code>SPECIFIC_DATABASE</code>, specify the database name using the
         *        <code>DatabaseName</code> parameter of the <code>Endpoint</code> object.</p>
         *        <p>
         *        Example: <code>targetDbType=MULTIPLE_DATABASES</code>
         * @see TargetDbType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetDbType
         */
        Builder targetDbType(TargetDbType targetDbType);

        /**
         * <p>
         * Specifies the maximum size (in KB) of any .csv file used to transfer data to a MySQL-compatible database.
         * </p>
         * <p>
         * Example: <code>maxFileSize=512</code>
         * </p>
         * 
         * @param maxFileSize
         *        Specifies the maximum size (in KB) of any .csv file used to transfer data to a MySQL-compatible
         *        database.</p>
         *        <p>
         *        Example: <code>maxFileSize=512</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxFileSize(Integer maxFileSize);

        /**
         * <p>
         * Improves performance when loading data into the MySQL-compatible target database. Specifies how many threads
         * to use to load the data into the MySQL-compatible target database. Setting a large number of threads can have
         * an adverse effect on database performance, because a separate connection is required for each thread. The
         * default is one.
         * </p>
         * <p>
         * Example: <code>parallelLoadThreads=1</code>
         * </p>
         * 
         * @param parallelLoadThreads
         *        Improves performance when loading data into the MySQL-compatible target database. Specifies how many
         *        threads to use to load the data into the MySQL-compatible target database. Setting a large number of
         *        threads can have an adverse effect on database performance, because a separate connection is required
         *        for each thread. The default is one.</p>
         *        <p>
         *        Example: <code>parallelLoadThreads=1</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parallelLoadThreads(Integer parallelLoadThreads);

        /**
         * <p>
         * Endpoint connection password.
         * </p>
         * 
         * @param password
         *        Endpoint connection password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * Endpoint TCP port.
         * </p>
         * 
         * @param port
         *        Endpoint TCP port.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The host name of the endpoint database.
         * </p>
         * <p>
         * For an Amazon RDS MySQL instance, this is the output of <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
         * >DescribeDBInstances</a>, in the
         * <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
         * field.
         * </p>
         * <p>
         * For an Aurora MySQL instance, this is the output of <a
         * href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
         * >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
         * </p>
         * 
         * @param serverName
         *        The host name of the endpoint database. </p>
         *        <p>
         *        For an Amazon RDS MySQL instance, this is the output of <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBInstances.html"
         *        >DescribeDBInstances</a>, in the
         *        <code> <a href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_Endpoint.html">Endpoint</a>.Address</code>
         *        field.
         *        </p>
         *        <p>
         *        For an Aurora MySQL instance, this is the output of <a
         *        href="https://docs.aws.amazon.com/AmazonRDS/latest/APIReference/API_DescribeDBClusters.html"
         *        >DescribeDBClusters</a>, in the <code>Endpoint</code> field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverName(String serverName);

        /**
         * <p>
         * Specifies the time zone for the source MySQL database.
         * </p>
         * <p>
         * Example: <code>serverTimezone=US/Pacific;</code>
         * </p>
         * <p>
         * Note: Do not enclose time zones in single quotes.
         * </p>
         * 
         * @param serverTimezone
         *        Specifies the time zone for the source MySQL database.</p>
         *        <p>
         *        Example: <code>serverTimezone=US/Pacific;</code>
         *        </p>
         *        <p>
         *        Note: Do not enclose time zones in single quotes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverTimezone(String serverTimezone);

        /**
         * <p>
         * Endpoint connection user name.
         * </p>
         * 
         * @param username
         *        Endpoint connection user name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and grants the
         * required permissions to access the value in <code>SecretsManagerSecret</code>. The role must allow the
         * <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of the Amazon Web Services
         * Secrets Manager secret that allows access to the MySQL endpoint.
         * </p>
         * <note>
         * <p>
         * You can specify one of two sets of values for these permissions. You can specify the values for this setting
         * and <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for <code>UserName</code>,
         * <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You can't specify both. For more
         * information on creating this <code>SecretsManagerSecret</code> and the
         * <code>SecretsManagerAccessRoleArn</code> and <code>SecretsManagerSecretId</code> required to access it, see
         * <a href="https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager">
         * Using secrets to access Database Migration Service resources</a> in the <i>Database Migration Service User
         * Guide</i>.
         * </p>
         * </note>
         * 
         * @param secretsManagerAccessRoleArn
         *        The full Amazon Resource Name (ARN) of the IAM role that specifies DMS as the trusted entity and
         *        grants the required permissions to access the value in <code>SecretsManagerSecret</code>. The role
         *        must allow the <code>iam:PassRole</code> action. <code>SecretsManagerSecret</code> has the value of
         *        the Amazon Web Services Secrets Manager secret that allows access to the MySQL endpoint.</p> <note>
         *        <p>
         *        You can specify one of two sets of values for these permissions. You can specify the values for this
         *        setting and <code>SecretsManagerSecretId</code>. Or you can specify clear-text values for
         *        <code>UserName</code>, <code>Password</code>, <code>ServerName</code>, and <code>Port</code>. You
         *        can't specify both. For more information on creating this <code>SecretsManagerSecret</code> and the
         *        <code>SecretsManagerAccessRoleArn</code> and <code>SecretsManagerSecretId</code> required to access
         *        it, see <a href=
         *        "https://docs.aws.amazon.com/dms/latest/userguide/CHAP_Security.html#security-iam-secretsmanager"
         *        >Using secrets to access Database Migration Service resources</a> in the <i>Database Migration Service
         *        User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretsManagerAccessRoleArn(String secretsManagerAccessRoleArn);

        /**
         * <p>
         * The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the MySQL
         * endpoint connection details.
         * </p>
         * 
         * @param secretsManagerSecretId
         *        The full ARN, partial ARN, or friendly name of the <code>SecretsManagerSecret</code> that contains the
         *        MySQL endpoint connection details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretsManagerSecretId(String secretsManagerSecretId);

        /**
         * <p>
         * Sets the client statement timeout (in seconds) for a MySQL source endpoint.
         * </p>
         * 
         * @param executeTimeout
         *        Sets the client statement timeout (in seconds) for a MySQL source endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executeTimeout(Integer executeTimeout);

        /**
         * <p>
         * The IAM role you can use to authenticate when connecting to your endpoint. Ensure to include
         * <code>iam:PassRole</code> and <code>rds-db:connect</code> actions in permission policy.
         * </p>
         * 
         * @param serviceAccessRoleArn
         *        The IAM role you can use to authenticate when connecting to your endpoint. Ensure to include
         *        <code>iam:PassRole</code> and <code>rds-db:connect</code> actions in permission policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessRoleArn(String serviceAccessRoleArn);

        /**
         * <p>
         * This attribute allows you to specify the authentication method as "iam auth".
         * </p>
         * 
         * @param authenticationMethod
         *        This attribute allows you to specify the authentication method as "iam auth".
         * @see MySQLAuthenticationMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MySQLAuthenticationMethod
         */
        Builder authenticationMethod(String authenticationMethod);

        /**
         * <p>
         * This attribute allows you to specify the authentication method as "iam auth".
         * </p>
         * 
         * @param authenticationMethod
         *        This attribute allows you to specify the authentication method as "iam auth".
         * @see MySQLAuthenticationMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MySQLAuthenticationMethod
         */
        Builder authenticationMethod(MySQLAuthenticationMethod authenticationMethod);
    }

    static final class BuilderImpl implements Builder {
        private String afterConnectScript;

        private Boolean cleanSourceMetadataOnMismatch;

        private String databaseName;

        private Integer eventsPollInterval;

        private String targetDbType;

        private Integer maxFileSize;

        private Integer parallelLoadThreads;

        private String password;

        private Integer port;

        private String serverName;

        private String serverTimezone;

        private String username;

        private String secretsManagerAccessRoleArn;

        private String secretsManagerSecretId;

        private Integer executeTimeout;

        private String serviceAccessRoleArn;

        private String authenticationMethod;

        private BuilderImpl() {
        }

        private BuilderImpl(MySQLSettings model) {
            afterConnectScript(model.afterConnectScript);
            cleanSourceMetadataOnMismatch(model.cleanSourceMetadataOnMismatch);
            databaseName(model.databaseName);
            eventsPollInterval(model.eventsPollInterval);
            targetDbType(model.targetDbType);
            maxFileSize(model.maxFileSize);
            parallelLoadThreads(model.parallelLoadThreads);
            password(model.password);
            port(model.port);
            serverName(model.serverName);
            serverTimezone(model.serverTimezone);
            username(model.username);
            secretsManagerAccessRoleArn(model.secretsManagerAccessRoleArn);
            secretsManagerSecretId(model.secretsManagerSecretId);
            executeTimeout(model.executeTimeout);
            serviceAccessRoleArn(model.serviceAccessRoleArn);
            authenticationMethod(model.authenticationMethod);
        }

        public final String getAfterConnectScript() {
            return afterConnectScript;
        }

        public final void setAfterConnectScript(String afterConnectScript) {
            this.afterConnectScript = afterConnectScript;
        }

        @Override
        public final Builder afterConnectScript(String afterConnectScript) {
            this.afterConnectScript = afterConnectScript;
            return this;
        }

        public final Boolean getCleanSourceMetadataOnMismatch() {
            return cleanSourceMetadataOnMismatch;
        }

        public final void setCleanSourceMetadataOnMismatch(Boolean cleanSourceMetadataOnMismatch) {
            this.cleanSourceMetadataOnMismatch = cleanSourceMetadataOnMismatch;
        }

        @Override
        public final Builder cleanSourceMetadataOnMismatch(Boolean cleanSourceMetadataOnMismatch) {
            this.cleanSourceMetadataOnMismatch = cleanSourceMetadataOnMismatch;
            return this;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final Integer getEventsPollInterval() {
            return eventsPollInterval;
        }

        public final void setEventsPollInterval(Integer eventsPollInterval) {
            this.eventsPollInterval = eventsPollInterval;
        }

        @Override
        public final Builder eventsPollInterval(Integer eventsPollInterval) {
            this.eventsPollInterval = eventsPollInterval;
            return this;
        }

        public final String getTargetDbType() {
            return targetDbType;
        }

        public final void setTargetDbType(String targetDbType) {
            this.targetDbType = targetDbType;
        }

        @Override
        public final Builder targetDbType(String targetDbType) {
            this.targetDbType = targetDbType;
            return this;
        }

        @Override
        public final Builder targetDbType(TargetDbType targetDbType) {
            this.targetDbType(targetDbType == null ? null : targetDbType.toString());
            return this;
        }

        public final Integer getMaxFileSize() {
            return maxFileSize;
        }

        public final void setMaxFileSize(Integer maxFileSize) {
            this.maxFileSize = maxFileSize;
        }

        @Override
        public final Builder maxFileSize(Integer maxFileSize) {
            this.maxFileSize = maxFileSize;
            return this;
        }

        public final Integer getParallelLoadThreads() {
            return parallelLoadThreads;
        }

        public final void setParallelLoadThreads(Integer parallelLoadThreads) {
            this.parallelLoadThreads = parallelLoadThreads;
        }

        @Override
        public final Builder parallelLoadThreads(Integer parallelLoadThreads) {
            this.parallelLoadThreads = parallelLoadThreads;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getServerName() {
            return serverName;
        }

        public final void setServerName(String serverName) {
            this.serverName = serverName;
        }

        @Override
        public final Builder serverName(String serverName) {
            this.serverName = serverName;
            return this;
        }

        public final String getServerTimezone() {
            return serverTimezone;
        }

        public final void setServerTimezone(String serverTimezone) {
            this.serverTimezone = serverTimezone;
        }

        @Override
        public final Builder serverTimezone(String serverTimezone) {
            this.serverTimezone = serverTimezone;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final String getSecretsManagerAccessRoleArn() {
            return secretsManagerAccessRoleArn;
        }

        public final void setSecretsManagerAccessRoleArn(String secretsManagerAccessRoleArn) {
            this.secretsManagerAccessRoleArn = secretsManagerAccessRoleArn;
        }

        @Override
        public final Builder secretsManagerAccessRoleArn(String secretsManagerAccessRoleArn) {
            this.secretsManagerAccessRoleArn = secretsManagerAccessRoleArn;
            return this;
        }

        public final String getSecretsManagerSecretId() {
            return secretsManagerSecretId;
        }

        public final void setSecretsManagerSecretId(String secretsManagerSecretId) {
            this.secretsManagerSecretId = secretsManagerSecretId;
        }

        @Override
        public final Builder secretsManagerSecretId(String secretsManagerSecretId) {
            this.secretsManagerSecretId = secretsManagerSecretId;
            return this;
        }

        public final Integer getExecuteTimeout() {
            return executeTimeout;
        }

        public final void setExecuteTimeout(Integer executeTimeout) {
            this.executeTimeout = executeTimeout;
        }

        @Override
        public final Builder executeTimeout(Integer executeTimeout) {
            this.executeTimeout = executeTimeout;
            return this;
        }

        public final String getServiceAccessRoleArn() {
            return serviceAccessRoleArn;
        }

        public final void setServiceAccessRoleArn(String serviceAccessRoleArn) {
            this.serviceAccessRoleArn = serviceAccessRoleArn;
        }

        @Override
        public final Builder serviceAccessRoleArn(String serviceAccessRoleArn) {
            this.serviceAccessRoleArn = serviceAccessRoleArn;
            return this;
        }

        public final String getAuthenticationMethod() {
            return authenticationMethod;
        }

        public final void setAuthenticationMethod(String authenticationMethod) {
            this.authenticationMethod = authenticationMethod;
        }

        @Override
        public final Builder authenticationMethod(String authenticationMethod) {
            this.authenticationMethod = authenticationMethod;
            return this;
        }

        @Override
        public final Builder authenticationMethod(MySQLAuthenticationMethod authenticationMethod) {
            this.authenticationMethod(authenticationMethod == null ? null : authenticationMethod.toString());
            return this;
        }

        @Override
        public MySQLSettings build() {
            return new MySQLSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
