/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentity;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.cognitoidentity.internal.CognitoIdentityServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.cognitoidentity.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.cognitoidentity.model.CognitoIdentityException;
import software.amazon.awssdk.services.cognitoidentity.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cognitoidentity.model.CreateIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.CreateIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeveloperUserAlreadyRegisteredException;
import software.amazon.awssdk.services.cognitoidentity.model.ExternalServiceException;
import software.amazon.awssdk.services.cognitoidentity.model.GetCredentialsForIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetCredentialsForIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdentityPoolRolesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdentityPoolRolesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenForDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenForDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetPrincipalTagAttributeMapRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetPrincipalTagAttributeMapResponse;
import software.amazon.awssdk.services.cognitoidentity.model.InternalErrorException;
import software.amazon.awssdk.services.cognitoidentity.model.InvalidIdentityPoolConfigurationException;
import software.amazon.awssdk.services.cognitoidentity.model.InvalidParameterException;
import software.amazon.awssdk.services.cognitoidentity.model.LimitExceededException;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentityPoolsRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentityPoolsResponse;
import software.amazon.awssdk.services.cognitoidentity.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cognitoidentity.model.LookupDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.LookupDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.MergeDeveloperIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.MergeDeveloperIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.NotAuthorizedException;
import software.amazon.awssdk.services.cognitoidentity.model.ResourceConflictException;
import software.amazon.awssdk.services.cognitoidentity.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cognitoidentity.model.SetIdentityPoolRolesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.SetIdentityPoolRolesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.SetPrincipalTagAttributeMapRequest;
import software.amazon.awssdk.services.cognitoidentity.model.SetPrincipalTagAttributeMapResponse;
import software.amazon.awssdk.services.cognitoidentity.model.TagResourceRequest;
import software.amazon.awssdk.services.cognitoidentity.model.TagResourceResponse;
import software.amazon.awssdk.services.cognitoidentity.model.TooManyRequestsException;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UntagResourceRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UntagResourceResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UpdateIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UpdateIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.transform.CreateIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetCredentialsForIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdentityPoolRolesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenForDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetPrincipalTagAttributeMapRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentityPoolsRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.LookupDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.MergeDeveloperIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.SetIdentityPoolRolesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.SetPrincipalTagAttributeMapRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UpdateIdentityPoolRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CognitoIdentityClient}.
 *
 * @see CognitoIdentityClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCognitoIdentityClient implements CognitoIdentityClient {
    private static final Logger log = Logger.loggerFor(DefaultCognitoIdentityClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCognitoIdentityClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Cognito_Identity" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new identity pool. The identity pool is a store of user identity information that is specific to your
     * Amazon Web Services account. The keys for <code>SupportedLoginProviders</code> are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Facebook: <code>graph.facebook.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Google: <code>accounts.google.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Sign in With Apple: <code>appleid.apple.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon: <code>www.amazon.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Twitter: <code>api.twitter.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Digits: <code>www.digits.com</code>
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * If you don't provide a value for a parameter, Amazon Cognito sets it to its default value.
     * </p>
     * </important>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param createIdentityPoolRequest
     *        Input to the CreateIdentityPool action.
     * @return Result of the CreateIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws LimitExceededException
     *         Thrown when the total number of user pools has exceeded a preset limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.CreateIdentityPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/CreateIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateIdentityPoolResponse createIdentityPool(CreateIdentityPoolRequest createIdentityPoolRequest)
            throws InvalidParameterException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, LimitExceededException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateIdentityPoolResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIdentityPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIdentityPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIdentityPool");

            return clientHandler.execute(new ClientExecutionParams<CreateIdentityPoolRequest, CreateIdentityPoolResponse>()
                    .withOperationName("CreateIdentityPool").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createIdentityPoolRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateIdentityPoolRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param deleteIdentitiesRequest
     *        Input to the <code>DeleteIdentities</code> action.
     * @return Result of the DeleteIdentities operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DeleteIdentities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DeleteIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteIdentitiesResponse deleteIdentities(DeleteIdentitiesRequest deleteIdentitiesRequest)
            throws InvalidParameterException, TooManyRequestsException, InternalErrorException, AwsServiceException,
            SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteIdentitiesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIdentitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIdentitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIdentities");

            return clientHandler.execute(new ClientExecutionParams<DeleteIdentitiesRequest, DeleteIdentitiesResponse>()
                    .withOperationName("DeleteIdentities").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteIdentitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIdentitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an identity pool. Once a pool is deleted, users will not be able to authenticate with the pool.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param deleteIdentityPoolRequest
     *        Input to the DeleteIdentityPool action.
     * @return Result of the DeleteIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DeleteIdentityPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DeleteIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteIdentityPoolResponse deleteIdentityPool(DeleteIdentityPoolRequest deleteIdentityPoolRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteIdentityPoolResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIdentityPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIdentityPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIdentityPool");

            return clientHandler.execute(new ClientExecutionParams<DeleteIdentityPoolRequest, DeleteIdentityPoolResponse>()
                    .withOperationName("DeleteIdentityPool").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteIdentityPoolRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIdentityPoolRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata related to the given identity, including when the identity was created and any associated linked
     * logins.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param describeIdentityRequest
     *        Input to the <code>DescribeIdentity</code> action.
     * @return Result of the DescribeIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DescribeIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DescribeIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeIdentityResponse describeIdentity(DescribeIdentityRequest describeIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeIdentityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeIdentityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentity");

            return clientHandler.execute(new ClientExecutionParams<DescribeIdentityRequest, DescribeIdentityResponse>()
                    .withOperationName("DescribeIdentity").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeIdentityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeIdentityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details about a particular identity pool, including the pool name, ID description, creation date, and
     * current number of users.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param describeIdentityPoolRequest
     *        Input to the DescribeIdentityPool action.
     * @return Result of the DescribeIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.DescribeIdentityPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DescribeIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeIdentityPoolResponse describeIdentityPool(DescribeIdentityPoolRequest describeIdentityPoolRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeIdentityPoolResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeIdentityPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIdentityPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentityPool");

            return clientHandler.execute(new ClientExecutionParams<DescribeIdentityPoolRequest, DescribeIdentityPoolResponse>()
                    .withOperationName("DescribeIdentityPool").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeIdentityPoolRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeIdentityPoolRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns credentials for the provided identity ID. Any provided logins will be validated against supported login
     * providers. If the token is for <code>cognito-identity.amazonaws.com</code>, it will be passed through to Security
     * Token Service with the appropriate role for the token.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getCredentialsForIdentityRequest
     *        Input to the <code>GetCredentialsForIdentity</code> action.
     * @return Result of the GetCredentialsForIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InvalidIdentityPoolConfigurationException
     *         If you provided authentication information in the request, the identity pool has no authenticated role
     *         configured, or STS returned an error response to the request to assume the authenticated role from the
     *         identity pool. If you provided no authentication information in the request, the identity pool has no
     *         unauthenticated role configured, or STS returned an error response to the request to assume the
     *         unauthenticated role from the identity pool.</p>
     *         <p>
     *         Your role trust policy must grant <code>AssumeRoleWithWebIdentity</code> permissions to
     *         <code>cognito-identity.amazonaws.com</code>.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetCredentialsForIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetCredentialsForIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCredentialsForIdentityResponse getCredentialsForIdentity(
            GetCredentialsForIdentityRequest getCredentialsForIdentityRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InvalidIdentityPoolConfigurationException, InternalErrorException, ExternalServiceException, AwsServiceException,
            SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCredentialsForIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCredentialsForIdentityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCredentialsForIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCredentialsForIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCredentialsForIdentity");

            return clientHandler
                    .execute(new ClientExecutionParams<GetCredentialsForIdentityRequest, GetCredentialsForIdentityResponse>()
                            .withOperationName("GetCredentialsForIdentity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getCredentialsForIdentityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetCredentialsForIdentityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates (or retrieves) IdentityID. Supplying multiple logins will create an implicit linked account.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getIdRequest
     *        Input to the GetId action.
     * @return Result of the GetId operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws LimitExceededException
     *         Thrown when the total number of user pools has exceeded a preset limit.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetId
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetIdResponse getId(GetIdRequest getIdRequest) throws InvalidParameterException, ResourceNotFoundException,
            NotAuthorizedException, ResourceConflictException, TooManyRequestsException, InternalErrorException,
            LimitExceededException, ExternalServiceException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIdResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIdRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetId");

            return clientHandler.execute(new ClientExecutionParams<GetIdRequest, GetIdResponse>().withOperationName("GetId")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getIdRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIdRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the roles for an identity pool.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param getIdentityPoolRolesRequest
     *        Input to the <code>GetIdentityPoolRoles</code> action.
     * @return Result of the GetIdentityPoolRoles operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetIdentityPoolRoles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetIdentityPoolRoles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetIdentityPoolRolesResponse getIdentityPoolRoles(GetIdentityPoolRolesRequest getIdentityPoolRolesRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIdentityPoolRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetIdentityPoolRolesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIdentityPoolRolesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdentityPoolRolesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIdentityPoolRoles");

            return clientHandler.execute(new ClientExecutionParams<GetIdentityPoolRolesRequest, GetIdentityPoolRolesResponse>()
                    .withOperationName("GetIdentityPoolRoles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getIdentityPoolRolesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIdentityPoolRolesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets an OpenID token, using a known Cognito ID. This known Cognito ID is returned by <a>GetId</a>. You can
     * optionally add additional logins for the identity. Supplying multiple logins creates an implicit link.
     * </p>
     * <p>
     * The OpenID token is valid for 10 minutes.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getOpenIdTokenRequest
     *        Input to the GetOpenIdToken action.
     * @return Result of the GetOpenIdToken operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetOpenIdToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetOpenIdToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOpenIdTokenResponse getOpenIdToken(GetOpenIdTokenRequest getOpenIdTokenRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, ExternalServiceException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOpenIdTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetOpenIdTokenResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOpenIdTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOpenIdTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOpenIdToken");

            return clientHandler.execute(new ClientExecutionParams<GetOpenIdTokenRequest, GetOpenIdTokenResponse>()
                    .withOperationName("GetOpenIdToken").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getOpenIdTokenRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetOpenIdTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Registers (or retrieves) a Cognito <code>IdentityId</code> and an OpenID Connect token for a user authenticated
     * by your backend authentication process. Supplying multiple logins will create an implicit linked account. You can
     * only specify one developer provider as part of the <code>Logins</code> map, which is linked to the identity pool.
     * The developer provider is the "domain" by which Cognito will refer to your users.
     * </p>
     * <p>
     * You can use <code>GetOpenIdTokenForDeveloperIdentity</code> to create a new identity and to link new logins (that
     * is, user credentials issued by a public provider or developer provider) to an existing identity. When you want to
     * create a new identity, the <code>IdentityId</code> should be null. When you want to associate a new login with an
     * existing authenticated/unauthenticated identity, you can do so by providing the existing <code>IdentityId</code>.
     * This API will create the identity in the specified <code>IdentityPoolId</code>.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param getOpenIdTokenForDeveloperIdentityRequest
     *        Input to the <code>GetOpenIdTokenForDeveloperIdentity</code> action.
     * @return Result of the GetOpenIdTokenForDeveloperIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws DeveloperUserAlreadyRegisteredException
     *         The provided developer user identifier is already registered with Cognito under a different identity ID.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetOpenIdTokenForDeveloperIdentity
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetOpenIdTokenForDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetOpenIdTokenForDeveloperIdentityResponse getOpenIdTokenForDeveloperIdentity(
            GetOpenIdTokenForDeveloperIdentityRequest getOpenIdTokenForDeveloperIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, DeveloperUserAlreadyRegisteredException, AwsServiceException,
            SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetOpenIdTokenForDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetOpenIdTokenForDeveloperIdentityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getOpenIdTokenForDeveloperIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getOpenIdTokenForDeveloperIdentityRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOpenIdTokenForDeveloperIdentity");

            return clientHandler
                    .execute(new ClientExecutionParams<GetOpenIdTokenForDeveloperIdentityRequest, GetOpenIdTokenForDeveloperIdentityResponse>()
                            .withOperationName("GetOpenIdTokenForDeveloperIdentity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getOpenIdTokenForDeveloperIdentityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetOpenIdTokenForDeveloperIdentityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use <code>GetPrincipalTagAttributeMap</code> to list all mappings between <code>PrincipalTags</code> and user
     * attributes.
     * </p>
     *
     * @param getPrincipalTagAttributeMapRequest
     * @return Result of the GetPrincipalTagAttributeMap operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.GetPrincipalTagAttributeMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetPrincipalTagAttributeMap"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPrincipalTagAttributeMapResponse getPrincipalTagAttributeMap(
            GetPrincipalTagAttributeMapRequest getPrincipalTagAttributeMapRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPrincipalTagAttributeMapResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPrincipalTagAttributeMapResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPrincipalTagAttributeMapRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPrincipalTagAttributeMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPrincipalTagAttributeMap");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPrincipalTagAttributeMapRequest, GetPrincipalTagAttributeMapResponse>()
                            .withOperationName("GetPrincipalTagAttributeMap").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getPrincipalTagAttributeMapRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPrincipalTagAttributeMapRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the identities in an identity pool.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param listIdentitiesRequest
     *        Input to the ListIdentities action.
     * @return Result of the ListIdentities operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.ListIdentities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIdentitiesResponse listIdentities(ListIdentitiesRequest listIdentitiesRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListIdentitiesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIdentitiesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentities");

            return clientHandler.execute(new ClientExecutionParams<ListIdentitiesRequest, ListIdentitiesResponse>()
                    .withOperationName("ListIdentities").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listIdentitiesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIdentitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all of the Cognito identity pools registered for your account.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param listIdentityPoolsRequest
     *        Input to the ListIdentityPools action.
     * @return Result of the ListIdentityPools operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.ListIdentityPools
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListIdentityPools"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListIdentityPoolsResponse listIdentityPools(ListIdentityPoolsRequest listIdentityPoolsRequest)
            throws InvalidParameterException, NotAuthorizedException, TooManyRequestsException, ResourceNotFoundException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIdentityPoolsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListIdentityPoolsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIdentityPoolsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentityPoolsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentityPools");

            return clientHandler.execute(new ClientExecutionParams<ListIdentityPoolsRequest, ListIdentityPoolsResponse>()
                    .withOperationName("ListIdentityPools").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listIdentityPoolsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIdentityPoolsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags that are assigned to an Amazon Cognito identity pool.
     * </p>
     * <p>
     * A tag is a label that you can apply to identity pools to categorize and manage them in different ways, such as by
     * purpose, owner, environment, or other criteria.
     * </p>
     * <p>
     * You can use this action up to 10 times per second, per account.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the <code>IdentityID</code> associated with a <code>DeveloperUserIdentifier</code> or the list of
     * <code>DeveloperUserIdentifier</code> values associated with an <code>IdentityId</code> for an existing identity.
     * Either <code>IdentityID</code> or <code>DeveloperUserIdentifier</code> must not be null. If you supply only one
     * of these values, the other value will be searched in the database and returned as a part of the response. If you
     * supply both, <code>DeveloperUserIdentifier</code> will be matched against <code>IdentityID</code>. If the values
     * are verified against the database, the response returns both values and is the same as the request. Otherwise, a
     * <code>ResourceConflictException</code> is thrown.
     * </p>
     * <p>
     * <code>LookupDeveloperIdentity</code> is intended for low-throughput control plane operations: for example, to
     * enable customer service to locate an identity ID by username. If you are using it for higher-volume operations
     * such as user authentication, your requests are likely to be throttled. <a>GetOpenIdTokenForDeveloperIdentity</a>
     * is a better option for higher-volume operations for user authentication.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param lookupDeveloperIdentityRequest
     *        Input to the <code>LookupDeveloperIdentityInput</code> action.
     * @return Result of the LookupDeveloperIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.LookupDeveloperIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/LookupDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public LookupDeveloperIdentityResponse lookupDeveloperIdentity(LookupDeveloperIdentityRequest lookupDeveloperIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<LookupDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, LookupDeveloperIdentityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(lookupDeveloperIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, lookupDeveloperIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LookupDeveloperIdentity");

            return clientHandler
                    .execute(new ClientExecutionParams<LookupDeveloperIdentityRequest, LookupDeveloperIdentityResponse>()
                            .withOperationName("LookupDeveloperIdentity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(lookupDeveloperIdentityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new LookupDeveloperIdentityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Merges two users having different <code>IdentityId</code>s, existing in the same identity pool, and identified by
     * the same developer provider. You can use this action to request that discrete users be merged and identified as a
     * single user in the Cognito environment. Cognito associates the given source user (
     * <code>SourceUserIdentifier</code>) with the <code>IdentityId</code> of the <code>DestinationUserIdentifier</code>
     * . Only developer-authenticated users can be merged. If the users to be merged are associated with the same public
     * provider, but as two different users, an exception will be thrown.
     * </p>
     * <p>
     * The number of linked logins is limited to 20. So, the number of linked logins for the source user,
     * <code>SourceUserIdentifier</code>, and the destination user, <code>DestinationUserIdentifier</code>, together
     * should not be larger than 20. Otherwise, an exception will be thrown.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param mergeDeveloperIdentitiesRequest
     *        Input to the <code>MergeDeveloperIdentities</code> action.
     * @return Result of the MergeDeveloperIdentities operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.MergeDeveloperIdentities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/MergeDeveloperIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public MergeDeveloperIdentitiesResponse mergeDeveloperIdentities(
            MergeDeveloperIdentitiesRequest mergeDeveloperIdentitiesRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<MergeDeveloperIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, MergeDeveloperIdentitiesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(mergeDeveloperIdentitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeDeveloperIdentitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeDeveloperIdentities");

            return clientHandler
                    .execute(new ClientExecutionParams<MergeDeveloperIdentitiesRequest, MergeDeveloperIdentitiesResponse>()
                            .withOperationName("MergeDeveloperIdentities").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(mergeDeveloperIdentitiesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new MergeDeveloperIdentitiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the roles for an identity pool. These roles are used when making calls to <a>GetCredentialsForIdentity</a>
     * action.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param setIdentityPoolRolesRequest
     *        Input to the <code>SetIdentityPoolRoles</code> action.
     * @return Result of the SetIdentityPoolRoles operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ConcurrentModificationException
     *         Thrown if there are parallel requests to modify a resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.SetIdentityPoolRoles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/SetIdentityPoolRoles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetIdentityPoolRolesResponse setIdentityPoolRoles(SetIdentityPoolRolesRequest setIdentityPoolRolesRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetIdentityPoolRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetIdentityPoolRolesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setIdentityPoolRolesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setIdentityPoolRolesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetIdentityPoolRoles");

            return clientHandler.execute(new ClientExecutionParams<SetIdentityPoolRolesRequest, SetIdentityPoolRolesResponse>()
                    .withOperationName("SetIdentityPoolRoles").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(setIdentityPoolRolesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetIdentityPoolRolesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use this operation to use default (username and clientID) attribute or custom attribute mappings.
     * </p>
     *
     * @param setPrincipalTagAttributeMapRequest
     * @return Result of the SetPrincipalTagAttributeMap operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.SetPrincipalTagAttributeMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/SetPrincipalTagAttributeMap"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SetPrincipalTagAttributeMapResponse setPrincipalTagAttributeMap(
            SetPrincipalTagAttributeMapRequest setPrincipalTagAttributeMapRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetPrincipalTagAttributeMapResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SetPrincipalTagAttributeMapResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(setPrincipalTagAttributeMapRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setPrincipalTagAttributeMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetPrincipalTagAttributeMap");

            return clientHandler
                    .execute(new ClientExecutionParams<SetPrincipalTagAttributeMapRequest, SetPrincipalTagAttributeMapResponse>()
                            .withOperationName("SetPrincipalTagAttributeMap").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(setPrincipalTagAttributeMapRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SetPrincipalTagAttributeMapRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns a set of tags to the specified Amazon Cognito identity pool. A tag is a label that you can use to
     * categorize and manage identity pools in different ways, such as by purpose, owner, environment, or other
     * criteria.
     * </p>
     * <p>
     * Each tag consists of a key and value, both of which you define. A key is a general category for more specific
     * values. For example, if you have two versions of an identity pool, one for testing and another for production,
     * you might assign an <code>Environment</code> tag key to both identity pools. The value of this key might be
     * <code>Test</code> for one identity pool and <code>Production</code> for the other.
     * </p>
     * <p>
     * Tags are useful for cost tracking and access control. You can activate your tags so that they appear on the
     * Billing and Cost Management console, where you can track the costs associated with your identity pools. In an IAM
     * policy, you can constrain permissions for identity pools based on specific tags or tag values.
     * </p>
     * <p>
     * You can use this action up to 5 times per second, per account. An identity pool can have as many as 50 tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Unlinks a <code>DeveloperUserIdentifier</code> from an existing identity. Unlinked developer users will be
     * considered new identities next time they are seen. If, for a given Cognito identity, you remove all federated
     * identities as well as the developer user identifier, the Cognito identity becomes inaccessible.
     * </p>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param unlinkDeveloperIdentityRequest
     *        Input to the <code>UnlinkDeveloperIdentity</code> action.
     * @return Result of the UnlinkDeveloperIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UnlinkDeveloperIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UnlinkDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnlinkDeveloperIdentityResponse unlinkDeveloperIdentity(UnlinkDeveloperIdentityRequest unlinkDeveloperIdentityRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnlinkDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UnlinkDeveloperIdentityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unlinkDeveloperIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unlinkDeveloperIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnlinkDeveloperIdentity");

            return clientHandler
                    .execute(new ClientExecutionParams<UnlinkDeveloperIdentityRequest, UnlinkDeveloperIdentityResponse>()
                            .withOperationName("UnlinkDeveloperIdentity").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(unlinkDeveloperIdentityRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UnlinkDeveloperIdentityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Unlinks a federated identity from an existing account. Unlinked logins will be considered new identities next
     * time they are seen. Removing the last linked login will make this identity inaccessible.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param unlinkIdentityRequest
     *        Input to the UnlinkIdentity action.
     * @return Result of the UnlinkIdentity operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ExternalServiceException
     *         An exception thrown when a dependent service such as Facebook or Twitter is not responding
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UnlinkIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UnlinkIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnlinkIdentityResponse unlinkIdentity(UnlinkIdentityRequest unlinkIdentityRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, ResourceConflictException, TooManyRequestsException,
            InternalErrorException, ExternalServiceException, AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnlinkIdentityResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UnlinkIdentityResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unlinkIdentityRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unlinkIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnlinkIdentity");

            return clientHandler.execute(new ClientExecutionParams<UnlinkIdentityRequest, UnlinkIdentityResponse>()
                    .withOperationName("UnlinkIdentity").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(unlinkIdentityRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UnlinkIdentityRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified Amazon Cognito identity pool. You can use this action up to 5 times
     * per second, per account
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterException,
            ResourceNotFoundException, NotAuthorizedException, TooManyRequestsException, InternalErrorException,
            AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an identity pool.
     * </p>
     * <important>
     * <p>
     * If you don't provide a value for a parameter, Amazon Cognito sets it to its default value.
     * </p>
     * </important>
     * <p>
     * You must use Amazon Web Services developer credentials to call this operation.
     * </p>
     *
     * @param updateIdentityPoolRequest
     *        An object representing an Amazon Cognito identity pool.
     * @return Result of the UpdateIdentityPool operation returned by the service.
     * @throws InvalidParameterException
     *         Thrown for missing or bad input parameter(s).
     * @throws ResourceNotFoundException
     *         Thrown when the requested resource (for example, a dataset or record) does not exist.
     * @throws NotAuthorizedException
     *         Thrown when a user is not authorized to access the requested resource.
     * @throws ResourceConflictException
     *         Thrown when a user tries to use a login which is already linked to another account.
     * @throws TooManyRequestsException
     *         Thrown when a request is throttled.
     * @throws InternalErrorException
     *         Thrown when the service encounters an error during processing the request.
     * @throws ConcurrentModificationException
     *         Thrown if there are parallel requests to modify a resource.
     * @throws LimitExceededException
     *         Thrown when the total number of user pools has exceeded a preset limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CognitoIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CognitoIdentityClient.UpdateIdentityPool
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UpdateIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateIdentityPoolResponse updateIdentityPool(UpdateIdentityPoolRequest updateIdentityPoolRequest)
            throws InvalidParameterException, ResourceNotFoundException, NotAuthorizedException, ResourceConflictException,
            TooManyRequestsException, InternalErrorException, ConcurrentModificationException, LimitExceededException,
            AwsServiceException, SdkClientException, CognitoIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateIdentityPoolResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ConcurrentModificationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConcurrentModificationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ConcurrentModificationException::builder).build());
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "DeveloperUserAlreadyRegisteredException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                        .httpStatusCode(400).exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build());
            case "ResourceConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceConflictException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceConflictException::builder).build());
            case "ExternalServiceException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExternalServiceException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExternalServiceException::builder).build());
            case "NotAuthorizedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotAuthorizedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotAuthorizedException::builder).build());
            case "InvalidIdentityPoolConfigurationException":
                return Optional
                        .of(ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .httpStatusCode(400).exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder)
                                .build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "TooManyRequestsException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateIdentityPoolRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIdentityPoolRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIdentityPool");

            return clientHandler.execute(new ClientExecutionParams<UpdateIdentityPoolRequest, UpdateIdentityPoolResponse>()
                    .withOperationName("UpdateIdentityPool").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateIdentityPoolRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateIdentityPoolRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        CognitoIdentityServiceClientConfigurationBuilder serviceConfigBuilder = new CognitoIdentityServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CognitoIdentityException::builder).protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1");
    }

    @Override
    public final CognitoIdentityServiceClientConfiguration serviceClientConfiguration() {
        return new CognitoIdentityServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
