/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a widget on a CloudTrail Lake dashboard.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RequestWidget implements SdkPojo, Serializable, ToCopyableBuilder<RequestWidget.Builder, RequestWidget> {
    private static final SdkField<String> QUERY_STATEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueryStatement").getter(getter(RequestWidget::queryStatement)).setter(setter(Builder::queryStatement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryStatement").build()).build();

    private static final SdkField<List<String>> QUERY_PARAMETERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("QueryParameters")
            .getter(getter(RequestWidget::queryParameters))
            .setter(setter(Builder::queryParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> VIEW_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ViewProperties")
            .getter(getter(RequestWidget::viewProperties))
            .setter(setter(Builder::viewProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ViewProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUERY_STATEMENT_FIELD,
            QUERY_PARAMETERS_FIELD, VIEW_PROPERTIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String queryStatement;

    private final List<String> queryParameters;

    private final Map<String, String> viewProperties;

    private RequestWidget(BuilderImpl builder) {
        this.queryStatement = builder.queryStatement;
        this.queryParameters = builder.queryParameters;
        this.viewProperties = builder.viewProperties;
    }

    /**
     * <p>
     * The query statement for the widget. For custom dashboard widgets, you can query across multiple event data stores
     * as long as all event data stores exist in your account.
     * </p>
     * <note>
     * <p>
     * When a query uses <code>?</code> with <code>eventTime</code>, <code>?</code> must be surrounded by single quotes
     * as follows: <code>'?'</code>.
     * </p>
     * </note>
     * 
     * @return The query statement for the widget. For custom dashboard widgets, you can query across multiple event
     *         data stores as long as all event data stores exist in your account. </p> <note>
     *         <p>
     *         When a query uses <code>?</code> with <code>eventTime</code>, <code>?</code> must be surrounded by single
     *         quotes as follows: <code>'?'</code>.
     *         </p>
     */
    public final String queryStatement() {
        return queryStatement;
    }

    /**
     * For responses, this returns true if the service returned a value for the QueryParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasQueryParameters() {
        return queryParameters != null && !(queryParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The optional query parameters. The following query parameters are valid: <code>$StartTime$</code>,
     * <code>$EndTime$</code>, and <code>$Period$</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQueryParameters} method.
     * </p>
     * 
     * @return The optional query parameters. The following query parameters are valid: <code>$StartTime$</code>,
     *         <code>$EndTime$</code>, and <code>$Period$</code>.
     */
    public final List<String> queryParameters() {
        return queryParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the ViewProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasViewProperties() {
        return viewProperties != null && !(viewProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The view properties for the widget. For more information about view properties, see <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-widget-properties.html"> View properties
     * for widgets </a> in the <i>CloudTrail User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasViewProperties} method.
     * </p>
     * 
     * @return The view properties for the widget. For more information about view properties, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-widget-properties.html"> View
     *         properties for widgets </a> in the <i>CloudTrail User Guide</i>.
     */
    public final Map<String, String> viewProperties() {
        return viewProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(queryStatement());
        hashCode = 31 * hashCode + Objects.hashCode(hasQueryParameters() ? queryParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasViewProperties() ? viewProperties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RequestWidget)) {
            return false;
        }
        RequestWidget other = (RequestWidget) obj;
        return Objects.equals(queryStatement(), other.queryStatement()) && hasQueryParameters() == other.hasQueryParameters()
                && Objects.equals(queryParameters(), other.queryParameters()) && hasViewProperties() == other.hasViewProperties()
                && Objects.equals(viewProperties(), other.viewProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RequestWidget").add("QueryStatement", queryStatement())
                .add("QueryParameters", hasQueryParameters() ? queryParameters() : null)
                .add("ViewProperties", hasViewProperties() ? viewProperties() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "QueryStatement":
            return Optional.ofNullable(clazz.cast(queryStatement()));
        case "QueryParameters":
            return Optional.ofNullable(clazz.cast(queryParameters()));
        case "ViewProperties":
            return Optional.ofNullable(clazz.cast(viewProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("QueryStatement", QUERY_STATEMENT_FIELD);
        map.put("QueryParameters", QUERY_PARAMETERS_FIELD);
        map.put("ViewProperties", VIEW_PROPERTIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RequestWidget, T> g) {
        return obj -> g.apply((RequestWidget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RequestWidget> {
        /**
         * <p>
         * The query statement for the widget. For custom dashboard widgets, you can query across multiple event data
         * stores as long as all event data stores exist in your account.
         * </p>
         * <note>
         * <p>
         * When a query uses <code>?</code> with <code>eventTime</code>, <code>?</code> must be surrounded by single
         * quotes as follows: <code>'?'</code>.
         * </p>
         * </note>
         * 
         * @param queryStatement
         *        The query statement for the widget. For custom dashboard widgets, you can query across multiple event
         *        data stores as long as all event data stores exist in your account. </p> <note>
         *        <p>
         *        When a query uses <code>?</code> with <code>eventTime</code>, <code>?</code> must be surrounded by
         *        single quotes as follows: <code>'?'</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryStatement(String queryStatement);

        /**
         * <p>
         * The optional query parameters. The following query parameters are valid: <code>$StartTime$</code>,
         * <code>$EndTime$</code>, and <code>$Period$</code>.
         * </p>
         * 
         * @param queryParameters
         *        The optional query parameters. The following query parameters are valid: <code>$StartTime$</code>,
         *        <code>$EndTime$</code>, and <code>$Period$</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryParameters(Collection<String> queryParameters);

        /**
         * <p>
         * The optional query parameters. The following query parameters are valid: <code>$StartTime$</code>,
         * <code>$EndTime$</code>, and <code>$Period$</code>.
         * </p>
         * 
         * @param queryParameters
         *        The optional query parameters. The following query parameters are valid: <code>$StartTime$</code>,
         *        <code>$EndTime$</code>, and <code>$Period$</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryParameters(String... queryParameters);

        /**
         * <p>
         * The view properties for the widget. For more information about view properties, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-widget-properties.html"> View
         * properties for widgets </a> in the <i>CloudTrail User Guide</i>.
         * </p>
         * 
         * @param viewProperties
         *        The view properties for the widget. For more information about view properties, see <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/lake-widget-properties.html"> View
         *        properties for widgets </a> in the <i>CloudTrail User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewProperties(Map<String, String> viewProperties);
    }

    static final class BuilderImpl implements Builder {
        private String queryStatement;

        private List<String> queryParameters = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> viewProperties = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RequestWidget model) {
            queryStatement(model.queryStatement);
            queryParameters(model.queryParameters);
            viewProperties(model.viewProperties);
        }

        public final String getQueryStatement() {
            return queryStatement;
        }

        public final void setQueryStatement(String queryStatement) {
            this.queryStatement = queryStatement;
        }

        @Override
        public final Builder queryStatement(String queryStatement) {
            this.queryStatement = queryStatement;
            return this;
        }

        public final Collection<String> getQueryParameters() {
            if (queryParameters instanceof SdkAutoConstructList) {
                return null;
            }
            return queryParameters;
        }

        public final void setQueryParameters(Collection<String> queryParameters) {
            this.queryParameters = QueryParametersCopier.copy(queryParameters);
        }

        @Override
        public final Builder queryParameters(Collection<String> queryParameters) {
            this.queryParameters = QueryParametersCopier.copy(queryParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder queryParameters(String... queryParameters) {
            queryParameters(Arrays.asList(queryParameters));
            return this;
        }

        public final Map<String, String> getViewProperties() {
            if (viewProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return viewProperties;
        }

        public final void setViewProperties(Map<String, String> viewProperties) {
            this.viewProperties = ViewPropertiesMapCopier.copy(viewProperties);
        }

        @Override
        public final Builder viewProperties(Map<String, String> viewProperties) {
            this.viewProperties = ViewPropertiesMapCopier.copy(viewProperties);
            return this;
        }

        @Override
        public RequestWidget build() {
            return new RequestWidget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
