/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.clouddirectory.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns the path to the <code>ObjectIdentifiers</code> that is associated with the directory.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PathToObjectIdentifiers implements SdkPojo, Serializable,
        ToCopyableBuilder<PathToObjectIdentifiers.Builder, PathToObjectIdentifiers> {
    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Path")
            .getter(getter(PathToObjectIdentifiers::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Path").build()).build();

    private static final SdkField<List<String>> OBJECT_IDENTIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ObjectIdentifiers")
            .getter(getter(PathToObjectIdentifiers::objectIdentifiers))
            .setter(setter(Builder::objectIdentifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectIdentifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PATH_FIELD,
            OBJECT_IDENTIFIERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String path;

    private final List<String> objectIdentifiers;

    private PathToObjectIdentifiers(BuilderImpl builder) {
        this.path = builder.path;
        this.objectIdentifiers = builder.objectIdentifiers;
    }

    /**
     * <p>
     * The path that is used to identify the object starting from directory root.
     * </p>
     * 
     * @return The path that is used to identify the object starting from directory root.
     */
    public final String path() {
        return path;
    }

    /**
     * For responses, this returns true if the service returned a value for the ObjectIdentifiers property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasObjectIdentifiers() {
        return objectIdentifiers != null && !(objectIdentifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Lists <code>ObjectIdentifiers</code> starting from directory root to the object in the request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasObjectIdentifiers} method.
     * </p>
     * 
     * @return Lists <code>ObjectIdentifiers</code> starting from directory root to the object in the request.
     */
    public final List<String> objectIdentifiers() {
        return objectIdentifiers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(hasObjectIdentifiers() ? objectIdentifiers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PathToObjectIdentifiers)) {
            return false;
        }
        PathToObjectIdentifiers other = (PathToObjectIdentifiers) obj;
        return Objects.equals(path(), other.path()) && hasObjectIdentifiers() == other.hasObjectIdentifiers()
                && Objects.equals(objectIdentifiers(), other.objectIdentifiers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PathToObjectIdentifiers").add("Path", path())
                .add("ObjectIdentifiers", hasObjectIdentifiers() ? objectIdentifiers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Path":
            return Optional.ofNullable(clazz.cast(path()));
        case "ObjectIdentifiers":
            return Optional.ofNullable(clazz.cast(objectIdentifiers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Path", PATH_FIELD);
        map.put("ObjectIdentifiers", OBJECT_IDENTIFIERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PathToObjectIdentifiers, T> g) {
        return obj -> g.apply((PathToObjectIdentifiers) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PathToObjectIdentifiers> {
        /**
         * <p>
         * The path that is used to identify the object starting from directory root.
         * </p>
         * 
         * @param path
         *        The path that is used to identify the object starting from directory root.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Lists <code>ObjectIdentifiers</code> starting from directory root to the object in the request.
         * </p>
         * 
         * @param objectIdentifiers
         *        Lists <code>ObjectIdentifiers</code> starting from directory root to the object in the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectIdentifiers(Collection<String> objectIdentifiers);

        /**
         * <p>
         * Lists <code>ObjectIdentifiers</code> starting from directory root to the object in the request.
         * </p>
         * 
         * @param objectIdentifiers
         *        Lists <code>ObjectIdentifiers</code> starting from directory root to the object in the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectIdentifiers(String... objectIdentifiers);
    }

    static final class BuilderImpl implements Builder {
        private String path;

        private List<String> objectIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PathToObjectIdentifiers model) {
            path(model.path);
            objectIdentifiers(model.objectIdentifiers);
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final Collection<String> getObjectIdentifiers() {
            if (objectIdentifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return objectIdentifiers;
        }

        public final void setObjectIdentifiers(Collection<String> objectIdentifiers) {
            this.objectIdentifiers = ObjectIdentifierListCopier.copy(objectIdentifiers);
        }

        @Override
        public final Builder objectIdentifiers(Collection<String> objectIdentifiers) {
            this.objectIdentifiers = ObjectIdentifierListCopier.copy(objectIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objectIdentifiers(String... objectIdentifiers) {
            objectIdentifiers(Arrays.asList(objectIdentifiers));
            return this;
        }

        @Override
        public PathToObjectIdentifiers build() {
            return new PathToObjectIdentifiers(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
