/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains metadata about a backup vault.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackupVaultListMember implements SdkPojo, Serializable,
        ToCopyableBuilder<BackupVaultListMember.Builder, BackupVaultListMember> {
    private static final SdkField<String> BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultName").getter(getter(BackupVaultListMember::backupVaultName))
            .setter(setter(Builder::backupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultName").build()).build();

    private static final SdkField<String> BACKUP_VAULT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultArn").getter(getter(BackupVaultListMember::backupVaultArn))
            .setter(setter(Builder::backupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultArn").build()).build();

    private static final SdkField<String> VAULT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VaultType").getter(getter(BackupVaultListMember::vaultTypeAsString)).setter(setter(Builder::vaultType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VaultType").build()).build();

    private static final SdkField<String> VAULT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VaultState").getter(getter(BackupVaultListMember::vaultStateAsString))
            .setter(setter(Builder::vaultState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VaultState").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(BackupVaultListMember::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<String> ENCRYPTION_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionKeyArn").getter(getter(BackupVaultListMember::encryptionKeyArn))
            .setter(setter(Builder::encryptionKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionKeyArn").build()).build();

    private static final SdkField<String> CREATOR_REQUEST_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatorRequestId").getter(getter(BackupVaultListMember::creatorRequestId))
            .setter(setter(Builder::creatorRequestId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatorRequestId").build()).build();

    private static final SdkField<Long> NUMBER_OF_RECOVERY_POINTS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("NumberOfRecoveryPoints").getter(getter(BackupVaultListMember::numberOfRecoveryPoints))
            .setter(setter(Builder::numberOfRecoveryPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfRecoveryPoints").build())
            .build();

    private static final SdkField<Boolean> LOCKED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Locked").getter(getter(BackupVaultListMember::locked)).setter(setter(Builder::locked))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Locked").build()).build();

    private static final SdkField<Long> MIN_RETENTION_DAYS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MinRetentionDays").getter(getter(BackupVaultListMember::minRetentionDays))
            .setter(setter(Builder::minRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinRetentionDays").build()).build();

    private static final SdkField<Long> MAX_RETENTION_DAYS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaxRetentionDays").getter(getter(BackupVaultListMember::maxRetentionDays))
            .setter(setter(Builder::maxRetentionDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRetentionDays").build()).build();

    private static final SdkField<Instant> LOCK_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LockDate").getter(getter(BackupVaultListMember::lockDate)).setter(setter(Builder::lockDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LockDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BACKUP_VAULT_NAME_FIELD,
            BACKUP_VAULT_ARN_FIELD, VAULT_TYPE_FIELD, VAULT_STATE_FIELD, CREATION_DATE_FIELD, ENCRYPTION_KEY_ARN_FIELD,
            CREATOR_REQUEST_ID_FIELD, NUMBER_OF_RECOVERY_POINTS_FIELD, LOCKED_FIELD, MIN_RETENTION_DAYS_FIELD,
            MAX_RETENTION_DAYS_FIELD, LOCK_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String backupVaultName;

    private final String backupVaultArn;

    private final String vaultType;

    private final String vaultState;

    private final Instant creationDate;

    private final String encryptionKeyArn;

    private final String creatorRequestId;

    private final Long numberOfRecoveryPoints;

    private final Boolean locked;

    private final Long minRetentionDays;

    private final Long maxRetentionDays;

    private final Instant lockDate;

    private BackupVaultListMember(BuilderImpl builder) {
        this.backupVaultName = builder.backupVaultName;
        this.backupVaultArn = builder.backupVaultArn;
        this.vaultType = builder.vaultType;
        this.vaultState = builder.vaultState;
        this.creationDate = builder.creationDate;
        this.encryptionKeyArn = builder.encryptionKeyArn;
        this.creatorRequestId = builder.creatorRequestId;
        this.numberOfRecoveryPoints = builder.numberOfRecoveryPoints;
        this.locked = builder.locked;
        this.minRetentionDays = builder.minRetentionDays;
        this.maxRetentionDays = builder.maxRetentionDays;
        this.lockDate = builder.lockDate;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the Amazon Web Services Region where they are created.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the Amazon Web Services Region where they are created.
     */
    public final String backupVaultName() {
        return backupVaultName;
    }

    /**
     * <p>
     * An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
     * </p>
     * 
     * @return An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
     */
    public final String backupVaultArn() {
        return backupVaultArn;
    }

    /**
     * <p>
     * The type of vault in which the described recovery point is stored.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vaultType} will
     * return {@link VaultType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vaultTypeAsString}.
     * </p>
     * 
     * @return The type of vault in which the described recovery point is stored.
     * @see VaultType
     */
    public final VaultType vaultType() {
        return VaultType.fromValue(vaultType);
    }

    /**
     * <p>
     * The type of vault in which the described recovery point is stored.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vaultType} will
     * return {@link VaultType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vaultTypeAsString}.
     * </p>
     * 
     * @return The type of vault in which the described recovery point is stored.
     * @see VaultType
     */
    public final String vaultTypeAsString() {
        return vaultType;
    }

    /**
     * <p>
     * The current state of the vault.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vaultState} will
     * return {@link VaultState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vaultStateAsString}.
     * </p>
     * 
     * @return The current state of the vault.
     * @see VaultState
     */
    public final VaultState vaultState() {
        return VaultState.fromValue(vaultState);
    }

    /**
     * <p>
     * The current state of the vault.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vaultState} will
     * return {@link VaultState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vaultStateAsString}.
     * </p>
     * 
     * @return The current state of the vault.
     * @see VaultState
     */
    public final String vaultStateAsString() {
        return vaultState;
    }

    /**
     * <p>
     * The date and time a resource backup is created, in Unix format and Coordinated Universal Time (UTC). The value of
     * <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents Friday,
     * January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time a resource backup is created, in Unix format and Coordinated Universal Time (UTC). The
     *         value of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
     *         represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * A server-side encryption key you can specify to encrypt your backups from services that support full Backup
     * management; for example, <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * . If you specify a key, you must specify its ARN, not its alias. If you do not specify a key, Backup creates a
     * KMS key for you by default.
     * </p>
     * <p>
     * To learn which Backup services support full Backup management and how Backup handles encryption for backups from
     * services that do not yet support full Backup, see <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/encryption.html"> Encryption for backups in
     * Backup</a>
     * </p>
     * 
     * @return A server-side encryption key you can specify to encrypt your backups from services that support full
     *         Backup management; for example,
     *         <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>. If you specify
     *         a key, you must specify its ARN, not its alias. If you do not specify a key, Backup creates a KMS key for
     *         you by default.</p>
     *         <p>
     *         To learn which Backup services support full Backup management and how Backup handles encryption for
     *         backups from services that do not yet support full Backup, see <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/encryption.html"> Encryption for backups in
     *         Backup</a>
     */
    public final String encryptionKeyArn() {
        return encryptionKeyArn;
    }

    /**
     * <p>
     * A unique string that identifies the request and allows failed requests to be retried without the risk of running
     * the operation twice. This parameter is optional.
     * </p>
     * <p>
     * If used, this parameter must contain 1 to 50 alphanumeric or '-_.' characters.
     * </p>
     * 
     * @return A unique string that identifies the request and allows failed requests to be retried without the risk of
     *         running the operation twice. This parameter is optional.</p>
     *         <p>
     *         If used, this parameter must contain 1 to 50 alphanumeric or '-_.' characters.
     */
    public final String creatorRequestId() {
        return creatorRequestId;
    }

    /**
     * <p>
     * The number of recovery points that are stored in a backup vault.
     * </p>
     * 
     * @return The number of recovery points that are stored in a backup vault.
     */
    public final Long numberOfRecoveryPoints() {
        return numberOfRecoveryPoints;
    }

    /**
     * <p>
     * A Boolean value that indicates whether Backup Vault Lock applies to the selected backup vault. If
     * <code>true</code>, Vault Lock prevents delete and update operations on the recovery points in the selected vault.
     * </p>
     * 
     * @return A Boolean value that indicates whether Backup Vault Lock applies to the selected backup vault. If
     *         <code>true</code>, Vault Lock prevents delete and update operations on the recovery points in the
     *         selected vault.
     */
    public final Boolean locked() {
        return locked;
    }

    /**
     * <p>
     * The Backup Vault Lock setting that specifies the minimum retention period that the vault retains its recovery
     * points. If this parameter is not specified, Vault Lock does not enforce a minimum retention period.
     * </p>
     * <p>
     * If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period equal to
     * or longer than the minimum retention period. If the job's retention period is shorter than that minimum retention
     * period, then the vault fails the backup or copy job, and you should either modify your lifecycle settings or use
     * a different vault. Recovery points already stored in the vault prior to Vault Lock are not affected.
     * </p>
     * 
     * @return The Backup Vault Lock setting that specifies the minimum retention period that the vault retains its
     *         recovery points. If this parameter is not specified, Vault Lock does not enforce a minimum retention
     *         period.</p>
     *         <p>
     *         If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period
     *         equal to or longer than the minimum retention period. If the job's retention period is shorter than that
     *         minimum retention period, then the vault fails the backup or copy job, and you should either modify your
     *         lifecycle settings or use a different vault. Recovery points already stored in the vault prior to Vault
     *         Lock are not affected.
     */
    public final Long minRetentionDays() {
        return minRetentionDays;
    }

    /**
     * <p>
     * The Backup Vault Lock setting that specifies the maximum retention period that the vault retains its recovery
     * points. If this parameter is not specified, Vault Lock does not enforce a maximum retention period on the
     * recovery points in the vault (allowing indefinite storage).
     * </p>
     * <p>
     * If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period equal to
     * or shorter than the maximum retention period. If the job's retention period is longer than that maximum retention
     * period, then the vault fails the backup or copy job, and you should either modify your lifecycle settings or use
     * a different vault. Recovery points already stored in the vault prior to Vault Lock are not affected.
     * </p>
     * 
     * @return The Backup Vault Lock setting that specifies the maximum retention period that the vault retains its
     *         recovery points. If this parameter is not specified, Vault Lock does not enforce a maximum retention
     *         period on the recovery points in the vault (allowing indefinite storage).</p>
     *         <p>
     *         If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period
     *         equal to or shorter than the maximum retention period. If the job's retention period is longer than that
     *         maximum retention period, then the vault fails the backup or copy job, and you should either modify your
     *         lifecycle settings or use a different vault. Recovery points already stored in the vault prior to Vault
     *         Lock are not affected.
     */
    public final Long maxRetentionDays() {
        return maxRetentionDays;
    }

    /**
     * <p>
     * The date and time when Backup Vault Lock configuration becomes immutable, meaning it cannot be changed or
     * deleted.
     * </p>
     * <p>
     * If you applied Vault Lock to your vault without specifying a lock date, you can change your Vault Lock settings,
     * or delete Vault Lock from the vault entirely, at any time.
     * </p>
     * <p>
     * This value is in Unix format, Coordinated Universal Time (UTC), and accurate to milliseconds. For example, the
     * value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time when Backup Vault Lock configuration becomes immutable, meaning it cannot be changed or
     *         deleted.</p>
     *         <p>
     *         If you applied Vault Lock to your vault without specifying a lock date, you can change your Vault Lock
     *         settings, or delete Vault Lock from the vault entirely, at any time.
     *         </p>
     *         <p>
     *         This value is in Unix format, Coordinated Universal Time (UTC), and accurate to milliseconds. For
     *         example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant lockDate() {
        return lockDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(vaultTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vaultStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(creatorRequestId());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfRecoveryPoints());
        hashCode = 31 * hashCode + Objects.hashCode(locked());
        hashCode = 31 * hashCode + Objects.hashCode(minRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(maxRetentionDays());
        hashCode = 31 * hashCode + Objects.hashCode(lockDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackupVaultListMember)) {
            return false;
        }
        BackupVaultListMember other = (BackupVaultListMember) obj;
        return Objects.equals(backupVaultName(), other.backupVaultName())
                && Objects.equals(backupVaultArn(), other.backupVaultArn())
                && Objects.equals(vaultTypeAsString(), other.vaultTypeAsString())
                && Objects.equals(vaultStateAsString(), other.vaultStateAsString())
                && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(encryptionKeyArn(), other.encryptionKeyArn())
                && Objects.equals(creatorRequestId(), other.creatorRequestId())
                && Objects.equals(numberOfRecoveryPoints(), other.numberOfRecoveryPoints())
                && Objects.equals(locked(), other.locked()) && Objects.equals(minRetentionDays(), other.minRetentionDays())
                && Objects.equals(maxRetentionDays(), other.maxRetentionDays()) && Objects.equals(lockDate(), other.lockDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BackupVaultListMember").add("BackupVaultName", backupVaultName())
                .add("BackupVaultArn", backupVaultArn()).add("VaultType", vaultTypeAsString())
                .add("VaultState", vaultStateAsString()).add("CreationDate", creationDate())
                .add("EncryptionKeyArn", encryptionKeyArn()).add("CreatorRequestId", creatorRequestId())
                .add("NumberOfRecoveryPoints", numberOfRecoveryPoints()).add("Locked", locked())
                .add("MinRetentionDays", minRetentionDays()).add("MaxRetentionDays", maxRetentionDays())
                .add("LockDate", lockDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BackupVaultName":
            return Optional.ofNullable(clazz.cast(backupVaultName()));
        case "BackupVaultArn":
            return Optional.ofNullable(clazz.cast(backupVaultArn()));
        case "VaultType":
            return Optional.ofNullable(clazz.cast(vaultTypeAsString()));
        case "VaultState":
            return Optional.ofNullable(clazz.cast(vaultStateAsString()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "EncryptionKeyArn":
            return Optional.ofNullable(clazz.cast(encryptionKeyArn()));
        case "CreatorRequestId":
            return Optional.ofNullable(clazz.cast(creatorRequestId()));
        case "NumberOfRecoveryPoints":
            return Optional.ofNullable(clazz.cast(numberOfRecoveryPoints()));
        case "Locked":
            return Optional.ofNullable(clazz.cast(locked()));
        case "MinRetentionDays":
            return Optional.ofNullable(clazz.cast(minRetentionDays()));
        case "MaxRetentionDays":
            return Optional.ofNullable(clazz.cast(maxRetentionDays()));
        case "LockDate":
            return Optional.ofNullable(clazz.cast(lockDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("BackupVaultName", BACKUP_VAULT_NAME_FIELD);
        map.put("BackupVaultArn", BACKUP_VAULT_ARN_FIELD);
        map.put("VaultType", VAULT_TYPE_FIELD);
        map.put("VaultState", VAULT_STATE_FIELD);
        map.put("CreationDate", CREATION_DATE_FIELD);
        map.put("EncryptionKeyArn", ENCRYPTION_KEY_ARN_FIELD);
        map.put("CreatorRequestId", CREATOR_REQUEST_ID_FIELD);
        map.put("NumberOfRecoveryPoints", NUMBER_OF_RECOVERY_POINTS_FIELD);
        map.put("Locked", LOCKED_FIELD);
        map.put("MinRetentionDays", MIN_RETENTION_DAYS_FIELD);
        map.put("MaxRetentionDays", MAX_RETENTION_DAYS_FIELD);
        map.put("LockDate", LOCK_DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BackupVaultListMember, T> g) {
        return obj -> g.apply((BackupVaultListMember) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackupVaultListMember> {
        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the Amazon Web Services Region where they are created.
         * </p>
         * 
         * @param backupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the Amazon Web Services Region where they are
         *        created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultName(String backupVaultName);

        /**
         * <p>
         * An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
         * </p>
         * 
         * @param backupVaultArn
         *        An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultArn(String backupVaultArn);

        /**
         * <p>
         * The type of vault in which the described recovery point is stored.
         * </p>
         * 
         * @param vaultType
         *        The type of vault in which the described recovery point is stored.
         * @see VaultType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VaultType
         */
        Builder vaultType(String vaultType);

        /**
         * <p>
         * The type of vault in which the described recovery point is stored.
         * </p>
         * 
         * @param vaultType
         *        The type of vault in which the described recovery point is stored.
         * @see VaultType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VaultType
         */
        Builder vaultType(VaultType vaultType);

        /**
         * <p>
         * The current state of the vault.
         * </p>
         * 
         * @param vaultState
         *        The current state of the vault.
         * @see VaultState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VaultState
         */
        Builder vaultState(String vaultState);

        /**
         * <p>
         * The current state of the vault.
         * </p>
         * 
         * @param vaultState
         *        The current state of the vault.
         * @see VaultState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VaultState
         */
        Builder vaultState(VaultState vaultState);

        /**
         * <p>
         * The date and time a resource backup is created, in Unix format and Coordinated Universal Time (UTC). The
         * value of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
         * represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param creationDate
         *        The date and time a resource backup is created, in Unix format and Coordinated Universal Time (UTC).
         *        The value of <code>CreationDate</code> is accurate to milliseconds. For example, the value
         *        1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * A server-side encryption key you can specify to encrypt your backups from services that support full Backup
         * management; for example,
         * <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>. If you specify a
         * key, you must specify its ARN, not its alias. If you do not specify a key, Backup creates a KMS key for you
         * by default.
         * </p>
         * <p>
         * To learn which Backup services support full Backup management and how Backup handles encryption for backups
         * from services that do not yet support full Backup, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/encryption.html"> Encryption for backups in
         * Backup</a>
         * </p>
         * 
         * @param encryptionKeyArn
         *        A server-side encryption key you can specify to encrypt your backups from services that support full
         *        Backup management; for example,
         *        <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>. If you
         *        specify a key, you must specify its ARN, not its alias. If you do not specify a key, Backup creates a
         *        KMS key for you by default.</p>
         *        <p>
         *        To learn which Backup services support full Backup management and how Backup handles encryption for
         *        backups from services that do not yet support full Backup, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/encryption.html"> Encryption for backups
         *        in Backup</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKeyArn(String encryptionKeyArn);

        /**
         * <p>
         * A unique string that identifies the request and allows failed requests to be retried without the risk of
         * running the operation twice. This parameter is optional.
         * </p>
         * <p>
         * If used, this parameter must contain 1 to 50 alphanumeric or '-_.' characters.
         * </p>
         * 
         * @param creatorRequestId
         *        A unique string that identifies the request and allows failed requests to be retried without the risk
         *        of running the operation twice. This parameter is optional.</p>
         *        <p>
         *        If used, this parameter must contain 1 to 50 alphanumeric or '-_.' characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorRequestId(String creatorRequestId);

        /**
         * <p>
         * The number of recovery points that are stored in a backup vault.
         * </p>
         * 
         * @param numberOfRecoveryPoints
         *        The number of recovery points that are stored in a backup vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfRecoveryPoints(Long numberOfRecoveryPoints);

        /**
         * <p>
         * A Boolean value that indicates whether Backup Vault Lock applies to the selected backup vault. If
         * <code>true</code>, Vault Lock prevents delete and update operations on the recovery points in the selected
         * vault.
         * </p>
         * 
         * @param locked
         *        A Boolean value that indicates whether Backup Vault Lock applies to the selected backup vault. If
         *        <code>true</code>, Vault Lock prevents delete and update operations on the recovery points in the
         *        selected vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locked(Boolean locked);

        /**
         * <p>
         * The Backup Vault Lock setting that specifies the minimum retention period that the vault retains its recovery
         * points. If this parameter is not specified, Vault Lock does not enforce a minimum retention period.
         * </p>
         * <p>
         * If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period equal
         * to or longer than the minimum retention period. If the job's retention period is shorter than that minimum
         * retention period, then the vault fails the backup or copy job, and you should either modify your lifecycle
         * settings or use a different vault. Recovery points already stored in the vault prior to Vault Lock are not
         * affected.
         * </p>
         * 
         * @param minRetentionDays
         *        The Backup Vault Lock setting that specifies the minimum retention period that the vault retains its
         *        recovery points. If this parameter is not specified, Vault Lock does not enforce a minimum retention
         *        period.</p>
         *        <p>
         *        If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period
         *        equal to or longer than the minimum retention period. If the job's retention period is shorter than
         *        that minimum retention period, then the vault fails the backup or copy job, and you should either
         *        modify your lifecycle settings or use a different vault. Recovery points already stored in the vault
         *        prior to Vault Lock are not affected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minRetentionDays(Long minRetentionDays);

        /**
         * <p>
         * The Backup Vault Lock setting that specifies the maximum retention period that the vault retains its recovery
         * points. If this parameter is not specified, Vault Lock does not enforce a maximum retention period on the
         * recovery points in the vault (allowing indefinite storage).
         * </p>
         * <p>
         * If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period equal
         * to or shorter than the maximum retention period. If the job's retention period is longer than that maximum
         * retention period, then the vault fails the backup or copy job, and you should either modify your lifecycle
         * settings or use a different vault. Recovery points already stored in the vault prior to Vault Lock are not
         * affected.
         * </p>
         * 
         * @param maxRetentionDays
         *        The Backup Vault Lock setting that specifies the maximum retention period that the vault retains its
         *        recovery points. If this parameter is not specified, Vault Lock does not enforce a maximum retention
         *        period on the recovery points in the vault (allowing indefinite storage).</p>
         *        <p>
         *        If specified, any backup or copy job to the vault must have a lifecycle policy with a retention period
         *        equal to or shorter than the maximum retention period. If the job's retention period is longer than
         *        that maximum retention period, then the vault fails the backup or copy job, and you should either
         *        modify your lifecycle settings or use a different vault. Recovery points already stored in the vault
         *        prior to Vault Lock are not affected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRetentionDays(Long maxRetentionDays);

        /**
         * <p>
         * The date and time when Backup Vault Lock configuration becomes immutable, meaning it cannot be changed or
         * deleted.
         * </p>
         * <p>
         * If you applied Vault Lock to your vault without specifying a lock date, you can change your Vault Lock
         * settings, or delete Vault Lock from the vault entirely, at any time.
         * </p>
         * <p>
         * This value is in Unix format, Coordinated Universal Time (UTC), and accurate to milliseconds. For example,
         * the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param lockDate
         *        The date and time when Backup Vault Lock configuration becomes immutable, meaning it cannot be changed
         *        or deleted.</p>
         *        <p>
         *        If you applied Vault Lock to your vault without specifying a lock date, you can change your Vault Lock
         *        settings, or delete Vault Lock from the vault entirely, at any time.
         *        </p>
         *        <p>
         *        This value is in Unix format, Coordinated Universal Time (UTC), and accurate to milliseconds. For
         *        example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lockDate(Instant lockDate);
    }

    static final class BuilderImpl implements Builder {
        private String backupVaultName;

        private String backupVaultArn;

        private String vaultType;

        private String vaultState;

        private Instant creationDate;

        private String encryptionKeyArn;

        private String creatorRequestId;

        private Long numberOfRecoveryPoints;

        private Boolean locked;

        private Long minRetentionDays;

        private Long maxRetentionDays;

        private Instant lockDate;

        private BuilderImpl() {
        }

        private BuilderImpl(BackupVaultListMember model) {
            backupVaultName(model.backupVaultName);
            backupVaultArn(model.backupVaultArn);
            vaultType(model.vaultType);
            vaultState(model.vaultState);
            creationDate(model.creationDate);
            encryptionKeyArn(model.encryptionKeyArn);
            creatorRequestId(model.creatorRequestId);
            numberOfRecoveryPoints(model.numberOfRecoveryPoints);
            locked(model.locked);
            minRetentionDays(model.minRetentionDays);
            maxRetentionDays(model.maxRetentionDays);
            lockDate(model.lockDate);
        }

        public final String getBackupVaultName() {
            return backupVaultName;
        }

        public final void setBackupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
        }

        @Override
        public final Builder backupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
            return this;
        }

        public final String getBackupVaultArn() {
            return backupVaultArn;
        }

        public final void setBackupVaultArn(String backupVaultArn) {
            this.backupVaultArn = backupVaultArn;
        }

        @Override
        public final Builder backupVaultArn(String backupVaultArn) {
            this.backupVaultArn = backupVaultArn;
            return this;
        }

        public final String getVaultType() {
            return vaultType;
        }

        public final void setVaultType(String vaultType) {
            this.vaultType = vaultType;
        }

        @Override
        public final Builder vaultType(String vaultType) {
            this.vaultType = vaultType;
            return this;
        }

        @Override
        public final Builder vaultType(VaultType vaultType) {
            this.vaultType(vaultType == null ? null : vaultType.toString());
            return this;
        }

        public final String getVaultState() {
            return vaultState;
        }

        public final void setVaultState(String vaultState) {
            this.vaultState = vaultState;
        }

        @Override
        public final Builder vaultState(String vaultState) {
            this.vaultState = vaultState;
            return this;
        }

        @Override
        public final Builder vaultState(VaultState vaultState) {
            this.vaultState(vaultState == null ? null : vaultState.toString());
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final String getEncryptionKeyArn() {
            return encryptionKeyArn;
        }

        public final void setEncryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
        }

        @Override
        public final Builder encryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
            return this;
        }

        public final String getCreatorRequestId() {
            return creatorRequestId;
        }

        public final void setCreatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
        }

        @Override
        public final Builder creatorRequestId(String creatorRequestId) {
            this.creatorRequestId = creatorRequestId;
            return this;
        }

        public final Long getNumberOfRecoveryPoints() {
            return numberOfRecoveryPoints;
        }

        public final void setNumberOfRecoveryPoints(Long numberOfRecoveryPoints) {
            this.numberOfRecoveryPoints = numberOfRecoveryPoints;
        }

        @Override
        public final Builder numberOfRecoveryPoints(Long numberOfRecoveryPoints) {
            this.numberOfRecoveryPoints = numberOfRecoveryPoints;
            return this;
        }

        public final Boolean getLocked() {
            return locked;
        }

        public final void setLocked(Boolean locked) {
            this.locked = locked;
        }

        @Override
        public final Builder locked(Boolean locked) {
            this.locked = locked;
            return this;
        }

        public final Long getMinRetentionDays() {
            return minRetentionDays;
        }

        public final void setMinRetentionDays(Long minRetentionDays) {
            this.minRetentionDays = minRetentionDays;
        }

        @Override
        public final Builder minRetentionDays(Long minRetentionDays) {
            this.minRetentionDays = minRetentionDays;
            return this;
        }

        public final Long getMaxRetentionDays() {
            return maxRetentionDays;
        }

        public final void setMaxRetentionDays(Long maxRetentionDays) {
            this.maxRetentionDays = maxRetentionDays;
        }

        @Override
        public final Builder maxRetentionDays(Long maxRetentionDays) {
            this.maxRetentionDays = maxRetentionDays;
            return this;
        }

        public final Instant getLockDate() {
            return lockDate;
        }

        public final void setLockDate(Instant lockDate) {
            this.lockDate = lockDate;
        }

        @Override
        public final Builder lockDate(Instant lockDate) {
            this.lockDate = lockDate;
            return this;
        }

        @Override
        public BackupVaultListMember build() {
            return new BackupVaultListMember(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
