/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appmesh.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A reference to an object that represents a Transport Layer Security (TLS) client policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClientPolicyTls implements SdkPojo, Serializable, ToCopyableBuilder<ClientPolicyTls.Builder, ClientPolicyTls> {
    private static final SdkField<ClientTlsCertificate> CERTIFICATE_FIELD = SdkField
            .<ClientTlsCertificate> builder(MarshallingType.SDK_POJO).memberName("certificate")
            .getter(getter(ClientPolicyTls::certificate)).setter(setter(Builder::certificate))
            .constructor(ClientTlsCertificate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificate").build()).build();

    private static final SdkField<Boolean> ENFORCE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enforce").getter(getter(ClientPolicyTls::enforce)).setter(setter(Builder::enforce))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enforce").build()).build();

    private static final SdkField<List<Integer>> PORTS_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("ports")
            .getter(getter(ClientPolicyTls::ports))
            .setter(setter(Builder::ports))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ports").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TlsValidationContext> VALIDATION_FIELD = SdkField
            .<TlsValidationContext> builder(MarshallingType.SDK_POJO).memberName("validation")
            .getter(getter(ClientPolicyTls::validation)).setter(setter(Builder::validation))
            .constructor(TlsValidationContext::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("validation").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CERTIFICATE_FIELD,
            ENFORCE_FIELD, PORTS_FIELD, VALIDATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final ClientTlsCertificate certificate;

    private final Boolean enforce;

    private final List<Integer> ports;

    private final TlsValidationContext validation;

    private ClientPolicyTls(BuilderImpl builder) {
        this.certificate = builder.certificate;
        this.enforce = builder.enforce;
        this.ports = builder.ports;
        this.validation = builder.validation;
    }

    /**
     * <p>
     * A reference to an object that represents a client's TLS certificate.
     * </p>
     * 
     * @return A reference to an object that represents a client's TLS certificate.
     */
    public final ClientTlsCertificate certificate() {
        return certificate;
    }

    /**
     * <p>
     * Whether the policy is enforced. The default is <code>True</code>, if a value isn't specified.
     * </p>
     * 
     * @return Whether the policy is enforced. The default is <code>True</code>, if a value isn't specified.
     */
    public final Boolean enforce() {
        return enforce;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ports property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasPorts() {
        return ports != null && !(ports instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more ports that the policy is enforced for.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPorts} method.
     * </p>
     * 
     * @return One or more ports that the policy is enforced for.
     */
    public final List<Integer> ports() {
        return ports;
    }

    /**
     * <p>
     * A reference to an object that represents a TLS validation context.
     * </p>
     * 
     * @return A reference to an object that represents a TLS validation context.
     */
    public final TlsValidationContext validation() {
        return validation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        hashCode = 31 * hashCode + Objects.hashCode(enforce());
        hashCode = 31 * hashCode + Objects.hashCode(hasPorts() ? ports() : null);
        hashCode = 31 * hashCode + Objects.hashCode(validation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClientPolicyTls)) {
            return false;
        }
        ClientPolicyTls other = (ClientPolicyTls) obj;
        return Objects.equals(certificate(), other.certificate()) && Objects.equals(enforce(), other.enforce())
                && hasPorts() == other.hasPorts() && Objects.equals(ports(), other.ports())
                && Objects.equals(validation(), other.validation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClientPolicyTls").add("Certificate", certificate()).add("Enforce", enforce())
                .add("Ports", hasPorts() ? ports() : null).add("Validation", validation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        case "enforce":
            return Optional.ofNullable(clazz.cast(enforce()));
        case "ports":
            return Optional.ofNullable(clazz.cast(ports()));
        case "validation":
            return Optional.ofNullable(clazz.cast(validation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("certificate", CERTIFICATE_FIELD);
        map.put("enforce", ENFORCE_FIELD);
        map.put("ports", PORTS_FIELD);
        map.put("validation", VALIDATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ClientPolicyTls, T> g) {
        return obj -> g.apply((ClientPolicyTls) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClientPolicyTls> {
        /**
         * <p>
         * A reference to an object that represents a client's TLS certificate.
         * </p>
         * 
         * @param certificate
         *        A reference to an object that represents a client's TLS certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(ClientTlsCertificate certificate);

        /**
         * <p>
         * A reference to an object that represents a client's TLS certificate.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClientTlsCertificate.Builder} avoiding
         * the need to create one manually via {@link ClientTlsCertificate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClientTlsCertificate.Builder#build()} is called immediately and
         * its result is passed to {@link #certificate(ClientTlsCertificate)}.
         * 
         * @param certificate
         *        a consumer that will call methods on {@link ClientTlsCertificate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificate(ClientTlsCertificate)
         */
        default Builder certificate(Consumer<ClientTlsCertificate.Builder> certificate) {
            return certificate(ClientTlsCertificate.builder().applyMutation(certificate).build());
        }

        /**
         * <p>
         * Whether the policy is enforced. The default is <code>True</code>, if a value isn't specified.
         * </p>
         * 
         * @param enforce
         *        Whether the policy is enforced. The default is <code>True</code>, if a value isn't specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enforce(Boolean enforce);

        /**
         * <p>
         * One or more ports that the policy is enforced for.
         * </p>
         * 
         * @param ports
         *        One or more ports that the policy is enforced for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ports(Collection<Integer> ports);

        /**
         * <p>
         * One or more ports that the policy is enforced for.
         * </p>
         * 
         * @param ports
         *        One or more ports that the policy is enforced for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ports(Integer... ports);

        /**
         * <p>
         * A reference to an object that represents a TLS validation context.
         * </p>
         * 
         * @param validation
         *        A reference to an object that represents a TLS validation context.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validation(TlsValidationContext validation);

        /**
         * <p>
         * A reference to an object that represents a TLS validation context.
         * </p>
         * This is a convenience method that creates an instance of the {@link TlsValidationContext.Builder} avoiding
         * the need to create one manually via {@link TlsValidationContext#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TlsValidationContext.Builder#build()} is called immediately and
         * its result is passed to {@link #validation(TlsValidationContext)}.
         * 
         * @param validation
         *        a consumer that will call methods on {@link TlsValidationContext.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validation(TlsValidationContext)
         */
        default Builder validation(Consumer<TlsValidationContext.Builder> validation) {
            return validation(TlsValidationContext.builder().applyMutation(validation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private ClientTlsCertificate certificate;

        private Boolean enforce;

        private List<Integer> ports = DefaultSdkAutoConstructList.getInstance();

        private TlsValidationContext validation;

        private BuilderImpl() {
        }

        private BuilderImpl(ClientPolicyTls model) {
            certificate(model.certificate);
            enforce(model.enforce);
            ports(model.ports);
            validation(model.validation);
        }

        public final ClientTlsCertificate.Builder getCertificate() {
            return certificate != null ? certificate.toBuilder() : null;
        }

        public final void setCertificate(ClientTlsCertificate.BuilderImpl certificate) {
            this.certificate = certificate != null ? certificate.build() : null;
        }

        @Override
        public final Builder certificate(ClientTlsCertificate certificate) {
            this.certificate = certificate;
            return this;
        }

        public final Boolean getEnforce() {
            return enforce;
        }

        public final void setEnforce(Boolean enforce) {
            this.enforce = enforce;
        }

        @Override
        public final Builder enforce(Boolean enforce) {
            this.enforce = enforce;
            return this;
        }

        public final Collection<Integer> getPorts() {
            if (ports instanceof SdkAutoConstructList) {
                return null;
            }
            return ports;
        }

        public final void setPorts(Collection<Integer> ports) {
            this.ports = PortSetCopier.copy(ports);
        }

        @Override
        public final Builder ports(Collection<Integer> ports) {
            this.ports = PortSetCopier.copy(ports);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ports(Integer... ports) {
            ports(Arrays.asList(ports));
            return this;
        }

        public final TlsValidationContext.Builder getValidation() {
            return validation != null ? validation.toBuilder() : null;
        }

        public final void setValidation(TlsValidationContext.BuilderImpl validation) {
            this.validation = validation != null ? validation.build() : null;
        }

        @Override
        public final Builder validation(TlsValidationContext validation) {
            this.validation = validation;
            return this;
        }

        @Override
        public ClientPolicyTls build() {
            return new ClientPolicyTls(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
