/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplify.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the association between a custom domain and an Amplify app.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DomainAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<DomainAssociation.Builder, DomainAssociation> {
    private static final SdkField<String> DOMAIN_ASSOCIATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainAssociationArn").getter(getter(DomainAssociation::domainAssociationArn))
            .setter(setter(Builder::domainAssociationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainAssociationArn").build())
            .build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainName").getter(getter(DomainAssociation::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainName").build()).build();

    private static final SdkField<Boolean> ENABLE_AUTO_SUB_DOMAIN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("enableAutoSubDomain").getter(getter(DomainAssociation::enableAutoSubDomain))
            .setter(setter(Builder::enableAutoSubDomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableAutoSubDomain").build())
            .build();

    private static final SdkField<List<String>> AUTO_SUB_DOMAIN_CREATION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("autoSubDomainCreationPatterns")
            .getter(getter(DomainAssociation::autoSubDomainCreationPatterns))
            .setter(setter(Builder::autoSubDomainCreationPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoSubDomainCreationPatterns")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> AUTO_SUB_DOMAIN_IAM_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("autoSubDomainIAMRole").getter(getter(DomainAssociation::autoSubDomainIAMRole))
            .setter(setter(Builder::autoSubDomainIAMRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoSubDomainIAMRole").build())
            .build();

    private static final SdkField<String> DOMAIN_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainStatus").getter(getter(DomainAssociation::domainStatusAsString))
            .setter(setter(Builder::domainStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainStatus").build()).build();

    private static final SdkField<String> UPDATE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("updateStatus").getter(getter(DomainAssociation::updateStatusAsString))
            .setter(setter(Builder::updateStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updateStatus").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(DomainAssociation::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<String> CERTIFICATE_VERIFICATION_DNS_RECORD_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("certificateVerificationDNSRecord")
            .getter(getter(DomainAssociation::certificateVerificationDNSRecord))
            .setter(setter(Builder::certificateVerificationDNSRecord))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateVerificationDNSRecord")
                    .build()).build();

    private static final SdkField<List<SubDomain>> SUB_DOMAINS_FIELD = SdkField
            .<List<SubDomain>> builder(MarshallingType.LIST)
            .memberName("subDomains")
            .getter(getter(DomainAssociation::subDomains))
            .setter(setter(Builder::subDomains))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subDomains").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SubDomain> builder(MarshallingType.SDK_POJO)
                                            .constructor(SubDomain::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Certificate> CERTIFICATE_FIELD = SdkField.<Certificate> builder(MarshallingType.SDK_POJO)
            .memberName("certificate").getter(getter(DomainAssociation::certificate)).setter(setter(Builder::certificate))
            .constructor(Certificate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_ASSOCIATION_ARN_FIELD,
            DOMAIN_NAME_FIELD, ENABLE_AUTO_SUB_DOMAIN_FIELD, AUTO_SUB_DOMAIN_CREATION_PATTERNS_FIELD,
            AUTO_SUB_DOMAIN_IAM_ROLE_FIELD, DOMAIN_STATUS_FIELD, UPDATE_STATUS_FIELD, STATUS_REASON_FIELD,
            CERTIFICATE_VERIFICATION_DNS_RECORD_FIELD, SUB_DOMAINS_FIELD, CERTIFICATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String domainAssociationArn;

    private final String domainName;

    private final Boolean enableAutoSubDomain;

    private final List<String> autoSubDomainCreationPatterns;

    private final String autoSubDomainIAMRole;

    private final String domainStatus;

    private final String updateStatus;

    private final String statusReason;

    private final String certificateVerificationDNSRecord;

    private final List<SubDomain> subDomains;

    private final Certificate certificate;

    private DomainAssociation(BuilderImpl builder) {
        this.domainAssociationArn = builder.domainAssociationArn;
        this.domainName = builder.domainName;
        this.enableAutoSubDomain = builder.enableAutoSubDomain;
        this.autoSubDomainCreationPatterns = builder.autoSubDomainCreationPatterns;
        this.autoSubDomainIAMRole = builder.autoSubDomainIAMRole;
        this.domainStatus = builder.domainStatus;
        this.updateStatus = builder.updateStatus;
        this.statusReason = builder.statusReason;
        this.certificateVerificationDNSRecord = builder.certificateVerificationDNSRecord;
        this.subDomains = builder.subDomains;
        this.certificate = builder.certificate;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the domain association.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the domain association.
     */
    public final String domainAssociationArn() {
        return domainAssociationArn;
    }

    /**
     * <p>
     * The name of the domain.
     * </p>
     * 
     * @return The name of the domain.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * Enables the automated creation of subdomains for branches.
     * </p>
     * 
     * @return Enables the automated creation of subdomains for branches.
     */
    public final Boolean enableAutoSubDomain() {
        return enableAutoSubDomain;
    }

    /**
     * For responses, this returns true if the service returned a value for the AutoSubDomainCreationPatterns property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAutoSubDomainCreationPatterns() {
        return autoSubDomainCreationPatterns != null && !(autoSubDomainCreationPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Sets branch patterns for automatic subdomain creation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAutoSubDomainCreationPatterns} method.
     * </p>
     * 
     * @return Sets branch patterns for automatic subdomain creation.
     */
    public final List<String> autoSubDomainCreationPatterns() {
        return autoSubDomainCreationPatterns;
    }

    /**
     * <p>
     * The required AWS Identity and Access Management (IAM) service role for the Amazon Resource Name (ARN) for
     * automatically creating subdomains.
     * </p>
     * 
     * @return The required AWS Identity and Access Management (IAM) service role for the Amazon Resource Name (ARN) for
     *         automatically creating subdomains.
     */
    public final String autoSubDomainIAMRole() {
        return autoSubDomainIAMRole;
    }

    /**
     * <p>
     * The current status of the domain association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #domainStatus} will
     * return {@link DomainStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #domainStatusAsString}.
     * </p>
     * 
     * @return The current status of the domain association.
     * @see DomainStatus
     */
    public final DomainStatus domainStatus() {
        return DomainStatus.fromValue(domainStatus);
    }

    /**
     * <p>
     * The current status of the domain association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #domainStatus} will
     * return {@link DomainStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #domainStatusAsString}.
     * </p>
     * 
     * @return The current status of the domain association.
     * @see DomainStatus
     */
    public final String domainStatusAsString() {
        return domainStatus;
    }

    /**
     * <p>
     * The status of the domain update operation that is currently in progress. The following list describes the valid
     * update states.
     * </p>
     * <dl>
     * <dt>REQUESTING_CERTIFICATE</dt>
     * <dd>
     * <p>
     * The certificate is in the process of being updated.
     * </p>
     * </dd>
     * <dt>PENDING_VERIFICATION</dt>
     * <dd>
     * <p>
     * Indicates that an Amplify managed certificate is in the process of being verified. This occurs during the
     * creation of a custom domain or when a custom domain is updated to use a managed certificate.
     * </p>
     * </dd>
     * <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
     * <dd>
     * <p>
     * Indicates that an Amplify custom certificate is in the process of being imported. This occurs during the creation
     * of a custom domain or when a custom domain is updated to use a custom certificate.
     * </p>
     * </dd>
     * <dt>PENDING_DEPLOYMENT</dt>
     * <dd>
     * <p>
     * Indicates that the subdomain or certificate changes are being propagated.
     * </p>
     * </dd>
     * <dt>AWAITING_APP_CNAME</dt>
     * <dd>
     * <p>
     * Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom domain is on
     * Route 53, Amplify handles this for you automatically. For more information about custom domains, see <a
     * href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom domains</a> in
     * the <i>Amplify Hosting User Guide</i>.
     * </p>
     * </dd>
     * <dt>UPDATE_COMPLETE</dt>
     * <dd>
     * <p>
     * The certificate has been associated with a domain.
     * </p>
     * </dd>
     * <dt>UPDATE_FAILED</dt>
     * <dd>
     * <p>
     * The certificate has failed to be provisioned or associated, and there is no existing active certificate to roll
     * back to.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link UpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #updateStatusAsString}.
     * </p>
     * 
     * @return The status of the domain update operation that is currently in progress. The following list describes the
     *         valid update states.</p>
     *         <dl>
     *         <dt>REQUESTING_CERTIFICATE</dt>
     *         <dd>
     *         <p>
     *         The certificate is in the process of being updated.
     *         </p>
     *         </dd>
     *         <dt>PENDING_VERIFICATION</dt>
     *         <dd>
     *         <p>
     *         Indicates that an Amplify managed certificate is in the process of being verified. This occurs during the
     *         creation of a custom domain or when a custom domain is updated to use a managed certificate.
     *         </p>
     *         </dd>
     *         <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
     *         <dd>
     *         <p>
     *         Indicates that an Amplify custom certificate is in the process of being imported. This occurs during the
     *         creation of a custom domain or when a custom domain is updated to use a custom certificate.
     *         </p>
     *         </dd>
     *         <dt>PENDING_DEPLOYMENT</dt>
     *         <dd>
     *         <p>
     *         Indicates that the subdomain or certificate changes are being propagated.
     *         </p>
     *         </dd>
     *         <dt>AWAITING_APP_CNAME</dt>
     *         <dd>
     *         <p>
     *         Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom domain
     *         is on Route 53, Amplify handles this for you automatically. For more information about custom domains,
     *         see <a href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom
     *         domains</a> in the <i>Amplify Hosting User Guide</i>.
     *         </p>
     *         </dd>
     *         <dt>UPDATE_COMPLETE</dt>
     *         <dd>
     *         <p>
     *         The certificate has been associated with a domain.
     *         </p>
     *         </dd>
     *         <dt>UPDATE_FAILED</dt>
     *         <dd>
     *         <p>
     *         The certificate has failed to be provisioned or associated, and there is no existing active certificate
     *         to roll back to.
     *         </p>
     *         </dd>
     * @see UpdateStatus
     */
    public final UpdateStatus updateStatus() {
        return UpdateStatus.fromValue(updateStatus);
    }

    /**
     * <p>
     * The status of the domain update operation that is currently in progress. The following list describes the valid
     * update states.
     * </p>
     * <dl>
     * <dt>REQUESTING_CERTIFICATE</dt>
     * <dd>
     * <p>
     * The certificate is in the process of being updated.
     * </p>
     * </dd>
     * <dt>PENDING_VERIFICATION</dt>
     * <dd>
     * <p>
     * Indicates that an Amplify managed certificate is in the process of being verified. This occurs during the
     * creation of a custom domain or when a custom domain is updated to use a managed certificate.
     * </p>
     * </dd>
     * <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
     * <dd>
     * <p>
     * Indicates that an Amplify custom certificate is in the process of being imported. This occurs during the creation
     * of a custom domain or when a custom domain is updated to use a custom certificate.
     * </p>
     * </dd>
     * <dt>PENDING_DEPLOYMENT</dt>
     * <dd>
     * <p>
     * Indicates that the subdomain or certificate changes are being propagated.
     * </p>
     * </dd>
     * <dt>AWAITING_APP_CNAME</dt>
     * <dd>
     * <p>
     * Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom domain is on
     * Route 53, Amplify handles this for you automatically. For more information about custom domains, see <a
     * href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom domains</a> in
     * the <i>Amplify Hosting User Guide</i>.
     * </p>
     * </dd>
     * <dt>UPDATE_COMPLETE</dt>
     * <dd>
     * <p>
     * The certificate has been associated with a domain.
     * </p>
     * </dd>
     * <dt>UPDATE_FAILED</dt>
     * <dd>
     * <p>
     * The certificate has failed to be provisioned or associated, and there is no existing active certificate to roll
     * back to.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #updateStatus} will
     * return {@link UpdateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #updateStatusAsString}.
     * </p>
     * 
     * @return The status of the domain update operation that is currently in progress. The following list describes the
     *         valid update states.</p>
     *         <dl>
     *         <dt>REQUESTING_CERTIFICATE</dt>
     *         <dd>
     *         <p>
     *         The certificate is in the process of being updated.
     *         </p>
     *         </dd>
     *         <dt>PENDING_VERIFICATION</dt>
     *         <dd>
     *         <p>
     *         Indicates that an Amplify managed certificate is in the process of being verified. This occurs during the
     *         creation of a custom domain or when a custom domain is updated to use a managed certificate.
     *         </p>
     *         </dd>
     *         <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
     *         <dd>
     *         <p>
     *         Indicates that an Amplify custom certificate is in the process of being imported. This occurs during the
     *         creation of a custom domain or when a custom domain is updated to use a custom certificate.
     *         </p>
     *         </dd>
     *         <dt>PENDING_DEPLOYMENT</dt>
     *         <dd>
     *         <p>
     *         Indicates that the subdomain or certificate changes are being propagated.
     *         </p>
     *         </dd>
     *         <dt>AWAITING_APP_CNAME</dt>
     *         <dd>
     *         <p>
     *         Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom domain
     *         is on Route 53, Amplify handles this for you automatically. For more information about custom domains,
     *         see <a href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom
     *         domains</a> in the <i>Amplify Hosting User Guide</i>.
     *         </p>
     *         </dd>
     *         <dt>UPDATE_COMPLETE</dt>
     *         <dd>
     *         <p>
     *         The certificate has been associated with a domain.
     *         </p>
     *         </dd>
     *         <dt>UPDATE_FAILED</dt>
     *         <dd>
     *         <p>
     *         The certificate has failed to be provisioned or associated, and there is no existing active certificate
     *         to roll back to.
     *         </p>
     *         </dd>
     * @see UpdateStatus
     */
    public final String updateStatusAsString() {
        return updateStatus;
    }

    /**
     * <p>
     * Additional information that describes why the domain association is in the current state.
     * </p>
     * 
     * @return Additional information that describes why the domain association is in the current state.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The DNS record for certificate verification.
     * </p>
     * 
     * @return The DNS record for certificate verification.
     */
    public final String certificateVerificationDNSRecord() {
        return certificateVerificationDNSRecord;
    }

    /**
     * For responses, this returns true if the service returned a value for the SubDomains property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSubDomains() {
        return subDomains != null && !(subDomains instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The subdomains for the domain association.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubDomains} method.
     * </p>
     * 
     * @return The subdomains for the domain association.
     */
    public final List<SubDomain> subDomains() {
        return subDomains;
    }

    /**
     * <p>
     * Describes the SSL/TLS certificate for the domain association. This can be your own custom certificate or the
     * default certificate that Amplify provisions for you.
     * </p>
     * <p>
     * If you are updating your domain to use a different certificate, <code>certificate</code> points to the new
     * certificate that is being created instead of the current active certificate. Otherwise, <code>certificate</code>
     * points to the current active certificate.
     * </p>
     * 
     * @return Describes the SSL/TLS certificate for the domain association. This can be your own custom certificate or
     *         the default certificate that Amplify provisions for you.</p>
     *         <p>
     *         If you are updating your domain to use a different certificate, <code>certificate</code> points to the
     *         new certificate that is being created instead of the current active certificate. Otherwise,
     *         <code>certificate</code> points to the current active certificate.
     */
    public final Certificate certificate() {
        return certificate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainAssociationArn());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(enableAutoSubDomain());
        hashCode = 31 * hashCode + Objects.hashCode(hasAutoSubDomainCreationPatterns() ? autoSubDomainCreationPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(autoSubDomainIAMRole());
        hashCode = 31 * hashCode + Objects.hashCode(domainStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updateStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(certificateVerificationDNSRecord());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubDomains() ? subDomains() : null);
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DomainAssociation)) {
            return false;
        }
        DomainAssociation other = (DomainAssociation) obj;
        return Objects.equals(domainAssociationArn(), other.domainAssociationArn())
                && Objects.equals(domainName(), other.domainName())
                && Objects.equals(enableAutoSubDomain(), other.enableAutoSubDomain())
                && hasAutoSubDomainCreationPatterns() == other.hasAutoSubDomainCreationPatterns()
                && Objects.equals(autoSubDomainCreationPatterns(), other.autoSubDomainCreationPatterns())
                && Objects.equals(autoSubDomainIAMRole(), other.autoSubDomainIAMRole())
                && Objects.equals(domainStatusAsString(), other.domainStatusAsString())
                && Objects.equals(updateStatusAsString(), other.updateStatusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(certificateVerificationDNSRecord(), other.certificateVerificationDNSRecord())
                && hasSubDomains() == other.hasSubDomains() && Objects.equals(subDomains(), other.subDomains())
                && Objects.equals(certificate(), other.certificate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("DomainAssociation")
                .add("DomainAssociationArn", domainAssociationArn())
                .add("DomainName", domainName())
                .add("EnableAutoSubDomain", enableAutoSubDomain())
                .add("AutoSubDomainCreationPatterns", hasAutoSubDomainCreationPatterns() ? autoSubDomainCreationPatterns() : null)
                .add("AutoSubDomainIAMRole", autoSubDomainIAMRole()).add("DomainStatus", domainStatusAsString())
                .add("UpdateStatus", updateStatusAsString()).add("StatusReason", statusReason())
                .add("CertificateVerificationDNSRecord", certificateVerificationDNSRecord())
                .add("SubDomains", hasSubDomains() ? subDomains() : null).add("Certificate", certificate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "domainAssociationArn":
            return Optional.ofNullable(clazz.cast(domainAssociationArn()));
        case "domainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "enableAutoSubDomain":
            return Optional.ofNullable(clazz.cast(enableAutoSubDomain()));
        case "autoSubDomainCreationPatterns":
            return Optional.ofNullable(clazz.cast(autoSubDomainCreationPatterns()));
        case "autoSubDomainIAMRole":
            return Optional.ofNullable(clazz.cast(autoSubDomainIAMRole()));
        case "domainStatus":
            return Optional.ofNullable(clazz.cast(domainStatusAsString()));
        case "updateStatus":
            return Optional.ofNullable(clazz.cast(updateStatusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "certificateVerificationDNSRecord":
            return Optional.ofNullable(clazz.cast(certificateVerificationDNSRecord()));
        case "subDomains":
            return Optional.ofNullable(clazz.cast(subDomains()));
        case "certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("domainAssociationArn", DOMAIN_ASSOCIATION_ARN_FIELD);
        map.put("domainName", DOMAIN_NAME_FIELD);
        map.put("enableAutoSubDomain", ENABLE_AUTO_SUB_DOMAIN_FIELD);
        map.put("autoSubDomainCreationPatterns", AUTO_SUB_DOMAIN_CREATION_PATTERNS_FIELD);
        map.put("autoSubDomainIAMRole", AUTO_SUB_DOMAIN_IAM_ROLE_FIELD);
        map.put("domainStatus", DOMAIN_STATUS_FIELD);
        map.put("updateStatus", UPDATE_STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("certificateVerificationDNSRecord", CERTIFICATE_VERIFICATION_DNS_RECORD_FIELD);
        map.put("subDomains", SUB_DOMAINS_FIELD);
        map.put("certificate", CERTIFICATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DomainAssociation, T> g) {
        return obj -> g.apply((DomainAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DomainAssociation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) for the domain association.
         * </p>
         * 
         * @param domainAssociationArn
         *        The Amazon Resource Name (ARN) for the domain association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainAssociationArn(String domainAssociationArn);

        /**
         * <p>
         * The name of the domain.
         * </p>
         * 
         * @param domainName
         *        The name of the domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * Enables the automated creation of subdomains for branches.
         * </p>
         * 
         * @param enableAutoSubDomain
         *        Enables the automated creation of subdomains for branches.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableAutoSubDomain(Boolean enableAutoSubDomain);

        /**
         * <p>
         * Sets branch patterns for automatic subdomain creation.
         * </p>
         * 
         * @param autoSubDomainCreationPatterns
         *        Sets branch patterns for automatic subdomain creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoSubDomainCreationPatterns(Collection<String> autoSubDomainCreationPatterns);

        /**
         * <p>
         * Sets branch patterns for automatic subdomain creation.
         * </p>
         * 
         * @param autoSubDomainCreationPatterns
         *        Sets branch patterns for automatic subdomain creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoSubDomainCreationPatterns(String... autoSubDomainCreationPatterns);

        /**
         * <p>
         * The required AWS Identity and Access Management (IAM) service role for the Amazon Resource Name (ARN) for
         * automatically creating subdomains.
         * </p>
         * 
         * @param autoSubDomainIAMRole
         *        The required AWS Identity and Access Management (IAM) service role for the Amazon Resource Name (ARN)
         *        for automatically creating subdomains.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoSubDomainIAMRole(String autoSubDomainIAMRole);

        /**
         * <p>
         * The current status of the domain association.
         * </p>
         * 
         * @param domainStatus
         *        The current status of the domain association.
         * @see DomainStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DomainStatus
         */
        Builder domainStatus(String domainStatus);

        /**
         * <p>
         * The current status of the domain association.
         * </p>
         * 
         * @param domainStatus
         *        The current status of the domain association.
         * @see DomainStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DomainStatus
         */
        Builder domainStatus(DomainStatus domainStatus);

        /**
         * <p>
         * The status of the domain update operation that is currently in progress. The following list describes the
         * valid update states.
         * </p>
         * <dl>
         * <dt>REQUESTING_CERTIFICATE</dt>
         * <dd>
         * <p>
         * The certificate is in the process of being updated.
         * </p>
         * </dd>
         * <dt>PENDING_VERIFICATION</dt>
         * <dd>
         * <p>
         * Indicates that an Amplify managed certificate is in the process of being verified. This occurs during the
         * creation of a custom domain or when a custom domain is updated to use a managed certificate.
         * </p>
         * </dd>
         * <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
         * <dd>
         * <p>
         * Indicates that an Amplify custom certificate is in the process of being imported. This occurs during the
         * creation of a custom domain or when a custom domain is updated to use a custom certificate.
         * </p>
         * </dd>
         * <dt>PENDING_DEPLOYMENT</dt>
         * <dd>
         * <p>
         * Indicates that the subdomain or certificate changes are being propagated.
         * </p>
         * </dd>
         * <dt>AWAITING_APP_CNAME</dt>
         * <dd>
         * <p>
         * Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom domain is
         * on Route 53, Amplify handles this for you automatically. For more information about custom domains, see <a
         * href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom domains</a>
         * in the <i>Amplify Hosting User Guide</i>.
         * </p>
         * </dd>
         * <dt>UPDATE_COMPLETE</dt>
         * <dd>
         * <p>
         * The certificate has been associated with a domain.
         * </p>
         * </dd>
         * <dt>UPDATE_FAILED</dt>
         * <dd>
         * <p>
         * The certificate has failed to be provisioned or associated, and there is no existing active certificate to
         * roll back to.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param updateStatus
         *        The status of the domain update operation that is currently in progress. The following list describes
         *        the valid update states.</p>
         *        <dl>
         *        <dt>REQUESTING_CERTIFICATE</dt>
         *        <dd>
         *        <p>
         *        The certificate is in the process of being updated.
         *        </p>
         *        </dd>
         *        <dt>PENDING_VERIFICATION</dt>
         *        <dd>
         *        <p>
         *        Indicates that an Amplify managed certificate is in the process of being verified. This occurs during
         *        the creation of a custom domain or when a custom domain is updated to use a managed certificate.
         *        </p>
         *        </dd>
         *        <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
         *        <dd>
         *        <p>
         *        Indicates that an Amplify custom certificate is in the process of being imported. This occurs during
         *        the creation of a custom domain or when a custom domain is updated to use a custom certificate.
         *        </p>
         *        </dd>
         *        <dt>PENDING_DEPLOYMENT</dt>
         *        <dd>
         *        <p>
         *        Indicates that the subdomain or certificate changes are being propagated.
         *        </p>
         *        </dd>
         *        <dt>AWAITING_APP_CNAME</dt>
         *        <dd>
         *        <p>
         *        Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom
         *        domain is on Route 53, Amplify handles this for you automatically. For more information about custom
         *        domains, see <a
         *        href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom
         *        domains</a> in the <i>Amplify Hosting User Guide</i>.
         *        </p>
         *        </dd>
         *        <dt>UPDATE_COMPLETE</dt>
         *        <dd>
         *        <p>
         *        The certificate has been associated with a domain.
         *        </p>
         *        </dd>
         *        <dt>UPDATE_FAILED</dt>
         *        <dd>
         *        <p>
         *        The certificate has failed to be provisioned or associated, and there is no existing active
         *        certificate to roll back to.
         *        </p>
         *        </dd>
         * @see UpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpdateStatus
         */
        Builder updateStatus(String updateStatus);

        /**
         * <p>
         * The status of the domain update operation that is currently in progress. The following list describes the
         * valid update states.
         * </p>
         * <dl>
         * <dt>REQUESTING_CERTIFICATE</dt>
         * <dd>
         * <p>
         * The certificate is in the process of being updated.
         * </p>
         * </dd>
         * <dt>PENDING_VERIFICATION</dt>
         * <dd>
         * <p>
         * Indicates that an Amplify managed certificate is in the process of being verified. This occurs during the
         * creation of a custom domain or when a custom domain is updated to use a managed certificate.
         * </p>
         * </dd>
         * <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
         * <dd>
         * <p>
         * Indicates that an Amplify custom certificate is in the process of being imported. This occurs during the
         * creation of a custom domain or when a custom domain is updated to use a custom certificate.
         * </p>
         * </dd>
         * <dt>PENDING_DEPLOYMENT</dt>
         * <dd>
         * <p>
         * Indicates that the subdomain or certificate changes are being propagated.
         * </p>
         * </dd>
         * <dt>AWAITING_APP_CNAME</dt>
         * <dd>
         * <p>
         * Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom domain is
         * on Route 53, Amplify handles this for you automatically. For more information about custom domains, see <a
         * href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom domains</a>
         * in the <i>Amplify Hosting User Guide</i>.
         * </p>
         * </dd>
         * <dt>UPDATE_COMPLETE</dt>
         * <dd>
         * <p>
         * The certificate has been associated with a domain.
         * </p>
         * </dd>
         * <dt>UPDATE_FAILED</dt>
         * <dd>
         * <p>
         * The certificate has failed to be provisioned or associated, and there is no existing active certificate to
         * roll back to.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param updateStatus
         *        The status of the domain update operation that is currently in progress. The following list describes
         *        the valid update states.</p>
         *        <dl>
         *        <dt>REQUESTING_CERTIFICATE</dt>
         *        <dd>
         *        <p>
         *        The certificate is in the process of being updated.
         *        </p>
         *        </dd>
         *        <dt>PENDING_VERIFICATION</dt>
         *        <dd>
         *        <p>
         *        Indicates that an Amplify managed certificate is in the process of being verified. This occurs during
         *        the creation of a custom domain or when a custom domain is updated to use a managed certificate.
         *        </p>
         *        </dd>
         *        <dt>IMPORTING_CUSTOM_CERTIFICATE</dt>
         *        <dd>
         *        <p>
         *        Indicates that an Amplify custom certificate is in the process of being imported. This occurs during
         *        the creation of a custom domain or when a custom domain is updated to use a custom certificate.
         *        </p>
         *        </dd>
         *        <dt>PENDING_DEPLOYMENT</dt>
         *        <dd>
         *        <p>
         *        Indicates that the subdomain or certificate changes are being propagated.
         *        </p>
         *        </dd>
         *        <dt>AWAITING_APP_CNAME</dt>
         *        <dd>
         *        <p>
         *        Amplify is waiting for CNAME records corresponding to subdomains to be propagated. If your custom
         *        domain is on Route 53, Amplify handles this for you automatically. For more information about custom
         *        domains, see <a
         *        href="https://docs.aws.amazon.com/amplify/latest/userguide/custom-domains.html">Setting up custom
         *        domains</a> in the <i>Amplify Hosting User Guide</i>.
         *        </p>
         *        </dd>
         *        <dt>UPDATE_COMPLETE</dt>
         *        <dd>
         *        <p>
         *        The certificate has been associated with a domain.
         *        </p>
         *        </dd>
         *        <dt>UPDATE_FAILED</dt>
         *        <dd>
         *        <p>
         *        The certificate has failed to be provisioned or associated, and there is no existing active
         *        certificate to roll back to.
         *        </p>
         *        </dd>
         * @see UpdateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpdateStatus
         */
        Builder updateStatus(UpdateStatus updateStatus);

        /**
         * <p>
         * Additional information that describes why the domain association is in the current state.
         * </p>
         * 
         * @param statusReason
         *        Additional information that describes why the domain association is in the current state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The DNS record for certificate verification.
         * </p>
         * 
         * @param certificateVerificationDNSRecord
         *        The DNS record for certificate verification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateVerificationDNSRecord(String certificateVerificationDNSRecord);

        /**
         * <p>
         * The subdomains for the domain association.
         * </p>
         * 
         * @param subDomains
         *        The subdomains for the domain association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subDomains(Collection<SubDomain> subDomains);

        /**
         * <p>
         * The subdomains for the domain association.
         * </p>
         * 
         * @param subDomains
         *        The subdomains for the domain association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subDomains(SubDomain... subDomains);

        /**
         * <p>
         * The subdomains for the domain association.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.amplify.model.SubDomain.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.amplify.model.SubDomain#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.amplify.model.SubDomain.Builder#build()} is called immediately and its
         * result is passed to {@link #subDomains(List<SubDomain>)}.
         * 
         * @param subDomains
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.amplify.model.SubDomain.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subDomains(java.util.Collection<SubDomain>)
         */
        Builder subDomains(Consumer<SubDomain.Builder>... subDomains);

        /**
         * <p>
         * Describes the SSL/TLS certificate for the domain association. This can be your own custom certificate or the
         * default certificate that Amplify provisions for you.
         * </p>
         * <p>
         * If you are updating your domain to use a different certificate, <code>certificate</code> points to the new
         * certificate that is being created instead of the current active certificate. Otherwise,
         * <code>certificate</code> points to the current active certificate.
         * </p>
         * 
         * @param certificate
         *        Describes the SSL/TLS certificate for the domain association. This can be your own custom certificate
         *        or the default certificate that Amplify provisions for you.</p>
         *        <p>
         *        If you are updating your domain to use a different certificate, <code>certificate</code> points to the
         *        new certificate that is being created instead of the current active certificate. Otherwise,
         *        <code>certificate</code> points to the current active certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(Certificate certificate);

        /**
         * <p>
         * Describes the SSL/TLS certificate for the domain association. This can be your own custom certificate or the
         * default certificate that Amplify provisions for you.
         * </p>
         * <p>
         * If you are updating your domain to use a different certificate, <code>certificate</code> points to the new
         * certificate that is being created instead of the current active certificate. Otherwise,
         * <code>certificate</code> points to the current active certificate.
         * </p>
         * This is a convenience method that creates an instance of the {@link Certificate.Builder} avoiding the need to
         * create one manually via {@link Certificate#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Certificate.Builder#build()} is called immediately and its result
         * is passed to {@link #certificate(Certificate)}.
         * 
         * @param certificate
         *        a consumer that will call methods on {@link Certificate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificate(Certificate)
         */
        default Builder certificate(Consumer<Certificate.Builder> certificate) {
            return certificate(Certificate.builder().applyMutation(certificate).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String domainAssociationArn;

        private String domainName;

        private Boolean enableAutoSubDomain;

        private List<String> autoSubDomainCreationPatterns = DefaultSdkAutoConstructList.getInstance();

        private String autoSubDomainIAMRole;

        private String domainStatus;

        private String updateStatus;

        private String statusReason;

        private String certificateVerificationDNSRecord;

        private List<SubDomain> subDomains = DefaultSdkAutoConstructList.getInstance();

        private Certificate certificate;

        private BuilderImpl() {
        }

        private BuilderImpl(DomainAssociation model) {
            domainAssociationArn(model.domainAssociationArn);
            domainName(model.domainName);
            enableAutoSubDomain(model.enableAutoSubDomain);
            autoSubDomainCreationPatterns(model.autoSubDomainCreationPatterns);
            autoSubDomainIAMRole(model.autoSubDomainIAMRole);
            domainStatus(model.domainStatus);
            updateStatus(model.updateStatus);
            statusReason(model.statusReason);
            certificateVerificationDNSRecord(model.certificateVerificationDNSRecord);
            subDomains(model.subDomains);
            certificate(model.certificate);
        }

        public final String getDomainAssociationArn() {
            return domainAssociationArn;
        }

        public final void setDomainAssociationArn(String domainAssociationArn) {
            this.domainAssociationArn = domainAssociationArn;
        }

        @Override
        public final Builder domainAssociationArn(String domainAssociationArn) {
            this.domainAssociationArn = domainAssociationArn;
            return this;
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final Boolean getEnableAutoSubDomain() {
            return enableAutoSubDomain;
        }

        public final void setEnableAutoSubDomain(Boolean enableAutoSubDomain) {
            this.enableAutoSubDomain = enableAutoSubDomain;
        }

        @Override
        public final Builder enableAutoSubDomain(Boolean enableAutoSubDomain) {
            this.enableAutoSubDomain = enableAutoSubDomain;
            return this;
        }

        public final Collection<String> getAutoSubDomainCreationPatterns() {
            if (autoSubDomainCreationPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return autoSubDomainCreationPatterns;
        }

        public final void setAutoSubDomainCreationPatterns(Collection<String> autoSubDomainCreationPatterns) {
            this.autoSubDomainCreationPatterns = AutoSubDomainCreationPatternsCopier.copy(autoSubDomainCreationPatterns);
        }

        @Override
        public final Builder autoSubDomainCreationPatterns(Collection<String> autoSubDomainCreationPatterns) {
            this.autoSubDomainCreationPatterns = AutoSubDomainCreationPatternsCopier.copy(autoSubDomainCreationPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoSubDomainCreationPatterns(String... autoSubDomainCreationPatterns) {
            autoSubDomainCreationPatterns(Arrays.asList(autoSubDomainCreationPatterns));
            return this;
        }

        public final String getAutoSubDomainIAMRole() {
            return autoSubDomainIAMRole;
        }

        public final void setAutoSubDomainIAMRole(String autoSubDomainIAMRole) {
            this.autoSubDomainIAMRole = autoSubDomainIAMRole;
        }

        @Override
        public final Builder autoSubDomainIAMRole(String autoSubDomainIAMRole) {
            this.autoSubDomainIAMRole = autoSubDomainIAMRole;
            return this;
        }

        public final String getDomainStatus() {
            return domainStatus;
        }

        public final void setDomainStatus(String domainStatus) {
            this.domainStatus = domainStatus;
        }

        @Override
        public final Builder domainStatus(String domainStatus) {
            this.domainStatus = domainStatus;
            return this;
        }

        @Override
        public final Builder domainStatus(DomainStatus domainStatus) {
            this.domainStatus(domainStatus == null ? null : domainStatus.toString());
            return this;
        }

        public final String getUpdateStatus() {
            return updateStatus;
        }

        public final void setUpdateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
        }

        @Override
        public final Builder updateStatus(String updateStatus) {
            this.updateStatus = updateStatus;
            return this;
        }

        @Override
        public final Builder updateStatus(UpdateStatus updateStatus) {
            this.updateStatus(updateStatus == null ? null : updateStatus.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final String getCertificateVerificationDNSRecord() {
            return certificateVerificationDNSRecord;
        }

        public final void setCertificateVerificationDNSRecord(String certificateVerificationDNSRecord) {
            this.certificateVerificationDNSRecord = certificateVerificationDNSRecord;
        }

        @Override
        public final Builder certificateVerificationDNSRecord(String certificateVerificationDNSRecord) {
            this.certificateVerificationDNSRecord = certificateVerificationDNSRecord;
            return this;
        }

        public final List<SubDomain.Builder> getSubDomains() {
            List<SubDomain.Builder> result = SubDomainsCopier.copyToBuilder(this.subDomains);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSubDomains(Collection<SubDomain.BuilderImpl> subDomains) {
            this.subDomains = SubDomainsCopier.copyFromBuilder(subDomains);
        }

        @Override
        public final Builder subDomains(Collection<SubDomain> subDomains) {
            this.subDomains = SubDomainsCopier.copy(subDomains);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subDomains(SubDomain... subDomains) {
            subDomains(Arrays.asList(subDomains));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subDomains(Consumer<SubDomain.Builder>... subDomains) {
            subDomains(Stream.of(subDomains).map(c -> SubDomain.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Certificate.Builder getCertificate() {
            return certificate != null ? certificate.toBuilder() : null;
        }

        public final void setCertificate(Certificate.BuilderImpl certificate) {
            this.certificate = certificate != null ? certificate.build() : null;
        }

        @Override
        public final Builder certificate(Certificate certificate) {
            this.certificate = certificate;
            return this;
        }

        @Override
        public DomainAssociation build() {
            return new DomainAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
