/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediastore;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediastore.internal.MediaStoreServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediastore.model.ContainerInUseException;
import software.amazon.awssdk.services.mediastore.model.ContainerNotFoundException;
import software.amazon.awssdk.services.mediastore.model.CorsPolicyNotFoundException;
import software.amazon.awssdk.services.mediastore.model.CreateContainerRequest;
import software.amazon.awssdk.services.mediastore.model.CreateContainerResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteContainerResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DeleteMetricPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.DeleteMetricPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.DescribeContainerRequest;
import software.amazon.awssdk.services.mediastore.model.DescribeContainerResponse;
import software.amazon.awssdk.services.mediastore.model.GetContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.GetMetricPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.GetMetricPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediastore.model.LimitExceededException;
import software.amazon.awssdk.services.mediastore.model.ListContainersRequest;
import software.amazon.awssdk.services.mediastore.model.ListContainersResponse;
import software.amazon.awssdk.services.mediastore.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediastore.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediastore.model.MediaStoreException;
import software.amazon.awssdk.services.mediastore.model.PolicyNotFoundException;
import software.amazon.awssdk.services.mediastore.model.PutContainerPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutContainerPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutCorsPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutCorsPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutLifecyclePolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutLifecyclePolicyResponse;
import software.amazon.awssdk.services.mediastore.model.PutMetricPolicyRequest;
import software.amazon.awssdk.services.mediastore.model.PutMetricPolicyResponse;
import software.amazon.awssdk.services.mediastore.model.StartAccessLoggingRequest;
import software.amazon.awssdk.services.mediastore.model.StartAccessLoggingResponse;
import software.amazon.awssdk.services.mediastore.model.StopAccessLoggingRequest;
import software.amazon.awssdk.services.mediastore.model.StopAccessLoggingResponse;
import software.amazon.awssdk.services.mediastore.model.TagResourceRequest;
import software.amazon.awssdk.services.mediastore.model.TagResourceResponse;
import software.amazon.awssdk.services.mediastore.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediastore.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediastore.transform.CreateContainerRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteContainerPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteContainerRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteCorsPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DeleteMetricPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.DescribeContainerRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetContainerPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetCorsPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.GetMetricPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.ListContainersRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutContainerPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutCorsPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutLifecyclePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.PutMetricPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.StartAccessLoggingRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.StopAccessLoggingRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediastore.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaStoreAsyncClient}.
 *
 * @see MediaStoreAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaStoreAsyncClient implements MediaStoreAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaStoreAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MediaStoreServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMediaStoreAsyncClient(MediaStoreServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a storage container to hold objects. A container is similar to a bucket in the Amazon S3 service.
     * </p>
     *
     * @param createContainerRequest
     * @return A Java Future containing the result of the CreateContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>LimitExceededException A service limit has been exceeded.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.CreateContainer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/CreateContainer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateContainerResponse> createContainer(CreateContainerRequest createContainerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createContainerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createContainerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateContainer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateContainerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateContainerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateContainerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateContainerRequest, CreateContainerResponse>()
                            .withOperationName("CreateContainer")
                            .withMarshaller(new CreateContainerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createContainerRequest));
            CompletableFuture<CreateContainerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified container. Before you make a <code>DeleteContainer</code> request, delete any objects in
     * the container or in any folders in the container. You can delete only empty containers.
     * </p>
     *
     * @param deleteContainerRequest
     * @return A Java Future containing the result of the DeleteContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteContainer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteContainerResponse> deleteContainer(DeleteContainerRequest deleteContainerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteContainerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContainerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContainer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteContainerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteContainerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteContainerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteContainerRequest, DeleteContainerResponse>()
                            .withOperationName("DeleteContainer")
                            .withMarshaller(new DeleteContainerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteContainerRequest));
            CompletableFuture<DeleteContainerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the access policy that is associated with the specified container.
     * </p>
     *
     * @param deleteContainerPolicyRequest
     * @return A Java Future containing the result of the DeleteContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteContainerPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteContainerPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteContainerPolicyResponse> deleteContainerPolicy(
            DeleteContainerPolicyRequest deleteContainerPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteContainerPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteContainerPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteContainerPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteContainerPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteContainerPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteContainerPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteContainerPolicyRequest, DeleteContainerPolicyResponse>()
                            .withOperationName("DeleteContainerPolicy")
                            .withMarshaller(new DeleteContainerPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteContainerPolicyRequest));
            CompletableFuture<DeleteContainerPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:DeleteCorsPolicy</code> action.
     * The container owner has this permission by default and can grant this permission to others.
     * </p>
     *
     * @param deleteCorsPolicyRequest
     * @return A Java Future containing the result of the DeleteCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>CorsPolicyNotFoundException The CORS policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteCorsPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteCorsPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCorsPolicyResponse> deleteCorsPolicy(DeleteCorsPolicyRequest deleteCorsPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCorsPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCorsPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCorsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCorsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCorsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCorsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCorsPolicyRequest, DeleteCorsPolicyResponse>()
                            .withOperationName("DeleteCorsPolicy")
                            .withMarshaller(new DeleteCorsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCorsPolicyRequest));
            CompletableFuture<DeleteCorsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an object lifecycle policy from a container. It takes up to 20 minutes for the change to take effect.
     * </p>
     *
     * @param deleteLifecyclePolicyRequest
     * @return A Java Future containing the result of the DeleteLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteLifecyclePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLifecyclePolicyResponse> deleteLifecyclePolicy(
            DeleteLifecyclePolicyRequest deleteLifecyclePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLifecyclePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLifecyclePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLifecyclePolicyRequest, DeleteLifecyclePolicyResponse>()
                            .withOperationName("DeleteLifecyclePolicy")
                            .withMarshaller(new DeleteLifecyclePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLifecyclePolicyRequest));
            CompletableFuture<DeleteLifecyclePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the metric policy that is associated with the specified container. If there is no metric policy
     * associated with the container, MediaStore doesn't send metrics to CloudWatch.
     * </p>
     *
     * @param deleteMetricPolicyRequest
     * @return A Java Future containing the result of the DeleteMetricPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DeleteMetricPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DeleteMetricPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMetricPolicyResponse> deleteMetricPolicy(DeleteMetricPolicyRequest deleteMetricPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMetricPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMetricPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMetricPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMetricPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMetricPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMetricPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMetricPolicyRequest, DeleteMetricPolicyResponse>()
                            .withOperationName("DeleteMetricPolicy")
                            .withMarshaller(new DeleteMetricPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMetricPolicyRequest));
            CompletableFuture<DeleteMetricPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the properties of the requested container. This request is commonly used to retrieve the endpoint of a
     * container. An endpoint is a value assigned by the service when a new container is created. A container's endpoint
     * does not change after it has been assigned. The <code>DescribeContainer</code> request returns a single
     * <code>Container</code> object based on <code>ContainerName</code>. To return all <code>Container</code> objects
     * that are associated with a specified AWS account, use <a>ListContainers</a>.
     * </p>
     *
     * @param describeContainerRequest
     * @return A Java Future containing the result of the DescribeContainer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.DescribeContainer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/DescribeContainer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeContainerResponse> describeContainer(DescribeContainerRequest describeContainerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeContainerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContainerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContainer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeContainerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeContainerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeContainerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeContainerRequest, DescribeContainerResponse>()
                            .withOperationName("DescribeContainer")
                            .withMarshaller(new DescribeContainerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeContainerRequest));
            CompletableFuture<DescribeContainerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the access policy for the specified container. For information about the data that is included in an
     * access policy, see the <a href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management
     * User Guide</a>.
     * </p>
     *
     * @param getContainerPolicyRequest
     * @return A Java Future containing the result of the GetContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetContainerPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetContainerPolicyResponse> getContainerPolicy(GetContainerPolicyRequest getContainerPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getContainerPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContainerPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContainerPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetContainerPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetContainerPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetContainerPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetContainerPolicyRequest, GetContainerPolicyResponse>()
                            .withOperationName("GetContainerPolicy")
                            .withMarshaller(new GetContainerPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getContainerPolicyRequest));
            CompletableFuture<GetContainerPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the cross-origin resource sharing (CORS) configuration information that is set for the container.
     * </p>
     * <p>
     * To use this operation, you must have permission to perform the <code>MediaStore:GetCorsPolicy</code> action. By
     * default, the container owner has this permission and can grant it to others.
     * </p>
     *
     * @param getCorsPolicyRequest
     * @return A Java Future containing the result of the GetCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>CorsPolicyNotFoundException The CORS policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetCorsPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCorsPolicyResponse> getCorsPolicy(GetCorsPolicyRequest getCorsPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCorsPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCorsPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCorsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCorsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetCorsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCorsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCorsPolicyRequest, GetCorsPolicyResponse>()
                            .withOperationName("GetCorsPolicy")
                            .withMarshaller(new GetCorsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCorsPolicyRequest));
            CompletableFuture<GetCorsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the object lifecycle policy that is assigned to a container.
     * </p>
     *
     * @param getLifecyclePolicyRequest
     * @return A Java Future containing the result of the GetLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLifecyclePolicyResponse> getLifecyclePolicy(GetLifecyclePolicyRequest getLifecyclePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLifecyclePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLifecyclePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLifecyclePolicyRequest, GetLifecyclePolicyResponse>()
                            .withOperationName("GetLifecyclePolicy")
                            .withMarshaller(new GetLifecyclePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLifecyclePolicyRequest));
            CompletableFuture<GetLifecyclePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the metric policy for the specified container.
     * </p>
     *
     * @param getMetricPolicyRequest
     * @return A Java Future containing the result of the GetMetricPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>PolicyNotFoundException The policy that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.GetMetricPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/GetMetricPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMetricPolicyResponse> getMetricPolicy(GetMetricPolicyRequest getMetricPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMetricPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMetricPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMetricPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMetricPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMetricPolicyRequest, GetMetricPolicyResponse>()
                            .withOperationName("GetMetricPolicy")
                            .withMarshaller(new GetMetricPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMetricPolicyRequest));
            CompletableFuture<GetMetricPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the properties of all containers in AWS Elemental MediaStore.
     * </p>
     * <p>
     * You can query to receive all the containers in one response. Or you can include the <code>MaxResults</code>
     * parameter to receive a limited number of containers in each response. In this case, the response includes a
     * token. To get the next set of containers, send the command again, this time with the <code>NextToken</code>
     * parameter (with the returned token as its value). The next set of responses appears, with a token if there are
     * still more containers to receive.
     * </p>
     * <p>
     * See also <a>DescribeContainer</a>, which gets the properties of one container.
     * </p>
     *
     * @param listContainersRequest
     * @return A Java Future containing the result of the ListContainers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListContainers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListContainers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListContainersResponse> listContainers(ListContainersRequest listContainersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContainersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContainersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContainers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContainersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListContainersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContainersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContainersRequest, ListContainersResponse>()
                            .withOperationName("ListContainers")
                            .withMarshaller(new ListContainersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listContainersRequest));
            CompletableFuture<ListContainersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the tags assigned to the specified container.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an access policy for the specified container to restrict the users and clients that can access it. For
     * information about the data that is included in an access policy, see the <a
     * href="https://aws.amazon.com/documentation/iam/">AWS Identity and Access Management User Guide</a>.
     * </p>
     * <p>
     * For this release of the REST API, you can create only one policy for a container. If you enter
     * <code>PutContainerPolicy</code> twice, the second command modifies the existing policy.
     * </p>
     *
     * @param putContainerPolicyRequest
     * @return A Java Future containing the result of the PutContainerPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutContainerPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutContainerPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutContainerPolicyResponse> putContainerPolicy(PutContainerPolicyRequest putContainerPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putContainerPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putContainerPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutContainerPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutContainerPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutContainerPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutContainerPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutContainerPolicyRequest, PutContainerPolicyResponse>()
                            .withOperationName("PutContainerPolicy")
                            .withMarshaller(new PutContainerPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putContainerPolicyRequest));
            CompletableFuture<PutContainerPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the cross-origin resource sharing (CORS) configuration on a container so that the container can service
     * cross-origin requests. For example, you might want to enable a request whose origin is http://www.example.com to
     * access your AWS Elemental MediaStore container at my.example.container.com by using the browser's XMLHttpRequest
     * capability.
     * </p>
     * <p>
     * To enable CORS on a container, you attach a CORS policy to the container. In the CORS policy, you configure rules
     * that identify origins and the HTTP methods that can be executed on your container. The policy can contain up to
     * 398,000 characters. You can add up to 100 rules to a CORS policy. If more than one rule applies, the service uses
     * the first applicable rule listed.
     * </p>
     * <p>
     * To learn more about CORS, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/cors-policy.html">Cross-Origin Resource Sharing (CORS) in
     * AWS Elemental MediaStore</a>.
     * </p>
     *
     * @param putCorsPolicyRequest
     * @return A Java Future containing the result of the PutCorsPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutCorsPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutCorsPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutCorsPolicyResponse> putCorsPolicy(PutCorsPolicyRequest putCorsPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putCorsPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCorsPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCorsPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutCorsPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutCorsPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutCorsPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutCorsPolicyRequest, PutCorsPolicyResponse>()
                            .withOperationName("PutCorsPolicy")
                            .withMarshaller(new PutCorsPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putCorsPolicyRequest));
            CompletableFuture<PutCorsPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Writes an object lifecycle policy to a container. If the container already has an object lifecycle policy, the
     * service replaces the existing policy with the new policy. It takes up to 20 minutes for the change to take
     * effect.
     * </p>
     * <p>
     * For information about how to construct an object lifecycle policy, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/policies-object-lifecycle-components.html">Components of
     * an Object Lifecycle Policy</a>.
     * </p>
     *
     * @param putLifecyclePolicyRequest
     * @return A Java Future containing the result of the PutLifecyclePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutLifecyclePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutLifecyclePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLifecyclePolicyResponse> putLifecyclePolicy(PutLifecyclePolicyRequest putLifecyclePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLifecyclePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLifecyclePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLifecyclePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLifecyclePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLifecyclePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLifecyclePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLifecyclePolicyRequest, PutLifecyclePolicyResponse>()
                            .withOperationName("PutLifecyclePolicy")
                            .withMarshaller(new PutLifecyclePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putLifecyclePolicyRequest));
            CompletableFuture<PutLifecyclePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The metric policy that you want to add to the container. A metric policy allows AWS Elemental MediaStore to send
     * metrics to Amazon CloudWatch. It takes up to 20 minutes for the new policy to take effect.
     * </p>
     *
     * @param putMetricPolicyRequest
     * @return A Java Future containing the result of the PutMetricPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.PutMetricPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/PutMetricPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutMetricPolicyResponse> putMetricPolicy(PutMetricPolicyRequest putMetricPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMetricPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutMetricPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutMetricPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutMetricPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutMetricPolicyRequest, PutMetricPolicyResponse>()
                            .withOperationName("PutMetricPolicy")
                            .withMarshaller(new PutMetricPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putMetricPolicyRequest));
            CompletableFuture<PutMetricPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts access logging on the specified container. When you enable access logging on a container, MediaStore
     * delivers access logs for objects stored in that container to Amazon CloudWatch Logs.
     * </p>
     *
     * @param startAccessLoggingRequest
     * @return A Java Future containing the result of the StartAccessLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.StartAccessLogging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StartAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartAccessLoggingResponse> startAccessLogging(StartAccessLoggingRequest startAccessLoggingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startAccessLoggingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAccessLoggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartAccessLogging");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartAccessLoggingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartAccessLoggingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartAccessLoggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartAccessLoggingRequest, StartAccessLoggingResponse>()
                            .withOperationName("StartAccessLogging")
                            .withMarshaller(new StartAccessLoggingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startAccessLoggingRequest));
            CompletableFuture<StartAccessLoggingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops access logging on the specified container. When you stop access logging on a container, MediaStore stops
     * sending access logs to Amazon CloudWatch Logs. These access logs are not saved and are not retrievable.
     * </p>
     *
     * @param stopAccessLoggingRequest
     * @return A Java Future containing the result of the StopAccessLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.StopAccessLogging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/StopAccessLogging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopAccessLoggingResponse> stopAccessLogging(StopAccessLoggingRequest stopAccessLoggingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopAccessLoggingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopAccessLoggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopAccessLogging");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopAccessLoggingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopAccessLoggingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopAccessLoggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopAccessLoggingRequest, StopAccessLoggingResponse>()
                            .withOperationName("StopAccessLogging")
                            .withMarshaller(new StopAccessLoggingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopAccessLoggingRequest));
            CompletableFuture<StopAccessLoggingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to the specified AWS Elemental MediaStore container. Tags are key:value pairs that you can associate
     * with AWS resources. For example, the tag key might be "customer" and the tag value might be "companyA." You can
     * specify one or more tags to add to each container. You can add up to 50 tags to each container. For more
     * information about tagging, including naming and usage conventions, see <a
     * href="https://docs.aws.amazon.com/mediastore/latest/ug/tagging.html">Tagging Resources in MediaStore</a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified container. You can specify one or more tags to remove.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ContainerInUseException The container that you specified in the request already exists or is being
     *         updated.</li>
     *         <li>ContainerNotFoundException The container that you specified in the request does not exist.</li>
     *         <li>InternalServerErrorException The service is temporarily unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaStoreException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaStoreAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediastore-2017-09-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaStore");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MediaStoreServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaStoreException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContainerInUseException")
                                .exceptionBuilderSupplier(ContainerInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CorsPolicyNotFoundException")
                                .exceptionBuilderSupplier(CorsPolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContainerNotFoundException")
                                .exceptionBuilderSupplier(ContainerNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyNotFoundException")
                                .exceptionBuilderSupplier(PolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        MediaStoreServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = MediaStoreServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
