/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconnect.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.mediaconnect.MediaConnectAsyncClient;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowRequest;
import software.amazon.awssdk.services.mediaconnect.model.DescribeFlowResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface MediaConnectAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeFlowRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowActive(DescribeFlowRequest describeFlowRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFlowRequest#builder()}
     *
     * @param describeFlowRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowActive(
            Consumer<DescribeFlowRequest.Builder> describeFlowRequest) {
        return waitUntilFlowActive(DescribeFlowRequest.builder().applyMutation(describeFlowRequest).build());
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowActive} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeFlowRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowActive(DescribeFlowRequest describeFlowRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowActive} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFlowRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowActive(
            Consumer<DescribeFlowRequest.Builder> describeFlowRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFlowActive(DescribeFlowRequest.builder().applyMutation(describeFlowRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeFlowRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowDeleted(DescribeFlowRequest describeFlowRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFlowRequest#builder()}
     *
     * @param describeFlowRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowDeleted(
            Consumer<DescribeFlowRequest.Builder> describeFlowRequest) {
        return waitUntilFlowDeleted(DescribeFlowRequest.builder().applyMutation(describeFlowRequest).build());
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeFlowRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowDeleted(DescribeFlowRequest describeFlowRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowDeleted} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFlowRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowDeleted(
            Consumer<DescribeFlowRequest.Builder> describeFlowRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFlowDeleted(DescribeFlowRequest.builder().applyMutation(describeFlowRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowStandby} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeFlowRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowStandby(DescribeFlowRequest describeFlowRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowStandby} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeFlowRequest#builder()}
     *
     * @param describeFlowRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowStandby(
            Consumer<DescribeFlowRequest.Builder> describeFlowRequest) {
        return waitUntilFlowStandby(DescribeFlowRequest.builder().applyMutation(describeFlowRequest).build());
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowStandby} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeFlowRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowStandby(DescribeFlowRequest describeFlowRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link MediaConnectAsyncClient#describeFlow} API until the desired condition {@code FlowStandby} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeFlowRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeFlowResponse>> waitUntilFlowStandby(
            Consumer<DescribeFlowRequest.Builder> describeFlowRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilFlowStandby(DescribeFlowRequest.builder().applyMutation(describeFlowRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link MediaConnectAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultMediaConnectAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link MediaConnectAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link MediaConnectAsyncClient} will be created to poll resources. It is recommended to share a
     * single instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link MediaConnectAsyncWaiter}
     */
    static MediaConnectAsyncWaiter create() {
        return DefaultMediaConnectAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link MediaConnectAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(MediaConnectAsyncClient client);

        /**
         * Builds an instance of {@link MediaConnectAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link MediaConnectAsyncWaiter}
         */
        MediaConnectAsyncWaiter build();
    }
}
