/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntime;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.AsyncStreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lexruntime.internal.LexRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.lexruntime.model.BadGatewayException;
import software.amazon.awssdk.services.lexruntime.model.BadRequestException;
import software.amazon.awssdk.services.lexruntime.model.ConflictException;
import software.amazon.awssdk.services.lexruntime.model.DeleteSessionRequest;
import software.amazon.awssdk.services.lexruntime.model.DeleteSessionResponse;
import software.amazon.awssdk.services.lexruntime.model.DependencyFailedException;
import software.amazon.awssdk.services.lexruntime.model.GetSessionRequest;
import software.amazon.awssdk.services.lexruntime.model.GetSessionResponse;
import software.amazon.awssdk.services.lexruntime.model.InternalFailureException;
import software.amazon.awssdk.services.lexruntime.model.LexRuntimeException;
import software.amazon.awssdk.services.lexruntime.model.LexRuntimeRequest;
import software.amazon.awssdk.services.lexruntime.model.LimitExceededException;
import software.amazon.awssdk.services.lexruntime.model.LoopDetectedException;
import software.amazon.awssdk.services.lexruntime.model.NotAcceptableException;
import software.amazon.awssdk.services.lexruntime.model.NotFoundException;
import software.amazon.awssdk.services.lexruntime.model.PostContentRequest;
import software.amazon.awssdk.services.lexruntime.model.PostContentResponse;
import software.amazon.awssdk.services.lexruntime.model.PostTextRequest;
import software.amazon.awssdk.services.lexruntime.model.PostTextResponse;
import software.amazon.awssdk.services.lexruntime.model.PutSessionRequest;
import software.amazon.awssdk.services.lexruntime.model.PutSessionResponse;
import software.amazon.awssdk.services.lexruntime.model.RequestTimeoutException;
import software.amazon.awssdk.services.lexruntime.model.UnsupportedMediaTypeException;
import software.amazon.awssdk.services.lexruntime.transform.DeleteSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.PostContentRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.PostTextRequestMarshaller;
import software.amazon.awssdk.services.lexruntime.transform.PutSessionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link LexRuntimeAsyncClient}.
 *
 * @see LexRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLexRuntimeAsyncClient implements LexRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLexRuntimeAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LexRuntimeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLexRuntimeAsyncClient(LexRuntimeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Removes session information for a specified bot, alias, and user ID.
     * </p>
     *
     * @param deleteSessionRequest
     * @return A Java Future containing the result of the DeleteSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (such as the Amazon Lex bot or an alias) that is referred to is not
     *         found.</li>
     *         <li>BadRequestException Request validation failed, there is no usable message in the context, or the bot
     *         build failed, is still in progress, or contains unbuilt changes.</li>
     *         <li>LimitExceededException Exceeded a limit.</li>
     *         <li>InternalFailureException Internal service error. Retry the call.</li>
     *         <li>ConflictException Two clients are using the same AWS account, Amazon Lex bot, and user ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeAsyncClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/DeleteSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSessionResponse> deleteSession(DeleteSessionRequest deleteSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSessionRequest, DeleteSessionResponse>()
                            .withOperationName("DeleteSession")
                            .withMarshaller(new DeleteSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSessionRequest));
            CompletableFuture<DeleteSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns session information for a specified bot, alias, and user ID.
     * </p>
     *
     * @param getSessionRequest
     * @return A Java Future containing the result of the GetSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (such as the Amazon Lex bot or an alias) that is referred to is not
     *         found.</li>
     *         <li>BadRequestException Request validation failed, there is no usable message in the context, or the bot
     *         build failed, is still in progress, or contains unbuilt changes.</li>
     *         <li>LimitExceededException Exceeded a limit.</li>
     *         <li>InternalFailureException Internal service error. Retry the call.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeAsyncClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/GetSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSessionResponse> getSession(GetSessionRequest getSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>().withOperationName("GetSession")
                            .withMarshaller(new GetSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSessionRequest));
            CompletableFuture<GetSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends user input (text or speech) to Amazon Lex. Clients use this API to send text and audio requests to Amazon
     * Lex at runtime. Amazon Lex interprets the user input using the machine learning model that it built for the bot.
     * </p>
     * <p>
     * The <code>PostContent</code> operation supports audio input at 8kHz and 16kHz. You can use 8kHz audio to achieve
     * higher speech recognition accuracy in telephone audio applications.
     * </p>
     * <p>
     * In response, Amazon Lex returns the next message to convey to the user. Consider the following example messages:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a user input "I would like a pizza," Amazon Lex might return a response with a message eliciting slot data
     * (for example, <code>PizzaSize</code>): "What size pizza would you like?".
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user provides all of the pizza order information, Amazon Lex might return a response with a message to
     * get user confirmation: "Order the pizza?".
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user replies "Yes" to the confirmation prompt, Amazon Lex might return a conclusion statement:
     * "Thank you, your cheese pizza has been ordered.".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Not all Amazon Lex messages require a response from the user. For example, conclusion statements do not require a
     * response. Some messages require only a yes or no response. In addition to the <code>message</code>, Amazon Lex
     * provides additional context about the message in the response that you can use to enhance client behavior, such
     * as displaying the appropriate client user interface. Consider the following examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the message is to elicit slot data, Amazon Lex returns the following context information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>x-amz-lex-dialog-state</code> header set to <code>ElicitSlot</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-lex-intent-name</code> header set to the intent name in the current context
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-lex-slot-to-elicit</code> header set to the slot name for which the <code>message</code> is eliciting
     * information
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>x-amz-lex-slots</code> header set to a map of slots configured for the intent with their current values
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * If the message is a confirmation prompt, the <code>x-amz-lex-dialog-state</code> header is set to
     * <code>Confirmation</code> and the <code>x-amz-lex-slot-to-elicit</code> header is omitted.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the message is a clarification prompt configured for the intent, indicating that the user intent is not
     * understood, the <code>x-amz-dialog-state</code> header is set to <code>ElicitIntent</code> and the
     * <code>x-amz-slot-to-elicit</code> header is omitted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, Amazon Lex also returns your application-specific <code>sessionAttributes</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/context-mgmt.html">Managing Conversation
     * Context</a>.
     * </p>
     *
     * @param postContentRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        User input in PCM or Opus audio format or text format as described in the <code>Content-Type</code> HTTP
     *        header.
     *        </p>
     *        <p>
     *        You can stream audio data to Amazon Lex or you can create a local buffer that captures all of the audio
     *        data before sending. In general, you get better performance if you stream audio data rather than buffering
     *        the data locally.
     *        </p>
     *        '
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The prompt (or statement) to convey to the user. This is based on the bot configuration and context. For
     *        example, if Amazon Lex did not understand the user intent, it sends the <code>clarificationPrompt</code>
     *        configured for the bot. If the intent requires confirmation before taking the fulfillment action, it sends
     *        the <code>confirmationPrompt</code>. Another example: Suppose that the Lambda function successfully
     *        fulfilled the intent, and sent a message to convey to the user. Then Amazon Lex sends that message in the
     *        response.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (such as the Amazon Lex bot or an alias) that is referred to is not
     *         found.</li>
     *         <li>BadRequestException Request validation failed, there is no usable message in the context, or the bot
     *         build failed, is still in progress, or contains unbuilt changes.</li>
     *         <li>LimitExceededException Exceeded a limit.</li>
     *         <li>InternalFailureException Internal service error. Retry the call.</li>
     *         <li>ConflictException Two clients are using the same AWS account, Amazon Lex bot, and user ID.</li>
     *         <li>UnsupportedMediaTypeException The Content-Type header (<code>PostContent</code> API) has an invalid
     *         value.</li>
     *         <li>NotAcceptableException The accept header in the request does not have a valid value.</li>
     *         <li>RequestTimeoutException The input speech is too long.</li>
     *         <li>DependencyFailedException One of the dependencies, such as AWS Lambda or Amazon Polly, threw an
     *         exception. For example, </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Amazon Lex does not have sufficient permissions to call a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a Lambda function takes longer than 30 seconds to execute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a fulfillment Lambda function returns a <code>Delegate</code> dialog action without removing any slot
     *         values.
     *         </p>
     *         </li></li>
     *         <li>BadGatewayException Either the Amazon Lex bot is still building, or one of the dependent services
     *         (Amazon Polly, AWS Lambda) failed with an internal service error.</li>
     *         <li>LoopDetectedException This exception is not used.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeAsyncClient.PostContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/PostContent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> postContent(PostContentRequest postContentRequest, AsyncRequestBody requestBody,
            AsyncResponseTransformer<PostContentResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postContentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostContent");
            Pair<AsyncResponseTransformer<PostContentResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            postContentRequest = applySignerOverride(postContentRequest, Aws4UnsignedPayloadSigner.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<PostContentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PostContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<PostContentRequest, PostContentResponse>()
                            .withOperationName("PostContent")
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new PostContentRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).transferEncoding(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncRequestBody(requestBody).withInput(postContentRequest), asyncResponseTransformer);
            AsyncResponseTransformer<PostContentResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<PostContentResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends user input to Amazon Lex. Client applications can use this API to send requests to Amazon Lex at runtime.
     * Amazon Lex then interprets the user input using the machine learning model it built for the bot.
     * </p>
     * <p>
     * In response, Amazon Lex returns the next <code>message</code> to convey to the user an optional
     * <code>responseCard</code> to display. Consider the following example messages:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a user input "I would like a pizza", Amazon Lex might return a response with a message eliciting slot data
     * (for example, PizzaSize): "What size pizza would you like?"
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user provides all of the pizza order information, Amazon Lex might return a response with a message to
     * obtain user confirmation "Proceed with the pizza order?".
     * </p>
     * </li>
     * <li>
     * <p>
     * After the user replies to a confirmation prompt with a "yes", Amazon Lex might return a conclusion statement:
     * "Thank you, your cheese pizza has been ordered.".
     * </p>
     * </li>
     * </ul>
     * <p>
     * Not all Amazon Lex messages require a user response. For example, a conclusion statement does not require a
     * response. Some messages require only a "yes" or "no" user response. In addition to the <code>message</code>,
     * Amazon Lex provides additional context about the message in the response that you might use to enhance client
     * behavior, for example, to display the appropriate client user interface. These are the <code>slotToElicit</code>,
     * <code>dialogState</code>, <code>intentName</code>, and <code>slots</code> fields in the response. Consider the
     * following examples:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the message is to elicit slot data, Amazon Lex returns the following context information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>dialogState</code> set to ElicitSlot
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>intentName</code> set to the intent name in the current context
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>slotToElicit</code> set to the slot name for which the <code>message</code> is eliciting information
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>slots</code> set to a map of slots, configured for the intent, with currently known values
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * If the message is a confirmation prompt, the <code>dialogState</code> is set to ConfirmIntent and
     * <code>SlotToElicit</code> is set to null.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the message is a clarification prompt (configured for the intent) that indicates that user intent is not
     * understood, the <code>dialogState</code> is set to ElicitIntent and <code>slotToElicit</code> is set to null.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In addition, Amazon Lex also returns your application-specific <code>sessionAttributes</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/context-mgmt.html">Managing Conversation
     * Context</a>.
     * </p>
     *
     * @param postTextRequest
     * @return A Java Future containing the result of the PostText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (such as the Amazon Lex bot or an alias) that is referred to is not
     *         found.</li>
     *         <li>BadRequestException Request validation failed, there is no usable message in the context, or the bot
     *         build failed, is still in progress, or contains unbuilt changes.</li>
     *         <li>LimitExceededException Exceeded a limit.</li>
     *         <li>InternalFailureException Internal service error. Retry the call.</li>
     *         <li>ConflictException Two clients are using the same AWS account, Amazon Lex bot, and user ID.</li>
     *         <li>DependencyFailedException One of the dependencies, such as AWS Lambda or Amazon Polly, threw an
     *         exception. For example, </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Amazon Lex does not have sufficient permissions to call a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a Lambda function takes longer than 30 seconds to execute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a fulfillment Lambda function returns a <code>Delegate</code> dialog action without removing any slot
     *         values.
     *         </p>
     *         </li></li>
     *         <li>BadGatewayException Either the Amazon Lex bot is still building, or one of the dependent services
     *         (Amazon Polly, AWS Lambda) failed with an internal service error.</li>
     *         <li>LoopDetectedException This exception is not used.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeAsyncClient.PostText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/PostText" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PostTextResponse> postText(PostTextRequest postTextRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(postTextRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, postTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PostText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PostTextResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PostTextResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PostTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PostTextRequest, PostTextResponse>().withOperationName("PostText")
                            .withMarshaller(new PostTextRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(postTextRequest));
            CompletableFuture<PostTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new session or modifies an existing session with an Amazon Lex bot. Use this operation to enable your
     * application to set the state of the bot.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/how-session-api.html">Managing
     * Sessions</a>.
     * </p>
     *
     * @param putSessionRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows '
     *        <p>
     *        The audio version of the message to convey to the user.
     *        </p>
     *        '.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException The resource (such as the Amazon Lex bot or an alias) that is referred to is not
     *         found.</li>
     *         <li>BadRequestException Request validation failed, there is no usable message in the context, or the bot
     *         build failed, is still in progress, or contains unbuilt changes.</li>
     *         <li>LimitExceededException Exceeded a limit.</li>
     *         <li>InternalFailureException Internal service error. Retry the call.</li>
     *         <li>ConflictException Two clients are using the same AWS account, Amazon Lex bot, and user ID.</li>
     *         <li>NotAcceptableException The accept header in the request does not have a valid value.</li>
     *         <li>DependencyFailedException One of the dependencies, such as AWS Lambda or Amazon Polly, threw an
     *         exception. For example, </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If Amazon Lex does not have sufficient permissions to call a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a Lambda function takes longer than 30 seconds to execute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a fulfillment Lambda function returns a <code>Delegate</code> dialog action without removing any slot
     *         values.
     *         </p>
     *         </li></li>
     *         <li>BadGatewayException Either the Amazon Lex bot is still building, or one of the dependent services
     *         (Amazon Polly, AWS Lambda) failed with an internal service error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LexRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LexRuntimeAsyncClient.PutSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/runtime.lex-2016-11-28/PutSession" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> putSession(PutSessionRequest putSessionRequest,
            AsyncResponseTransformer<PutSessionResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Runtime Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSession");
            Pair<AsyncResponseTransformer<PutSessionResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<PutSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<PutSessionRequest, PutSessionResponse>().withOperationName("PutSession")
                            .withMarshaller(new PutSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putSessionRequest), asyncResponseTransformer);
            AsyncResponseTransformer<PutSessionResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<PutSessionResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LexRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LexRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTimeoutException")
                                .exceptionBuilderSupplier(RequestTimeoutException::builder).httpStatusCode(408).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedMediaTypeException")
                                .exceptionBuilderSupplier(UnsupportedMediaTypeException::builder).httpStatusCode(415).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAcceptableException")
                                .exceptionBuilderSupplier(NotAcceptableException::builder).httpStatusCode(406).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LoopDetectedException")
                                .exceptionBuilderSupplier(LoopDetectedException::builder).httpStatusCode(508).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LexRuntimeRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private static boolean isSignerOverridden(SdkClientConfiguration clientConfiguration) {
        return Boolean.TRUE.equals(clientConfiguration.option(SdkClientOption.SIGNER_OVERRIDDEN));
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        LexRuntimeServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = LexRuntimeServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
