/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexmodelbuilding;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lexmodelbuilding.internal.LexModelBuildingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.lexmodelbuilding.model.AccessDeniedException;
import software.amazon.awssdk.services.lexmodelbuilding.model.BadRequestException;
import software.amazon.awssdk.services.lexmodelbuilding.model.ConflictException;
import software.amazon.awssdk.services.lexmodelbuilding.model.CreateBotVersionRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.CreateBotVersionResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.CreateIntentVersionRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.CreateIntentVersionResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.CreateSlotTypeVersionRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.CreateSlotTypeVersionResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotAliasRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotAliasResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotChannelAssociationRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotChannelAssociationResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotVersionRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteBotVersionResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteIntentRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteIntentResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteIntentVersionRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteIntentVersionResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteSlotTypeRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteSlotTypeResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteSlotTypeVersionRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteSlotTypeVersionResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteUtterancesRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.DeleteUtterancesResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotAliasRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotAliasResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotAliasesRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotAliasesResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotChannelAssociationRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotChannelAssociationResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotChannelAssociationsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotChannelAssociationsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotVersionsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotVersionsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBotsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBuiltinIntentRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBuiltinIntentResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBuiltinIntentsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBuiltinIntentsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBuiltinSlotTypesRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetBuiltinSlotTypesResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetExportRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetExportResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetImportRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetImportResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentVersionsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentVersionsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetIntentsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetMigrationRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetMigrationResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetMigrationsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetMigrationsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypeVersionsResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypesRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetSlotTypesResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetUtterancesViewRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.GetUtterancesViewResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.InternalFailureException;
import software.amazon.awssdk.services.lexmodelbuilding.model.LexModelBuildingException;
import software.amazon.awssdk.services.lexmodelbuilding.model.LimitExceededException;
import software.amazon.awssdk.services.lexmodelbuilding.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.NotFoundException;
import software.amazon.awssdk.services.lexmodelbuilding.model.PreconditionFailedException;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutBotAliasRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutBotAliasResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutBotRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutBotResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutIntentRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutIntentResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutSlotTypeRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.PutSlotTypeResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.ResourceInUseException;
import software.amazon.awssdk.services.lexmodelbuilding.model.StartImportRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.StartImportResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.StartMigrationRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.StartMigrationResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.TagResourceRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.TagResourceResponse;
import software.amazon.awssdk.services.lexmodelbuilding.model.UntagResourceRequest;
import software.amazon.awssdk.services.lexmodelbuilding.model.UntagResourceResponse;
import software.amazon.awssdk.services.lexmodelbuilding.transform.CreateBotVersionRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.CreateIntentVersionRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.CreateSlotTypeVersionRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteBotAliasRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteBotChannelAssociationRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteBotRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteBotVersionRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteIntentRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteIntentVersionRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteSlotTypeRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteSlotTypeVersionRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.DeleteUtterancesRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotAliasRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotAliasesRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotChannelAssociationRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotChannelAssociationsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotVersionsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBotsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBuiltinIntentRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBuiltinIntentsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetBuiltinSlotTypesRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetExportRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetImportRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetIntentRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetIntentVersionsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetIntentsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetMigrationRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetMigrationsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetSlotTypeRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetSlotTypeVersionsRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetSlotTypesRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.GetUtterancesViewRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.PutBotAliasRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.PutBotRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.PutIntentRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.PutSlotTypeRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.StartImportRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.StartMigrationRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lexmodelbuilding.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LexModelBuildingClient}.
 *
 * @see LexModelBuildingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLexModelBuildingClient implements LexModelBuildingClient {
    private static final Logger log = Logger.loggerFor(DefaultLexModelBuildingClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LexModelBuildingServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLexModelBuildingClient(LexModelBuildingServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new version of the bot based on the <code>&#36LATEST</code> version. If the <code>&#36LATEST</code>
     * version of this resource hasn't changed since you created the last version, Amazon Lex doesn't create a new
     * version. It returns the last created version.
     * </p>
     * <note>
     * <p>
     * You can update only the <code>&#36LATEST</code> version of the bot. You can't update the numbered versions that
     * you create with the <code>CreateBotVersion</code> operation.
     * </p>
     * </note>
     * <p>
     * When you create the first version of a bot, Amazon Lex sets the version to 1. Subsequent versions increment by 1.
     * For more information, see <a>versioning-intro</a>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lex:CreateBotVersion</code> action.
     * </p>
     *
     * @param createBotVersionRequest
     * @return Result of the CreateBotVersion operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.CreateBotVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/CreateBotVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateBotVersionResponse createBotVersion(CreateBotVersionRequest createBotVersionRequest) throws NotFoundException,
            ConflictException, LimitExceededException, InternalFailureException, BadRequestException,
            PreconditionFailedException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBotVersionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBotVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBotVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBotVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBotVersion");

            return clientHandler.execute(new ClientExecutionParams<CreateBotVersionRequest, CreateBotVersionResponse>()
                    .withOperationName("CreateBotVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createBotVersionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBotVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new version of an intent based on the <code>&#36LATEST</code> version of the intent. If the
     * <code>&#36LATEST</code> version of this intent hasn't changed since you last updated it, Amazon Lex doesn't
     * create a new version. It returns the last version you created.
     * </p>
     * <note>
     * <p>
     * You can update only the <code>&#36LATEST</code> version of the intent. You can't update the numbered versions
     * that you create with the <code>CreateIntentVersion</code> operation.
     * </p>
     * </note>
     * <p>
     * When you create a version of an intent, Amazon Lex sets the version to 1. Subsequent versions increment by 1. For
     * more information, see <a>versioning-intro</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lex:CreateIntentVersion</code> action.
     * </p>
     *
     * @param createIntentVersionRequest
     * @return Result of the CreateIntentVersion operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.CreateIntentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/CreateIntentVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateIntentVersionResponse createIntentVersion(CreateIntentVersionRequest createIntentVersionRequest)
            throws NotFoundException, ConflictException, LimitExceededException, InternalFailureException, BadRequestException,
            PreconditionFailedException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateIntentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateIntentVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIntentVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIntentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIntentVersion");

            return clientHandler.execute(new ClientExecutionParams<CreateIntentVersionRequest, CreateIntentVersionResponse>()
                    .withOperationName("CreateIntentVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createIntentVersionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateIntentVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new version of a slot type based on the <code>&#36LATEST</code> version of the specified slot type. If
     * the <code>&#36LATEST</code> version of this resource has not changed since the last version that you created,
     * Amazon Lex doesn't create a new version. It returns the last version that you created.
     * </p>
     * <note>
     * <p>
     * You can update only the <code>&#36LATEST</code> version of a slot type. You can't update the numbered versions
     * that you create with the <code>CreateSlotTypeVersion</code> operation.
     * </p>
     * </note>
     * <p>
     * When you create a version of a slot type, Amazon Lex sets the version to 1. Subsequent versions increment by 1.
     * For more information, see <a>versioning-intro</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:CreateSlotTypeVersion</code> action.
     * </p>
     *
     * @param createSlotTypeVersionRequest
     * @return Result of the CreateSlotTypeVersion operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.CreateSlotTypeVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/CreateSlotTypeVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSlotTypeVersionResponse createSlotTypeVersion(CreateSlotTypeVersionRequest createSlotTypeVersionRequest)
            throws NotFoundException, ConflictException, LimitExceededException, InternalFailureException, BadRequestException,
            PreconditionFailedException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSlotTypeVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSlotTypeVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSlotTypeVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSlotTypeVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSlotTypeVersion");

            return clientHandler.execute(new ClientExecutionParams<CreateSlotTypeVersionRequest, CreateSlotTypeVersionResponse>()
                    .withOperationName("CreateSlotTypeVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createSlotTypeVersionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSlotTypeVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all versions of the bot, including the <code>&#36LATEST</code> version. To delete a specific version of
     * the bot, use the <a>DeleteBotVersion</a> operation. The <code>DeleteBot</code> operation doesn't immediately
     * remove the bot schema. Instead, it is marked for deletion and removed later.
     * </p>
     * <p>
     * Amazon Lex stores utterances indefinitely for improving the ability of your bot to respond to user inputs. These
     * utterances are not removed when the bot is deleted. To remove the utterances, use the <a>DeleteUtterances</a>
     * operation.
     * </p>
     * <p>
     * If a bot has an alias, you can't delete it. Instead, the <code>DeleteBot</code> operation returns a
     * <code>ResourceInUseException</code> exception that includes a reference to the alias that refers to the bot. To
     * remove the reference to the bot, delete the alias. If you get the same exception again, delete the referring
     * alias until the <code>DeleteBot</code> operation is successful.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:DeleteBot</code> action.
     * </p>
     *
     * @param deleteBotRequest
     * @return Result of the DeleteBot operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBotResponse deleteBot(DeleteBotRequest deleteBotRequest) throws NotFoundException, ConflictException,
            LimitExceededException, InternalFailureException, BadRequestException, ResourceInUseException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBot");

            return clientHandler.execute(new ClientExecutionParams<DeleteBotRequest, DeleteBotResponse>()
                    .withOperationName("DeleteBot").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteBotRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an alias for the specified bot.
     * </p>
     * <p>
     * You can't delete an alias that is used in the association between a bot and a messaging channel. If an alias is
     * used in a channel association, the <code>DeleteBot</code> operation returns a <code>ResourceInUseException</code>
     * exception that includes a reference to the channel association that refers to the bot. You can remove the
     * reference to the alias by deleting the channel association. If you get the same exception again, delete the
     * referring association until the <code>DeleteBotAlias</code> operation is successful.
     * </p>
     *
     * @param deleteBotAliasRequest
     * @return Result of the DeleteBotAlias operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteBotAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteBotAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBotAliasResponse deleteBotAlias(DeleteBotAliasRequest deleteBotAliasRequest) throws NotFoundException,
            ConflictException, LimitExceededException, InternalFailureException, BadRequestException, ResourceInUseException,
            AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBotAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBotAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBotAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBotAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBotAlias");

            return clientHandler.execute(new ClientExecutionParams<DeleteBotAliasRequest, DeleteBotAliasResponse>()
                    .withOperationName("DeleteBotAlias").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteBotAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBotAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the association between an Amazon Lex bot and a messaging platform.
     * </p>
     * <p>
     * This operation requires permission for the <code>lex:DeleteBotChannelAssociation</code> action.
     * </p>
     *
     * @param deleteBotChannelAssociationRequest
     * @return Result of the DeleteBotChannelAssociation operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteBotChannelAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteBotChannelAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteBotChannelAssociationResponse deleteBotChannelAssociation(
            DeleteBotChannelAssociationRequest deleteBotChannelAssociationRequest) throws NotFoundException, ConflictException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBotChannelAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBotChannelAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBotChannelAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBotChannelAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBotChannelAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteBotChannelAssociationRequest, DeleteBotChannelAssociationResponse>()
                            .withOperationName("DeleteBotChannelAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(deleteBotChannelAssociationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteBotChannelAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specific version of a bot. To delete all versions of a bot, use the <a>DeleteBot</a> operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:DeleteBotVersion</code> action.
     * </p>
     *
     * @param deleteBotVersionRequest
     * @return Result of the DeleteBotVersion operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteBotVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteBotVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteBotVersionResponse deleteBotVersion(DeleteBotVersionRequest deleteBotVersionRequest) throws NotFoundException,
            ConflictException, LimitExceededException, InternalFailureException, BadRequestException, ResourceInUseException,
            AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBotVersionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBotVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBotVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBotVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBotVersion");

            return clientHandler.execute(new ClientExecutionParams<DeleteBotVersionRequest, DeleteBotVersionResponse>()
                    .withOperationName("DeleteBotVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteBotVersionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBotVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all versions of the intent, including the <code>&#36LATEST</code> version. To delete a specific version
     * of the intent, use the <a>DeleteIntentVersion</a> operation.
     * </p>
     * <p>
     * You can delete a version of an intent only if it is not referenced. To delete an intent that is referred to in
     * one or more bots (see <a>how-it-works</a>), you must remove those references first.
     * </p>
     * <note>
     * <p>
     * If you get the <code>ResourceInUseException</code> exception, it provides an example reference that shows where
     * the intent is referenced. To remove the reference to the intent, either update the bot or delete it. If you get
     * the same exception when you attempt to delete the intent again, repeat until the intent has no references and the
     * call to <code>DeleteIntent</code> is successful.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code>lex:DeleteIntent</code> action.
     * </p>
     *
     * @param deleteIntentRequest
     * @return Result of the DeleteIntent operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteIntent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteIntent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteIntentResponse deleteIntent(DeleteIntentRequest deleteIntentRequest) throws NotFoundException,
            ConflictException, LimitExceededException, InternalFailureException, BadRequestException, ResourceInUseException,
            AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIntentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteIntentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIntentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIntentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIntent");

            return clientHandler.execute(new ClientExecutionParams<DeleteIntentRequest, DeleteIntentResponse>()
                    .withOperationName("DeleteIntent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteIntentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIntentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specific version of an intent. To delete all versions of a intent, use the <a>DeleteIntent</a>
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:DeleteIntentVersion</code> action.
     * </p>
     *
     * @param deleteIntentVersionRequest
     * @return Result of the DeleteIntentVersion operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteIntentVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteIntentVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteIntentVersionResponse deleteIntentVersion(DeleteIntentVersionRequest deleteIntentVersionRequest)
            throws NotFoundException, ConflictException, LimitExceededException, InternalFailureException, BadRequestException,
            ResourceInUseException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIntentVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteIntentVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIntentVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIntentVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIntentVersion");

            return clientHandler.execute(new ClientExecutionParams<DeleteIntentVersionRequest, DeleteIntentVersionResponse>()
                    .withOperationName("DeleteIntentVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteIntentVersionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIntentVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all versions of the slot type, including the <code>&#36LATEST</code> version. To delete a specific
     * version of the slot type, use the <a>DeleteSlotTypeVersion</a> operation.
     * </p>
     * <p>
     * You can delete a version of a slot type only if it is not referenced. To delete a slot type that is referred to
     * in one or more intents, you must remove those references first.
     * </p>
     * <note>
     * <p>
     * If you get the <code>ResourceInUseException</code> exception, the exception provides an example reference that
     * shows the intent where the slot type is referenced. To remove the reference to the slot type, either update the
     * intent or delete it. If you get the same exception when you attempt to delete the slot type again, repeat until
     * the slot type has no references and the <code>DeleteSlotType</code> call is successful.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code>lex:DeleteSlotType</code> action.
     * </p>
     *
     * @param deleteSlotTypeRequest
     * @return Result of the DeleteSlotType operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteSlotType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteSlotType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSlotTypeResponse deleteSlotType(DeleteSlotTypeRequest deleteSlotTypeRequest) throws NotFoundException,
            ConflictException, LimitExceededException, InternalFailureException, BadRequestException, ResourceInUseException,
            AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSlotTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSlotTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlotTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSlotTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlotType");

            return clientHandler.execute(new ClientExecutionParams<DeleteSlotTypeRequest, DeleteSlotTypeResponse>()
                    .withOperationName("DeleteSlotType").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteSlotTypeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSlotTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specific version of a slot type. To delete all versions of a slot type, use the <a>DeleteSlotType</a>
     * operation.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:DeleteSlotTypeVersion</code> action.
     * </p>
     *
     * @param deleteSlotTypeVersionRequest
     * @return Result of the DeleteSlotTypeVersion operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ResourceInUseException
     *         The resource that you are attempting to delete is referred to by another resource. Use this information
     *         to remove references to the resource that you are trying to delete.</p>
     *         <p>
     *         The body of the exception contains a JSON object that describes the resource.
     *         </p>
     *         <p>
     *         <code>{ "resourceType": BOT | BOTALIAS | BOTCHANNEL | INTENT,</code>
     *         </p>
     *         <p>
     *         <code>"resourceReference": {</code>
     *         </p>
     *         <p>
     *         <code>"name": <i>string</i>, "version": <i>string</i> } }</code>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteSlotTypeVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteSlotTypeVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSlotTypeVersionResponse deleteSlotTypeVersion(DeleteSlotTypeVersionRequest deleteSlotTypeVersionRequest)
            throws NotFoundException, ConflictException, LimitExceededException, InternalFailureException, BadRequestException,
            ResourceInUseException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSlotTypeVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSlotTypeVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlotTypeVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSlotTypeVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlotTypeVersion");

            return clientHandler.execute(new ClientExecutionParams<DeleteSlotTypeVersionRequest, DeleteSlotTypeVersionResponse>()
                    .withOperationName("DeleteSlotTypeVersion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteSlotTypeVersionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSlotTypeVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes stored utterances.
     * </p>
     * <p>
     * Amazon Lex stores the utterances that users send to your bot. Utterances are stored for 15 days for use with the
     * <a>GetUtterancesView</a> operation, and then stored indefinitely for use in improving the ability of your bot to
     * respond to user input.
     * </p>
     * <p>
     * Use the <code>DeleteUtterances</code> operation to manually delete stored utterances for a specific user. When
     * you use the <code>DeleteUtterances</code> operation, utterances stored for improving your bot's ability to
     * respond to user input are deleted immediately. Utterances stored for use with the <code>GetUtterancesView</code>
     * operation are deleted after 15 days.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:DeleteUtterances</code> action.
     * </p>
     *
     * @param deleteUtterancesRequest
     * @return Result of the DeleteUtterances operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.DeleteUtterances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/DeleteUtterances" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteUtterancesResponse deleteUtterances(DeleteUtterancesRequest deleteUtterancesRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUtterancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUtterancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteUtterancesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUtterancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUtterances");

            return clientHandler.execute(new ClientExecutionParams<DeleteUtterancesRequest, DeleteUtterancesResponse>()
                    .withOperationName("DeleteUtterances").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteUtterancesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteUtterancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata information for a specific bot. You must provide the bot name and the bot version or alias.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetBot</code> action.
     * </p>
     *
     * @param getBotRequest
     * @return Result of the GetBot operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBotResponse getBot(GetBotRequest getBotRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBot");

            return clientHandler.execute(new ClientExecutionParams<GetBotRequest, GetBotResponse>().withOperationName("GetBot")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getBotRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetBotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an Amazon Lex bot alias. For more information about aliases, see
     * <a>versioning-aliases</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetBotAlias</code> action.
     * </p>
     *
     * @param getBotAliasRequest
     * @return Result of the GetBotAlias operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBotAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBotAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBotAliasResponse getBotAlias(GetBotAliasRequest getBotAliasRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBotAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBotAlias");

            return clientHandler.execute(new ClientExecutionParams<GetBotAliasRequest, GetBotAliasResponse>()
                    .withOperationName("GetBotAlias").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBotAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBotAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of aliases for a specified Amazon Lex bot.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetBotAliases</code> action.
     * </p>
     *
     * @param getBotAliasesRequest
     * @return Result of the GetBotAliases operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBotAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBotAliases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBotAliasesResponse getBotAliases(GetBotAliasesRequest getBotAliasesRequest) throws LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotAliasesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBotAliasesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotAliasesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBotAliases");

            return clientHandler.execute(new ClientExecutionParams<GetBotAliasesRequest, GetBotAliasesResponse>()
                    .withOperationName("GetBotAliases").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBotAliasesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBotAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the association between an Amazon Lex bot and a messaging platform.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetBotChannelAssociation</code> action.
     * </p>
     *
     * @param getBotChannelAssociationRequest
     * @return Result of the GetBotChannelAssociation operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBotChannelAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBotChannelAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBotChannelAssociationResponse getBotChannelAssociation(
            GetBotChannelAssociationRequest getBotChannelAssociationRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotChannelAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBotChannelAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotChannelAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotChannelAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBotChannelAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBotChannelAssociationRequest, GetBotChannelAssociationResponse>()
                            .withOperationName("GetBotChannelAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getBotChannelAssociationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBotChannelAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all of the channels associated with the specified bot.
     * </p>
     * <p>
     * The <code>GetBotChannelAssociations</code> operation requires permissions for the
     * <code>lex:GetBotChannelAssociations</code> action.
     * </p>
     *
     * @param getBotChannelAssociationsRequest
     * @return Result of the GetBotChannelAssociations operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBotChannelAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBotChannelAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBotChannelAssociationsResponse getBotChannelAssociations(
            GetBotChannelAssociationsRequest getBotChannelAssociationsRequest) throws LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotChannelAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBotChannelAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotChannelAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotChannelAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBotChannelAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBotChannelAssociationsRequest, GetBotChannelAssociationsResponse>()
                            .withOperationName("GetBotChannelAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(getBotChannelAssociationsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBotChannelAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about all of the versions of a bot.
     * </p>
     * <p>
     * The <code>GetBotVersions</code> operation returns a <code>BotMetadata</code> object for each version of a bot.
     * For example, if a bot has three numbered versions, the <code>GetBotVersions</code> operation returns four
     * <code>BotMetadata</code> objects in the response, one for each numbered version and one for the
     * <code>&#36LATEST</code> version.
     * </p>
     * <p>
     * The <code>GetBotVersions</code> operation always returns at least one version, the <code>&#36LATEST</code>
     * version.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetBotVersions</code> action.
     * </p>
     *
     * @param getBotVersionsRequest
     * @return Result of the GetBotVersions operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBotVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBotVersions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBotVersionsResponse getBotVersions(GetBotVersionsRequest getBotVersionsRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotVersionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBotVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotVersionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBotVersions");

            return clientHandler.execute(new ClientExecutionParams<GetBotVersionsRequest, GetBotVersionsResponse>()
                    .withOperationName("GetBotVersions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBotVersionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBotVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns bot information as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you provide the <code>nameContains</code> field, the response includes information for the
     * <code>&#36LATEST</code> version of all bots whose name contains the specified string.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't specify the <code>nameContains</code> field, the operation returns information about the
     * <code>&#36LATEST</code> version of all of your bots.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires permission for the <code>lex:GetBots</code> action.
     * </p>
     *
     * @param getBotsRequest
     * @return Result of the GetBots operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBotsResponse getBots(GetBotsRequest getBotsRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBotsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBots");

            return clientHandler.execute(new ClientExecutionParams<GetBotsRequest, GetBotsResponse>()
                    .withOperationName("GetBots").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBotsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a built-in intent.
     * </p>
     * <p>
     * This operation requires permission for the <code>lex:GetBuiltinIntent</code> action.
     * </p>
     *
     * @param getBuiltinIntentRequest
     * @return Result of the GetBuiltinIntent operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBuiltinIntent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBuiltinIntent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetBuiltinIntentResponse getBuiltinIntent(GetBuiltinIntentRequest getBuiltinIntentRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBuiltinIntentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBuiltinIntentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBuiltinIntentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBuiltinIntentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBuiltinIntent");

            return clientHandler.execute(new ClientExecutionParams<GetBuiltinIntentRequest, GetBuiltinIntentResponse>()
                    .withOperationName("GetBuiltinIntent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBuiltinIntentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBuiltinIntentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of built-in intents that meet the specified criteria.
     * </p>
     * <p>
     * This operation requires permission for the <code>lex:GetBuiltinIntents</code> action.
     * </p>
     *
     * @param getBuiltinIntentsRequest
     * @return Result of the GetBuiltinIntents operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBuiltinIntents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBuiltinIntents" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetBuiltinIntentsResponse getBuiltinIntents(GetBuiltinIntentsRequest getBuiltinIntentsRequest)
            throws LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBuiltinIntentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBuiltinIntentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBuiltinIntentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBuiltinIntentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBuiltinIntents");

            return clientHandler.execute(new ClientExecutionParams<GetBuiltinIntentsRequest, GetBuiltinIntentsResponse>()
                    .withOperationName("GetBuiltinIntents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBuiltinIntentsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBuiltinIntentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of built-in slot types that meet the specified criteria.
     * </p>
     * <p>
     * For a list of built-in slot types, see <a href=
     * "https://developer.amazon.com/public/solutions/alexa/alexa-skills-kit/docs/built-in-intent-ref/slot-type-reference"
     * >Slot Type Reference</a> in the <i>Alexa Skills Kit</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>lex:GetBuiltInSlotTypes</code> action.
     * </p>
     *
     * @param getBuiltinSlotTypesRequest
     * @return Result of the GetBuiltinSlotTypes operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetBuiltinSlotTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetBuiltinSlotTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBuiltinSlotTypesResponse getBuiltinSlotTypes(GetBuiltinSlotTypesRequest getBuiltinSlotTypesRequest)
            throws LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBuiltinSlotTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBuiltinSlotTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBuiltinSlotTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBuiltinSlotTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBuiltinSlotTypes");

            return clientHandler.execute(new ClientExecutionParams<GetBuiltinSlotTypesRequest, GetBuiltinSlotTypesResponse>()
                    .withOperationName("GetBuiltinSlotTypes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBuiltinSlotTypesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBuiltinSlotTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports the contents of a Amazon Lex resource in a specified format.
     * </p>
     *
     * @param getExportRequest
     * @return Result of the GetExport operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetExport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExportResponse getExport(GetExportRequest getExportRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExport");

            return clientHandler.execute(new ClientExecutionParams<GetExportRequest, GetExportResponse>()
                    .withOperationName("GetExport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getExportRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about an import job started with the <code>StartImport</code> operation.
     * </p>
     *
     * @param getImportRequest
     * @return Result of the GetImport operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetImportResponse getImport(GetImportRequest getImportRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetImportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getImportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImport");

            return clientHandler.execute(new ClientExecutionParams<GetImportRequest, GetImportResponse>()
                    .withOperationName("GetImport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getImportRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an intent. In addition to the intent name, you must specify the intent version.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lex:GetIntent</code> action.
     * </p>
     *
     * @param getIntentRequest
     * @return Result of the GetIntent operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetIntent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetIntent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetIntentResponse getIntent(GetIntentRequest getIntentRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIntentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIntentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIntentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIntentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIntent");

            return clientHandler.execute(new ClientExecutionParams<GetIntentRequest, GetIntentResponse>()
                    .withOperationName("GetIntent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getIntentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIntentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about all of the versions of an intent.
     * </p>
     * <p>
     * The <code>GetIntentVersions</code> operation returns an <code>IntentMetadata</code> object for each version of an
     * intent. For example, if an intent has three numbered versions, the <code>GetIntentVersions</code> operation
     * returns four <code>IntentMetadata</code> objects in the response, one for each numbered version and one for the
     * <code>&#36LATEST</code> version.
     * </p>
     * <p>
     * The <code>GetIntentVersions</code> operation always returns at least one version, the <code>&#36LATEST</code>
     * version.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetIntentVersions</code> action.
     * </p>
     *
     * @param getIntentVersionsRequest
     * @return Result of the GetIntentVersions operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetIntentVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetIntentVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetIntentVersionsResponse getIntentVersions(GetIntentVersionsRequest getIntentVersionsRequest)
            throws NotFoundException, LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIntentVersionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIntentVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIntentVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIntentVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIntentVersions");

            return clientHandler.execute(new ClientExecutionParams<GetIntentVersionsRequest, GetIntentVersionsResponse>()
                    .withOperationName("GetIntentVersions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getIntentVersionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIntentVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns intent information as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify the <code>nameContains</code> field, returns the <code>&#36LATEST</code> version of all intents
     * that contain the specified string.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't specify the <code>nameContains</code> field, returns information about the <code>&#36LATEST</code>
     * version of all intents.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The operation requires permission for the <code>lex:GetIntents</code> action.
     * </p>
     *
     * @param getIntentsRequest
     * @return Result of the GetIntents operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetIntents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetIntents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetIntentsResponse getIntents(GetIntentsRequest getIntentsRequest) throws NotFoundException, LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetIntentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetIntentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIntentsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIntentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIntents");

            return clientHandler.execute(new ClientExecutionParams<GetIntentsRequest, GetIntentsResponse>()
                    .withOperationName("GetIntents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getIntentsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetIntentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides details about an ongoing or complete migration from an Amazon Lex V1 bot to an Amazon Lex V2 bot. Use
     * this operation to view the migration alerts and warnings related to the migration.
     * </p>
     *
     * @param getMigrationRequest
     * @return Result of the GetMigration operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetMigration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetMigration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMigrationResponse getMigration(GetMigrationRequest getMigrationRequest) throws LimitExceededException,
            InternalFailureException, BadRequestException, NotFoundException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMigrationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMigrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMigrationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMigrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMigration");

            return clientHandler.execute(new ClientExecutionParams<GetMigrationRequest, GetMigrationResponse>()
                    .withOperationName("GetMigration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getMigrationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMigrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of migrations between Amazon Lex V1 and Amazon Lex V2.
     * </p>
     *
     * @param getMigrationsRequest
     * @return Result of the GetMigrations operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetMigrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetMigrations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMigrationsResponse getMigrations(GetMigrationsRequest getMigrationsRequest) throws LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMigrationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMigrationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMigrationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMigrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMigrations");

            return clientHandler.execute(new ClientExecutionParams<GetMigrationsRequest, GetMigrationsResponse>()
                    .withOperationName("GetMigrations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getMigrationsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMigrationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a specific version of a slot type. In addition to specifying the slot type name, you
     * must specify the slot type version.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetSlotType</code> action.
     * </p>
     *
     * @param getSlotTypeRequest
     * @return Result of the GetSlotType operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetSlotType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetSlotType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSlotTypeResponse getSlotType(GetSlotTypeRequest getSlotTypeRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSlotTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSlotTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSlotTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSlotTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSlotType");

            return clientHandler.execute(new ClientExecutionParams<GetSlotTypeRequest, GetSlotTypeResponse>()
                    .withOperationName("GetSlotType").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSlotTypeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSlotTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about all versions of a slot type.
     * </p>
     * <p>
     * The <code>GetSlotTypeVersions</code> operation returns a <code>SlotTypeMetadata</code> object for each version of
     * a slot type. For example, if a slot type has three numbered versions, the <code>GetSlotTypeVersions</code>
     * operation returns four <code>SlotTypeMetadata</code> objects in the response, one for each numbered version and
     * one for the <code>&#36LATEST</code> version.
     * </p>
     * <p>
     * The <code>GetSlotTypeVersions</code> operation always returns at least one version, the <code>&#36LATEST</code>
     * version.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetSlotTypeVersions</code> action.
     * </p>
     *
     * @param getSlotTypeVersionsRequest
     * @return Result of the GetSlotTypeVersions operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetSlotTypeVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetSlotTypeVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSlotTypeVersionsResponse getSlotTypeVersions(GetSlotTypeVersionsRequest getSlotTypeVersionsRequest)
            throws NotFoundException, LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSlotTypeVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSlotTypeVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSlotTypeVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSlotTypeVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSlotTypeVersions");

            return clientHandler.execute(new ClientExecutionParams<GetSlotTypeVersionsRequest, GetSlotTypeVersionsResponse>()
                    .withOperationName("GetSlotTypeVersions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSlotTypeVersionsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSlotTypeVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns slot type information as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify the <code>nameContains</code> field, returns the <code>&#36LATEST</code> version of all slot types
     * that contain the specified string.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you don't specify the <code>nameContains</code> field, returns information about the <code>&#36LATEST</code>
     * version of all slot types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The operation requires permission for the <code>lex:GetSlotTypes</code> action.
     * </p>
     *
     * @param getSlotTypesRequest
     * @return Result of the GetSlotTypes operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetSlotTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetSlotTypes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSlotTypesResponse getSlotTypes(GetSlotTypesRequest getSlotTypesRequest) throws NotFoundException,
            LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSlotTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSlotTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSlotTypesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSlotTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSlotTypes");

            return clientHandler.execute(new ClientExecutionParams<GetSlotTypesRequest, GetSlotTypesResponse>()
                    .withOperationName("GetSlotTypes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSlotTypesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSlotTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use the <code>GetUtterancesView</code> operation to get information about the utterances that your users have
     * made to your bot. You can use this list to tune the utterances that your bot responds to.
     * </p>
     * <p>
     * For example, say that you have created a bot to order flowers. After your users have used your bot for a while,
     * use the <code>GetUtterancesView</code> operation to see the requests that they have made and whether they have
     * been successful. You might find that the utterance "I want flowers" is not being recognized. You could add this
     * utterance to the <code>OrderFlowers</code> intent so that your bot recognizes that utterance.
     * </p>
     * <p>
     * After you publish a new version of a bot, you can get information about the old version and the new so that you
     * can compare the performance across the two versions.
     * </p>
     * <p>
     * Utterance statistics are generated once a day. Data is available for the last 15 days. You can request
     * information for up to 5 versions of your bot in each request. Amazon Lex returns the most frequent utterances
     * received by the bot in the last 15 days. The response contains information about a maximum of 100 utterances for
     * each version.
     * </p>
     * <p>
     * If you set <code>childDirected</code> field to true when you created your bot, if you are using slot obfuscation
     * with one or more slots, or if you opted out of participating in improving Amazon Lex, utterances are not
     * available.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:GetUtterancesView</code> action.
     * </p>
     *
     * @param getUtterancesViewRequest
     * @return Result of the GetUtterancesView operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.GetUtterancesView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/GetUtterancesView" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetUtterancesViewResponse getUtterancesView(GetUtterancesViewRequest getUtterancesViewRequest)
            throws LimitExceededException, InternalFailureException, BadRequestException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUtterancesViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUtterancesViewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getUtterancesViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUtterancesViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUtterancesView");

            return clientHandler.execute(new ClientExecutionParams<GetUtterancesViewRequest, GetUtterancesViewResponse>()
                    .withOperationName("GetUtterancesView").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getUtterancesViewRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetUtterancesViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of tags associated with the specified resource. Only bots, bot aliases, and bot channels can have
     * tags associated with them.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws NotFoundException, BadRequestException, InternalFailureException, LimitExceededException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsForResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Lex conversational bot or replaces an existing bot. When you create or update a bot you are
     * only required to specify a name, a locale, and whether the bot is directed toward children under age 13. You can
     * use this to add intents later, or to remove intents from an existing bot. When you create a bot with the minimum
     * information, the bot is created or updated but Amazon Lex returns the <code/> response <code>FAILED</code>. You
     * can build the bot after you add one or more intents. For more information about Amazon Lex bots, see
     * <a>how-it-works</a>.
     * </p>
     * <p>
     * If you specify the name of an existing bot, the fields in the request replace the existing values in the
     * <code>&#36LATEST</code> version of the bot. Amazon Lex removes any fields that you don't provide values for in
     * the request, except for the <code>idleTTLInSeconds</code> and <code>privacySettings</code> fields, which are set
     * to their default values. If you don't specify values for required fields, Amazon Lex throws an exception.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:PutBot</code> action. For more information, see
     * <a>security-iam</a>.
     * </p>
     *
     * @param putBotRequest
     * @return Result of the PutBot operation returned by the service.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.PutBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/PutBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutBotResponse putBot(PutBotRequest putBotRequest) throws ConflictException, LimitExceededException,
            InternalFailureException, BadRequestException, PreconditionFailedException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutBotResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBot");

            return clientHandler.execute(new ClientExecutionParams<PutBotRequest, PutBotResponse>().withOperationName("PutBot")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putBotRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new PutBotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an alias for the specified version of the bot or replaces an alias for the specified bot. To change the
     * version of the bot that the alias points to, replace the alias. For more information about aliases, see
     * <a>versioning-aliases</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:PutBotAlias</code> action.
     * </p>
     *
     * @param putBotAliasRequest
     * @return Result of the PutBotAlias operation returned by the service.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.PutBotAlias
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/PutBotAlias" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutBotAliasResponse putBotAlias(PutBotAliasRequest putBotAliasRequest) throws ConflictException,
            LimitExceededException, InternalFailureException, BadRequestException, PreconditionFailedException,
            AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBotAliasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutBotAliasResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putBotAliasRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBotAliasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBotAlias");

            return clientHandler.execute(new ClientExecutionParams<PutBotAliasRequest, PutBotAliasResponse>()
                    .withOperationName("PutBotAlias").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putBotAliasRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutBotAliasRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an intent or replaces an existing intent.
     * </p>
     * <p>
     * To define the interaction between the user and your bot, you use one or more intents. For a pizza ordering bot,
     * for example, you would create an <code>OrderPizza</code> intent.
     * </p>
     * <p>
     * To create an intent or replace an existing intent, you must provide the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Intent name. For example, <code>OrderPizza</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Sample utterances. For example, "Can I order a pizza, please." and "I want to order a pizza."
     * </p>
     * </li>
     * <li>
     * <p>
     * Information to be gathered. You specify slot types for the information that your bot will request from the user.
     * You can specify standard slot types, such as a date or a time, or custom slot types such as the size and crust of
     * a pizza.
     * </p>
     * </li>
     * <li>
     * <p>
     * How the intent will be fulfilled. You can provide a Lambda function or configure the intent to return the intent
     * information to the client application. If you use a Lambda function, when all of the intent information is
     * available, Amazon Lex invokes your Lambda function. If you configure your intent to return the intent information
     * to the client application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can specify other optional information in the request, such as:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A confirmation prompt to ask the user to confirm an intent. For example, "Shall I order your pizza?"
     * </p>
     * </li>
     * <li>
     * <p>
     * A conclusion statement to send to the user after the intent has been fulfilled. For example,
     * "I placed your pizza order."
     * </p>
     * </li>
     * <li>
     * <p>
     * A follow-up prompt that asks the user for additional activity. For example, asking
     * "Do you want to order a drink with your pizza?"
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify an existing intent name to update the intent, Amazon Lex replaces the values in the
     * <code>&#36LATEST</code> version of the intent with the values in the request. Amazon Lex removes fields that you
     * don't provide in the request. If you don't specify the required fields, Amazon Lex throws an exception. When you
     * update the <code>&#36LATEST</code> version of an intent, the <code>status</code> field of any bot that uses the
     * <code>&#36LATEST</code> version of the intent is set to <code>NOT_BUILT</code>.
     * </p>
     * <p>
     * For more information, see <a>how-it-works</a>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:PutIntent</code> action.
     * </p>
     *
     * @param putIntentRequest
     * @return Result of the PutIntent operation returned by the service.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.PutIntent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/PutIntent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutIntentResponse putIntent(PutIntentRequest putIntentRequest) throws ConflictException, LimitExceededException,
            InternalFailureException, BadRequestException, PreconditionFailedException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutIntentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutIntentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putIntentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putIntentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutIntent");

            return clientHandler.execute(new ClientExecutionParams<PutIntentRequest, PutIntentResponse>()
                    .withOperationName("PutIntent").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putIntentRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutIntentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a custom slot type or replaces an existing custom slot type.
     * </p>
     * <p>
     * To create a custom slot type, specify a name for the slot type and a set of enumeration values, which are the
     * values that a slot of this type can assume. For more information, see <a>how-it-works</a>.
     * </p>
     * <p>
     * If you specify the name of an existing slot type, the fields in the request replace the existing values in the
     * <code>&#36LATEST</code> version of the slot type. Amazon Lex removes the fields that you don't provide in the
     * request. If you don't specify required fields, Amazon Lex throws an exception. When you update the
     * <code>&#36LATEST</code> version of a slot type, if a bot uses the <code>&#36LATEST</code> version of an intent
     * that contains the slot type, the bot's <code>status</code> field is set to <code>NOT_BUILT</code>.
     * </p>
     * <p>
     * This operation requires permissions for the <code>lex:PutSlotType</code> action.
     * </p>
     *
     * @param putSlotTypeRequest
     * @return Result of the PutSlotType operation returned by the service.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws PreconditionFailedException
     *         The checksum of the resource that you are trying to change does not match the checksum in the request.
     *         Check the resource's checksum and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.PutSlotType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/PutSlotType" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutSlotTypeResponse putSlotType(PutSlotTypeRequest putSlotTypeRequest) throws ConflictException,
            LimitExceededException, InternalFailureException, BadRequestException, PreconditionFailedException,
            AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutSlotTypeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutSlotTypeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putSlotTypeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putSlotTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutSlotType");

            return clientHandler.execute(new ClientExecutionParams<PutSlotTypeRequest, PutSlotTypeResponse>()
                    .withOperationName("PutSlotType").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putSlotTypeRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutSlotTypeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a job to import a resource to Amazon Lex.
     * </p>
     *
     * @param startImportRequest
     * @return Result of the StartImport operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.StartImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/StartImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartImportResponse startImport(StartImportRequest startImportRequest) throws LimitExceededException,
            InternalFailureException, BadRequestException, AwsServiceException, SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartImportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startImportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImport");

            return clientHandler.execute(new ClientExecutionParams<StartImportRequest, StartImportResponse>()
                    .withOperationName("StartImport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startImportRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts migrating a bot from Amazon Lex V1 to Amazon Lex V2. Migrate your bot when you want to take advantage of
     * the new features of Amazon Lex V2.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/lex/latest/dg/migrate.html">Migrating a bot</a> in
     * the <i>Amazon Lex developer guide</i>.
     * </p>
     *
     * @param startMigrationRequest
     * @return Result of the StartMigration operation returned by the service.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws AccessDeniedException
     *         Your IAM user or role does not have permission to call the Amazon Lex V2 APIs required to migrate your
     *         bot.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.StartMigration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/StartMigration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartMigrationResponse startMigration(StartMigrationRequest startMigrationRequest) throws LimitExceededException,
            InternalFailureException, BadRequestException, AccessDeniedException, NotFoundException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMigrationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartMigrationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMigrationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMigrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMigration");

            return clientHandler.execute(new ClientExecutionParams<StartMigrationRequest, StartMigrationResponse>()
                    .withOperationName("StartMigration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startMigrationRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartMigrationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource. If a tag key already exists, the existing value is replaced
     * with the new value.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws NotFoundException, BadRequestException,
            ConflictException, InternalFailureException, LimitExceededException, AwsServiceException, SdkClientException,
            LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a bot, bot alias or bot channel.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws NotFoundException
     *         The resource specified in the request was not found. Check the resource and try again.
     * @throws BadRequestException
     *         The request is not well formed. For example, a value is invalid or a required field is missing. Check the
     *         field values, and try again.
     * @throws ConflictException
     *         There was a conflict processing the request. Try your request again.
     * @throws InternalFailureException
     *         An internal Amazon Lex error occurred. Try your request again.
     * @throws LimitExceededException
     *         The request exceeded a limit. Try your request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LexModelBuildingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LexModelBuildingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lex-models-2017-04-19/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws NotFoundException,
            BadRequestException, ConflictException, InternalFailureException, LimitExceededException, AwsServiceException,
            SdkClientException, LexModelBuildingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Lex Model Building Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        LexModelBuildingServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = LexModelBuildingServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LexModelBuildingException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionFailedException")
                                .exceptionBuilderSupplier(PreconditionFailedException::builder).httpStatusCode(412).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public final LexModelBuildingServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
