/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kinesisanalytics.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Identifies the S3 bucket and object that contains the reference data. Also identifies the IAM role Amazon Kinesis
 * Analytics can assume to read this object on your behalf.
 * </p>
 * <p>
 * An Amazon Kinesis Analytics application loads reference data only once. If the data changes, you call the <a
 * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_UpdateApplication.html">UpdateApplication</a>
 * operation to trigger reloading of data into your application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3ReferenceDataSource implements SdkPojo, Serializable,
        ToCopyableBuilder<S3ReferenceDataSource.Builder, S3ReferenceDataSource> {
    private static final SdkField<String> BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketARN").getter(getter(S3ReferenceDataSource::bucketARN)).setter(setter(Builder::bucketARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketARN").build()).build();

    private static final SdkField<String> FILE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileKey").getter(getter(S3ReferenceDataSource::fileKey)).setter(setter(Builder::fileKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileKey").build()).build();

    private static final SdkField<String> REFERENCE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReferenceRoleARN").getter(getter(S3ReferenceDataSource::referenceRoleARN))
            .setter(setter(Builder::referenceRoleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferenceRoleARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_ARN_FIELD,
            FILE_KEY_FIELD, REFERENCE_ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketARN;

    private final String fileKey;

    private final String referenceRoleARN;

    private S3ReferenceDataSource(BuilderImpl builder) {
        this.bucketARN = builder.bucketARN;
        this.fileKey = builder.fileKey;
        this.referenceRoleARN = builder.referenceRoleARN;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) of the S3 bucket.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) of the S3 bucket.
     */
    public final String bucketARN() {
        return bucketARN;
    }

    /**
     * <p>
     * Object key name containing reference data.
     * </p>
     * 
     * @return Object key name containing reference data.
     */
    public final String fileKey() {
        return fileKey;
    }

    /**
     * <p>
     * ARN of the IAM role that the service can assume to read data on your behalf. This role must have permission for
     * the <code>s3:GetObject</code> action on the object and trust policy that allows Amazon Kinesis Analytics service
     * principal to assume this role.
     * </p>
     * 
     * @return ARN of the IAM role that the service can assume to read data on your behalf. This role must have
     *         permission for the <code>s3:GetObject</code> action on the object and trust policy that allows Amazon
     *         Kinesis Analytics service principal to assume this role.
     */
    public final String referenceRoleARN() {
        return referenceRoleARN;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketARN());
        hashCode = 31 * hashCode + Objects.hashCode(fileKey());
        hashCode = 31 * hashCode + Objects.hashCode(referenceRoleARN());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3ReferenceDataSource)) {
            return false;
        }
        S3ReferenceDataSource other = (S3ReferenceDataSource) obj;
        return Objects.equals(bucketARN(), other.bucketARN()) && Objects.equals(fileKey(), other.fileKey())
                && Objects.equals(referenceRoleARN(), other.referenceRoleARN());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3ReferenceDataSource").add("BucketARN", bucketARN()).add("FileKey", fileKey())
                .add("ReferenceRoleARN", referenceRoleARN()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketARN":
            return Optional.ofNullable(clazz.cast(bucketARN()));
        case "FileKey":
            return Optional.ofNullable(clazz.cast(fileKey()));
        case "ReferenceRoleARN":
            return Optional.ofNullable(clazz.cast(referenceRoleARN()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3ReferenceDataSource, T> g) {
        return obj -> g.apply((S3ReferenceDataSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3ReferenceDataSource> {
        /**
         * <p>
         * Amazon Resource Name (ARN) of the S3 bucket.
         * </p>
         * 
         * @param bucketARN
         *        Amazon Resource Name (ARN) of the S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketARN(String bucketARN);

        /**
         * <p>
         * Object key name containing reference data.
         * </p>
         * 
         * @param fileKey
         *        Object key name containing reference data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileKey(String fileKey);

        /**
         * <p>
         * ARN of the IAM role that the service can assume to read data on your behalf. This role must have permission
         * for the <code>s3:GetObject</code> action on the object and trust policy that allows Amazon Kinesis Analytics
         * service principal to assume this role.
         * </p>
         * 
         * @param referenceRoleARN
         *        ARN of the IAM role that the service can assume to read data on your behalf. This role must have
         *        permission for the <code>s3:GetObject</code> action on the object and trust policy that allows Amazon
         *        Kinesis Analytics service principal to assume this role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceRoleARN(String referenceRoleARN);
    }

    static final class BuilderImpl implements Builder {
        private String bucketARN;

        private String fileKey;

        private String referenceRoleARN;

        private BuilderImpl() {
        }

        private BuilderImpl(S3ReferenceDataSource model) {
            bucketARN(model.bucketARN);
            fileKey(model.fileKey);
            referenceRoleARN(model.referenceRoleARN);
        }

        public final String getBucketARN() {
            return bucketARN;
        }

        public final void setBucketARN(String bucketARN) {
            this.bucketARN = bucketARN;
        }

        @Override
        public final Builder bucketARN(String bucketARN) {
            this.bucketARN = bucketARN;
            return this;
        }

        public final String getFileKey() {
            return fileKey;
        }

        public final void setFileKey(String fileKey) {
            this.fileKey = fileKey;
        }

        @Override
        public final Builder fileKey(String fileKey) {
            this.fileKey = fileKey;
            return this;
        }

        public final String getReferenceRoleARN() {
            return referenceRoleARN;
        }

        public final void setReferenceRoleARN(String referenceRoleARN) {
            this.referenceRoleARN = referenceRoleARN;
        }

        @Override
        public final Builder referenceRoleARN(String referenceRoleARN) {
            this.referenceRoleARN = referenceRoleARN;
            return this;
        }

        @Override
        public S3ReferenceDataSource build() {
            return new S3ReferenceDataSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
