/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The information that DataSync Discovery collects about a volume in your on-premises storage system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetAppONTAPVolume implements SdkPojo, Serializable,
        ToCopyableBuilder<NetAppONTAPVolume.Builder, NetAppONTAPVolume> {
    private static final SdkField<String> VOLUME_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VolumeName").getter(getter(NetAppONTAPVolume::volumeName)).setter(setter(Builder::volumeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeName").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(NetAppONTAPVolume::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<Long> CIFS_SHARE_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CifsShareCount").getter(getter(NetAppONTAPVolume::cifsShareCount))
            .setter(setter(Builder::cifsShareCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CifsShareCount").build()).build();

    private static final SdkField<String> SECURITY_STYLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityStyle").getter(getter(NetAppONTAPVolume::securityStyle)).setter(setter(Builder::securityStyle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityStyle").build()).build();

    private static final SdkField<String> SVM_UUID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SvmUuid").getter(getter(NetAppONTAPVolume::svmUuid)).setter(setter(Builder::svmUuid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SvmUuid").build()).build();

    private static final SdkField<String> SVM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SvmName").getter(getter(NetAppONTAPVolume::svmName)).setter(setter(Builder::svmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SvmName").build()).build();

    private static final SdkField<Long> CAPACITY_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CapacityUsed").getter(getter(NetAppONTAPVolume::capacityUsed)).setter(setter(Builder::capacityUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityUsed").build()).build();

    private static final SdkField<Long> CAPACITY_PROVISIONED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CapacityProvisioned").getter(getter(NetAppONTAPVolume::capacityProvisioned))
            .setter(setter(Builder::capacityProvisioned))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityProvisioned").build())
            .build();

    private static final SdkField<Long> LOGICAL_CAPACITY_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("LogicalCapacityUsed").getter(getter(NetAppONTAPVolume::logicalCapacityUsed))
            .setter(setter(Builder::logicalCapacityUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalCapacityUsed").build())
            .build();

    private static final SdkField<Boolean> NFS_EXPORTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("NfsExported").getter(getter(NetAppONTAPVolume::nfsExported)).setter(setter(Builder::nfsExported))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NfsExported").build()).build();

    private static final SdkField<Long> SNAPSHOT_CAPACITY_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("SnapshotCapacityUsed").getter(getter(NetAppONTAPVolume::snapshotCapacityUsed))
            .setter(setter(Builder::snapshotCapacityUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotCapacityUsed").build())
            .build();

    private static final SdkField<MaxP95Performance> MAX_P95_PERFORMANCE_FIELD = SdkField
            .<MaxP95Performance> builder(MarshallingType.SDK_POJO).memberName("MaxP95Performance")
            .getter(getter(NetAppONTAPVolume::maxP95Performance)).setter(setter(Builder::maxP95Performance))
            .constructor(MaxP95Performance::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxP95Performance").build()).build();

    private static final SdkField<List<Recommendation>> RECOMMENDATIONS_FIELD = SdkField
            .<List<Recommendation>> builder(MarshallingType.LIST)
            .memberName("Recommendations")
            .getter(getter(NetAppONTAPVolume::recommendations))
            .setter(setter(Builder::recommendations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recommendations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Recommendation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Recommendation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RECOMMENDATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecommendationStatus").getter(getter(NetAppONTAPVolume::recommendationStatusAsString))
            .setter(setter(Builder::recommendationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecommendationStatus").build())
            .build();

    private static final SdkField<Long> LUN_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("LunCount")
            .getter(getter(NetAppONTAPVolume::lunCount)).setter(setter(Builder::lunCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LunCount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOLUME_NAME_FIELD,
            RESOURCE_ID_FIELD, CIFS_SHARE_COUNT_FIELD, SECURITY_STYLE_FIELD, SVM_UUID_FIELD, SVM_NAME_FIELD, CAPACITY_USED_FIELD,
            CAPACITY_PROVISIONED_FIELD, LOGICAL_CAPACITY_USED_FIELD, NFS_EXPORTED_FIELD, SNAPSHOT_CAPACITY_USED_FIELD,
            MAX_P95_PERFORMANCE_FIELD, RECOMMENDATIONS_FIELD, RECOMMENDATION_STATUS_FIELD, LUN_COUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String volumeName;

    private final String resourceId;

    private final Long cifsShareCount;

    private final String securityStyle;

    private final String svmUuid;

    private final String svmName;

    private final Long capacityUsed;

    private final Long capacityProvisioned;

    private final Long logicalCapacityUsed;

    private final Boolean nfsExported;

    private final Long snapshotCapacityUsed;

    private final MaxP95Performance maxP95Performance;

    private final List<Recommendation> recommendations;

    private final String recommendationStatus;

    private final Long lunCount;

    private NetAppONTAPVolume(BuilderImpl builder) {
        this.volumeName = builder.volumeName;
        this.resourceId = builder.resourceId;
        this.cifsShareCount = builder.cifsShareCount;
        this.securityStyle = builder.securityStyle;
        this.svmUuid = builder.svmUuid;
        this.svmName = builder.svmName;
        this.capacityUsed = builder.capacityUsed;
        this.capacityProvisioned = builder.capacityProvisioned;
        this.logicalCapacityUsed = builder.logicalCapacityUsed;
        this.nfsExported = builder.nfsExported;
        this.snapshotCapacityUsed = builder.snapshotCapacityUsed;
        this.maxP95Performance = builder.maxP95Performance;
        this.recommendations = builder.recommendations;
        this.recommendationStatus = builder.recommendationStatus;
        this.lunCount = builder.lunCount;
    }

    /**
     * <p>
     * The name of the volume.
     * </p>
     * 
     * @return The name of the volume.
     */
    public final String volumeName() {
        return volumeName;
    }

    /**
     * <p>
     * The universally unique identifier (UUID) of the volume.
     * </p>
     * 
     * @return The universally unique identifier (UUID) of the volume.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The number of CIFS shares in the volume.
     * </p>
     * 
     * @return The number of CIFS shares in the volume.
     */
    public final Long cifsShareCount() {
        return cifsShareCount;
    }

    /**
     * <p>
     * The volume's security style (such as Unix or NTFS).
     * </p>
     * 
     * @return The volume's security style (such as Unix or NTFS).
     */
    public final String securityStyle() {
        return securityStyle;
    }

    /**
     * <p>
     * The UUID of the storage virtual machine (SVM) associated with the volume.
     * </p>
     * 
     * @return The UUID of the storage virtual machine (SVM) associated with the volume.
     */
    public final String svmUuid() {
        return svmUuid;
    }

    /**
     * <p>
     * The name of the SVM associated with the volume.
     * </p>
     * 
     * @return The name of the SVM associated with the volume.
     */
    public final String svmName() {
        return svmName;
    }

    /**
     * <p>
     * The storage space that's being used in the volume.
     * </p>
     * 
     * @return The storage space that's being used in the volume.
     */
    public final Long capacityUsed() {
        return capacityUsed;
    }

    /**
     * <p>
     * The total storage space that's available in the volume.
     * </p>
     * 
     * @return The total storage space that's available in the volume.
     */
    public final Long capacityProvisioned() {
        return capacityProvisioned;
    }

    /**
     * <p>
     * The storage space that's being used in the volume without accounting for compression or deduplication.
     * </p>
     * 
     * @return The storage space that's being used in the volume without accounting for compression or deduplication.
     */
    public final Long logicalCapacityUsed() {
        return logicalCapacityUsed;
    }

    /**
     * <p>
     * The number of NFS volumes in the volume.
     * </p>
     * 
     * @return The number of NFS volumes in the volume.
     */
    public final Boolean nfsExported() {
        return nfsExported;
    }

    /**
     * <p>
     * The amount of storage in the volume that's being used for snapshots.
     * </p>
     * 
     * @return The amount of storage in the volume that's being used for snapshots.
     */
    public final Long snapshotCapacityUsed() {
        return snapshotCapacityUsed;
    }

    /**
     * <p>
     * The performance data that DataSync Discovery collects about the volume.
     * </p>
     * 
     * @return The performance data that DataSync Discovery collects about the volume.
     */
    public final MaxP95Performance maxP95Performance() {
        return maxP95Performance;
    }

    /**
     * For responses, this returns true if the service returned a value for the Recommendations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendations() {
        return recommendations != null && !(recommendations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more information,
     * see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html">
     * Recommendations provided by DataSync Discovery</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendations} method.
     * </p>
     * 
     * @return The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
     *         >Recommendations provided by DataSync Discovery</a>.
     */
    public final List<Recommendation> recommendations() {
        return recommendations;
    }

    /**
     * <p>
     * Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or can't be
     * determined.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     * >Recommendation statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recommendationStatus} will return {@link RecommendationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #recommendationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or
     *         can't be determined.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     *         >Recommendation statuses</a>.
     * @see RecommendationStatus
     */
    public final RecommendationStatus recommendationStatus() {
        return RecommendationStatus.fromValue(recommendationStatus);
    }

    /**
     * <p>
     * Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or can't be
     * determined.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     * >Recommendation statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recommendationStatus} will return {@link RecommendationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #recommendationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or
     *         can't be determined.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     *         >Recommendation statuses</a>.
     * @see RecommendationStatus
     */
    public final String recommendationStatusAsString() {
        return recommendationStatus;
    }

    /**
     * <p>
     * The number of LUNs (logical unit numbers) in the volume.
     * </p>
     * 
     * @return The number of LUNs (logical unit numbers) in the volume.
     */
    public final Long lunCount() {
        return lunCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(volumeName());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(cifsShareCount());
        hashCode = 31 * hashCode + Objects.hashCode(securityStyle());
        hashCode = 31 * hashCode + Objects.hashCode(svmUuid());
        hashCode = 31 * hashCode + Objects.hashCode(svmName());
        hashCode = 31 * hashCode + Objects.hashCode(capacityUsed());
        hashCode = 31 * hashCode + Objects.hashCode(capacityProvisioned());
        hashCode = 31 * hashCode + Objects.hashCode(logicalCapacityUsed());
        hashCode = 31 * hashCode + Objects.hashCode(nfsExported());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotCapacityUsed());
        hashCode = 31 * hashCode + Objects.hashCode(maxP95Performance());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendations() ? recommendations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recommendationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lunCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetAppONTAPVolume)) {
            return false;
        }
        NetAppONTAPVolume other = (NetAppONTAPVolume) obj;
        return Objects.equals(volumeName(), other.volumeName()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(cifsShareCount(), other.cifsShareCount())
                && Objects.equals(securityStyle(), other.securityStyle()) && Objects.equals(svmUuid(), other.svmUuid())
                && Objects.equals(svmName(), other.svmName()) && Objects.equals(capacityUsed(), other.capacityUsed())
                && Objects.equals(capacityProvisioned(), other.capacityProvisioned())
                && Objects.equals(logicalCapacityUsed(), other.logicalCapacityUsed())
                && Objects.equals(nfsExported(), other.nfsExported())
                && Objects.equals(snapshotCapacityUsed(), other.snapshotCapacityUsed())
                && Objects.equals(maxP95Performance(), other.maxP95Performance())
                && hasRecommendations() == other.hasRecommendations()
                && Objects.equals(recommendations(), other.recommendations())
                && Objects.equals(recommendationStatusAsString(), other.recommendationStatusAsString())
                && Objects.equals(lunCount(), other.lunCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetAppONTAPVolume").add("VolumeName", volumeName()).add("ResourceId", resourceId())
                .add("CifsShareCount", cifsShareCount()).add("SecurityStyle", securityStyle()).add("SvmUuid", svmUuid())
                .add("SvmName", svmName()).add("CapacityUsed", capacityUsed()).add("CapacityProvisioned", capacityProvisioned())
                .add("LogicalCapacityUsed", logicalCapacityUsed()).add("NfsExported", nfsExported())
                .add("SnapshotCapacityUsed", snapshotCapacityUsed()).add("MaxP95Performance", maxP95Performance())
                .add("Recommendations", hasRecommendations() ? recommendations() : null)
                .add("RecommendationStatus", recommendationStatusAsString()).add("LunCount", lunCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VolumeName":
            return Optional.ofNullable(clazz.cast(volumeName()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "CifsShareCount":
            return Optional.ofNullable(clazz.cast(cifsShareCount()));
        case "SecurityStyle":
            return Optional.ofNullable(clazz.cast(securityStyle()));
        case "SvmUuid":
            return Optional.ofNullable(clazz.cast(svmUuid()));
        case "SvmName":
            return Optional.ofNullable(clazz.cast(svmName()));
        case "CapacityUsed":
            return Optional.ofNullable(clazz.cast(capacityUsed()));
        case "CapacityProvisioned":
            return Optional.ofNullable(clazz.cast(capacityProvisioned()));
        case "LogicalCapacityUsed":
            return Optional.ofNullable(clazz.cast(logicalCapacityUsed()));
        case "NfsExported":
            return Optional.ofNullable(clazz.cast(nfsExported()));
        case "SnapshotCapacityUsed":
            return Optional.ofNullable(clazz.cast(snapshotCapacityUsed()));
        case "MaxP95Performance":
            return Optional.ofNullable(clazz.cast(maxP95Performance()));
        case "Recommendations":
            return Optional.ofNullable(clazz.cast(recommendations()));
        case "RecommendationStatus":
            return Optional.ofNullable(clazz.cast(recommendationStatusAsString()));
        case "LunCount":
            return Optional.ofNullable(clazz.cast(lunCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NetAppONTAPVolume, T> g) {
        return obj -> g.apply((NetAppONTAPVolume) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetAppONTAPVolume> {
        /**
         * <p>
         * The name of the volume.
         * </p>
         * 
         * @param volumeName
         *        The name of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeName(String volumeName);

        /**
         * <p>
         * The universally unique identifier (UUID) of the volume.
         * </p>
         * 
         * @param resourceId
         *        The universally unique identifier (UUID) of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The number of CIFS shares in the volume.
         * </p>
         * 
         * @param cifsShareCount
         *        The number of CIFS shares in the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cifsShareCount(Long cifsShareCount);

        /**
         * <p>
         * The volume's security style (such as Unix or NTFS).
         * </p>
         * 
         * @param securityStyle
         *        The volume's security style (such as Unix or NTFS).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityStyle(String securityStyle);

        /**
         * <p>
         * The UUID of the storage virtual machine (SVM) associated with the volume.
         * </p>
         * 
         * @param svmUuid
         *        The UUID of the storage virtual machine (SVM) associated with the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder svmUuid(String svmUuid);

        /**
         * <p>
         * The name of the SVM associated with the volume.
         * </p>
         * 
         * @param svmName
         *        The name of the SVM associated with the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder svmName(String svmName);

        /**
         * <p>
         * The storage space that's being used in the volume.
         * </p>
         * 
         * @param capacityUsed
         *        The storage space that's being used in the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityUsed(Long capacityUsed);

        /**
         * <p>
         * The total storage space that's available in the volume.
         * </p>
         * 
         * @param capacityProvisioned
         *        The total storage space that's available in the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityProvisioned(Long capacityProvisioned);

        /**
         * <p>
         * The storage space that's being used in the volume without accounting for compression or deduplication.
         * </p>
         * 
         * @param logicalCapacityUsed
         *        The storage space that's being used in the volume without accounting for compression or deduplication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalCapacityUsed(Long logicalCapacityUsed);

        /**
         * <p>
         * The number of NFS volumes in the volume.
         * </p>
         * 
         * @param nfsExported
         *        The number of NFS volumes in the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExported(Boolean nfsExported);

        /**
         * <p>
         * The amount of storage in the volume that's being used for snapshots.
         * </p>
         * 
         * @param snapshotCapacityUsed
         *        The amount of storage in the volume that's being used for snapshots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotCapacityUsed(Long snapshotCapacityUsed);

        /**
         * <p>
         * The performance data that DataSync Discovery collects about the volume.
         * </p>
         * 
         * @param maxP95Performance
         *        The performance data that DataSync Discovery collects about the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxP95Performance(MaxP95Performance maxP95Performance);

        /**
         * <p>
         * The performance data that DataSync Discovery collects about the volume.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaxP95Performance.Builder} avoiding the
         * need to create one manually via {@link MaxP95Performance#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaxP95Performance.Builder#build()} is called immediately and its
         * result is passed to {@link #maxP95Performance(MaxP95Performance)}.
         * 
         * @param maxP95Performance
         *        a consumer that will call methods on {@link MaxP95Performance.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maxP95Performance(MaxP95Performance)
         */
        default Builder maxP95Performance(Consumer<MaxP95Performance.Builder> maxP95Performance) {
            return maxP95Performance(MaxP95Performance.builder().applyMutation(maxP95Performance).build());
        }

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * 
         * @param recommendations
         *        The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         *        >Recommendations provided by DataSync Discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Collection<Recommendation> recommendations);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * 
         * @param recommendations
         *        The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         *        >Recommendations provided by DataSync Discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Recommendation... recommendations);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the volume. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.Recommendation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder#build()} is called immediately
         * and its result is passed to {@link #recommendations(List<Recommendation>)}.
         * 
         * @param recommendations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendations(java.util.Collection<Recommendation>)
         */
        Builder recommendations(Consumer<Recommendation.Builder>... recommendations);

        /**
         * <p>
         * Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or can't
         * be determined.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         * >Recommendation statuses</a>.
         * </p>
         * 
         * @param recommendationStatus
         *        Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or
         *        can't be determined.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         *        >Recommendation statuses</a>.
         * @see RecommendationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationStatus
         */
        Builder recommendationStatus(String recommendationStatus);

        /**
         * <p>
         * Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or can't
         * be determined.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         * >Recommendation statuses</a>.
         * </p>
         * 
         * @param recommendationStatus
         *        Indicates whether DataSync Discovery recommendations for the volume are ready to view, incomplete, or
         *        can't be determined.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         *        >Recommendation statuses</a>.
         * @see RecommendationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationStatus
         */
        Builder recommendationStatus(RecommendationStatus recommendationStatus);

        /**
         * <p>
         * The number of LUNs (logical unit numbers) in the volume.
         * </p>
         * 
         * @param lunCount
         *        The number of LUNs (logical unit numbers) in the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lunCount(Long lunCount);
    }

    static final class BuilderImpl implements Builder {
        private String volumeName;

        private String resourceId;

        private Long cifsShareCount;

        private String securityStyle;

        private String svmUuid;

        private String svmName;

        private Long capacityUsed;

        private Long capacityProvisioned;

        private Long logicalCapacityUsed;

        private Boolean nfsExported;

        private Long snapshotCapacityUsed;

        private MaxP95Performance maxP95Performance;

        private List<Recommendation> recommendations = DefaultSdkAutoConstructList.getInstance();

        private String recommendationStatus;

        private Long lunCount;

        private BuilderImpl() {
        }

        private BuilderImpl(NetAppONTAPVolume model) {
            volumeName(model.volumeName);
            resourceId(model.resourceId);
            cifsShareCount(model.cifsShareCount);
            securityStyle(model.securityStyle);
            svmUuid(model.svmUuid);
            svmName(model.svmName);
            capacityUsed(model.capacityUsed);
            capacityProvisioned(model.capacityProvisioned);
            logicalCapacityUsed(model.logicalCapacityUsed);
            nfsExported(model.nfsExported);
            snapshotCapacityUsed(model.snapshotCapacityUsed);
            maxP95Performance(model.maxP95Performance);
            recommendations(model.recommendations);
            recommendationStatus(model.recommendationStatus);
            lunCount(model.lunCount);
        }

        public final String getVolumeName() {
            return volumeName;
        }

        public final void setVolumeName(String volumeName) {
            this.volumeName = volumeName;
        }

        @Override
        public final Builder volumeName(String volumeName) {
            this.volumeName = volumeName;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final Long getCifsShareCount() {
            return cifsShareCount;
        }

        public final void setCifsShareCount(Long cifsShareCount) {
            this.cifsShareCount = cifsShareCount;
        }

        @Override
        public final Builder cifsShareCount(Long cifsShareCount) {
            this.cifsShareCount = cifsShareCount;
            return this;
        }

        public final String getSecurityStyle() {
            return securityStyle;
        }

        public final void setSecurityStyle(String securityStyle) {
            this.securityStyle = securityStyle;
        }

        @Override
        public final Builder securityStyle(String securityStyle) {
            this.securityStyle = securityStyle;
            return this;
        }

        public final String getSvmUuid() {
            return svmUuid;
        }

        public final void setSvmUuid(String svmUuid) {
            this.svmUuid = svmUuid;
        }

        @Override
        public final Builder svmUuid(String svmUuid) {
            this.svmUuid = svmUuid;
            return this;
        }

        public final String getSvmName() {
            return svmName;
        }

        public final void setSvmName(String svmName) {
            this.svmName = svmName;
        }

        @Override
        public final Builder svmName(String svmName) {
            this.svmName = svmName;
            return this;
        }

        public final Long getCapacityUsed() {
            return capacityUsed;
        }

        public final void setCapacityUsed(Long capacityUsed) {
            this.capacityUsed = capacityUsed;
        }

        @Override
        public final Builder capacityUsed(Long capacityUsed) {
            this.capacityUsed = capacityUsed;
            return this;
        }

        public final Long getCapacityProvisioned() {
            return capacityProvisioned;
        }

        public final void setCapacityProvisioned(Long capacityProvisioned) {
            this.capacityProvisioned = capacityProvisioned;
        }

        @Override
        public final Builder capacityProvisioned(Long capacityProvisioned) {
            this.capacityProvisioned = capacityProvisioned;
            return this;
        }

        public final Long getLogicalCapacityUsed() {
            return logicalCapacityUsed;
        }

        public final void setLogicalCapacityUsed(Long logicalCapacityUsed) {
            this.logicalCapacityUsed = logicalCapacityUsed;
        }

        @Override
        public final Builder logicalCapacityUsed(Long logicalCapacityUsed) {
            this.logicalCapacityUsed = logicalCapacityUsed;
            return this;
        }

        public final Boolean getNfsExported() {
            return nfsExported;
        }

        public final void setNfsExported(Boolean nfsExported) {
            this.nfsExported = nfsExported;
        }

        @Override
        public final Builder nfsExported(Boolean nfsExported) {
            this.nfsExported = nfsExported;
            return this;
        }

        public final Long getSnapshotCapacityUsed() {
            return snapshotCapacityUsed;
        }

        public final void setSnapshotCapacityUsed(Long snapshotCapacityUsed) {
            this.snapshotCapacityUsed = snapshotCapacityUsed;
        }

        @Override
        public final Builder snapshotCapacityUsed(Long snapshotCapacityUsed) {
            this.snapshotCapacityUsed = snapshotCapacityUsed;
            return this;
        }

        public final MaxP95Performance.Builder getMaxP95Performance() {
            return maxP95Performance != null ? maxP95Performance.toBuilder() : null;
        }

        public final void setMaxP95Performance(MaxP95Performance.BuilderImpl maxP95Performance) {
            this.maxP95Performance = maxP95Performance != null ? maxP95Performance.build() : null;
        }

        @Override
        public final Builder maxP95Performance(MaxP95Performance maxP95Performance) {
            this.maxP95Performance = maxP95Performance;
            return this;
        }

        public final List<Recommendation.Builder> getRecommendations() {
            List<Recommendation.Builder> result = RecommendationsCopier.copyToBuilder(this.recommendations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendations(Collection<Recommendation.BuilderImpl> recommendations) {
            this.recommendations = RecommendationsCopier.copyFromBuilder(recommendations);
        }

        @Override
        public final Builder recommendations(Collection<Recommendation> recommendations) {
            this.recommendations = RecommendationsCopier.copy(recommendations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Recommendation... recommendations) {
            recommendations(Arrays.asList(recommendations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Consumer<Recommendation.Builder>... recommendations) {
            recommendations(Stream.of(recommendations).map(c -> Recommendation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRecommendationStatus() {
            return recommendationStatus;
        }

        public final void setRecommendationStatus(String recommendationStatus) {
            this.recommendationStatus = recommendationStatus;
        }

        @Override
        public final Builder recommendationStatus(String recommendationStatus) {
            this.recommendationStatus = recommendationStatus;
            return this;
        }

        @Override
        public final Builder recommendationStatus(RecommendationStatus recommendationStatus) {
            this.recommendationStatus(recommendationStatus == null ? null : recommendationStatus.toString());
            return this;
        }

        public final Long getLunCount() {
            return lunCount;
        }

        public final void setLunCount(Long lunCount) {
            this.lunCount = lunCount;
        }

        @Override
        public final Builder lunCount(Long lunCount) {
            this.lunCount = lunCount;
            return this;
        }

        @Override
        public NetAppONTAPVolume build() {
            return new NetAppONTAPVolume(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
