/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.athena.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a data catalog in an Amazon Web Services account.
 * </p>
 * <note>
 * <p>
 * In the Athena console, data catalogs are listed as "data sources" on the <b>Data sources</b> page under the <b>Data
 * source name</b> column.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataCatalog implements SdkPojo, Serializable, ToCopyableBuilder<DataCatalog.Builder, DataCatalog> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DataCatalog::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(DataCatalog::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DataCatalog::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<Map<String, String>> PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Parameters")
            .getter(getter(DataCatalog::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            TYPE_FIELD, PARAMETERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final String type;

    private final Map<String, String> parameters;

    private DataCatalog(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.type = builder.type;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * The name of the data catalog. The catalog name must be unique for the Amazon Web Services account and can use a
     * maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the length constraint of
     * 256 is reserved for use by Athena.
     * </p>
     * 
     * @return The name of the data catalog. The catalog name must be unique for the Amazon Web Services account and can
     *         use a maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the length
     *         constraint of 256 is reserved for use by Athena.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * An optional description of the data catalog.
     * </p>
     * 
     * @return An optional description of the data catalog.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for an
     * external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataCatalogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for an
     *         external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
     * @see DataCatalogType
     */
    public final DataCatalogType type() {
        return DataCatalogType.fromValue(type);
    }

    /**
     * <p>
     * The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for an
     * external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataCatalogType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for an
     *         external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
     * @see DataCatalogType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the Lambda function or functions to use for the data catalog. This is a mapping whose values depend on
     * the catalog type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For the <code>HIVE</code> data catalog type, use the following syntax. The <code>metadata-function</code>
     * parameter is required. <code>The sdk-version</code> parameter is optional and defaults to the currently supported
     * version.
     * </p>
     * <p>
     * <code>metadata-function=<i>lambda_arn</i>, sdk-version=<i>version_number</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * For the <code>LAMBDA</code> data catalog type, use one of the following sets of required parameters, but not
     * both.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you have one Lambda function that processes metadata and another for reading the actual data, use the
     * following syntax. Both parameters are required.
     * </p>
     * <p>
     * <code>metadata-function=<i>lambda_arn</i>, record-function=<i>lambda_arn</i> </code>
     * </p>
     * </li>
     * <li>
     * <p>
     * If you have a composite Lambda function that processes both metadata and data, use the following syntax to
     * specify your Lambda function.
     * </p>
     * <p>
     * <code>function=<i>lambda_arn</i> </code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * The <code>GLUE</code> type takes a catalog ID parameter and is required. The <code> <i>catalog_id</i> </code> is
     * the account ID of the Amazon Web Services account to which the Glue catalog belongs.
     * </p>
     * <p>
     * <code>catalog-id=<i>catalog_id</i> </code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>GLUE</code> data catalog type also applies to the default <code>AwsDataCatalog</code> that already
     * exists in your account, of which you can have only one and cannot modify.
     * </p>
     * </li>
     * <li>
     * <p>
     * Queries that specify a Glue Data Catalog other than the default <code>AwsDataCatalog</code> must be run on Athena
     * engine version 2.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return Specifies the Lambda function or functions to use for the data catalog. This is a mapping whose values
     *         depend on the catalog type. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For the <code>HIVE</code> data catalog type, use the following syntax. The <code>metadata-function</code>
     *         parameter is required. <code>The sdk-version</code> parameter is optional and defaults to the currently
     *         supported version.
     *         </p>
     *         <p>
     *         <code>metadata-function=<i>lambda_arn</i>, sdk-version=<i>version_number</i> </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For the <code>LAMBDA</code> data catalog type, use one of the following sets of required parameters, but
     *         not both.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you have one Lambda function that processes metadata and another for reading the actual data, use the
     *         following syntax. Both parameters are required.
     *         </p>
     *         <p>
     *         <code>metadata-function=<i>lambda_arn</i>, record-function=<i>lambda_arn</i> </code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you have a composite Lambda function that processes both metadata and data, use the following syntax
     *         to specify your Lambda function.
     *         </p>
     *         <p>
     *         <code>function=<i>lambda_arn</i> </code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>GLUE</code> type takes a catalog ID parameter and is required. The
     *         <code> <i>catalog_id</i> </code> is the account ID of the Amazon Web Services account to which the Glue
     *         catalog belongs.
     *         </p>
     *         <p>
     *         <code>catalog-id=<i>catalog_id</i> </code>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>GLUE</code> data catalog type also applies to the default <code>AwsDataCatalog</code> that
     *         already exists in your account, of which you can have only one and cannot modify.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Queries that specify a Glue Data Catalog other than the default <code>AwsDataCatalog</code> must be run
     *         on Athena engine version 2.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public final Map<String, String> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataCatalog)) {
            return false;
        }
        DataCatalog other = (DataCatalog) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(typeAsString(), other.typeAsString()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataCatalog").add("Name", name()).add("Description", description()).add("Type", typeAsString())
                .add("Parameters", hasParameters() ? parameters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataCatalog, T> g) {
        return obj -> g.apply((DataCatalog) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataCatalog> {
        /**
         * <p>
         * The name of the data catalog. The catalog name must be unique for the Amazon Web Services account and can use
         * a maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the length
         * constraint of 256 is reserved for use by Athena.
         * </p>
         * 
         * @param name
         *        The name of the data catalog. The catalog name must be unique for the Amazon Web Services account and
         *        can use a maximum of 127 alphanumeric, underscore, at sign, or hyphen characters. The remainder of the
         *        length constraint of 256 is reserved for use by Athena.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * An optional description of the data catalog.
         * </p>
         * 
         * @param description
         *        An optional description of the data catalog.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for an
         * external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
         * </p>
         * 
         * @param type
         *        The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for
         *        an external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
         * @see DataCatalogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCatalogType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for an
         * external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
         * </p>
         * 
         * @param type
         *        The type of data catalog to create: <code>LAMBDA</code> for a federated catalog, <code>HIVE</code> for
         *        an external hive metastore, or <code>GLUE</code> for an Glue Data Catalog.
         * @see DataCatalogType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataCatalogType
         */
        Builder type(DataCatalogType type);

        /**
         * <p>
         * Specifies the Lambda function or functions to use for the data catalog. This is a mapping whose values depend
         * on the catalog type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For the <code>HIVE</code> data catalog type, use the following syntax. The <code>metadata-function</code>
         * parameter is required. <code>The sdk-version</code> parameter is optional and defaults to the currently
         * supported version.
         * </p>
         * <p>
         * <code>metadata-function=<i>lambda_arn</i>, sdk-version=<i>version_number</i> </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * For the <code>LAMBDA</code> data catalog type, use one of the following sets of required parameters, but not
         * both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you have one Lambda function that processes metadata and another for reading the actual data, use the
         * following syntax. Both parameters are required.
         * </p>
         * <p>
         * <code>metadata-function=<i>lambda_arn</i>, record-function=<i>lambda_arn</i> </code>
         * </p>
         * </li>
         * <li>
         * <p>
         * If you have a composite Lambda function that processes both metadata and data, use the following syntax to
         * specify your Lambda function.
         * </p>
         * <p>
         * <code>function=<i>lambda_arn</i> </code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * The <code>GLUE</code> type takes a catalog ID parameter and is required. The <code> <i>catalog_id</i> </code>
         * is the account ID of the Amazon Web Services account to which the Glue catalog belongs.
         * </p>
         * <p>
         * <code>catalog-id=<i>catalog_id</i> </code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * The <code>GLUE</code> data catalog type also applies to the default <code>AwsDataCatalog</code> that already
         * exists in your account, of which you can have only one and cannot modify.
         * </p>
         * </li>
         * <li>
         * <p>
         * Queries that specify a Glue Data Catalog other than the default <code>AwsDataCatalog</code> must be run on
         * Athena engine version 2.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param parameters
         *        Specifies the Lambda function or functions to use for the data catalog. This is a mapping whose values
         *        depend on the catalog type. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For the <code>HIVE</code> data catalog type, use the following syntax. The
         *        <code>metadata-function</code> parameter is required. <code>The sdk-version</code> parameter is
         *        optional and defaults to the currently supported version.
         *        </p>
         *        <p>
         *        <code>metadata-function=<i>lambda_arn</i>, sdk-version=<i>version_number</i> </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For the <code>LAMBDA</code> data catalog type, use one of the following sets of required parameters,
         *        but not both.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you have one Lambda function that processes metadata and another for reading the actual data, use
         *        the following syntax. Both parameters are required.
         *        </p>
         *        <p>
         *        <code>metadata-function=<i>lambda_arn</i>, record-function=<i>lambda_arn</i> </code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you have a composite Lambda function that processes both metadata and data, use the following
         *        syntax to specify your Lambda function.
         *        </p>
         *        <p>
         *        <code>function=<i>lambda_arn</i> </code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>GLUE</code> type takes a catalog ID parameter and is required. The
         *        <code> <i>catalog_id</i> </code> is the account ID of the Amazon Web Services account to which the
         *        Glue catalog belongs.
         *        </p>
         *        <p>
         *        <code>catalog-id=<i>catalog_id</i> </code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The <code>GLUE</code> data catalog type also applies to the default <code>AwsDataCatalog</code> that
         *        already exists in your account, of which you can have only one and cannot modify.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Queries that specify a Glue Data Catalog other than the default <code>AwsDataCatalog</code> must be
         *        run on Athena engine version 2.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private String type;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataCatalog model) {
            name(model.name);
            description(model.description);
            type(model.type);
            parameters(model.parameters);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DataCatalogType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Map<String, String> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
            return this;
        }

        @Override
        public DataCatalog build() {
            return new DataCatalog(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
