/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an app block.
 * </p>
 * <p>
 * App blocks are an Amazon AppStream 2.0 resource that stores the details about the virtual hard disk in an S3 bucket.
 * It also stores the setup script with details about how to mount the virtual hard disk. The virtual hard disk includes
 * the application binaries and other files necessary to launch your applications. Multiple applications can be assigned
 * to a single app block.
 * </p>
 * <p>
 * This is only supported for Elastic fleets.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppBlock implements SdkPojo, Serializable, ToCopyableBuilder<AppBlock.Builder, AppBlock> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AppBlock::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(AppBlock::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(AppBlock::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(AppBlock::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<S3Location> SOURCE_S3_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .memberName("SourceS3Location").getter(getter(AppBlock::sourceS3Location)).setter(setter(Builder::sourceS3Location))
            .constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceS3Location").build()).build();

    private static final SdkField<ScriptDetails> SETUP_SCRIPT_DETAILS_FIELD = SdkField
            .<ScriptDetails> builder(MarshallingType.SDK_POJO).memberName("SetupScriptDetails")
            .getter(getter(AppBlock::setupScriptDetails)).setter(setter(Builder::setupScriptDetails))
            .constructor(ScriptDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SetupScriptDetails").build())
            .build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(AppBlock::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<ScriptDetails> POST_SETUP_SCRIPT_DETAILS_FIELD = SdkField
            .<ScriptDetails> builder(MarshallingType.SDK_POJO).memberName("PostSetupScriptDetails")
            .getter(getter(AppBlock::postSetupScriptDetails)).setter(setter(Builder::postSetupScriptDetails))
            .constructor(ScriptDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PostSetupScriptDetails").build())
            .build();

    private static final SdkField<String> PACKAGING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PackagingType").getter(getter(AppBlock::packagingTypeAsString)).setter(setter(Builder::packagingType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PackagingType").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(AppBlock::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<List<ErrorDetails>> APP_BLOCK_ERRORS_FIELD = SdkField
            .<List<ErrorDetails>> builder(MarshallingType.LIST)
            .memberName("AppBlockErrors")
            .getter(getter(AppBlock::appBlockErrors))
            .setter(setter(Builder::appBlockErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppBlockErrors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ErrorDetails> builder(MarshallingType.SDK_POJO)
                                            .constructor(ErrorDetails::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            DESCRIPTION_FIELD, DISPLAY_NAME_FIELD, SOURCE_S3_LOCATION_FIELD, SETUP_SCRIPT_DETAILS_FIELD, CREATED_TIME_FIELD,
            POST_SETUP_SCRIPT_DETAILS_FIELD, PACKAGING_TYPE_FIELD, STATE_FIELD, APP_BLOCK_ERRORS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String description;

    private final String displayName;

    private final S3Location sourceS3Location;

    private final ScriptDetails setupScriptDetails;

    private final Instant createdTime;

    private final ScriptDetails postSetupScriptDetails;

    private final String packagingType;

    private final String state;

    private final List<ErrorDetails> appBlockErrors;

    private AppBlock(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.description = builder.description;
        this.displayName = builder.displayName;
        this.sourceS3Location = builder.sourceS3Location;
        this.setupScriptDetails = builder.setupScriptDetails;
        this.createdTime = builder.createdTime;
        this.postSetupScriptDetails = builder.postSetupScriptDetails;
        this.packagingType = builder.packagingType;
        this.state = builder.state;
        this.appBlockErrors = builder.appBlockErrors;
    }

    /**
     * <p>
     * The name of the app block.
     * </p>
     * 
     * @return The name of the app block.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the app block.
     * </p>
     * 
     * @return The ARN of the app block.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The description of the app block.
     * </p>
     * 
     * @return The description of the app block.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The display name of the app block.
     * </p>
     * 
     * @return The display name of the app block.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The source S3 location of the app block.
     * </p>
     * 
     * @return The source S3 location of the app block.
     */
    public final S3Location sourceS3Location() {
        return sourceS3Location;
    }

    /**
     * <p>
     * The setup script details of the app block.
     * </p>
     * <p>
     * This only applies to app blocks with PackagingType <code>CUSTOM</code>.
     * </p>
     * 
     * @return The setup script details of the app block.</p>
     *         <p>
     *         This only applies to app blocks with PackagingType <code>CUSTOM</code>.
     */
    public final ScriptDetails setupScriptDetails() {
        return setupScriptDetails;
    }

    /**
     * <p>
     * The created time of the app block.
     * </p>
     * 
     * @return The created time of the app block.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The post setup script details of the app block.
     * </p>
     * <p>
     * This only applies to app blocks with PackagingType <code>APPSTREAM2</code>.
     * </p>
     * 
     * @return The post setup script details of the app block.</p>
     *         <p>
     *         This only applies to app blocks with PackagingType <code>APPSTREAM2</code>.
     */
    public final ScriptDetails postSetupScriptDetails() {
        return postSetupScriptDetails;
    }

    /**
     * <p>
     * The packaging type of the app block.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #packagingType}
     * will return {@link PackagingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #packagingTypeAsString}.
     * </p>
     * 
     * @return The packaging type of the app block.
     * @see PackagingType
     */
    public final PackagingType packagingType() {
        return PackagingType.fromValue(packagingType);
    }

    /**
     * <p>
     * The packaging type of the app block.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #packagingType}
     * will return {@link PackagingType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #packagingTypeAsString}.
     * </p>
     * 
     * @return The packaging type of the app block.
     * @see PackagingType
     */
    public final String packagingTypeAsString() {
        return packagingType;
    }

    /**
     * <p>
     * The state of the app block.
     * </p>
     * <p>
     * An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application package
     * (VHD) is assigned to it. After an application package (VHD) is created by an app block builder for an app block,
     * it becomes <code>ACTIVE</code>.
     * </p>
     * <p>
     * Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AppBlockState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the app block.</p>
     *         <p>
     *         An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application
     *         package (VHD) is assigned to it. After an application package (VHD) is created by an app block builder
     *         for an app block, it becomes <code>ACTIVE</code>.
     *         </p>
     *         <p>
     *         Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
     * @see AppBlockState
     */
    public final AppBlockState state() {
        return AppBlockState.fromValue(state);
    }

    /**
     * <p>
     * The state of the app block.
     * </p>
     * <p>
     * An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application package
     * (VHD) is assigned to it. After an application package (VHD) is created by an app block builder for an app block,
     * it becomes <code>ACTIVE</code>.
     * </p>
     * <p>
     * Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AppBlockState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the app block.</p>
     *         <p>
     *         An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application
     *         package (VHD) is assigned to it. After an application package (VHD) is created by an app block builder
     *         for an app block, it becomes <code>ACTIVE</code>.
     *         </p>
     *         <p>
     *         Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
     * @see AppBlockState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the AppBlockErrors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAppBlockErrors() {
        return appBlockErrors != null && !(appBlockErrors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The errors of the app block.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAppBlockErrors} method.
     * </p>
     * 
     * @return The errors of the app block.
     */
    public final List<ErrorDetails> appBlockErrors() {
        return appBlockErrors;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(sourceS3Location());
        hashCode = 31 * hashCode + Objects.hashCode(setupScriptDetails());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(postSetupScriptDetails());
        hashCode = 31 * hashCode + Objects.hashCode(packagingTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAppBlockErrors() ? appBlockErrors() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppBlock)) {
            return false;
        }
        AppBlock other = (AppBlock) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(description(), other.description()) && Objects.equals(displayName(), other.displayName())
                && Objects.equals(sourceS3Location(), other.sourceS3Location())
                && Objects.equals(setupScriptDetails(), other.setupScriptDetails())
                && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(postSetupScriptDetails(), other.postSetupScriptDetails())
                && Objects.equals(packagingTypeAsString(), other.packagingTypeAsString())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasAppBlockErrors() == other.hasAppBlockErrors()
                && Objects.equals(appBlockErrors(), other.appBlockErrors());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AppBlock").add("Name", name()).add("Arn", arn()).add("Description", description())
                .add("DisplayName", displayName()).add("SourceS3Location", sourceS3Location())
                .add("SetupScriptDetails", setupScriptDetails()).add("CreatedTime", createdTime())
                .add("PostSetupScriptDetails", postSetupScriptDetails()).add("PackagingType", packagingTypeAsString())
                .add("State", stateAsString()).add("AppBlockErrors", hasAppBlockErrors() ? appBlockErrors() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "SourceS3Location":
            return Optional.ofNullable(clazz.cast(sourceS3Location()));
        case "SetupScriptDetails":
            return Optional.ofNullable(clazz.cast(setupScriptDetails()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "PostSetupScriptDetails":
            return Optional.ofNullable(clazz.cast(postSetupScriptDetails()));
        case "PackagingType":
            return Optional.ofNullable(clazz.cast(packagingTypeAsString()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "AppBlockErrors":
            return Optional.ofNullable(clazz.cast(appBlockErrors()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AppBlock, T> g) {
        return obj -> g.apply((AppBlock) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppBlock> {
        /**
         * <p>
         * The name of the app block.
         * </p>
         * 
         * @param name
         *        The name of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the app block.
         * </p>
         * 
         * @param arn
         *        The ARN of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The description of the app block.
         * </p>
         * 
         * @param description
         *        The description of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The display name of the app block.
         * </p>
         * 
         * @param displayName
         *        The display name of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The source S3 location of the app block.
         * </p>
         * 
         * @param sourceS3Location
         *        The source S3 location of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceS3Location(S3Location sourceS3Location);

        /**
         * <p>
         * The source S3 location of the app block.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Location.Builder} avoiding the need to
         * create one manually via {@link S3Location#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #sourceS3Location(S3Location)}.
         * 
         * @param sourceS3Location
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceS3Location(S3Location)
         */
        default Builder sourceS3Location(Consumer<S3Location.Builder> sourceS3Location) {
            return sourceS3Location(S3Location.builder().applyMutation(sourceS3Location).build());
        }

        /**
         * <p>
         * The setup script details of the app block.
         * </p>
         * <p>
         * This only applies to app blocks with PackagingType <code>CUSTOM</code>.
         * </p>
         * 
         * @param setupScriptDetails
         *        The setup script details of the app block.</p>
         *        <p>
         *        This only applies to app blocks with PackagingType <code>CUSTOM</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder setupScriptDetails(ScriptDetails setupScriptDetails);

        /**
         * <p>
         * The setup script details of the app block.
         * </p>
         * <p>
         * This only applies to app blocks with PackagingType <code>CUSTOM</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScriptDetails.Builder} avoiding the need
         * to create one manually via {@link ScriptDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScriptDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #setupScriptDetails(ScriptDetails)}.
         * 
         * @param setupScriptDetails
         *        a consumer that will call methods on {@link ScriptDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #setupScriptDetails(ScriptDetails)
         */
        default Builder setupScriptDetails(Consumer<ScriptDetails.Builder> setupScriptDetails) {
            return setupScriptDetails(ScriptDetails.builder().applyMutation(setupScriptDetails).build());
        }

        /**
         * <p>
         * The created time of the app block.
         * </p>
         * 
         * @param createdTime
         *        The created time of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The post setup script details of the app block.
         * </p>
         * <p>
         * This only applies to app blocks with PackagingType <code>APPSTREAM2</code>.
         * </p>
         * 
         * @param postSetupScriptDetails
         *        The post setup script details of the app block.</p>
         *        <p>
         *        This only applies to app blocks with PackagingType <code>APPSTREAM2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder postSetupScriptDetails(ScriptDetails postSetupScriptDetails);

        /**
         * <p>
         * The post setup script details of the app block.
         * </p>
         * <p>
         * This only applies to app blocks with PackagingType <code>APPSTREAM2</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScriptDetails.Builder} avoiding the need
         * to create one manually via {@link ScriptDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScriptDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #postSetupScriptDetails(ScriptDetails)}.
         * 
         * @param postSetupScriptDetails
         *        a consumer that will call methods on {@link ScriptDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #postSetupScriptDetails(ScriptDetails)
         */
        default Builder postSetupScriptDetails(Consumer<ScriptDetails.Builder> postSetupScriptDetails) {
            return postSetupScriptDetails(ScriptDetails.builder().applyMutation(postSetupScriptDetails).build());
        }

        /**
         * <p>
         * The packaging type of the app block.
         * </p>
         * 
         * @param packagingType
         *        The packaging type of the app block.
         * @see PackagingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackagingType
         */
        Builder packagingType(String packagingType);

        /**
         * <p>
         * The packaging type of the app block.
         * </p>
         * 
         * @param packagingType
         *        The packaging type of the app block.
         * @see PackagingType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PackagingType
         */
        Builder packagingType(PackagingType packagingType);

        /**
         * <p>
         * The state of the app block.
         * </p>
         * <p>
         * An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application
         * package (VHD) is assigned to it. After an application package (VHD) is created by an app block builder for an
         * app block, it becomes <code>ACTIVE</code>.
         * </p>
         * <p>
         * Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
         * </p>
         * 
         * @param state
         *        The state of the app block.</p>
         *        <p>
         *        An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application
         *        package (VHD) is assigned to it. After an application package (VHD) is created by an app block builder
         *        for an app block, it becomes <code>ACTIVE</code>.
         *        </p>
         *        <p>
         *        Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
         * @see AppBlockState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppBlockState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the app block.
         * </p>
         * <p>
         * An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application
         * package (VHD) is assigned to it. After an application package (VHD) is created by an app block builder for an
         * app block, it becomes <code>ACTIVE</code>.
         * </p>
         * <p>
         * Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
         * </p>
         * 
         * @param state
         *        The state of the app block.</p>
         *        <p>
         *        An app block with AppStream 2.0 packaging will be in the <code>INACTIVE</code> state if no application
         *        package (VHD) is assigned to it. After an application package (VHD) is created by an app block builder
         *        for an app block, it becomes <code>ACTIVE</code>.
         *        </p>
         *        <p>
         *        Custom app blocks are always in the <code>ACTIVE</code> state and no action is required to use them.
         * @see AppBlockState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppBlockState
         */
        Builder state(AppBlockState state);

        /**
         * <p>
         * The errors of the app block.
         * </p>
         * 
         * @param appBlockErrors
         *        The errors of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBlockErrors(Collection<ErrorDetails> appBlockErrors);

        /**
         * <p>
         * The errors of the app block.
         * </p>
         * 
         * @param appBlockErrors
         *        The errors of the app block.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBlockErrors(ErrorDetails... appBlockErrors);

        /**
         * <p>
         * The errors of the app block.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.ErrorDetails.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.appstream.model.ErrorDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.ErrorDetails.Builder#build()} is called immediately
         * and its result is passed to {@link #appBlockErrors(List<ErrorDetails>)}.
         * 
         * @param appBlockErrors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.ErrorDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #appBlockErrors(java.util.Collection<ErrorDetails>)
         */
        Builder appBlockErrors(Consumer<ErrorDetails.Builder>... appBlockErrors);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String description;

        private String displayName;

        private S3Location sourceS3Location;

        private ScriptDetails setupScriptDetails;

        private Instant createdTime;

        private ScriptDetails postSetupScriptDetails;

        private String packagingType;

        private String state;

        private List<ErrorDetails> appBlockErrors = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AppBlock model) {
            name(model.name);
            arn(model.arn);
            description(model.description);
            displayName(model.displayName);
            sourceS3Location(model.sourceS3Location);
            setupScriptDetails(model.setupScriptDetails);
            createdTime(model.createdTime);
            postSetupScriptDetails(model.postSetupScriptDetails);
            packagingType(model.packagingType);
            state(model.state);
            appBlockErrors(model.appBlockErrors);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final S3Location.Builder getSourceS3Location() {
            return sourceS3Location != null ? sourceS3Location.toBuilder() : null;
        }

        public final void setSourceS3Location(S3Location.BuilderImpl sourceS3Location) {
            this.sourceS3Location = sourceS3Location != null ? sourceS3Location.build() : null;
        }

        @Override
        public final Builder sourceS3Location(S3Location sourceS3Location) {
            this.sourceS3Location = sourceS3Location;
            return this;
        }

        public final ScriptDetails.Builder getSetupScriptDetails() {
            return setupScriptDetails != null ? setupScriptDetails.toBuilder() : null;
        }

        public final void setSetupScriptDetails(ScriptDetails.BuilderImpl setupScriptDetails) {
            this.setupScriptDetails = setupScriptDetails != null ? setupScriptDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder setupScriptDetails(ScriptDetails setupScriptDetails) {
            this.setupScriptDetails = setupScriptDetails;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final ScriptDetails.Builder getPostSetupScriptDetails() {
            return postSetupScriptDetails != null ? postSetupScriptDetails.toBuilder() : null;
        }

        public final void setPostSetupScriptDetails(ScriptDetails.BuilderImpl postSetupScriptDetails) {
            this.postSetupScriptDetails = postSetupScriptDetails != null ? postSetupScriptDetails.build() : null;
        }

        @Override
        public final Builder postSetupScriptDetails(ScriptDetails postSetupScriptDetails) {
            this.postSetupScriptDetails = postSetupScriptDetails;
            return this;
        }

        public final String getPackagingType() {
            return packagingType;
        }

        public final void setPackagingType(String packagingType) {
            this.packagingType = packagingType;
        }

        @Override
        public final Builder packagingType(String packagingType) {
            this.packagingType = packagingType;
            return this;
        }

        @Override
        public final Builder packagingType(PackagingType packagingType) {
            this.packagingType(packagingType == null ? null : packagingType.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(AppBlockState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<ErrorDetails.Builder> getAppBlockErrors() {
            List<ErrorDetails.Builder> result = ErrorDetailsListCopier.copyToBuilder(this.appBlockErrors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAppBlockErrors(Collection<ErrorDetails.BuilderImpl> appBlockErrors) {
            this.appBlockErrors = ErrorDetailsListCopier.copyFromBuilder(appBlockErrors);
        }

        @Override
        public final Builder appBlockErrors(Collection<ErrorDetails> appBlockErrors) {
            this.appBlockErrors = ErrorDetailsListCopier.copy(appBlockErrors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appBlockErrors(ErrorDetails... appBlockErrors) {
            appBlockErrors(Arrays.asList(appBlockErrors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appBlockErrors(Consumer<ErrorDetails.Builder>... appBlockErrors) {
            appBlockErrors(Stream.of(appBlockErrors).map(c -> ErrorDetails.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public AppBlock build() {
            return new AppBlock(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
