/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.budgets.model.AccessDeniedException;
import software.amazon.awssdk.services.budgets.model.BudgetsException;
import software.amazon.awssdk.services.budgets.model.CreateBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.CreateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.CreateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.CreateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.CreationLimitExceededException;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetNotificationsForAccountRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetNotificationsForAccountResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DuplicateRecordException;
import software.amazon.awssdk.services.budgets.model.ExecuteBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.ExecuteBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.budgets.model.InternalErrorException;
import software.amazon.awssdk.services.budgets.model.InvalidNextTokenException;
import software.amazon.awssdk.services.budgets.model.InvalidParameterException;
import software.amazon.awssdk.services.budgets.model.NotFoundException;
import software.amazon.awssdk.services.budgets.model.ResourceLockedException;
import software.amazon.awssdk.services.budgets.model.ThrottlingException;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.budgets.transform.CreateBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionHistoriesRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionsForAccountRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionsForBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetNotificationsForAccountRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetPerformanceHistoryRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetsRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeNotificationsForBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeSubscribersForNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.ExecuteBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateSubscriberRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BudgetsAsyncClient}.
 *
 * @see BudgetsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBudgetsAsyncClient implements BudgetsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBudgetsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final BudgetsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultBudgetsAsyncClient(BudgetsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_CreateBudget.html#API_CreateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param createBudgetRequest
     *        Request of CreateBudget
     * @return A Java Future containing the result of the CreateBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateBudget
     */
    @Override
    public CompletableFuture<CreateBudgetResponse> createBudget(CreateBudgetRequest createBudgetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBudget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBudgetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBudgetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBudgetRequest, CreateBudgetResponse>()
                            .withOperationName("CreateBudget").withMarshaller(new CreateBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBudgetRequest));
            CompletableFuture<CreateBudgetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a budget action.
     * </p>
     *
     * @param createBudgetActionRequest
     * @return A Java Future containing the result of the CreateBudgetAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateBudgetAction
     */
    @Override
    public CompletableFuture<CreateBudgetActionResponse> createBudgetAction(CreateBudgetActionRequest createBudgetActionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBudgetAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBudgetActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBudgetActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBudgetActionRequest, CreateBudgetActionResponse>()
                            .withOperationName("CreateBudgetAction")
                            .withMarshaller(new CreateBudgetActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBudgetActionRequest));
            CompletableFuture<CreateBudgetActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     *
     * @param createNotificationRequest
     *        Request of CreateNotification
     * @return A Java Future containing the result of the CreateNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateNotification
     */
    @Override
    public CompletableFuture<CreateNotificationResponse> createNotification(CreateNotificationRequest createNotificationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNotificationRequest, CreateNotificationResponse>()
                            .withOperationName("CreateNotification")
                            .withMarshaller(new CreateNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createNotificationRequest));
            CompletableFuture<CreateNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     *
     * @param createSubscriberRequest
     *        Request of CreateSubscriber
     * @return A Java Future containing the result of the CreateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateSubscriber
     */
    @Override
    public CompletableFuture<CreateSubscriberResponse> createSubscriber(CreateSubscriberRequest createSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriberRequest, CreateSubscriberResponse>()
                            .withOperationName("CreateSubscriber")
                            .withMarshaller(new CreateSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSubscriberRequest));
            CompletableFuture<CreateSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <important>
     * <p>
     * Deleting a budget also deletes the notifications and subscribers that are associated with that budget.
     * </p>
     * </important>
     *
     * @param deleteBudgetRequest
     *        Request of DeleteBudget
     * @return A Java Future containing the result of the DeleteBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteBudget
     */
    @Override
    public CompletableFuture<DeleteBudgetResponse> deleteBudget(DeleteBudgetRequest deleteBudgetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBudget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBudgetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBudgetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBudgetRequest, DeleteBudgetResponse>()
                            .withOperationName("DeleteBudget").withMarshaller(new DeleteBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBudgetRequest));
            CompletableFuture<DeleteBudgetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a budget action.
     * </p>
     *
     * @param deleteBudgetActionRequest
     * @return A Java Future containing the result of the DeleteBudgetAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ResourceLockedException The request was received and recognized by the server, but the server
     *         rejected that particular method for the requested resource.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteBudgetAction
     */
    @Override
    public CompletableFuture<DeleteBudgetActionResponse> deleteBudgetAction(DeleteBudgetActionRequest deleteBudgetActionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBudgetAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBudgetActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBudgetActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBudgetActionRequest, DeleteBudgetActionResponse>()
                            .withOperationName("DeleteBudgetAction")
                            .withMarshaller(new DeleteBudgetActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBudgetActionRequest));
            CompletableFuture<DeleteBudgetActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <important>
     * <p>
     * Deleting a notification also deletes the subscribers that are associated with the notification.
     * </p>
     * </important>
     *
     * @param deleteNotificationRequest
     *        Request of DeleteNotification
     * @return A Java Future containing the result of the DeleteNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteNotification
     */
    @Override
    public CompletableFuture<DeleteNotificationResponse> deleteNotification(DeleteNotificationRequest deleteNotificationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationRequest, DeleteNotificationResponse>()
                            .withOperationName("DeleteNotification")
                            .withMarshaller(new DeleteNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteNotificationRequest));
            CompletableFuture<DeleteNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <important>
     * <p>
     * Deleting the last subscriber to a notification also deletes the notification.
     * </p>
     * </important>
     *
     * @param deleteSubscriberRequest
     *        Request of DeleteSubscriber
     * @return A Java Future containing the result of the DeleteSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteSubscriber
     */
    @Override
    public CompletableFuture<DeleteSubscriberResponse> deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubscriberRequest, DeleteSubscriberResponse>()
                            .withOperationName("DeleteSubscriber")
                            .withMarshaller(new DeleteSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSubscriberRequest));
            CompletableFuture<DeleteSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudget.html#API_DescribeBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param describeBudgetRequest
     *        Request of DescribeBudget
     * @return A Java Future containing the result of the DescribeBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudget
     */
    @Override
    public CompletableFuture<DescribeBudgetResponse> describeBudget(DescribeBudgetRequest describeBudgetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBudgetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetRequest, DescribeBudgetResponse>()
                            .withOperationName("DescribeBudget")
                            .withMarshaller(new DescribeBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetRequest));
            CompletableFuture<DescribeBudgetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a budget action detail.
     * </p>
     *
     * @param describeBudgetActionRequest
     * @return A Java Future containing the result of the DescribeBudgetAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgetAction
     */
    @Override
    public CompletableFuture<DescribeBudgetActionResponse> describeBudgetAction(
            DescribeBudgetActionRequest describeBudgetActionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBudgetActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionRequest, DescribeBudgetActionResponse>()
                            .withOperationName("DescribeBudgetAction")
                            .withMarshaller(new DescribeBudgetActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetActionRequest));
            CompletableFuture<DescribeBudgetActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a budget action history detail.
     * </p>
     *
     * @param describeBudgetActionHistoriesRequest
     * @return A Java Future containing the result of the DescribeBudgetActionHistories operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgetActionHistories
     */
    @Override
    public CompletableFuture<DescribeBudgetActionHistoriesResponse> describeBudgetActionHistories(
            DescribeBudgetActionHistoriesRequest describeBudgetActionHistoriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetActionHistoriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetActionHistories");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetActionHistoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBudgetActionHistoriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetActionHistoriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionHistoriesRequest, DescribeBudgetActionHistoriesResponse>()
                            .withOperationName("DescribeBudgetActionHistories")
                            .withMarshaller(new DescribeBudgetActionHistoriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetActionHistoriesRequest));
            CompletableFuture<DescribeBudgetActionHistoriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes all of the budget actions for an account.
     * </p>
     *
     * @param describeBudgetActionsForAccountRequest
     * @return A Java Future containing the result of the DescribeBudgetActionsForAccount operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgetActionsForAccount
     */
    @Override
    public CompletableFuture<DescribeBudgetActionsForAccountResponse> describeBudgetActionsForAccount(
            DescribeBudgetActionsForAccountRequest describeBudgetActionsForAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetActionsForAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetActionsForAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetActionsForAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBudgetActionsForAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetActionsForAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionsForAccountRequest, DescribeBudgetActionsForAccountResponse>()
                            .withOperationName("DescribeBudgetActionsForAccount")
                            .withMarshaller(new DescribeBudgetActionsForAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetActionsForAccountRequest));
            CompletableFuture<DescribeBudgetActionsForAccountResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes all of the budget actions for a budget.
     * </p>
     *
     * @param describeBudgetActionsForBudgetRequest
     * @return A Java Future containing the result of the DescribeBudgetActionsForBudget operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgetActionsForBudget
     */
    @Override
    public CompletableFuture<DescribeBudgetActionsForBudgetResponse> describeBudgetActionsForBudget(
            DescribeBudgetActionsForBudgetRequest describeBudgetActionsForBudgetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetActionsForBudgetRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetActionsForBudget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetActionsForBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBudgetActionsForBudgetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetActionsForBudgetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionsForBudgetRequest, DescribeBudgetActionsForBudgetResponse>()
                            .withOperationName("DescribeBudgetActionsForBudget")
                            .withMarshaller(new DescribeBudgetActionsForBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetActionsForBudgetRequest));
            CompletableFuture<DescribeBudgetActionsForBudgetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the budget names and notifications that are associated with an account.
     * </p>
     *
     * @param describeBudgetNotificationsForAccountRequest
     * @return A Java Future containing the result of the DescribeBudgetNotificationsForAccount operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgetNotificationsForAccount
     */
    @Override
    public CompletableFuture<DescribeBudgetNotificationsForAccountResponse> describeBudgetNotificationsForAccount(
            DescribeBudgetNotificationsForAccountRequest describeBudgetNotificationsForAccountRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetNotificationsForAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetNotificationsForAccount");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetNotificationsForAccountResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeBudgetNotificationsForAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetNotificationsForAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetNotificationsForAccountRequest, DescribeBudgetNotificationsForAccountResponse>()
                            .withOperationName("DescribeBudgetNotificationsForAccount")
                            .withMarshaller(new DescribeBudgetNotificationsForAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetNotificationsForAccountRequest));
            CompletableFuture<DescribeBudgetNotificationsForAccountResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the history for <code>DAILY</code>, <code>MONTHLY</code>, and <code>QUARTERLY</code> budgets. Budget
     * history isn't available for <code>ANNUAL</code> budgets.
     * </p>
     *
     * @param describeBudgetPerformanceHistoryRequest
     * @return A Java Future containing the result of the DescribeBudgetPerformanceHistory operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgetPerformanceHistory
     */
    @Override
    public CompletableFuture<DescribeBudgetPerformanceHistoryResponse> describeBudgetPerformanceHistory(
            DescribeBudgetPerformanceHistoryRequest describeBudgetPerformanceHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetPerformanceHistoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetPerformanceHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetPerformanceHistoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeBudgetPerformanceHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetPerformanceHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetPerformanceHistoryRequest, DescribeBudgetPerformanceHistoryResponse>()
                            .withOperationName("DescribeBudgetPerformanceHistory")
                            .withMarshaller(new DescribeBudgetPerformanceHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetPerformanceHistoryRequest));
            CompletableFuture<DescribeBudgetPerformanceHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the budgets that are associated with an account.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgets.html#API_DescribeBudgets_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param describeBudgetsRequest
     *        Request of DescribeBudgets
     * @return A Java Future containing the result of the DescribeBudgets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgets
     */
    @Override
    public CompletableFuture<DescribeBudgetsResponse> describeBudgets(DescribeBudgetsRequest describeBudgetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBudgetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBudgetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBudgetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBudgetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetsRequest, DescribeBudgetsResponse>()
                            .withOperationName("DescribeBudgets")
                            .withMarshaller(new DescribeBudgetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBudgetsRequest));
            CompletableFuture<DescribeBudgetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the notifications that are associated with a budget.
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        Request of DescribeNotificationsForBudget
     * @return A Java Future containing the result of the DescribeNotificationsForBudget operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeNotificationsForBudget
     */
    @Override
    public CompletableFuture<DescribeNotificationsForBudgetResponse> describeNotificationsForBudget(
            DescribeNotificationsForBudgetRequest describeNotificationsForBudgetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationsForBudgetRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationsForBudget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNotificationsForBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeNotificationsForBudgetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNotificationsForBudgetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationsForBudgetRequest, DescribeNotificationsForBudgetResponse>()
                            .withOperationName("DescribeNotificationsForBudget")
                            .withMarshaller(new DescribeNotificationsForBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeNotificationsForBudgetRequest));
            CompletableFuture<DescribeNotificationsForBudgetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the subscribers that are associated with a notification.
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        Request of DescribeSubscribersForNotification
     * @return A Java Future containing the result of the DescribeSubscribersForNotification operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeSubscribersForNotification
     */
    @Override
    public CompletableFuture<DescribeSubscribersForNotificationResponse> describeSubscribersForNotification(
            DescribeSubscribersForNotificationRequest describeSubscribersForNotificationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSubscribersForNotificationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSubscribersForNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSubscribersForNotificationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeSubscribersForNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSubscribersForNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSubscribersForNotificationRequest, DescribeSubscribersForNotificationResponse>()
                            .withOperationName("DescribeSubscribersForNotification")
                            .withMarshaller(new DescribeSubscribersForNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSubscribersForNotificationRequest));
            CompletableFuture<DescribeSubscribersForNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Executes a budget action.
     * </p>
     *
     * @param executeBudgetActionRequest
     * @return A Java Future containing the result of the ExecuteBudgetAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ResourceLockedException The request was received and recognized by the server, but the server
     *         rejected that particular method for the requested resource.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.ExecuteBudgetAction
     */
    @Override
    public CompletableFuture<ExecuteBudgetActionResponse> executeBudgetAction(
            ExecuteBudgetActionRequest executeBudgetActionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteBudgetAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExecuteBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExecuteBudgetActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExecuteBudgetActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExecuteBudgetActionRequest, ExecuteBudgetActionResponse>()
                            .withOperationName("ExecuteBudgetAction")
                            .withMarshaller(new ExecuteBudgetActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(executeBudgetActionRequest));
            CompletableFuture<ExecuteBudgetActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When you modify a budget, the <code>calculatedSpend</code> drops to zero until
     * Amazon Web Services has new usage data to use for forecasting.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_UpdateBudget.html#API_UpdateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param updateBudgetRequest
     *        Request of UpdateBudget
     * @return A Java Future containing the result of the UpdateBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateBudget
     */
    @Override
    public CompletableFuture<UpdateBudgetResponse> updateBudget(UpdateBudgetRequest updateBudgetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBudget");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateBudgetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBudgetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBudgetRequest, UpdateBudgetResponse>()
                            .withOperationName("UpdateBudget").withMarshaller(new UpdateBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBudgetRequest));
            CompletableFuture<UpdateBudgetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a budget action.
     * </p>
     *
     * @param updateBudgetActionRequest
     * @return A Java Future containing the result of the UpdateBudgetAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ResourceLockedException The request was received and recognized by the server, but the server
     *         rejected that particular method for the requested resource.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateBudgetAction
     */
    @Override
    public CompletableFuture<UpdateBudgetActionResponse> updateBudgetAction(UpdateBudgetActionRequest updateBudgetActionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBudgetAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBudgetActionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBudgetActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBudgetActionRequest, UpdateBudgetActionResponse>()
                            .withOperationName("UpdateBudgetAction")
                            .withMarshaller(new UpdateBudgetActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBudgetActionRequest));
            CompletableFuture<UpdateBudgetActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     *
     * @param updateNotificationRequest
     *        Request of UpdateNotification
     * @return A Java Future containing the result of the UpdateNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateNotification
     */
    @Override
    public CompletableFuture<UpdateNotificationResponse> updateNotification(UpdateNotificationRequest updateNotificationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotification");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNotificationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNotificationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationRequest, UpdateNotificationResponse>()
                            .withOperationName("UpdateNotification")
                            .withMarshaller(new UpdateNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateNotificationRequest));
            CompletableFuture<UpdateNotificationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     *
     * @param updateSubscriberRequest
     *        Request of UpdateSubscriber
     * @return A Java Future containing the result of the UpdateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The number of API requests has exceeded the maximum allowed API request
     *         throttling limit for the account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateSubscriber
     */
    @Override
    public CompletableFuture<UpdateSubscriberResponse> updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriber");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubscriberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriberRequest, UpdateSubscriberResponse>()
                            .withOperationName("UpdateSubscriber")
                            .withMarshaller(new UpdateSubscriberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSubscriberRequest));
            CompletableFuture<UpdateSubscriberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BudgetsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BudgetsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLockedException")
                                .exceptionBuilderSupplier(ResourceLockedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateRecordException")
                                .exceptionBuilderSupplier(DuplicateRecordException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CreationLimitExceededException")
                                .exceptionBuilderSupplier(CreationLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExpiredNextTokenException")
                                .exceptionBuilderSupplier(ExpiredNextTokenException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
