/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The subscriber to a budget notification. The subscriber consists of a subscription type and either an Amazon SNS
 * topic or an email address.
 * </p>
 * <p>
 * For example, an email subscriber would have the following parameters:
 * </p>
 * <ul>
 * <li>
 * <p>
 * A <code>subscriptionType</code> of <code>EMAIL</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * An <code>address</code> of <code>example@example.com</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Subscriber implements SdkPojo, Serializable, ToCopyableBuilder<Subscriber.Builder, Subscriber> {
    private static final SdkField<String> SUBSCRIPTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Subscriber::subscriptionTypeAsString)).setter(setter(Builder::subscriptionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubscriptionType").build()).build();

    private static final SdkField<String> ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Subscriber::address)).setter(setter(Builder::address))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Address").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBSCRIPTION_TYPE_FIELD,
            ADDRESS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String subscriptionType;

    private final String address;

    private Subscriber(BuilderImpl builder) {
        this.subscriptionType = builder.subscriptionType;
        this.address = builder.address;
    }

    /**
     * <p>
     * The type of notification that AWS sends to a subscriber.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subscriptionType}
     * will return {@link SubscriptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #subscriptionTypeAsString}.
     * </p>
     * 
     * @return The type of notification that AWS sends to a subscriber.
     * @see SubscriptionType
     */
    public SubscriptionType subscriptionType() {
        return SubscriptionType.fromValue(subscriptionType);
    }

    /**
     * <p>
     * The type of notification that AWS sends to a subscriber.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subscriptionType}
     * will return {@link SubscriptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #subscriptionTypeAsString}.
     * </p>
     * 
     * @return The type of notification that AWS sends to a subscriber.
     * @see SubscriptionType
     */
    public String subscriptionTypeAsString() {
        return subscriptionType;
    }

    /**
     * <p>
     * The address that AWS sends budget notifications to, either an SNS topic or an email.
     * </p>
     * <p>
     * When you create a subscriber, the value of <code>Address</code> can't contain line breaks.
     * </p>
     * 
     * @return The address that AWS sends budget notifications to, either an SNS topic or an email.</p>
     *         <p>
     *         When you create a subscriber, the value of <code>Address</code> can't contain line breaks.
     */
    public String address() {
        return address;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(subscriptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(address());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Subscriber)) {
            return false;
        }
        Subscriber other = (Subscriber) obj;
        return Objects.equals(subscriptionTypeAsString(), other.subscriptionTypeAsString())
                && Objects.equals(address(), other.address());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Subscriber").add("SubscriptionType", subscriptionTypeAsString())
                .add("Address", address() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubscriptionType":
            return Optional.ofNullable(clazz.cast(subscriptionTypeAsString()));
        case "Address":
            return Optional.ofNullable(clazz.cast(address()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Subscriber, T> g) {
        return obj -> g.apply((Subscriber) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Subscriber> {
        /**
         * <p>
         * The type of notification that AWS sends to a subscriber.
         * </p>
         * 
         * @param subscriptionType
         *        The type of notification that AWS sends to a subscriber.
         * @see SubscriptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriptionType
         */
        Builder subscriptionType(String subscriptionType);

        /**
         * <p>
         * The type of notification that AWS sends to a subscriber.
         * </p>
         * 
         * @param subscriptionType
         *        The type of notification that AWS sends to a subscriber.
         * @see SubscriptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriptionType
         */
        Builder subscriptionType(SubscriptionType subscriptionType);

        /**
         * <p>
         * The address that AWS sends budget notifications to, either an SNS topic or an email.
         * </p>
         * <p>
         * When you create a subscriber, the value of <code>Address</code> can't contain line breaks.
         * </p>
         * 
         * @param address
         *        The address that AWS sends budget notifications to, either an SNS topic or an email.</p>
         *        <p>
         *        When you create a subscriber, the value of <code>Address</code> can't contain line breaks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder address(String address);
    }

    static final class BuilderImpl implements Builder {
        private String subscriptionType;

        private String address;

        private BuilderImpl() {
        }

        private BuilderImpl(Subscriber model) {
            subscriptionType(model.subscriptionType);
            address(model.address);
        }

        public final String getSubscriptionType() {
            return subscriptionType;
        }

        @Override
        public final Builder subscriptionType(String subscriptionType) {
            this.subscriptionType = subscriptionType;
            return this;
        }

        @Override
        public final Builder subscriptionType(SubscriptionType subscriptionType) {
            this.subscriptionType(subscriptionType == null ? null : subscriptionType.toString());
            return this;
        }

        public final void setSubscriptionType(String subscriptionType) {
            this.subscriptionType = subscriptionType;
        }

        public final String getAddress() {
            return address;
        }

        @Override
        public final Builder address(String address) {
            this.address = address;
            return this;
        }

        public final void setAddress(String address) {
            this.address = address;
        }

        @Override
        public Subscriber build() {
            return new Subscriber(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
