/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents the details for an Batch job queue.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobQueueDetail implements SdkPojo, Serializable, ToCopyableBuilder<JobQueueDetail.Builder, JobQueueDetail> {
    private static final SdkField<String> JOB_QUEUE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobQueueName").getter(getter(JobQueueDetail::jobQueueName)).setter(setter(Builder::jobQueueName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobQueueName").build()).build();

    private static final SdkField<String> JOB_QUEUE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobQueueArn").getter(getter(JobQueueDetail::jobQueueArn)).setter(setter(Builder::jobQueueArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobQueueArn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(JobQueueDetail::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> SCHEDULING_POLICY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("schedulingPolicyArn").getter(getter(JobQueueDetail::schedulingPolicyArn))
            .setter(setter(Builder::schedulingPolicyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schedulingPolicyArn").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(JobQueueDetail::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(JobQueueDetail::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("priority").getter(getter(JobQueueDetail::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("priority").build()).build();

    private static final SdkField<List<ComputeEnvironmentOrder>> COMPUTE_ENVIRONMENT_ORDER_FIELD = SdkField
            .<List<ComputeEnvironmentOrder>> builder(MarshallingType.LIST)
            .memberName("computeEnvironmentOrder")
            .getter(getter(JobQueueDetail::computeEnvironmentOrder))
            .setter(setter(Builder::computeEnvironmentOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeEnvironmentOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ComputeEnvironmentOrder> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComputeEnvironmentOrder::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ServiceEnvironmentOrder>> SERVICE_ENVIRONMENT_ORDER_FIELD = SdkField
            .<List<ServiceEnvironmentOrder>> builder(MarshallingType.LIST)
            .memberName("serviceEnvironmentOrder")
            .getter(getter(JobQueueDetail::serviceEnvironmentOrder))
            .setter(setter(Builder::serviceEnvironmentOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceEnvironmentOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ServiceEnvironmentOrder> builder(MarshallingType.SDK_POJO)
                                            .constructor(ServiceEnvironmentOrder::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> JOB_QUEUE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobQueueType").getter(getter(JobQueueDetail::jobQueueTypeAsString))
            .setter(setter(Builder::jobQueueType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobQueueType").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(JobQueueDetail::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<JobStateTimeLimitAction>> JOB_STATE_TIME_LIMIT_ACTIONS_FIELD = SdkField
            .<List<JobStateTimeLimitAction>> builder(MarshallingType.LIST)
            .memberName("jobStateTimeLimitActions")
            .getter(getter(JobQueueDetail::jobStateTimeLimitActions))
            .setter(setter(Builder::jobStateTimeLimitActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobStateTimeLimitActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<JobStateTimeLimitAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(JobStateTimeLimitAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_QUEUE_NAME_FIELD,
            JOB_QUEUE_ARN_FIELD, STATE_FIELD, SCHEDULING_POLICY_ARN_FIELD, STATUS_FIELD, STATUS_REASON_FIELD, PRIORITY_FIELD,
            COMPUTE_ENVIRONMENT_ORDER_FIELD, SERVICE_ENVIRONMENT_ORDER_FIELD, JOB_QUEUE_TYPE_FIELD, TAGS_FIELD,
            JOB_STATE_TIME_LIMIT_ACTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String jobQueueName;

    private final String jobQueueArn;

    private final String state;

    private final String schedulingPolicyArn;

    private final String status;

    private final String statusReason;

    private final Integer priority;

    private final List<ComputeEnvironmentOrder> computeEnvironmentOrder;

    private final List<ServiceEnvironmentOrder> serviceEnvironmentOrder;

    private final String jobQueueType;

    private final Map<String, String> tags;

    private final List<JobStateTimeLimitAction> jobStateTimeLimitActions;

    private JobQueueDetail(BuilderImpl builder) {
        this.jobQueueName = builder.jobQueueName;
        this.jobQueueArn = builder.jobQueueArn;
        this.state = builder.state;
        this.schedulingPolicyArn = builder.schedulingPolicyArn;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.priority = builder.priority;
        this.computeEnvironmentOrder = builder.computeEnvironmentOrder;
        this.serviceEnvironmentOrder = builder.serviceEnvironmentOrder;
        this.jobQueueType = builder.jobQueueType;
        this.tags = builder.tags;
        this.jobStateTimeLimitActions = builder.jobStateTimeLimitActions;
    }

    /**
     * <p>
     * The job queue name.
     * </p>
     * 
     * @return The job queue name.
     */
    public final String jobQueueName() {
        return jobQueueName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the job queue.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the job queue.
     */
    public final String jobQueueArn() {
        return jobQueueArn;
    }

    /**
     * <p>
     * Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>, it can
     * accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs
     * already in the queue can finish.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JQState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>, it
     *         can accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue,
     *         but jobs already in the queue can finish.
     * @see JQState
     */
    public final JQState state() {
        return JQState.fromValue(state);
    }

    /**
     * <p>
     * Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>, it can
     * accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs
     * already in the queue can finish.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JQState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>, it
     *         can accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue,
     *         but jobs already in the queue can finish.
     * @see JQState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the scheduling policy. The format is
     * <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>. For example,
     * <code>aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the scheduling policy. The format is
     *         <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>. For
     *         example, <code>aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy</code>.
     */
    public final String schedulingPolicyArn() {
        return schedulingPolicyArn;
    }

    /**
     * <p>
     * The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JQStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
     * @see JQStatus
     */
    public final JQStatus status() {
        return JQStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JQStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
     * @see JQStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A short, human-readable string to provide additional details for the current status of the job queue.
     * </p>
     * 
     * @return A short, human-readable string to provide additional details for the current status of the job queue.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The priority of the job queue. Job queue priority determines the order that job queues are evaluated when
     * multiple queues dispatch jobs within a shared compute environment. A higher value for <code>priority</code>
     * indicates a higher priority. Queues are evaluated in cycles, in descending order by priority. For example, a job
     * queue with a priority value of <code>10</code> is evaluated before a queue with a priority value of
     * <code>1</code>. All of the compute environments must be either Amazon EC2 (<code>EC2</code> or <code>SPOT</code>)
     * or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>). Amazon EC2 and Fargate compute environments can't
     * be mixed.
     * </p>
     * <note>
     * <p>
     * Job queue priority doesn't guarantee that a particular job executes before a job in a lower priority queue. Jobs
     * added to higher priority queues during the queue evaluation cycle might not be evaluated until the next cycle. A
     * job is dispatched from a queue only if resources are available when the queue is evaluated. If there are
     * insufficient resources available at that time, the cycle proceeds to the next queue. This means that jobs added
     * to higher priority queues might have to wait for jobs in multiple lower priority queues to complete before they
     * are dispatched. You can use job dependencies to control the order for jobs from queues with different priorities.
     * For more information, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/job_dependencies.html">Job
     * Dependencies</a> in the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * 
     * @return The priority of the job queue. Job queue priority determines the order that job queues are evaluated when
     *         multiple queues dispatch jobs within a shared compute environment. A higher value for
     *         <code>priority</code> indicates a higher priority. Queues are evaluated in cycles, in descending order by
     *         priority. For example, a job queue with a priority value of <code>10</code> is evaluated before a queue
     *         with a priority value of <code>1</code>. All of the compute environments must be either Amazon EC2 (
     *         <code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>).
     *         Amazon EC2 and Fargate compute environments can't be mixed.</p> <note>
     *         <p>
     *         Job queue priority doesn't guarantee that a particular job executes before a job in a lower priority
     *         queue. Jobs added to higher priority queues during the queue evaluation cycle might not be evaluated
     *         until the next cycle. A job is dispatched from a queue only if resources are available when the queue is
     *         evaluated. If there are insufficient resources available at that time, the cycle proceeds to the next
     *         queue. This means that jobs added to higher priority queues might have to wait for jobs in multiple lower
     *         priority queues to complete before they are dispatched. You can use job dependencies to control the order
     *         for jobs from queues with different priorities. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/job_dependencies.html">Job Dependencies</a> in
     *         the <i>Batch User Guide</i>.
     *         </p>
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * For responses, this returns true if the service returned a value for the ComputeEnvironmentOrder property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasComputeEnvironmentOrder() {
        return computeEnvironmentOrder != null && !(computeEnvironmentOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The compute environments that are attached to the job queue and the order that job placement is preferred.
     * Compute environments are selected for job placement in ascending order.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasComputeEnvironmentOrder} method.
     * </p>
     * 
     * @return The compute environments that are attached to the job queue and the order that job placement is
     *         preferred. Compute environments are selected for job placement in ascending order.
     */
    public final List<ComputeEnvironmentOrder> computeEnvironmentOrder() {
        return computeEnvironmentOrder;
    }

    /**
     * For responses, this returns true if the service returned a value for the ServiceEnvironmentOrder property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasServiceEnvironmentOrder() {
        return serviceEnvironmentOrder != null && !(serviceEnvironmentOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The order of the service environment associated with the job queue. Job queues with a higher priority are
     * evaluated first when associated with the same service environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasServiceEnvironmentOrder} method.
     * </p>
     * 
     * @return The order of the service environment associated with the job queue. Job queues with a higher priority are
     *         evaluated first when associated with the same service environment.
     */
    public final List<ServiceEnvironmentOrder> serviceEnvironmentOrder() {
        return serviceEnvironmentOrder;
    }

    /**
     * <p>
     * The type of job queue. For service jobs that run on SageMaker Training, this value is
     * <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>, <code>ECS</code>, or
     * <code>ECS_FARGATE</code> depending on the compute environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobQueueType} will
     * return {@link JobQueueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobQueueTypeAsString}.
     * </p>
     * 
     * @return The type of job queue. For service jobs that run on SageMaker Training, this value is
     *         <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>,
     *         <code>ECS</code>, or <code>ECS_FARGATE</code> depending on the compute environment.
     * @see JobQueueType
     */
    public final JobQueueType jobQueueType() {
        return JobQueueType.fromValue(jobQueueType);
    }

    /**
     * <p>
     * The type of job queue. For service jobs that run on SageMaker Training, this value is
     * <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>, <code>ECS</code>, or
     * <code>ECS_FARGATE</code> depending on the compute environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobQueueType} will
     * return {@link JobQueueType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobQueueTypeAsString}.
     * </p>
     * 
     * @return The type of job queue. For service jobs that run on SageMaker Training, this value is
     *         <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>,
     *         <code>ECS</code>, or <code>ECS_FARGATE</code> depending on the compute environment.
     * @see JobQueueType
     */
    public final String jobQueueTypeAsString() {
        return jobQueueType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that are applied to the job queue. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch resources</a> in
     * <i>Batch User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that are applied to the job queue. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch
     *         resources</a> in <i>Batch User Guide</i>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the JobStateTimeLimitActions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasJobStateTimeLimitActions() {
        return jobStateTimeLimitActions != null && !(jobStateTimeLimitActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of actions that Batch perform on jobs that remain at the head of the job queue in the specified state
     * longer than specified times. Batch will perform each action after <code>maxTimeSeconds</code> has passed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobStateTimeLimitActions} method.
     * </p>
     * 
     * @return The set of actions that Batch perform on jobs that remain at the head of the job queue in the specified
     *         state longer than specified times. Batch will perform each action after <code>maxTimeSeconds</code> has
     *         passed.
     */
    public final List<JobStateTimeLimitAction> jobStateTimeLimitActions() {
        return jobStateTimeLimitActions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobQueueName());
        hashCode = 31 * hashCode + Objects.hashCode(jobQueueArn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schedulingPolicyArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(hasComputeEnvironmentOrder() ? computeEnvironmentOrder() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasServiceEnvironmentOrder() ? serviceEnvironmentOrder() : null);
        hashCode = 31 * hashCode + Objects.hashCode(jobQueueTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasJobStateTimeLimitActions() ? jobStateTimeLimitActions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobQueueDetail)) {
            return false;
        }
        JobQueueDetail other = (JobQueueDetail) obj;
        return Objects.equals(jobQueueName(), other.jobQueueName()) && Objects.equals(jobQueueArn(), other.jobQueueArn())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(schedulingPolicyArn(), other.schedulingPolicyArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason()) && Objects.equals(priority(), other.priority())
                && hasComputeEnvironmentOrder() == other.hasComputeEnvironmentOrder()
                && Objects.equals(computeEnvironmentOrder(), other.computeEnvironmentOrder())
                && hasServiceEnvironmentOrder() == other.hasServiceEnvironmentOrder()
                && Objects.equals(serviceEnvironmentOrder(), other.serviceEnvironmentOrder())
                && Objects.equals(jobQueueTypeAsString(), other.jobQueueTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasJobStateTimeLimitActions() == other.hasJobStateTimeLimitActions()
                && Objects.equals(jobStateTimeLimitActions(), other.jobStateTimeLimitActions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobQueueDetail").add("JobQueueName", jobQueueName()).add("JobQueueArn", jobQueueArn())
                .add("State", stateAsString()).add("SchedulingPolicyArn", schedulingPolicyArn()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("Priority", priority())
                .add("ComputeEnvironmentOrder", hasComputeEnvironmentOrder() ? computeEnvironmentOrder() : null)
                .add("ServiceEnvironmentOrder", hasServiceEnvironmentOrder() ? serviceEnvironmentOrder() : null)
                .add("JobQueueType", jobQueueTypeAsString()).add("Tags", hasTags() ? tags() : null)
                .add("JobStateTimeLimitActions", hasJobStateTimeLimitActions() ? jobStateTimeLimitActions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "jobQueueName":
            return Optional.ofNullable(clazz.cast(jobQueueName()));
        case "jobQueueArn":
            return Optional.ofNullable(clazz.cast(jobQueueArn()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "schedulingPolicyArn":
            return Optional.ofNullable(clazz.cast(schedulingPolicyArn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "computeEnvironmentOrder":
            return Optional.ofNullable(clazz.cast(computeEnvironmentOrder()));
        case "serviceEnvironmentOrder":
            return Optional.ofNullable(clazz.cast(serviceEnvironmentOrder()));
        case "jobQueueType":
            return Optional.ofNullable(clazz.cast(jobQueueTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "jobStateTimeLimitActions":
            return Optional.ofNullable(clazz.cast(jobStateTimeLimitActions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("jobQueueName", JOB_QUEUE_NAME_FIELD);
        map.put("jobQueueArn", JOB_QUEUE_ARN_FIELD);
        map.put("state", STATE_FIELD);
        map.put("schedulingPolicyArn", SCHEDULING_POLICY_ARN_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusReason", STATUS_REASON_FIELD);
        map.put("priority", PRIORITY_FIELD);
        map.put("computeEnvironmentOrder", COMPUTE_ENVIRONMENT_ORDER_FIELD);
        map.put("serviceEnvironmentOrder", SERVICE_ENVIRONMENT_ORDER_FIELD);
        map.put("jobQueueType", JOB_QUEUE_TYPE_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("jobStateTimeLimitActions", JOB_STATE_TIME_LIMIT_ACTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<JobQueueDetail, T> g) {
        return obj -> g.apply((JobQueueDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobQueueDetail> {
        /**
         * <p>
         * The job queue name.
         * </p>
         * 
         * @param jobQueueName
         *        The job queue name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobQueueName(String jobQueueName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the job queue.
         * </p>
         * 
         * @param jobQueueArn
         *        The Amazon Resource Name (ARN) of the job queue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobQueueArn(String jobQueueArn);

        /**
         * <p>
         * Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>, it can
         * accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs
         * already in the queue can finish.
         * </p>
         * 
         * @param state
         *        Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>,
         *        it can accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the
         *        queue, but jobs already in the queue can finish.
         * @see JQState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JQState
         */
        Builder state(String state);

        /**
         * <p>
         * Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>, it can
         * accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs
         * already in the queue can finish.
         * </p>
         * 
         * @param state
         *        Describes the ability of the queue to accept new jobs. If the job queue state is <code>ENABLED</code>,
         *        it can accept jobs. If the job queue state is <code>DISABLED</code>, new jobs can't be added to the
         *        queue, but jobs already in the queue can finish.
         * @see JQState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JQState
         */
        Builder state(JQState state);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the scheduling policy. The format is
         * <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>. For
         * example, <code>aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy</code>.
         * </p>
         * 
         * @param schedulingPolicyArn
         *        The Amazon Resource Name (ARN) of the scheduling policy. The format is
         *        <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>.
         *        For example, <code>aws:aws:batch:us-west-2:123456789012:scheduling-policy/MySchedulingPolicy</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedulingPolicyArn(String schedulingPolicyArn);

        /**
         * <p>
         * The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
         * </p>
         * 
         * @param status
         *        The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
         * @see JQStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JQStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
         * </p>
         * 
         * @param status
         *        The status of the job queue (for example, <code>CREATING</code> or <code>VALID</code>).
         * @see JQStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JQStatus
         */
        Builder status(JQStatus status);

        /**
         * <p>
         * A short, human-readable string to provide additional details for the current status of the job queue.
         * </p>
         * 
         * @param statusReason
         *        A short, human-readable string to provide additional details for the current status of the job queue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The priority of the job queue. Job queue priority determines the order that job queues are evaluated when
         * multiple queues dispatch jobs within a shared compute environment. A higher value for <code>priority</code>
         * indicates a higher priority. Queues are evaluated in cycles, in descending order by priority. For example, a
         * job queue with a priority value of <code>10</code> is evaluated before a queue with a priority value of
         * <code>1</code>. All of the compute environments must be either Amazon EC2 (<code>EC2</code> or
         * <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>). Amazon EC2 and Fargate
         * compute environments can't be mixed.
         * </p>
         * <note>
         * <p>
         * Job queue priority doesn't guarantee that a particular job executes before a job in a lower priority queue.
         * Jobs added to higher priority queues during the queue evaluation cycle might not be evaluated until the next
         * cycle. A job is dispatched from a queue only if resources are available when the queue is evaluated. If there
         * are insufficient resources available at that time, the cycle proceeds to the next queue. This means that jobs
         * added to higher priority queues might have to wait for jobs in multiple lower priority queues to complete
         * before they are dispatched. You can use job dependencies to control the order for jobs from queues with
         * different priorities. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/job_dependencies.html">Job Dependencies</a> in the
         * <i>Batch User Guide</i>.
         * </p>
         * </note>
         * 
         * @param priority
         *        The priority of the job queue. Job queue priority determines the order that job queues are evaluated
         *        when multiple queues dispatch jobs within a shared compute environment. A higher value for
         *        <code>priority</code> indicates a higher priority. Queues are evaluated in cycles, in descending order
         *        by priority. For example, a job queue with a priority value of <code>10</code> is evaluated before a
         *        queue with a priority value of <code>1</code>. All of the compute environments must be either Amazon
         *        EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or
         *        <code>FARGATE_SPOT</code>). Amazon EC2 and Fargate compute environments can't be mixed.</p> <note>
         *        <p>
         *        Job queue priority doesn't guarantee that a particular job executes before a job in a lower priority
         *        queue. Jobs added to higher priority queues during the queue evaluation cycle might not be evaluated
         *        until the next cycle. A job is dispatched from a queue only if resources are available when the queue
         *        is evaluated. If there are insufficient resources available at that time, the cycle proceeds to the
         *        next queue. This means that jobs added to higher priority queues might have to wait for jobs in
         *        multiple lower priority queues to complete before they are dispatched. You can use job dependencies to
         *        control the order for jobs from queues with different priorities. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/job_dependencies.html">Job Dependencies</a>
         *        in the <i>Batch User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The compute environments that are attached to the job queue and the order that job placement is preferred.
         * Compute environments are selected for job placement in ascending order.
         * </p>
         * 
         * @param computeEnvironmentOrder
         *        The compute environments that are attached to the job queue and the order that job placement is
         *        preferred. Compute environments are selected for job placement in ascending order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentOrder(Collection<ComputeEnvironmentOrder> computeEnvironmentOrder);

        /**
         * <p>
         * The compute environments that are attached to the job queue and the order that job placement is preferred.
         * Compute environments are selected for job placement in ascending order.
         * </p>
         * 
         * @param computeEnvironmentOrder
         *        The compute environments that are attached to the job queue and the order that job placement is
         *        preferred. Compute environments are selected for job placement in ascending order.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentOrder(ComputeEnvironmentOrder... computeEnvironmentOrder);

        /**
         * <p>
         * The compute environments that are attached to the job queue and the order that job placement is preferred.
         * Compute environments are selected for job placement in ascending order.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.ComputeEnvironmentOrder.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.batch.model.ComputeEnvironmentOrder#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.ComputeEnvironmentOrder.Builder#build()} is called
         * immediately and its result is passed to {@link #computeEnvironmentOrder(List<ComputeEnvironmentOrder>)}.
         * 
         * @param computeEnvironmentOrder
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.ComputeEnvironmentOrder.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #computeEnvironmentOrder(java.util.Collection<ComputeEnvironmentOrder>)
         */
        Builder computeEnvironmentOrder(Consumer<ComputeEnvironmentOrder.Builder>... computeEnvironmentOrder);

        /**
         * <p>
         * The order of the service environment associated with the job queue. Job queues with a higher priority are
         * evaluated first when associated with the same service environment.
         * </p>
         * 
         * @param serviceEnvironmentOrder
         *        The order of the service environment associated with the job queue. Job queues with a higher priority
         *        are evaluated first when associated with the same service environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceEnvironmentOrder(Collection<ServiceEnvironmentOrder> serviceEnvironmentOrder);

        /**
         * <p>
         * The order of the service environment associated with the job queue. Job queues with a higher priority are
         * evaluated first when associated with the same service environment.
         * </p>
         * 
         * @param serviceEnvironmentOrder
         *        The order of the service environment associated with the job queue. Job queues with a higher priority
         *        are evaluated first when associated with the same service environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceEnvironmentOrder(ServiceEnvironmentOrder... serviceEnvironmentOrder);

        /**
         * <p>
         * The order of the service environment associated with the job queue. Job queues with a higher priority are
         * evaluated first when associated with the same service environment.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.ServiceEnvironmentOrder.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.batch.model.ServiceEnvironmentOrder#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.ServiceEnvironmentOrder.Builder#build()} is called
         * immediately and its result is passed to {@link #serviceEnvironmentOrder(List<ServiceEnvironmentOrder>)}.
         * 
         * @param serviceEnvironmentOrder
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.ServiceEnvironmentOrder.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serviceEnvironmentOrder(java.util.Collection<ServiceEnvironmentOrder>)
         */
        Builder serviceEnvironmentOrder(Consumer<ServiceEnvironmentOrder.Builder>... serviceEnvironmentOrder);

        /**
         * <p>
         * The type of job queue. For service jobs that run on SageMaker Training, this value is
         * <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>, <code>ECS</code>
         * , or <code>ECS_FARGATE</code> depending on the compute environment.
         * </p>
         * 
         * @param jobQueueType
         *        The type of job queue. For service jobs that run on SageMaker Training, this value is
         *        <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>,
         *        <code>ECS</code>, or <code>ECS_FARGATE</code> depending on the compute environment.
         * @see JobQueueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobQueueType
         */
        Builder jobQueueType(String jobQueueType);

        /**
         * <p>
         * The type of job queue. For service jobs that run on SageMaker Training, this value is
         * <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>, <code>ECS</code>
         * , or <code>ECS_FARGATE</code> depending on the compute environment.
         * </p>
         * 
         * @param jobQueueType
         *        The type of job queue. For service jobs that run on SageMaker Training, this value is
         *        <code>SAGEMAKER_TRAINING</code>. For regular container jobs, this value is <code>EKS</code>,
         *        <code>ECS</code>, or <code>ECS_FARGATE</code> depending on the compute environment.
         * @see JobQueueType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobQueueType
         */
        Builder jobQueueType(JobQueueType jobQueueType);

        /**
         * <p>
         * The tags that are applied to the job queue. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch resources</a> in
         * <i>Batch User Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tags that are applied to the job queue. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch
         *        resources</a> in <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The set of actions that Batch perform on jobs that remain at the head of the job queue in the specified state
         * longer than specified times. Batch will perform each action after <code>maxTimeSeconds</code> has passed.
         * </p>
         * 
         * @param jobStateTimeLimitActions
         *        The set of actions that Batch perform on jobs that remain at the head of the job queue in the
         *        specified state longer than specified times. Batch will perform each action after
         *        <code>maxTimeSeconds</code> has passed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStateTimeLimitActions(Collection<JobStateTimeLimitAction> jobStateTimeLimitActions);

        /**
         * <p>
         * The set of actions that Batch perform on jobs that remain at the head of the job queue in the specified state
         * longer than specified times. Batch will perform each action after <code>maxTimeSeconds</code> has passed.
         * </p>
         * 
         * @param jobStateTimeLimitActions
         *        The set of actions that Batch perform on jobs that remain at the head of the job queue in the
         *        specified state longer than specified times. Batch will perform each action after
         *        <code>maxTimeSeconds</code> has passed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStateTimeLimitActions(JobStateTimeLimitAction... jobStateTimeLimitActions);

        /**
         * <p>
         * The set of actions that Batch perform on jobs that remain at the head of the job queue in the specified state
         * longer than specified times. Batch will perform each action after <code>maxTimeSeconds</code> has passed.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.JobStateTimeLimitAction.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.batch.model.JobStateTimeLimitAction#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.JobStateTimeLimitAction.Builder#build()} is called
         * immediately and its result is passed to {@link #jobStateTimeLimitActions(List<JobStateTimeLimitAction>)}.
         * 
         * @param jobStateTimeLimitActions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.JobStateTimeLimitAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobStateTimeLimitActions(java.util.Collection<JobStateTimeLimitAction>)
         */
        Builder jobStateTimeLimitActions(Consumer<JobStateTimeLimitAction.Builder>... jobStateTimeLimitActions);
    }

    static final class BuilderImpl implements Builder {
        private String jobQueueName;

        private String jobQueueArn;

        private String state;

        private String schedulingPolicyArn;

        private String status;

        private String statusReason;

        private Integer priority;

        private List<ComputeEnvironmentOrder> computeEnvironmentOrder = DefaultSdkAutoConstructList.getInstance();

        private List<ServiceEnvironmentOrder> serviceEnvironmentOrder = DefaultSdkAutoConstructList.getInstance();

        private String jobQueueType;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private List<JobStateTimeLimitAction> jobStateTimeLimitActions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobQueueDetail model) {
            jobQueueName(model.jobQueueName);
            jobQueueArn(model.jobQueueArn);
            state(model.state);
            schedulingPolicyArn(model.schedulingPolicyArn);
            status(model.status);
            statusReason(model.statusReason);
            priority(model.priority);
            computeEnvironmentOrder(model.computeEnvironmentOrder);
            serviceEnvironmentOrder(model.serviceEnvironmentOrder);
            jobQueueType(model.jobQueueType);
            tags(model.tags);
            jobStateTimeLimitActions(model.jobStateTimeLimitActions);
        }

        public final String getJobQueueName() {
            return jobQueueName;
        }

        public final void setJobQueueName(String jobQueueName) {
            this.jobQueueName = jobQueueName;
        }

        @Override
        public final Builder jobQueueName(String jobQueueName) {
            this.jobQueueName = jobQueueName;
            return this;
        }

        public final String getJobQueueArn() {
            return jobQueueArn;
        }

        public final void setJobQueueArn(String jobQueueArn) {
            this.jobQueueArn = jobQueueArn;
        }

        @Override
        public final Builder jobQueueArn(String jobQueueArn) {
            this.jobQueueArn = jobQueueArn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(JQState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getSchedulingPolicyArn() {
            return schedulingPolicyArn;
        }

        public final void setSchedulingPolicyArn(String schedulingPolicyArn) {
            this.schedulingPolicyArn = schedulingPolicyArn;
        }

        @Override
        public final Builder schedulingPolicyArn(String schedulingPolicyArn) {
            this.schedulingPolicyArn = schedulingPolicyArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(JQStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final List<ComputeEnvironmentOrder.Builder> getComputeEnvironmentOrder() {
            List<ComputeEnvironmentOrder.Builder> result = ComputeEnvironmentOrdersCopier
                    .copyToBuilder(this.computeEnvironmentOrder);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setComputeEnvironmentOrder(Collection<ComputeEnvironmentOrder.BuilderImpl> computeEnvironmentOrder) {
            this.computeEnvironmentOrder = ComputeEnvironmentOrdersCopier.copyFromBuilder(computeEnvironmentOrder);
        }

        @Override
        public final Builder computeEnvironmentOrder(Collection<ComputeEnvironmentOrder> computeEnvironmentOrder) {
            this.computeEnvironmentOrder = ComputeEnvironmentOrdersCopier.copy(computeEnvironmentOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder computeEnvironmentOrder(ComputeEnvironmentOrder... computeEnvironmentOrder) {
            computeEnvironmentOrder(Arrays.asList(computeEnvironmentOrder));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder computeEnvironmentOrder(Consumer<ComputeEnvironmentOrder.Builder>... computeEnvironmentOrder) {
            computeEnvironmentOrder(Stream.of(computeEnvironmentOrder)
                    .map(c -> ComputeEnvironmentOrder.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<ServiceEnvironmentOrder.Builder> getServiceEnvironmentOrder() {
            List<ServiceEnvironmentOrder.Builder> result = ServiceEnvironmentOrdersCopier
                    .copyToBuilder(this.serviceEnvironmentOrder);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setServiceEnvironmentOrder(Collection<ServiceEnvironmentOrder.BuilderImpl> serviceEnvironmentOrder) {
            this.serviceEnvironmentOrder = ServiceEnvironmentOrdersCopier.copyFromBuilder(serviceEnvironmentOrder);
        }

        @Override
        public final Builder serviceEnvironmentOrder(Collection<ServiceEnvironmentOrder> serviceEnvironmentOrder) {
            this.serviceEnvironmentOrder = ServiceEnvironmentOrdersCopier.copy(serviceEnvironmentOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceEnvironmentOrder(ServiceEnvironmentOrder... serviceEnvironmentOrder) {
            serviceEnvironmentOrder(Arrays.asList(serviceEnvironmentOrder));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder serviceEnvironmentOrder(Consumer<ServiceEnvironmentOrder.Builder>... serviceEnvironmentOrder) {
            serviceEnvironmentOrder(Stream.of(serviceEnvironmentOrder)
                    .map(c -> ServiceEnvironmentOrder.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getJobQueueType() {
            return jobQueueType;
        }

        public final void setJobQueueType(String jobQueueType) {
            this.jobQueueType = jobQueueType;
        }

        @Override
        public final Builder jobQueueType(String jobQueueType) {
            this.jobQueueType = jobQueueType;
            return this;
        }

        @Override
        public final Builder jobQueueType(JobQueueType jobQueueType) {
            this.jobQueueType(jobQueueType == null ? null : jobQueueType.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
            return this;
        }

        public final List<JobStateTimeLimitAction.Builder> getJobStateTimeLimitActions() {
            List<JobStateTimeLimitAction.Builder> result = JobStateTimeLimitActionsCopier
                    .copyToBuilder(this.jobStateTimeLimitActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setJobStateTimeLimitActions(Collection<JobStateTimeLimitAction.BuilderImpl> jobStateTimeLimitActions) {
            this.jobStateTimeLimitActions = JobStateTimeLimitActionsCopier.copyFromBuilder(jobStateTimeLimitActions);
        }

        @Override
        public final Builder jobStateTimeLimitActions(Collection<JobStateTimeLimitAction> jobStateTimeLimitActions) {
            this.jobStateTimeLimitActions = JobStateTimeLimitActionsCopier.copy(jobStateTimeLimitActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStateTimeLimitActions(JobStateTimeLimitAction... jobStateTimeLimitActions) {
            jobStateTimeLimitActions(Arrays.asList(jobStateTimeLimitActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStateTimeLimitActions(Consumer<JobStateTimeLimitAction.Builder>... jobStateTimeLimitActions) {
            jobStateTimeLimitActions(Stream.of(jobStateTimeLimitActions)
                    .map(c -> JobStateTimeLimitAction.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public JobQueueDetail build() {
            return new JobQueueDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
