/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a scheduled task used to back up a selection of resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackupRule implements SdkPojo, Serializable, ToCopyableBuilder<BackupRule.Builder, BackupRule> {
    private static final SdkField<String> RULE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleName").getter(getter(BackupRule::ruleName)).setter(setter(Builder::ruleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleName").build()).build();

    private static final SdkField<String> TARGET_BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetBackupVaultName").getter(getter(BackupRule::targetBackupVaultName))
            .setter(setter(Builder::targetBackupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetBackupVaultName").build())
            .build();

    private static final SdkField<String> TARGET_LOGICALLY_AIR_GAPPED_BACKUP_VAULT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetLogicallyAirGappedBackupVaultArn")
            .getter(getter(BackupRule::targetLogicallyAirGappedBackupVaultArn))
            .setter(setter(Builder::targetLogicallyAirGappedBackupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("TargetLogicallyAirGappedBackupVaultArn").build()).build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScheduleExpression").getter(getter(BackupRule::scheduleExpression))
            .setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<Long> START_WINDOW_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("StartWindowMinutes").getter(getter(BackupRule::startWindowMinutes))
            .setter(setter(Builder::startWindowMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartWindowMinutes").build())
            .build();

    private static final SdkField<Long> COMPLETION_WINDOW_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CompletionWindowMinutes").getter(getter(BackupRule::completionWindowMinutes))
            .setter(setter(Builder::completionWindowMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionWindowMinutes").build())
            .build();

    private static final SdkField<Lifecycle> LIFECYCLE_FIELD = SdkField.<Lifecycle> builder(MarshallingType.SDK_POJO)
            .memberName("Lifecycle").getter(getter(BackupRule::lifecycle)).setter(setter(Builder::lifecycle))
            .constructor(Lifecycle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<Map<String, String>> RECOVERY_POINT_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RecoveryPointTags")
            .getter(getter(BackupRule::recoveryPointTags))
            .setter(setter(Builder::recoveryPointTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("RuleId")
            .getter(getter(BackupRule::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleId").build()).build();

    private static final SdkField<List<CopyAction>> COPY_ACTIONS_FIELD = SdkField
            .<List<CopyAction>> builder(MarshallingType.LIST)
            .memberName("CopyActions")
            .getter(getter(BackupRule::copyActions))
            .setter(setter(Builder::copyActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CopyAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(CopyAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ENABLE_CONTINUOUS_BACKUP_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableContinuousBackup").getter(getter(BackupRule::enableContinuousBackup))
            .setter(setter(Builder::enableContinuousBackup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableContinuousBackup").build())
            .build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_TIMEZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ScheduleExpressionTimezone")
            .getter(getter(BackupRule::scheduleExpressionTimezone))
            .setter(setter(Builder::scheduleExpressionTimezone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpressionTimezone").build())
            .build();

    private static final SdkField<List<IndexAction>> INDEX_ACTIONS_FIELD = SdkField
            .<List<IndexAction>> builder(MarshallingType.LIST)
            .memberName("IndexActions")
            .getter(getter(BackupRule::indexActions))
            .setter(setter(Builder::indexActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IndexAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(IndexAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ScanAction>> SCAN_ACTIONS_FIELD = SdkField
            .<List<ScanAction>> builder(MarshallingType.LIST)
            .memberName("ScanActions")
            .getter(getter(BackupRule::scanActions))
            .setter(setter(Builder::scanActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScanActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ScanAction> builder(MarshallingType.SDK_POJO)
                                            .constructor(ScanAction::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_NAME_FIELD,
            TARGET_BACKUP_VAULT_NAME_FIELD, TARGET_LOGICALLY_AIR_GAPPED_BACKUP_VAULT_ARN_FIELD, SCHEDULE_EXPRESSION_FIELD,
            START_WINDOW_MINUTES_FIELD, COMPLETION_WINDOW_MINUTES_FIELD, LIFECYCLE_FIELD, RECOVERY_POINT_TAGS_FIELD,
            RULE_ID_FIELD, COPY_ACTIONS_FIELD, ENABLE_CONTINUOUS_BACKUP_FIELD, SCHEDULE_EXPRESSION_TIMEZONE_FIELD,
            INDEX_ACTIONS_FIELD, SCAN_ACTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ruleName;

    private final String targetBackupVaultName;

    private final String targetLogicallyAirGappedBackupVaultArn;

    private final String scheduleExpression;

    private final Long startWindowMinutes;

    private final Long completionWindowMinutes;

    private final Lifecycle lifecycle;

    private final Map<String, String> recoveryPointTags;

    private final String ruleId;

    private final List<CopyAction> copyActions;

    private final Boolean enableContinuousBackup;

    private final String scheduleExpressionTimezone;

    private final List<IndexAction> indexActions;

    private final List<ScanAction> scanActions;

    private BackupRule(BuilderImpl builder) {
        this.ruleName = builder.ruleName;
        this.targetBackupVaultName = builder.targetBackupVaultName;
        this.targetLogicallyAirGappedBackupVaultArn = builder.targetLogicallyAirGappedBackupVaultArn;
        this.scheduleExpression = builder.scheduleExpression;
        this.startWindowMinutes = builder.startWindowMinutes;
        this.completionWindowMinutes = builder.completionWindowMinutes;
        this.lifecycle = builder.lifecycle;
        this.recoveryPointTags = builder.recoveryPointTags;
        this.ruleId = builder.ruleId;
        this.copyActions = builder.copyActions;
        this.enableContinuousBackup = builder.enableContinuousBackup;
        this.scheduleExpressionTimezone = builder.scheduleExpressionTimezone;
        this.indexActions = builder.indexActions;
        this.scanActions = builder.scanActions;
    }

    /**
     * <p>
     * A display name for a backup rule. Must contain 1 to 50 alphanumeric or '-_.' characters.
     * </p>
     * 
     * @return A display name for a backup rule. Must contain 1 to 50 alphanumeric or '-_.' characters.
     */
    public final String ruleName() {
        return ruleName;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the Amazon Web Services Region where they are created.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the Amazon Web Services Region where they are created.
     */
    public final String targetBackupVaultName() {
        return targetBackupVaultName;
    }

    /**
     * <p>
     * The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided, supported fully
     * managed resources back up directly to logically air-gapped vault, while other supported resources create a
     * temporary (billable) snapshot in backup vault, then copy it to logically air-gapped vault. Unsupported resources
     * only back up to the specified backup vault.
     * </p>
     * 
     * @return The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided,
     *         supported fully managed resources back up directly to logically air-gapped vault, while other supported
     *         resources create a temporary (billable) snapshot in backup vault, then copy it to logically air-gapped
     *         vault. Unsupported resources only back up to the specified backup vault.
     */
    public final String targetLogicallyAirGappedBackupVaultArn() {
        return targetLogicallyAirGappedBackupVaultArn;
    }

    /**
     * <p>
     * A cron expression in UTC specifying when Backup initiates a backup job. When no CRON expression is provided,
     * Backup will use the default expression <code>cron(0 5 ? * * *)</code>.
     * </p>
     * <p>
     * For more information about Amazon Web Services cron expressions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html">Schedule Expressions for
     * Rules</a> in the <i>Amazon CloudWatch Events User Guide</i>.
     * </p>
     * <p>
     * Two examples of Amazon Web Services cron expressions are <code> 15 * ? * * *</code> (take a backup every hour at
     * 15 minutes past the hour) and <code>0 12 * * ? *</code> (take a backup every day at 12 noon UTC).
     * </p>
     * <p>
     * For a table of examples, click the preceding link and scroll down the page.
     * </p>
     * 
     * @return A cron expression in UTC specifying when Backup initiates a backup job. When no CRON expression is
     *         provided, Backup will use the default expression <code>cron(0 5 ? * * *)</code>.</p>
     *         <p>
     *         For more information about Amazon Web Services cron expressions, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html">Schedule
     *         Expressions for Rules</a> in the <i>Amazon CloudWatch Events User Guide</i>.
     *         </p>
     *         <p>
     *         Two examples of Amazon Web Services cron expressions are <code> 15 * ? * * *</code> (take a backup every
     *         hour at 15 minutes past the hour) and <code>0 12 * * ? *</code> (take a backup every day at 12 noon UTC).
     *         </p>
     *         <p>
     *         For a table of examples, click the preceding link and scroll down the page.
     */
    public final String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start successfully.
     * This value is optional. If this value is included, it must be at least 60 minutes to avoid errors.
     * </p>
     * <p>
     * During the start window, the backup job status remains in <code>CREATED</code> status until it has successfully
     * begun or until the start window time has run out. If within the start window time Backup receives an error that
     * allows the job to be retried, Backup will automatically retry to begin the job at least every 10 minutes until
     * the backup successfully begins (the job status changes to <code>RUNNING</code>) or until the job status changes
     * to <code>EXPIRED</code> (which is expected to occur when the start window time is over).
     * </p>
     * 
     * @return A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start
     *         successfully. This value is optional. If this value is included, it must be at least 60 minutes to avoid
     *         errors.</p>
     *         <p>
     *         During the start window, the backup job status remains in <code>CREATED</code> status until it has
     *         successfully begun or until the start window time has run out. If within the start window time Backup
     *         receives an error that allows the job to be retried, Backup will automatically retry to begin the job at
     *         least every 10 minutes until the backup successfully begins (the job status changes to
     *         <code>RUNNING</code>) or until the job status changes to <code>EXPIRED</code> (which is expected to occur
     *         when the start window time is over).
     */
    public final Long startWindowMinutes() {
        return startWindowMinutes;
    }

    /**
     * <p>
     * A value in minutes after a backup job is successfully started before it must be completed or it will be canceled
     * by Backup. This value is optional.
     * </p>
     * 
     * @return A value in minutes after a backup job is successfully started before it must be completed or it will be
     *         canceled by Backup. This value is optional.
     */
    public final Long completionWindowMinutes() {
        return completionWindowMinutes;
    }

    /**
     * <p>
     * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup
     * transitions and expires backups automatically according to the lifecycle that you define.
     * </p>
     * <p>
     * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
     * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition to
     * cold after days” setting cannot be changed after a backup has been transitioned to cold.
     * </p>
     * <p>
     * Resource types that can transition to cold storage are listed in the <a href=
     * "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
     * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
     * </p>
     * 
     * @return The lifecycle defines when a protected resource is transitioned to cold storage and when it expires.
     *         Backup transitions and expires backups automatically according to the lifecycle that you define. </p>
     *         <p>
     *         Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore,
     *         the “retention” setting must be 90 days greater than the “transition to cold after days” setting. The
     *         “transition to cold after days” setting cannot be changed after a backup has been transitioned to cold.
     *         </p>
     *         <p>
     *         Resource types that can transition to cold storage are listed in the <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
     *         >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
     */
    public final Lifecycle lifecycle() {
        return lifecycle;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecoveryPointTags property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecoveryPointTags() {
        return recoveryPointTags != null && !(recoveryPointTags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that are assigned to resources that are associated with this rule when restored from backup.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecoveryPointTags} method.
     * </p>
     * 
     * @return The tags that are assigned to resources that are associated with this rule when restored from backup.
     */
    public final Map<String, String> recoveryPointTags() {
        return recoveryPointTags;
    }

    /**
     * <p>
     * Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
     * </p>
     * 
     * @return Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
     */
    public final String ruleId() {
        return ruleId;
    }

    /**
     * For responses, this returns true if the service returned a value for the CopyActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCopyActions() {
        return copyActions != null && !(copyActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCopyActions} method.
     * </p>
     * 
     * @return An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
     */
    public final List<CopyAction> copyActions() {
        return copyActions;
    }

    /**
     * <p>
     * Specifies whether Backup creates continuous backups. True causes Backup to create continuous backups capable of
     * point-in-time restore (PITR). False (or not specified) causes Backup to create snapshot backups.
     * </p>
     * 
     * @return Specifies whether Backup creates continuous backups. True causes Backup to create continuous backups
     *         capable of point-in-time restore (PITR). False (or not specified) causes Backup to create snapshot
     *         backups.
     */
    public final Boolean enableContinuousBackup() {
        return enableContinuousBackup;
    }

    /**
     * <p>
     * The timezone in which the schedule expression is set. By default, ScheduleExpressions are in UTC. You can modify
     * this to a specified timezone.
     * </p>
     * 
     * @return The timezone in which the schedule expression is set. By default, ScheduleExpressions are in UTC. You can
     *         modify this to a specified timezone.
     */
    public final String scheduleExpressionTimezone() {
        return scheduleExpressionTimezone;
    }

    /**
     * For responses, this returns true if the service returned a value for the IndexActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIndexActions() {
        return indexActions != null && !(indexActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * IndexActions is an array you use to specify how backup data should be indexed.
     * </p>
     * <p>
     * eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with it.
     * </p>
     * <p>
     * Within the array is ResourceType. Only one will be accepted for each BackupRule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIndexActions} method.
     * </p>
     * 
     * @return IndexActions is an array you use to specify how backup data should be indexed.</p>
     *         <p>
     *         eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with it.
     *         </p>
     *         <p>
     *         Within the array is ResourceType. Only one will be accepted for each BackupRule.
     */
    public final List<IndexAction> indexActions() {
        return indexActions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ScanActions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasScanActions() {
        return scanActions != null && !(scanActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains your scanning configuration for the backup rule and includes the malware scanner, and scan mode of
     * either full or incremental.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScanActions} method.
     * </p>
     * 
     * @return Contains your scanning configuration for the backup rule and includes the malware scanner, and scan mode
     *         of either full or incremental.
     */
    public final List<ScanAction> scanActions() {
        return scanActions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleName());
        hashCode = 31 * hashCode + Objects.hashCode(targetBackupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(targetLogicallyAirGappedBackupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(startWindowMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(completionWindowMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycle());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecoveryPointTags() ? recoveryPointTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        hashCode = 31 * hashCode + Objects.hashCode(hasCopyActions() ? copyActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(enableContinuousBackup());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpressionTimezone());
        hashCode = 31 * hashCode + Objects.hashCode(hasIndexActions() ? indexActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasScanActions() ? scanActions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackupRule)) {
            return false;
        }
        BackupRule other = (BackupRule) obj;
        return Objects.equals(ruleName(), other.ruleName())
                && Objects.equals(targetBackupVaultName(), other.targetBackupVaultName())
                && Objects.equals(targetLogicallyAirGappedBackupVaultArn(), other.targetLogicallyAirGappedBackupVaultArn())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(startWindowMinutes(), other.startWindowMinutes())
                && Objects.equals(completionWindowMinutes(), other.completionWindowMinutes())
                && Objects.equals(lifecycle(), other.lifecycle()) && hasRecoveryPointTags() == other.hasRecoveryPointTags()
                && Objects.equals(recoveryPointTags(), other.recoveryPointTags()) && Objects.equals(ruleId(), other.ruleId())
                && hasCopyActions() == other.hasCopyActions() && Objects.equals(copyActions(), other.copyActions())
                && Objects.equals(enableContinuousBackup(), other.enableContinuousBackup())
                && Objects.equals(scheduleExpressionTimezone(), other.scheduleExpressionTimezone())
                && hasIndexActions() == other.hasIndexActions() && Objects.equals(indexActions(), other.indexActions())
                && hasScanActions() == other.hasScanActions() && Objects.equals(scanActions(), other.scanActions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BackupRule").add("RuleName", ruleName()).add("TargetBackupVaultName", targetBackupVaultName())
                .add("TargetLogicallyAirGappedBackupVaultArn", targetLogicallyAirGappedBackupVaultArn())
                .add("ScheduleExpression", scheduleExpression()).add("StartWindowMinutes", startWindowMinutes())
                .add("CompletionWindowMinutes", completionWindowMinutes()).add("Lifecycle", lifecycle())
                .add("RecoveryPointTags", recoveryPointTags() == null ? null : "*** Sensitive Data Redacted ***")
                .add("RuleId", ruleId()).add("CopyActions", hasCopyActions() ? copyActions() : null)
                .add("EnableContinuousBackup", enableContinuousBackup())
                .add("ScheduleExpressionTimezone", scheduleExpressionTimezone())
                .add("IndexActions", hasIndexActions() ? indexActions() : null)
                .add("ScanActions", hasScanActions() ? scanActions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleName":
            return Optional.ofNullable(clazz.cast(ruleName()));
        case "TargetBackupVaultName":
            return Optional.ofNullable(clazz.cast(targetBackupVaultName()));
        case "TargetLogicallyAirGappedBackupVaultArn":
            return Optional.ofNullable(clazz.cast(targetLogicallyAirGappedBackupVaultArn()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "StartWindowMinutes":
            return Optional.ofNullable(clazz.cast(startWindowMinutes()));
        case "CompletionWindowMinutes":
            return Optional.ofNullable(clazz.cast(completionWindowMinutes()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycle()));
        case "RecoveryPointTags":
            return Optional.ofNullable(clazz.cast(recoveryPointTags()));
        case "RuleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        case "CopyActions":
            return Optional.ofNullable(clazz.cast(copyActions()));
        case "EnableContinuousBackup":
            return Optional.ofNullable(clazz.cast(enableContinuousBackup()));
        case "ScheduleExpressionTimezone":
            return Optional.ofNullable(clazz.cast(scheduleExpressionTimezone()));
        case "IndexActions":
            return Optional.ofNullable(clazz.cast(indexActions()));
        case "ScanActions":
            return Optional.ofNullable(clazz.cast(scanActions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RuleName", RULE_NAME_FIELD);
        map.put("TargetBackupVaultName", TARGET_BACKUP_VAULT_NAME_FIELD);
        map.put("TargetLogicallyAirGappedBackupVaultArn", TARGET_LOGICALLY_AIR_GAPPED_BACKUP_VAULT_ARN_FIELD);
        map.put("ScheduleExpression", SCHEDULE_EXPRESSION_FIELD);
        map.put("StartWindowMinutes", START_WINDOW_MINUTES_FIELD);
        map.put("CompletionWindowMinutes", COMPLETION_WINDOW_MINUTES_FIELD);
        map.put("Lifecycle", LIFECYCLE_FIELD);
        map.put("RecoveryPointTags", RECOVERY_POINT_TAGS_FIELD);
        map.put("RuleId", RULE_ID_FIELD);
        map.put("CopyActions", COPY_ACTIONS_FIELD);
        map.put("EnableContinuousBackup", ENABLE_CONTINUOUS_BACKUP_FIELD);
        map.put("ScheduleExpressionTimezone", SCHEDULE_EXPRESSION_TIMEZONE_FIELD);
        map.put("IndexActions", INDEX_ACTIONS_FIELD);
        map.put("ScanActions", SCAN_ACTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BackupRule, T> g) {
        return obj -> g.apply((BackupRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackupRule> {
        /**
         * <p>
         * A display name for a backup rule. Must contain 1 to 50 alphanumeric or '-_.' characters.
         * </p>
         * 
         * @param ruleName
         *        A display name for a backup rule. Must contain 1 to 50 alphanumeric or '-_.' characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleName(String ruleName);

        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the Amazon Web Services Region where they are created.
         * </p>
         * 
         * @param targetBackupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the Amazon Web Services Region where they are
         *        created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetBackupVaultName(String targetBackupVaultName);

        /**
         * <p>
         * The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided, supported
         * fully managed resources back up directly to logically air-gapped vault, while other supported resources
         * create a temporary (billable) snapshot in backup vault, then copy it to logically air-gapped vault.
         * Unsupported resources only back up to the specified backup vault.
         * </p>
         * 
         * @param targetLogicallyAirGappedBackupVaultArn
         *        The ARN of a logically air-gapped vault. ARN must be in the same account and Region. If provided,
         *        supported fully managed resources back up directly to logically air-gapped vault, while other
         *        supported resources create a temporary (billable) snapshot in backup vault, then copy it to logically
         *        air-gapped vault. Unsupported resources only back up to the specified backup vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetLogicallyAirGappedBackupVaultArn(String targetLogicallyAirGappedBackupVaultArn);

        /**
         * <p>
         * A cron expression in UTC specifying when Backup initiates a backup job. When no CRON expression is provided,
         * Backup will use the default expression <code>cron(0 5 ? * * *)</code>.
         * </p>
         * <p>
         * For more information about Amazon Web Services cron expressions, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html">Schedule Expressions
         * for Rules</a> in the <i>Amazon CloudWatch Events User Guide</i>.
         * </p>
         * <p>
         * Two examples of Amazon Web Services cron expressions are <code> 15 * ? * * *</code> (take a backup every hour
         * at 15 minutes past the hour) and <code>0 12 * * ? *</code> (take a backup every day at 12 noon UTC).
         * </p>
         * <p>
         * For a table of examples, click the preceding link and scroll down the page.
         * </p>
         * 
         * @param scheduleExpression
         *        A cron expression in UTC specifying when Backup initiates a backup job. When no CRON expression is
         *        provided, Backup will use the default expression <code>cron(0 5 ? * * *)</code>.</p>
         *        <p>
         *        For more information about Amazon Web Services cron expressions, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/ScheduledEvents.html">Schedule
         *        Expressions for Rules</a> in the <i>Amazon CloudWatch Events User Guide</i>.
         *        </p>
         *        <p>
         *        Two examples of Amazon Web Services cron expressions are <code> 15 * ? * * *</code> (take a backup
         *        every hour at 15 minutes past the hour) and <code>0 12 * * ? *</code> (take a backup every day at 12
         *        noon UTC).
         *        </p>
         *        <p>
         *        For a table of examples, click the preceding link and scroll down the page.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start
         * successfully. This value is optional. If this value is included, it must be at least 60 minutes to avoid
         * errors.
         * </p>
         * <p>
         * During the start window, the backup job status remains in <code>CREATED</code> status until it has
         * successfully begun or until the start window time has run out. If within the start window time Backup
         * receives an error that allows the job to be retried, Backup will automatically retry to begin the job at
         * least every 10 minutes until the backup successfully begins (the job status changes to <code>RUNNING</code>)
         * or until the job status changes to <code>EXPIRED</code> (which is expected to occur when the start window
         * time is over).
         * </p>
         * 
         * @param startWindowMinutes
         *        A value in minutes after a backup is scheduled before a job will be canceled if it doesn't start
         *        successfully. This value is optional. If this value is included, it must be at least 60 minutes to
         *        avoid errors.</p>
         *        <p>
         *        During the start window, the backup job status remains in <code>CREATED</code> status until it has
         *        successfully begun or until the start window time has run out. If within the start window time Backup
         *        receives an error that allows the job to be retried, Backup will automatically retry to begin the job
         *        at least every 10 minutes until the backup successfully begins (the job status changes to
         *        <code>RUNNING</code>) or until the job status changes to <code>EXPIRED</code> (which is expected to
         *        occur when the start window time is over).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startWindowMinutes(Long startWindowMinutes);

        /**
         * <p>
         * A value in minutes after a backup job is successfully started before it must be completed or it will be
         * canceled by Backup. This value is optional.
         * </p>
         * 
         * @param completionWindowMinutes
         *        A value in minutes after a backup job is successfully started before it must be completed or it will
         *        be canceled by Backup. This value is optional.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionWindowMinutes(Long completionWindowMinutes);

        /**
         * <p>
         * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup
         * transitions and expires backups automatically according to the lifecycle that you define.
         * </p>
         * <p>
         * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
         * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition
         * to cold after days” setting cannot be changed after a backup has been transitioned to cold.
         * </p>
         * <p>
         * Resource types that can transition to cold storage are listed in the <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
         * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
         * </p>
         * 
         * @param lifecycle
         *        The lifecycle defines when a protected resource is transitioned to cold storage and when it expires.
         *        Backup transitions and expires backups automatically according to the lifecycle that you define. </p>
         *        <p>
         *        Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days.
         *        Therefore, the “retention” setting must be 90 days greater than the “transition to cold after days”
         *        setting. The “transition to cold after days” setting cannot be changed after a backup has been
         *        transitioned to cold.
         *        </p>
         *        <p>
         *        Resource types that can transition to cold storage are listed in the <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
         *        >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycle(Lifecycle lifecycle);

        /**
         * <p>
         * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup
         * transitions and expires backups automatically according to the lifecycle that you define.
         * </p>
         * <p>
         * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
         * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition
         * to cold after days” setting cannot be changed after a backup has been transitioned to cold.
         * </p>
         * <p>
         * Resource types that can transition to cold storage are listed in the <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
         * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
         * </p>
         * This is a convenience method that creates an instance of the {@link Lifecycle.Builder} avoiding the need to
         * create one manually via {@link Lifecycle#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Lifecycle.Builder#build()} is called immediately and its result
         * is passed to {@link #lifecycle(Lifecycle)}.
         * 
         * @param lifecycle
         *        a consumer that will call methods on {@link Lifecycle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycle(Lifecycle)
         */
        default Builder lifecycle(Consumer<Lifecycle.Builder> lifecycle) {
            return lifecycle(Lifecycle.builder().applyMutation(lifecycle).build());
        }

        /**
         * <p>
         * The tags that are assigned to resources that are associated with this rule when restored from backup.
         * </p>
         * 
         * @param recoveryPointTags
         *        The tags that are assigned to resources that are associated with this rule when restored from backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTags(Map<String, String> recoveryPointTags);

        /**
         * <p>
         * Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
         * </p>
         * 
         * @param ruleId
         *        Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);

        /**
         * <p>
         * An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
         * </p>
         * 
         * @param copyActions
         *        An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyActions(Collection<CopyAction> copyActions);

        /**
         * <p>
         * An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
         * </p>
         * 
         * @param copyActions
         *        An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyActions(CopyAction... copyActions);

        /**
         * <p>
         * An array of <code>CopyAction</code> objects, which contains the details of the copy operation.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.backup.model.CopyAction.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.backup.model.CopyAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.backup.model.CopyAction.Builder#build()} is called immediately and its
         * result is passed to {@link #copyActions(List<CopyAction>)}.
         * 
         * @param copyActions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.backup.model.CopyAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #copyActions(java.util.Collection<CopyAction>)
         */
        Builder copyActions(Consumer<CopyAction.Builder>... copyActions);

        /**
         * <p>
         * Specifies whether Backup creates continuous backups. True causes Backup to create continuous backups capable
         * of point-in-time restore (PITR). False (or not specified) causes Backup to create snapshot backups.
         * </p>
         * 
         * @param enableContinuousBackup
         *        Specifies whether Backup creates continuous backups. True causes Backup to create continuous backups
         *        capable of point-in-time restore (PITR). False (or not specified) causes Backup to create snapshot
         *        backups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableContinuousBackup(Boolean enableContinuousBackup);

        /**
         * <p>
         * The timezone in which the schedule expression is set. By default, ScheduleExpressions are in UTC. You can
         * modify this to a specified timezone.
         * </p>
         * 
         * @param scheduleExpressionTimezone
         *        The timezone in which the schedule expression is set. By default, ScheduleExpressions are in UTC. You
         *        can modify this to a specified timezone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpressionTimezone(String scheduleExpressionTimezone);

        /**
         * <p>
         * IndexActions is an array you use to specify how backup data should be indexed.
         * </p>
         * <p>
         * eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with it.
         * </p>
         * <p>
         * Within the array is ResourceType. Only one will be accepted for each BackupRule.
         * </p>
         * 
         * @param indexActions
         *        IndexActions is an array you use to specify how backup data should be indexed.</p>
         *        <p>
         *        eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with
         *        it.
         *        </p>
         *        <p>
         *        Within the array is ResourceType. Only one will be accepted for each BackupRule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexActions(Collection<IndexAction> indexActions);

        /**
         * <p>
         * IndexActions is an array you use to specify how backup data should be indexed.
         * </p>
         * <p>
         * eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with it.
         * </p>
         * <p>
         * Within the array is ResourceType. Only one will be accepted for each BackupRule.
         * </p>
         * 
         * @param indexActions
         *        IndexActions is an array you use to specify how backup data should be indexed.</p>
         *        <p>
         *        eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with
         *        it.
         *        </p>
         *        <p>
         *        Within the array is ResourceType. Only one will be accepted for each BackupRule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexActions(IndexAction... indexActions);

        /**
         * <p>
         * IndexActions is an array you use to specify how backup data should be indexed.
         * </p>
         * <p>
         * eEach BackupRule can have 0 or 1 IndexAction, as each backup can have up to one index associated with it.
         * </p>
         * <p>
         * Within the array is ResourceType. Only one will be accepted for each BackupRule.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.backup.model.IndexAction.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.backup.model.IndexAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.backup.model.IndexAction.Builder#build()} is called immediately and
         * its result is passed to {@link #indexActions(List<IndexAction>)}.
         * 
         * @param indexActions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.backup.model.IndexAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #indexActions(java.util.Collection<IndexAction>)
         */
        Builder indexActions(Consumer<IndexAction.Builder>... indexActions);

        /**
         * <p>
         * Contains your scanning configuration for the backup rule and includes the malware scanner, and scan mode of
         * either full or incremental.
         * </p>
         * 
         * @param scanActions
         *        Contains your scanning configuration for the backup rule and includes the malware scanner, and scan
         *        mode of either full or incremental.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanActions(Collection<ScanAction> scanActions);

        /**
         * <p>
         * Contains your scanning configuration for the backup rule and includes the malware scanner, and scan mode of
         * either full or incremental.
         * </p>
         * 
         * @param scanActions
         *        Contains your scanning configuration for the backup rule and includes the malware scanner, and scan
         *        mode of either full or incremental.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanActions(ScanAction... scanActions);

        /**
         * <p>
         * Contains your scanning configuration for the backup rule and includes the malware scanner, and scan mode of
         * either full or incremental.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.backup.model.ScanAction.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.backup.model.ScanAction#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.backup.model.ScanAction.Builder#build()} is called immediately and its
         * result is passed to {@link #scanActions(List<ScanAction>)}.
         * 
         * @param scanActions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.backup.model.ScanAction.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scanActions(java.util.Collection<ScanAction>)
         */
        Builder scanActions(Consumer<ScanAction.Builder>... scanActions);
    }

    static final class BuilderImpl implements Builder {
        private String ruleName;

        private String targetBackupVaultName;

        private String targetLogicallyAirGappedBackupVaultArn;

        private String scheduleExpression;

        private Long startWindowMinutes;

        private Long completionWindowMinutes;

        private Lifecycle lifecycle;

        private Map<String, String> recoveryPointTags = DefaultSdkAutoConstructMap.getInstance();

        private String ruleId;

        private List<CopyAction> copyActions = DefaultSdkAutoConstructList.getInstance();

        private Boolean enableContinuousBackup;

        private String scheduleExpressionTimezone;

        private List<IndexAction> indexActions = DefaultSdkAutoConstructList.getInstance();

        private List<ScanAction> scanActions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BackupRule model) {
            ruleName(model.ruleName);
            targetBackupVaultName(model.targetBackupVaultName);
            targetLogicallyAirGappedBackupVaultArn(model.targetLogicallyAirGappedBackupVaultArn);
            scheduleExpression(model.scheduleExpression);
            startWindowMinutes(model.startWindowMinutes);
            completionWindowMinutes(model.completionWindowMinutes);
            lifecycle(model.lifecycle);
            recoveryPointTags(model.recoveryPointTags);
            ruleId(model.ruleId);
            copyActions(model.copyActions);
            enableContinuousBackup(model.enableContinuousBackup);
            scheduleExpressionTimezone(model.scheduleExpressionTimezone);
            indexActions(model.indexActions);
            scanActions(model.scanActions);
        }

        public final String getRuleName() {
            return ruleName;
        }

        public final void setRuleName(String ruleName) {
            this.ruleName = ruleName;
        }

        @Override
        public final Builder ruleName(String ruleName) {
            this.ruleName = ruleName;
            return this;
        }

        public final String getTargetBackupVaultName() {
            return targetBackupVaultName;
        }

        public final void setTargetBackupVaultName(String targetBackupVaultName) {
            this.targetBackupVaultName = targetBackupVaultName;
        }

        @Override
        public final Builder targetBackupVaultName(String targetBackupVaultName) {
            this.targetBackupVaultName = targetBackupVaultName;
            return this;
        }

        public final String getTargetLogicallyAirGappedBackupVaultArn() {
            return targetLogicallyAirGappedBackupVaultArn;
        }

        public final void setTargetLogicallyAirGappedBackupVaultArn(String targetLogicallyAirGappedBackupVaultArn) {
            this.targetLogicallyAirGappedBackupVaultArn = targetLogicallyAirGappedBackupVaultArn;
        }

        @Override
        public final Builder targetLogicallyAirGappedBackupVaultArn(String targetLogicallyAirGappedBackupVaultArn) {
            this.targetLogicallyAirGappedBackupVaultArn = targetLogicallyAirGappedBackupVaultArn;
            return this;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final Long getStartWindowMinutes() {
            return startWindowMinutes;
        }

        public final void setStartWindowMinutes(Long startWindowMinutes) {
            this.startWindowMinutes = startWindowMinutes;
        }

        @Override
        public final Builder startWindowMinutes(Long startWindowMinutes) {
            this.startWindowMinutes = startWindowMinutes;
            return this;
        }

        public final Long getCompletionWindowMinutes() {
            return completionWindowMinutes;
        }

        public final void setCompletionWindowMinutes(Long completionWindowMinutes) {
            this.completionWindowMinutes = completionWindowMinutes;
        }

        @Override
        public final Builder completionWindowMinutes(Long completionWindowMinutes) {
            this.completionWindowMinutes = completionWindowMinutes;
            return this;
        }

        public final Lifecycle.Builder getLifecycle() {
            return lifecycle != null ? lifecycle.toBuilder() : null;
        }

        public final void setLifecycle(Lifecycle.BuilderImpl lifecycle) {
            this.lifecycle = lifecycle != null ? lifecycle.build() : null;
        }

        @Override
        public final Builder lifecycle(Lifecycle lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        public final Map<String, String> getRecoveryPointTags() {
            if (recoveryPointTags instanceof SdkAutoConstructMap) {
                return null;
            }
            return recoveryPointTags;
        }

        public final void setRecoveryPointTags(Map<String, String> recoveryPointTags) {
            this.recoveryPointTags = TagsCopier.copy(recoveryPointTags);
        }

        @Override
        public final Builder recoveryPointTags(Map<String, String> recoveryPointTags) {
            this.recoveryPointTags = TagsCopier.copy(recoveryPointTags);
            return this;
        }

        public final String getRuleId() {
            return ruleId;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final List<CopyAction.Builder> getCopyActions() {
            List<CopyAction.Builder> result = CopyActionsCopier.copyToBuilder(this.copyActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCopyActions(Collection<CopyAction.BuilderImpl> copyActions) {
            this.copyActions = CopyActionsCopier.copyFromBuilder(copyActions);
        }

        @Override
        public final Builder copyActions(Collection<CopyAction> copyActions) {
            this.copyActions = CopyActionsCopier.copy(copyActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder copyActions(CopyAction... copyActions) {
            copyActions(Arrays.asList(copyActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder copyActions(Consumer<CopyAction.Builder>... copyActions) {
            copyActions(Stream.of(copyActions).map(c -> CopyAction.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEnableContinuousBackup() {
            return enableContinuousBackup;
        }

        public final void setEnableContinuousBackup(Boolean enableContinuousBackup) {
            this.enableContinuousBackup = enableContinuousBackup;
        }

        @Override
        public final Builder enableContinuousBackup(Boolean enableContinuousBackup) {
            this.enableContinuousBackup = enableContinuousBackup;
            return this;
        }

        public final String getScheduleExpressionTimezone() {
            return scheduleExpressionTimezone;
        }

        public final void setScheduleExpressionTimezone(String scheduleExpressionTimezone) {
            this.scheduleExpressionTimezone = scheduleExpressionTimezone;
        }

        @Override
        public final Builder scheduleExpressionTimezone(String scheduleExpressionTimezone) {
            this.scheduleExpressionTimezone = scheduleExpressionTimezone;
            return this;
        }

        public final List<IndexAction.Builder> getIndexActions() {
            List<IndexAction.Builder> result = IndexActionsCopier.copyToBuilder(this.indexActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIndexActions(Collection<IndexAction.BuilderImpl> indexActions) {
            this.indexActions = IndexActionsCopier.copyFromBuilder(indexActions);
        }

        @Override
        public final Builder indexActions(Collection<IndexAction> indexActions) {
            this.indexActions = IndexActionsCopier.copy(indexActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder indexActions(IndexAction... indexActions) {
            indexActions(Arrays.asList(indexActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder indexActions(Consumer<IndexAction.Builder>... indexActions) {
            indexActions(Stream.of(indexActions).map(c -> IndexAction.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ScanAction.Builder> getScanActions() {
            List<ScanAction.Builder> result = ScanActionsCopier.copyToBuilder(this.scanActions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setScanActions(Collection<ScanAction.BuilderImpl> scanActions) {
            this.scanActions = ScanActionsCopier.copyFromBuilder(scanActions);
        }

        @Override
        public final Builder scanActions(Collection<ScanAction> scanActions) {
            this.scanActions = ScanActionsCopier.copy(scanActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scanActions(ScanAction... scanActions) {
            scanActions(Arrays.asList(scanActions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scanActions(Consumer<ScanAction.Builder>... scanActions) {
            scanActions(Stream.of(scanActions).map(c -> ScanAction.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public BackupRule build() {
            return new BackupRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
