/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth.signer;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.SdkTestInternalApi;
import software.amazon.awssdk.auth.AwsExecutionAttributes;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.signer.AbstractAwsSigner;
import software.amazon.awssdk.auth.signer.Presigner;
import software.amazon.awssdk.auth.signer.RegionAwareSigner;
import software.amazon.awssdk.auth.signer.ServiceAwareSigner;
import software.amazon.awssdk.auth.signer.SignerKey;
import software.amazon.awssdk.auth.signer.SigningAlgorithm;
import software.amazon.awssdk.auth.signer.internal.Aws4SignerRequestParams;
import software.amazon.awssdk.auth.signer.internal.Aws4SignerUtils;
import software.amazon.awssdk.auth.util.CredentialUtils;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.internal.collections.FifoCache;
import software.amazon.awssdk.core.runtime.auth.SdkClock;
import software.amazon.awssdk.core.util.DateUtils;
import software.amazon.awssdk.core.util.StringUtils;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.utils.BinaryUtils;
import software.amazon.awssdk.utils.http.SdkHttpUtils;

public class Aws4Signer
extends AbstractAwsSigner
implements ServiceAwareSigner,
RegionAwareSigner,
Presigner {
    private static final Logger LOG = LoggerFactory.getLogger(Aws4Signer.class);
    private static final int SIGNER_CACHE_MAX_SIZE = 300;
    private static final FifoCache<SignerKey> SIGNER_CACHE = new FifoCache(300);
    private static final List<String> LIST_OF_HEADERS_TO_IGNORE_IN_LOWER_CASE = Arrays.asList("connection", "x-amzn-trace-id");
    protected String serviceName;
    protected String regionName;
    private Date overriddenDate;
    private boolean doubleUrlEncode;
    private final SdkClock clock;

    public Aws4Signer() {
        this(true);
    }

    public Aws4Signer(boolean doubleUrlEncoding) {
        this(doubleUrlEncoding, SdkClock.STANDARD);
    }

    @SdkTestInternalApi
    public Aws4Signer(SdkClock clock) {
        this(true, clock);
    }

    private Aws4Signer(boolean doubleUrlEncode, SdkClock clock) {
        this.doubleUrlEncode = doubleUrlEncode;
        this.clock = clock;
    }

    @SdkTestInternalApi
    public void setOverrideDate(Date overriddenDate) {
        this.overriddenDate = overriddenDate != null ? new Date(overriddenDate.getTime()) : null;
    }

    public String getRegionName() {
        return this.regionName;
    }

    @Override
    public void setRegionName(String regionName) {
        this.regionName = regionName;
    }

    public String getServiceName() {
        return this.serviceName;
    }

    @Override
    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    public SdkHttpFullRequest sign(Context.BeforeTransmission execution, ExecutionAttributes executionAttributes) {
        if (CredentialUtils.isAnonymous((AwsCredentials)executionAttributes.getAttribute(AwsExecutionAttributes.AWS_CREDENTIALS))) {
            return execution.httpRequest();
        }
        return (SdkHttpFullRequest)execution.httpRequest().copy(b -> this.doSign((SdkHttpFullRequest.Builder)b, execution, executionAttributes));
    }

    private SdkHttpFullRequest.Builder doSign(SdkHttpFullRequest.Builder mutableRequest, Context.BeforeTransmission execution, ExecutionAttributes executionAttributes) {
        AwsCredentials sanitizedCredentials = this.sanitizeCredentials((AwsCredentials)executionAttributes.getAttribute(AwsExecutionAttributes.AWS_CREDENTIALS));
        if (sanitizedCredentials instanceof AwsSessionCredentials) {
            this.addSessionCredentials(mutableRequest, (AwsSessionCredentials)sanitizedCredentials);
        }
        Aws4SignerRequestParams signerParams = new Aws4SignerRequestParams(execution.request(), mutableRequest, executionAttributes, this.overriddenDate, this.regionName, this.serviceName, "AWS4-HMAC-SHA256");
        this.addHostHeader(mutableRequest);
        mutableRequest.header("X-Amz-Date", signerParams.getFormattedSigningDateTime());
        String contentSha256 = this.calculateContentHash(signerParams, mutableRequest);
        mutableRequest.firstMatchingHeader("x-amz-content-sha256").filter(h -> h.equals("required")).ifPresent(h -> mutableRequest.header("x-amz-content-sha256", contentSha256));
        String canonicalRequest = this.createCanonicalRequest(mutableRequest, contentSha256);
        String stringToSign = this.createStringToSign(canonicalRequest, signerParams);
        byte[] signingKey = this.deriveSigningKey(sanitizedCredentials, signerParams);
        byte[] signature = this.computeSignature(stringToSign, signingKey);
        mutableRequest.header("Authorization", this.buildAuthorizationHeader(signature, sanitizedCredentials, signerParams));
        this.processRequestPayload(mutableRequest, signature, signingKey, signerParams);
        return mutableRequest;
    }

    @Override
    public SdkHttpFullRequest presign(Context.BeforeTransmission execution, ExecutionAttributes executionAttributes, Date userSpecifiedExpirationDate) {
        if (CredentialUtils.isAnonymous((AwsCredentials)executionAttributes.getAttribute(AwsExecutionAttributes.AWS_CREDENTIALS))) {
            return execution.httpRequest();
        }
        SdkHttpFullRequest.Builder mutableRequest = (SdkHttpFullRequest.Builder)execution.httpRequest().toBuilder();
        long expirationInSeconds = this.generateExpirationDate(userSpecifiedExpirationDate);
        this.addHostHeader(mutableRequest);
        AwsCredentials sanitizedCredentials = this.sanitizeCredentials((AwsCredentials)executionAttributes.getAttribute(AwsExecutionAttributes.AWS_CREDENTIALS));
        if (sanitizedCredentials instanceof AwsSessionCredentials) {
            mutableRequest.rawQueryParameter("X-Amz-Security-Token", ((AwsSessionCredentials)sanitizedCredentials).sessionToken());
        }
        Aws4SignerRequestParams signerRequestParams = new Aws4SignerRequestParams(execution.request(), mutableRequest, executionAttributes, this.overriddenDate, this.regionName, this.serviceName, "AWS4-HMAC-SHA256");
        String timeStamp = signerRequestParams.getFormattedSigningDateTime();
        this.addPreSignInformationToRequest(mutableRequest, sanitizedCredentials, signerRequestParams, timeStamp, expirationInSeconds);
        String contentSha256 = this.calculateContentHashPresign(signerRequestParams, mutableRequest);
        String canonicalRequest = this.createCanonicalRequest(mutableRequest, contentSha256);
        String stringToSign = this.createStringToSign(canonicalRequest, signerRequestParams);
        byte[] signingKey = this.deriveSigningKey(sanitizedCredentials, signerRequestParams);
        byte[] signature = this.computeSignature(stringToSign, signingKey);
        mutableRequest.rawQueryParameter("X-Amz-Signature", BinaryUtils.toHex((byte[])signature));
        return (SdkHttpFullRequest)mutableRequest.build();
    }

    private String createCanonicalRequest(SdkHttpFullRequest.Builder request, String contentSha256) {
        String canonicalRequest = request.method().toString() + "\n" + this.getCanonicalizedResourcePath(request.encodedPath(), this.doubleUrlEncode) + "\n" + this.getCanonicalizedQueryString(request.rawQueryParameters()) + "\n" + this.getCanonicalizedHeaderString(request.headers()) + "\n" + this.getSignedHeadersString(request.headers()) + "\n" + contentSha256;
        if (LOG.isDebugEnabled()) {
            LOG.debug("AWS4 Canonical Request: '\"" + canonicalRequest + "\"");
        }
        return canonicalRequest;
    }

    private String createStringToSign(String canonicalRequest, Aws4SignerRequestParams signerParams) {
        String stringToSign = signerParams.getSigningAlgorithm() + "\n" + signerParams.getFormattedSigningDateTime() + "\n" + signerParams.getScope() + "\n" + BinaryUtils.toHex((byte[])this.hash(canonicalRequest));
        if (LOG.isDebugEnabled()) {
            LOG.debug("AWS4 String to Sign: '\"" + stringToSign + "\"");
        }
        return stringToSign;
    }

    private byte[] deriveSigningKey(AwsCredentials credentials, Aws4SignerRequestParams signerRequestParams) {
        String cacheKey = this.computeSigningCacheKeyName(credentials, signerRequestParams);
        long daysSinceEpochSigningDate = DateUtils.numberOfDaysSinceEpoch((long)signerRequestParams.getSigningDateTimeMilli());
        SignerKey signerKey = (SignerKey)SIGNER_CACHE.get(cacheKey);
        if (signerKey != null && daysSinceEpochSigningDate == signerKey.getNumberOfDaysSinceEpoch()) {
            return signerKey.getSigningKey();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Generating a new signing key as the signing key not available in the cache for the date " + TimeUnit.DAYS.toMillis(daysSinceEpochSigningDate));
        }
        byte[] signingKey = this.newSigningKey(credentials, signerRequestParams.getFormattedSigningDate(), signerRequestParams.getRegionName(), signerRequestParams.getServiceName());
        SIGNER_CACHE.add(cacheKey, (Object)new SignerKey(daysSinceEpochSigningDate, signingKey));
        return signingKey;
    }

    private String computeSigningCacheKeyName(AwsCredentials credentials, Aws4SignerRequestParams signerRequestParams) {
        return credentials.secretAccessKey() + "-" + signerRequestParams.getRegionName() + "-" + signerRequestParams.getServiceName();
    }

    private byte[] computeSignature(String stringToSign, byte[] signingKey) {
        return this.sign(stringToSign.getBytes(Charset.forName("UTF-8")), signingKey, SigningAlgorithm.HmacSHA256);
    }

    private String buildAuthorizationHeader(byte[] signature, AwsCredentials credentials, Aws4SignerRequestParams signerParams) {
        String signingCredentials = credentials.accessKeyId() + "/" + signerParams.getScope();
        String credential = "Credential=" + signingCredentials;
        String signerHeaders = "SignedHeaders=" + this.getSignedHeadersString(signerParams.httpRequest().headers());
        String signatureHeader = "Signature=" + BinaryUtils.toHex((byte[])signature);
        return "AWS4-HMAC-SHA256 " + credential + ", " + signerHeaders + ", " + signatureHeader;
    }

    private void addPreSignInformationToRequest(SdkHttpFullRequest.Builder mutableRequest, AwsCredentials sanitizedCredentials, Aws4SignerRequestParams signerParams, String timeStamp, long expirationInSeconds) {
        String signingCredentials = sanitizedCredentials.accessKeyId() + "/" + signerParams.getScope();
        mutableRequest.rawQueryParameter("X-Amz-Algorithm", "AWS4-HMAC-SHA256");
        mutableRequest.rawQueryParameter("X-Amz-Date", timeStamp);
        mutableRequest.rawQueryParameter("X-Amz-SignedHeaders", this.getSignedHeadersString(signerParams.httpRequest().headers()));
        mutableRequest.rawQueryParameter("X-Amz-Expires", Long.toString(expirationInSeconds));
        mutableRequest.rawQueryParameter("X-Amz-Credential", signingCredentials);
    }

    @Override
    protected void addSessionCredentials(SdkHttpFullRequest.Builder mutableRequest, AwsSessionCredentials credentials) {
        mutableRequest.header("X-Amz-Security-Token", credentials.sessionToken());
    }

    private String getCanonicalizedHeaderString(Map<String, List<String>> headers) {
        ArrayList<String> sortedHeaders = new ArrayList<String>(headers.keySet());
        sortedHeaders.sort(String.CASE_INSENSITIVE_ORDER);
        Map<String, List<String>> requestHeaders = headers;
        StringBuilder buffer = new StringBuilder();
        for (String header : sortedHeaders) {
            if (this.shouldExcludeHeaderFromSigning(header)) continue;
            String key = software.amazon.awssdk.utils.StringUtils.lowerCase((String)header);
            for (String headerValue : requestHeaders.get(header)) {
                StringUtils.appendCompactedString((StringBuilder)buffer, (String)key);
                buffer.append(":");
                if (headerValue != null) {
                    StringUtils.appendCompactedString((StringBuilder)buffer, (String)headerValue);
                }
                buffer.append("\n");
            }
        }
        return buffer.toString();
    }

    private String getSignedHeadersString(Map<String, List<String>> headers) {
        ArrayList<String> sortedHeaders = new ArrayList<String>(headers.keySet());
        sortedHeaders.sort(String.CASE_INSENSITIVE_ORDER);
        StringBuilder buffer = new StringBuilder();
        for (String header : sortedHeaders) {
            if (this.shouldExcludeHeaderFromSigning(header)) continue;
            if (buffer.length() > 0) {
                buffer.append(";");
            }
            buffer.append(software.amazon.awssdk.utils.StringUtils.lowerCase((String)header));
        }
        return buffer.toString();
    }

    private boolean shouldExcludeHeaderFromSigning(String header) {
        return LIST_OF_HEADERS_TO_IGNORE_IN_LOWER_CASE.contains(software.amazon.awssdk.utils.StringUtils.lowerCase((String)header));
    }

    private void addHostHeader(SdkHttpFullRequest.Builder mutableRequest) {
        StringBuilder hostHeaderBuilder = new StringBuilder(mutableRequest.host());
        if (!SdkHttpUtils.isUsingStandardPort((String)mutableRequest.protocol(), (Integer)mutableRequest.port())) {
            hostHeaderBuilder.append(":").append(mutableRequest.port());
        }
        mutableRequest.header("Host", hostHeaderBuilder.toString());
    }

    protected String calculateContentHash(Aws4SignerRequestParams signerRequestParams, SdkHttpFullRequest.Builder requestBuilder) {
        SdkHttpFullRequest.Builder requestToSign = signerRequestParams.httpRequest();
        InputStream payloadStream = this.getBinaryRequestPayloadStream(requestToSign.content());
        payloadStream.mark(AbstractAwsSigner.getReadLimit(signerRequestParams));
        String contentSha256 = BinaryUtils.toHex((byte[])this.hash(payloadStream));
        try {
            payloadStream.reset();
        }
        catch (IOException e) {
            throw new SdkClientException("Unable to reset stream after calculating AWS4 signature", (Throwable)e);
        }
        return contentSha256;
    }

    protected void processRequestPayload(SdkHttpFullRequest.Builder requestBuilder, byte[] signature, byte[] signingKey, Aws4SignerRequestParams signerRequestParams) {
    }

    protected String calculateContentHashPresign(Aws4SignerRequestParams signerRequestParams, SdkHttpFullRequest.Builder requestBuilder) {
        return this.calculateContentHash(signerRequestParams, requestBuilder);
    }

    private long generateExpirationDate(Date expirationDate) {
        long expirationInSeconds;
        long l = expirationInSeconds = expirationDate != null ? (expirationDate.getTime() - this.clock.currentTimeMillis()) / 1000L : 604800L;
        if (expirationInSeconds > 604800L) {
            throw new SdkClientException("Requests that are pre-signed by SigV4 algorithm are valid for at most 7 days. The expiration date set on the current request [" + Aws4SignerUtils.formatTimestamp(expirationDate.getTime()) + "] has exceeded this limit.");
        }
        return expirationInSeconds;
    }

    private byte[] newSigningKey(AwsCredentials credentials, String dateStamp, String regionName, String serviceName) {
        byte[] kSecret = ("AWS4" + credentials.secretAccessKey()).getBytes(Charset.forName("UTF-8"));
        byte[] kDate = this.sign(dateStamp, kSecret, SigningAlgorithm.HmacSHA256);
        byte[] kRegion = this.sign(regionName, kDate, SigningAlgorithm.HmacSHA256);
        byte[] kService = this.sign(serviceName, kRegion, SigningAlgorithm.HmacSHA256);
        return this.sign("aws4_request", kService, SigningAlgorithm.HmacSHA256);
    }
}

