/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutScalingPolicyRequest extends ApplicationAutoScalingRequest implements
        ToCopyableBuilder<PutScalingPolicyRequest.Builder, PutScalingPolicyRequest> {
    private static final SdkField<String> POLICY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutScalingPolicyRequest::policyName)).setter(setter(Builder::policyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyName").build()).build();

    private static final SdkField<String> SERVICE_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutScalingPolicyRequest::serviceNamespaceAsString)).setter(setter(Builder::serviceNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceNamespace").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutScalingPolicyRequest::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> SCALABLE_DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutScalingPolicyRequest::scalableDimensionAsString)).setter(setter(Builder::scalableDimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalableDimension").build()).build();

    private static final SdkField<String> POLICY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PutScalingPolicyRequest::policyTypeAsString)).setter(setter(Builder::policyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyType").build()).build();

    private static final SdkField<StepScalingPolicyConfiguration> STEP_SCALING_POLICY_CONFIGURATION_FIELD = SdkField
            .<StepScalingPolicyConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(PutScalingPolicyRequest::stepScalingPolicyConfiguration))
            .setter(setter(Builder::stepScalingPolicyConfiguration))
            .constructor(StepScalingPolicyConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepScalingPolicyConfiguration")
                    .build()).build();

    private static final SdkField<TargetTrackingScalingPolicyConfiguration> TARGET_TRACKING_SCALING_POLICY_CONFIGURATION_FIELD = SdkField
            .<TargetTrackingScalingPolicyConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(PutScalingPolicyRequest::targetTrackingScalingPolicyConfiguration))
            .setter(setter(Builder::targetTrackingScalingPolicyConfiguration))
            .constructor(TargetTrackingScalingPolicyConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("TargetTrackingScalingPolicyConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_NAME_FIELD,
            SERVICE_NAMESPACE_FIELD, RESOURCE_ID_FIELD, SCALABLE_DIMENSION_FIELD, POLICY_TYPE_FIELD,
            STEP_SCALING_POLICY_CONFIGURATION_FIELD, TARGET_TRACKING_SCALING_POLICY_CONFIGURATION_FIELD));

    private final String policyName;

    private final String serviceNamespace;

    private final String resourceId;

    private final String scalableDimension;

    private final String policyType;

    private final StepScalingPolicyConfiguration stepScalingPolicyConfiguration;

    private final TargetTrackingScalingPolicyConfiguration targetTrackingScalingPolicyConfiguration;

    private PutScalingPolicyRequest(BuilderImpl builder) {
        super(builder);
        this.policyName = builder.policyName;
        this.serviceNamespace = builder.serviceNamespace;
        this.resourceId = builder.resourceId;
        this.scalableDimension = builder.scalableDimension;
        this.policyType = builder.policyType;
        this.stepScalingPolicyConfiguration = builder.stepScalingPolicyConfiguration;
        this.targetTrackingScalingPolicyConfiguration = builder.targetTrackingScalingPolicyConfiguration;
    }

    /**
     * <p>
     * The name of the scaling policy.
     * </p>
     * 
     * @return The name of the scaling policy.
     */
    public String policyName() {
        return policyName;
    }

    /**
     * <p>
     * The namespace of the AWS service that provides the resource. For a resource provided by your own application or
     * service, use <code>custom-resource</code> instead.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service that provides the resource. For a resource provided by your own
     *         application or service, use <code>custom-resource</code> instead.
     * @see ServiceNamespace
     */
    public ServiceNamespace serviceNamespace() {
        return ServiceNamespace.fromValue(serviceNamespace);
    }

    /**
     * <p>
     * The namespace of the AWS service that provides the resource. For a resource provided by your own application or
     * service, use <code>custom-resource</code> instead.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the AWS service that provides the resource. For a resource provided by your own
     *         application or service, use <code>custom-resource</code> instead.
     * @see ServiceNamespace
     */
    public String serviceNamespaceAsString() {
        return serviceNamespace;
    }

    /**
     * <p>
     * The identifier of the resource associated with the scaling policy. This string consists of the resource type and
     * unique identifier.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and service
     * name. Example: <code>service/default/sample-webapp</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot
     * Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID and
     * instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet name.
     * Example: <code>fleet/sample-fleet</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name. Example:
     * <code>table/my-table</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the index
     * name. Example: <code>table/my-table/index/my-table-index</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
     * Example: <code>cluster:my-db-cluster</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is the
     * resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom resources are not supported with a resource type. This parameter must specify the <code>OutputValue</code>
     * from the CloudFormation template stack used to access the resources. The unique identifier is defined by the
     * service provider. More information is available in our <a
     * href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Comprehend document classification endpoint - The resource type and unique identifier are specified using
     * the endpoint ARN. Example:
     * <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is the
     * function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
     * <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the table name.
     * Example: <code>keyspace/mykeyspace/table/mytable</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier of the resource associated with the scaling policy. This string consists of the resource
     *         type and unique identifier.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
     *         service name. Example: <code>service/default/sample-webapp</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
     *         the Spot Fleet request ID. Example:
     *         <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID
     *         and instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet
     *         name. Example: <code>fleet/sample-fleet</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
     *         Example: <code>table/my-table</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
     *         the index name. Example: <code>table/my-table/index/my-table-index</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
     *         name. Example: <code>cluster:my-db-cluster</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier
     *         is the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Custom resources are not supported with a resource type. This parameter must specify the
     *         <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The unique
     *         identifier is defined by the service provider. More information is available in our <a
     *         href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Comprehend document classification endpoint - The resource type and unique identifier are
     *         specified using the endpoint ARN. Example:
     *         <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is
     *         the function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
     *         <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the table
     *         name. Example: <code>keyspace/mykeyspace/table/mytable</code>.
     *         </p>
     *         </li>
     */
    public String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker model
     * endpoint variant.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
     * your own application or service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend document classification endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension. This string consists of the service namespace, resource type, and scaling
     *         property.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
     *         model endpoint variant.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
     *         provided by your own application or service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
     *         units for an Amazon Comprehend document classification endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public ScalableDimension scalableDimension() {
        return ScalableDimension.fromValue(scalableDimension);
    }

    /**
     * <p>
     * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker model
     * endpoint variant.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
     * your own application or service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend document classification endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension. This string consists of the service namespace, resource type, and scaling
     *         property.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
     *         model endpoint variant.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
     *         provided by your own application or service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
     *         units for an Amazon Comprehend document classification endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public String scalableDimensionAsString() {
        return scalableDimension;
    }

    /**
     * <p>
     * The policy type. This parameter is required if you are creating a scaling policy.
     * </p>
     * <p>
     * The following policy types are supported:
     * </p>
     * <p>
     * <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
     * </p>
     * <p>
     * <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces (for Apache
     * Cassandra).
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     * >Target Tracking Scaling Policies</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     * >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The policy type. This parameter is required if you are creating a scaling policy.</p>
     *         <p>
     *         The following policy types are supported:
     *         </p>
     *         <p>
     *         <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
     *         </p>
     *         <p>
     *         <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces (for
     *         Apache Cassandra).
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     *         >Target Tracking Scaling Policies</a> and <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     *         >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * @see PolicyType
     */
    public PolicyType policyType() {
        return PolicyType.fromValue(policyType);
    }

    /**
     * <p>
     * The policy type. This parameter is required if you are creating a scaling policy.
     * </p>
     * <p>
     * The following policy types are supported:
     * </p>
     * <p>
     * <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
     * </p>
     * <p>
     * <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces (for Apache
     * Cassandra).
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     * >Target Tracking Scaling Policies</a> and <a href=
     * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     * >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The policy type. This parameter is required if you are creating a scaling policy.</p>
     *         <p>
     *         The following policy types are supported:
     *         </p>
     *         <p>
     *         <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
     *         </p>
     *         <p>
     *         <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces (for
     *         Apache Cassandra).
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
     *         >Target Tracking Scaling Policies</a> and <a href=
     *         "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
     *         >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
     * @see PolicyType
     */
    public String policyTypeAsString() {
        return policyType;
    }

    /**
     * <p>
     * A step scaling policy.
     * </p>
     * <p>
     * This parameter is required if you are creating a policy and the policy type is <code>StepScaling</code>.
     * </p>
     * 
     * @return A step scaling policy.</p>
     *         <p>
     *         This parameter is required if you are creating a policy and the policy type is <code>StepScaling</code>.
     */
    public StepScalingPolicyConfiguration stepScalingPolicyConfiguration() {
        return stepScalingPolicyConfiguration;
    }

    /**
     * <p>
     * A target tracking scaling policy. Includes support for predefined or customized metrics.
     * </p>
     * <p>
     * This parameter is required if you are creating a policy and the policy type is <code>TargetTrackingScaling</code>
     * .
     * </p>
     * 
     * @return A target tracking scaling policy. Includes support for predefined or customized metrics.</p>
     *         <p>
     *         This parameter is required if you are creating a policy and the policy type is
     *         <code>TargetTrackingScaling</code>.
     */
    public TargetTrackingScalingPolicyConfiguration targetTrackingScalingPolicyConfiguration() {
        return targetTrackingScalingPolicyConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(policyName());
        hashCode = 31 * hashCode + Objects.hashCode(serviceNamespaceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(scalableDimensionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stepScalingPolicyConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(targetTrackingScalingPolicyConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutScalingPolicyRequest)) {
            return false;
        }
        PutScalingPolicyRequest other = (PutScalingPolicyRequest) obj;
        return Objects.equals(policyName(), other.policyName())
                && Objects.equals(serviceNamespaceAsString(), other.serviceNamespaceAsString())
                && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(scalableDimensionAsString(), other.scalableDimensionAsString())
                && Objects.equals(policyTypeAsString(), other.policyTypeAsString())
                && Objects.equals(stepScalingPolicyConfiguration(), other.stepScalingPolicyConfiguration())
                && Objects.equals(targetTrackingScalingPolicyConfiguration(), other.targetTrackingScalingPolicyConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutScalingPolicyRequest").add("PolicyName", policyName())
                .add("ServiceNamespace", serviceNamespaceAsString()).add("ResourceId", resourceId())
                .add("ScalableDimension", scalableDimensionAsString()).add("PolicyType", policyTypeAsString())
                .add("StepScalingPolicyConfiguration", stepScalingPolicyConfiguration())
                .add("TargetTrackingScalingPolicyConfiguration", targetTrackingScalingPolicyConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyName":
            return Optional.ofNullable(clazz.cast(policyName()));
        case "ServiceNamespace":
            return Optional.ofNullable(clazz.cast(serviceNamespaceAsString()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ScalableDimension":
            return Optional.ofNullable(clazz.cast(scalableDimensionAsString()));
        case "PolicyType":
            return Optional.ofNullable(clazz.cast(policyTypeAsString()));
        case "StepScalingPolicyConfiguration":
            return Optional.ofNullable(clazz.cast(stepScalingPolicyConfiguration()));
        case "TargetTrackingScalingPolicyConfiguration":
            return Optional.ofNullable(clazz.cast(targetTrackingScalingPolicyConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutScalingPolicyRequest, T> g) {
        return obj -> g.apply((PutScalingPolicyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApplicationAutoScalingRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, PutScalingPolicyRequest> {
        /**
         * <p>
         * The name of the scaling policy.
         * </p>
         * 
         * @param policyName
         *        The name of the scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policyName(String policyName);

        /**
         * <p>
         * The namespace of the AWS service that provides the resource. For a resource provided by your own application
         * or service, use <code>custom-resource</code> instead.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service that provides the resource. For a resource provided by your own
         *        application or service, use <code>custom-resource</code> instead.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(String serviceNamespace);

        /**
         * <p>
         * The namespace of the AWS service that provides the resource. For a resource provided by your own application
         * or service, use <code>custom-resource</code> instead.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the AWS service that provides the resource. For a resource provided by your own
         *        application or service, use <code>custom-resource</code> instead.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(ServiceNamespace serviceNamespace);

        /**
         * <p>
         * The identifier of the resource associated with the scaling policy. This string consists of the resource type
         * and unique identifier.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
         * service name. Example: <code>service/default/sample-webapp</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is the
         * Spot Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID and
         * instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet name.
         * Example: <code>fleet/sample-fleet</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
         * Example: <code>table/my-table</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
         * index name. Example: <code>table/my-table/index/my-table-index</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
         * Example: <code>cluster:my-db-cluster</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is
         * the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Custom resources are not supported with a resource type. This parameter must specify the
         * <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The unique
         * identifier is defined by the service provider. More information is available in our <a
         * href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Comprehend document classification endpoint - The resource type and unique identifier are specified
         * using the endpoint ARN. Example:
         * <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is the
         * function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
         * <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the table name.
         * Example: <code>keyspace/mykeyspace/table/mytable</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceId
         *        The identifier of the resource associated with the scaling policy. This string consists of the
         *        resource type and unique identifier.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name
         *        and service name. Example: <code>service/default/sample-webapp</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleet request - The resource type is <code>spot-fleet-request</code> and the unique identifier is
         *        the Spot Fleet request ID. Example:
         *        <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster
         *        ID and instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet
         *        name. Example: <code>fleet/sample-fleet</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
         *        Example: <code>table/my-table</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
         *        the index name. Example: <code>table/my-table/index/my-table-index</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
         *        name. Example: <code>cluster:my-db-cluster</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon SageMaker endpoint variant - The resource type is <code>variant</code> and the unique
         *        identifier is the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Custom resources are not supported with a resource type. This parameter must specify the
         *        <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The
         *        unique identifier is defined by the service provider. More information is available in our <a
         *        href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Comprehend document classification endpoint - The resource type and unique identifier are
         *        specified using the endpoint ARN. Example:
         *        <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier
         *        is the function name with a function version or alias name suffix that is not <code>$LATEST</code>.
         *        Example: <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the
         *        table name. Example: <code>keyspace/mykeyspace/table/mytable</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
         * model endpoint variant.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
         * your own application or service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units
         * for an Amazon Comprehend document classification endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
         * table.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension. This string consists of the service namespace, resource type, and scaling
         *        property.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance
         *        Group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon
         *        SageMaker model endpoint variant.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
         *        provided by your own application or service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend document classification endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda
         *        function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
         *        table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon
         *        Keyspaces table.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(String scalableDimension);

        /**
         * <p>
         * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon SageMaker
         * model endpoint variant.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
         * your own application or service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units
         * for an Amazon Comprehend document classification endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
         * table.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension. This string consists of the service namespace, resource type, and scaling
         *        property.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance
         *        Group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for an Amazon
         *        SageMaker model endpoint variant.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
         *        provided by your own application or service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend document classification endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda
         *        function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
         *        table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon
         *        Keyspaces table.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(ScalableDimension scalableDimension);

        /**
         * <p>
         * The policy type. This parameter is required if you are creating a scaling policy.
         * </p>
         * <p>
         * The following policy types are supported:
         * </p>
         * <p>
         * <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
         * </p>
         * <p>
         * <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces (for
         * Apache Cassandra).
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
         * >Target Tracking Scaling Policies</a> and <a href=
         * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
         * >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param policyType
         *        The policy type. This parameter is required if you are creating a scaling policy.</p>
         *        <p>
         *        The following policy types are supported:
         *        </p>
         *        <p>
         *        <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
         *        </p>
         *        <p>
         *        <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces
         *        (for Apache Cassandra).
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
         *        >Target Tracking Scaling Policies</a> and <a href=
         *        "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
         *        >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * The policy type. This parameter is required if you are creating a scaling policy.
         * </p>
         * <p>
         * The following policy types are supported:
         * </p>
         * <p>
         * <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
         * </p>
         * <p>
         * <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces (for
         * Apache Cassandra).
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
         * >Target Tracking Scaling Policies</a> and <a href=
         * "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
         * >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
         * </p>
         * 
         * @param policyType
         *        The policy type. This parameter is required if you are creating a scaling policy.</p>
         *        <p>
         *        The following policy types are supported:
         *        </p>
         *        <p>
         *        <code>TargetTrackingScaling</code>—Not supported for Amazon EMR
         *        </p>
         *        <p>
         *        <code>StepScaling</code>—Not supported for DynamoDB, Amazon Comprehend, Lambda, or Amazon Keyspaces
         *        (for Apache Cassandra).
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-target-tracking.html"
         *        >Target Tracking Scaling Policies</a> and <a href=
         *        "https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-step-scaling-policies.html"
         *        >Step Scaling Policies</a> in the <i>Application Auto Scaling User Guide</i>.
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(PolicyType policyType);

        /**
         * <p>
         * A step scaling policy.
         * </p>
         * <p>
         * This parameter is required if you are creating a policy and the policy type is <code>StepScaling</code>.
         * </p>
         * 
         * @param stepScalingPolicyConfiguration
         *        A step scaling policy.</p>
         *        <p>
         *        This parameter is required if you are creating a policy and the policy type is
         *        <code>StepScaling</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepScalingPolicyConfiguration(StepScalingPolicyConfiguration stepScalingPolicyConfiguration);

        /**
         * <p>
         * A step scaling policy.
         * </p>
         * <p>
         * This parameter is required if you are creating a policy and the policy type is <code>StepScaling</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link StepScalingPolicyConfiguration.Builder} avoiding
         * the need to create one manually via {@link StepScalingPolicyConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link StepScalingPolicyConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #stepScalingPolicyConfiguration(StepScalingPolicyConfiguration)}.
         * 
         * @param stepScalingPolicyConfiguration
         *        a consumer that will call methods on {@link StepScalingPolicyConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stepScalingPolicyConfiguration(StepScalingPolicyConfiguration)
         */
        default Builder stepScalingPolicyConfiguration(
                Consumer<StepScalingPolicyConfiguration.Builder> stepScalingPolicyConfiguration) {
            return stepScalingPolicyConfiguration(StepScalingPolicyConfiguration.builder()
                    .applyMutation(stepScalingPolicyConfiguration).build());
        }

        /**
         * <p>
         * A target tracking scaling policy. Includes support for predefined or customized metrics.
         * </p>
         * <p>
         * This parameter is required if you are creating a policy and the policy type is
         * <code>TargetTrackingScaling</code>.
         * </p>
         * 
         * @param targetTrackingScalingPolicyConfiguration
         *        A target tracking scaling policy. Includes support for predefined or customized metrics.</p>
         *        <p>
         *        This parameter is required if you are creating a policy and the policy type is
         *        <code>TargetTrackingScaling</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetTrackingScalingPolicyConfiguration(
                TargetTrackingScalingPolicyConfiguration targetTrackingScalingPolicyConfiguration);

        /**
         * <p>
         * A target tracking scaling policy. Includes support for predefined or customized metrics.
         * </p>
         * <p>
         * This parameter is required if you are creating a policy and the policy type is
         * <code>TargetTrackingScaling</code>.
         * </p>
         * This is a convenience that creates an instance of the
         * {@link TargetTrackingScalingPolicyConfiguration.Builder} avoiding the need to create one manually via
         * {@link TargetTrackingScalingPolicyConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link TargetTrackingScalingPolicyConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #targetTrackingScalingPolicyConfiguration(TargetTrackingScalingPolicyConfiguration)}.
         * 
         * @param targetTrackingScalingPolicyConfiguration
         *        a consumer that will call methods on {@link TargetTrackingScalingPolicyConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetTrackingScalingPolicyConfiguration(TargetTrackingScalingPolicyConfiguration)
         */
        default Builder targetTrackingScalingPolicyConfiguration(
                Consumer<TargetTrackingScalingPolicyConfiguration.Builder> targetTrackingScalingPolicyConfiguration) {
            return targetTrackingScalingPolicyConfiguration(TargetTrackingScalingPolicyConfiguration.builder()
                    .applyMutation(targetTrackingScalingPolicyConfiguration).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApplicationAutoScalingRequest.BuilderImpl implements Builder {
        private String policyName;

        private String serviceNamespace;

        private String resourceId;

        private String scalableDimension;

        private String policyType;

        private StepScalingPolicyConfiguration stepScalingPolicyConfiguration;

        private TargetTrackingScalingPolicyConfiguration targetTrackingScalingPolicyConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(PutScalingPolicyRequest model) {
            super(model);
            policyName(model.policyName);
            serviceNamespace(model.serviceNamespace);
            resourceId(model.resourceId);
            scalableDimension(model.scalableDimension);
            policyType(model.policyType);
            stepScalingPolicyConfiguration(model.stepScalingPolicyConfiguration);
            targetTrackingScalingPolicyConfiguration(model.targetTrackingScalingPolicyConfiguration);
        }

        public final String getPolicyName() {
            return policyName;
        }

        @Override
        public final Builder policyName(String policyName) {
            this.policyName = policyName;
            return this;
        }

        public final void setPolicyName(String policyName) {
            this.policyName = policyName;
        }

        public final String getServiceNamespace() {
            return serviceNamespace;
        }

        @Override
        public final Builder serviceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
            return this;
        }

        @Override
        public final Builder serviceNamespace(ServiceNamespace serviceNamespace) {
            this.serviceNamespace(serviceNamespace == null ? null : serviceNamespace.toString());
            return this;
        }

        public final void setServiceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
        }

        public final String getResourceId() {
            return resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        public final String getScalableDimension() {
            return scalableDimension;
        }

        @Override
        public final Builder scalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
            return this;
        }

        @Override
        public final Builder scalableDimension(ScalableDimension scalableDimension) {
            this.scalableDimension(scalableDimension == null ? null : scalableDimension.toString());
            return this;
        }

        public final void setScalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
        }

        public final String getPolicyType() {
            return policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        @Override
        public final Builder policyType(PolicyType policyType) {
            this.policyType(policyType == null ? null : policyType.toString());
            return this;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        public final StepScalingPolicyConfiguration.Builder getStepScalingPolicyConfiguration() {
            return stepScalingPolicyConfiguration != null ? stepScalingPolicyConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder stepScalingPolicyConfiguration(StepScalingPolicyConfiguration stepScalingPolicyConfiguration) {
            this.stepScalingPolicyConfiguration = stepScalingPolicyConfiguration;
            return this;
        }

        public final void setStepScalingPolicyConfiguration(
                StepScalingPolicyConfiguration.BuilderImpl stepScalingPolicyConfiguration) {
            this.stepScalingPolicyConfiguration = stepScalingPolicyConfiguration != null ? stepScalingPolicyConfiguration.build()
                    : null;
        }

        public final TargetTrackingScalingPolicyConfiguration.Builder getTargetTrackingScalingPolicyConfiguration() {
            return targetTrackingScalingPolicyConfiguration != null ? targetTrackingScalingPolicyConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder targetTrackingScalingPolicyConfiguration(
                TargetTrackingScalingPolicyConfiguration targetTrackingScalingPolicyConfiguration) {
            this.targetTrackingScalingPolicyConfiguration = targetTrackingScalingPolicyConfiguration;
            return this;
        }

        public final void setTargetTrackingScalingPolicyConfiguration(
                TargetTrackingScalingPolicyConfiguration.BuilderImpl targetTrackingScalingPolicyConfiguration) {
            this.targetTrackingScalingPolicyConfiguration = targetTrackingScalingPolicyConfiguration != null ? targetTrackingScalingPolicyConfiguration
                    .build() : null;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutScalingPolicyRequest build() {
            return new PutScalingPolicyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
