/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * If specified, API Gateway performs two-way authentication between the client and the server. Clients must present a
 * trusted certificate to access your custom domain name.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MutualTlsAuthentication implements SdkPojo, Serializable,
        ToCopyableBuilder<MutualTlsAuthentication.Builder, MutualTlsAuthentication> {
    private static final SdkField<String> TRUSTSTORE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("truststoreUri").getter(getter(MutualTlsAuthentication::truststoreUri))
            .setter(setter(Builder::truststoreUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("truststoreUri").build()).build();

    private static final SdkField<String> TRUSTSTORE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("truststoreVersion").getter(getter(MutualTlsAuthentication::truststoreVersion))
            .setter(setter(Builder::truststoreVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("truststoreVersion").build()).build();

    private static final SdkField<List<String>> TRUSTSTORE_WARNINGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("truststoreWarnings")
            .getter(getter(MutualTlsAuthentication::truststoreWarnings))
            .setter(setter(Builder::truststoreWarnings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("truststoreWarnings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRUSTSTORE_URI_FIELD,
            TRUSTSTORE_VERSION_FIELD, TRUSTSTORE_WARNINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String truststoreUri;

    private final String truststoreVersion;

    private final List<String> truststoreWarnings;

    private MutualTlsAuthentication(BuilderImpl builder) {
        this.truststoreUri = builder.truststoreUri;
        this.truststoreVersion = builder.truststoreVersion;
        this.truststoreWarnings = builder.truststoreWarnings;
    }

    /**
     * <p>
     * An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example
     * <code>s3://bucket-name/key-name</code>. The truststore can contain certificates from public or private
     * certificate authorities. To update the truststore, upload a new version to S3, and then update your custom domain
     * name to use the new version. To update the truststore, you must have permissions to access the S3 object.
     * </p>
     * 
     * @return An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example
     *         <code>s3://bucket-name/key-name</code>. The truststore can contain certificates from public or private
     *         certificate authorities. To update the truststore, upload a new version to S3, and then update your
     *         custom domain name to use the new version. To update the truststore, you must have permissions to access
     *         the S3 object.
     */
    public final String truststoreUri() {
        return truststoreUri;
    }

    /**
     * <p>
     * The version of the S3 object that contains your truststore. To specify a version, you must have versioning
     * enabled for the S3 bucket.
     * </p>
     * 
     * @return The version of the S3 object that contains your truststore. To specify a version, you must have
     *         versioning enabled for the S3 bucket.
     */
    public final String truststoreVersion() {
        return truststoreVersion;
    }

    /**
     * Returns true if the TruststoreWarnings property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasTruststoreWarnings() {
        return truststoreWarnings != null && !(truststoreWarnings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of warnings that API Gateway returns while processing your truststore. Invalid certificates produce
     * warnings. Mutual TLS is still enabled, but some clients might not be able to access your API. To resolve
     * warnings, upload a new truststore to S3, and then update you domain name to use the new version.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTruststoreWarnings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of warnings that API Gateway returns while processing your truststore. Invalid certificates
     *         produce warnings. Mutual TLS is still enabled, but some clients might not be able to access your API. To
     *         resolve warnings, upload a new truststore to S3, and then update you domain name to use the new version.
     */
    public final List<String> truststoreWarnings() {
        return truststoreWarnings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(truststoreUri());
        hashCode = 31 * hashCode + Objects.hashCode(truststoreVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTruststoreWarnings() ? truststoreWarnings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MutualTlsAuthentication)) {
            return false;
        }
        MutualTlsAuthentication other = (MutualTlsAuthentication) obj;
        return Objects.equals(truststoreUri(), other.truststoreUri())
                && Objects.equals(truststoreVersion(), other.truststoreVersion())
                && hasTruststoreWarnings() == other.hasTruststoreWarnings()
                && Objects.equals(truststoreWarnings(), other.truststoreWarnings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MutualTlsAuthentication").add("TruststoreUri", truststoreUri())
                .add("TruststoreVersion", truststoreVersion())
                .add("TruststoreWarnings", hasTruststoreWarnings() ? truststoreWarnings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "truststoreUri":
            return Optional.ofNullable(clazz.cast(truststoreUri()));
        case "truststoreVersion":
            return Optional.ofNullable(clazz.cast(truststoreVersion()));
        case "truststoreWarnings":
            return Optional.ofNullable(clazz.cast(truststoreWarnings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MutualTlsAuthentication, T> g) {
        return obj -> g.apply((MutualTlsAuthentication) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MutualTlsAuthentication> {
        /**
         * <p>
         * An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example
         * <code>s3://bucket-name/key-name</code>. The truststore can contain certificates from public or private
         * certificate authorities. To update the truststore, upload a new version to S3, and then update your custom
         * domain name to use the new version. To update the truststore, you must have permissions to access the S3
         * object.
         * </p>
         * 
         * @param truststoreUri
         *        An Amazon S3 URL that specifies the truststore for mutual TLS authentication, for example
         *        <code>s3://bucket-name/key-name</code>. The truststore can contain certificates from public or private
         *        certificate authorities. To update the truststore, upload a new version to S3, and then update your
         *        custom domain name to use the new version. To update the truststore, you must have permissions to
         *        access the S3 object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truststoreUri(String truststoreUri);

        /**
         * <p>
         * The version of the S3 object that contains your truststore. To specify a version, you must have versioning
         * enabled for the S3 bucket.
         * </p>
         * 
         * @param truststoreVersion
         *        The version of the S3 object that contains your truststore. To specify a version, you must have
         *        versioning enabled for the S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truststoreVersion(String truststoreVersion);

        /**
         * <p>
         * A list of warnings that API Gateway returns while processing your truststore. Invalid certificates produce
         * warnings. Mutual TLS is still enabled, but some clients might not be able to access your API. To resolve
         * warnings, upload a new truststore to S3, and then update you domain name to use the new version.
         * </p>
         * 
         * @param truststoreWarnings
         *        A list of warnings that API Gateway returns while processing your truststore. Invalid certificates
         *        produce warnings. Mutual TLS is still enabled, but some clients might not be able to access your API.
         *        To resolve warnings, upload a new truststore to S3, and then update you domain name to use the new
         *        version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truststoreWarnings(Collection<String> truststoreWarnings);

        /**
         * <p>
         * A list of warnings that API Gateway returns while processing your truststore. Invalid certificates produce
         * warnings. Mutual TLS is still enabled, but some clients might not be able to access your API. To resolve
         * warnings, upload a new truststore to S3, and then update you domain name to use the new version.
         * </p>
         * 
         * @param truststoreWarnings
         *        A list of warnings that API Gateway returns while processing your truststore. Invalid certificates
         *        produce warnings. Mutual TLS is still enabled, but some clients might not be able to access your API.
         *        To resolve warnings, upload a new truststore to S3, and then update you domain name to use the new
         *        version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truststoreWarnings(String... truststoreWarnings);
    }

    static final class BuilderImpl implements Builder {
        private String truststoreUri;

        private String truststoreVersion;

        private List<String> truststoreWarnings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MutualTlsAuthentication model) {
            truststoreUri(model.truststoreUri);
            truststoreVersion(model.truststoreVersion);
            truststoreWarnings(model.truststoreWarnings);
        }

        public final String getTruststoreUri() {
            return truststoreUri;
        }

        @Override
        public final Builder truststoreUri(String truststoreUri) {
            this.truststoreUri = truststoreUri;
            return this;
        }

        public final void setTruststoreUri(String truststoreUri) {
            this.truststoreUri = truststoreUri;
        }

        public final String getTruststoreVersion() {
            return truststoreVersion;
        }

        @Override
        public final Builder truststoreVersion(String truststoreVersion) {
            this.truststoreVersion = truststoreVersion;
            return this;
        }

        public final void setTruststoreVersion(String truststoreVersion) {
            this.truststoreVersion = truststoreVersion;
        }

        public final Collection<String> getTruststoreWarnings() {
            if (truststoreWarnings instanceof SdkAutoConstructList) {
                return null;
            }
            return truststoreWarnings;
        }

        @Override
        public final Builder truststoreWarnings(Collection<String> truststoreWarnings) {
            this.truststoreWarnings = ListOfStringCopier.copy(truststoreWarnings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder truststoreWarnings(String... truststoreWarnings) {
            truststoreWarnings(Arrays.asList(truststoreWarnings));
            return this;
        }

        public final void setTruststoreWarnings(Collection<String> truststoreWarnings) {
            this.truststoreWarnings = ListOfStringCopier.copy(truststoreWarnings);
        }

        @Override
        public MutualTlsAuthentication build() {
            return new MutualTlsAuthentication(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
