/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.accessanalyzer.model.AccessAnalyzerException;
import software.amazon.awssdk.services.accessanalyzer.model.AccessAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.AccessDeniedException;
import software.amazon.awssdk.services.accessanalyzer.model.ConflictException;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.CreateArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.CreateArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.DeleteArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzedResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzedResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzerRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetAnalyzerResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.GetFindingRequest;
import software.amazon.awssdk.services.accessanalyzer.model.GetFindingResponse;
import software.amazon.awssdk.services.accessanalyzer.model.InternalServerException;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.accessanalyzer.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.accessanalyzer.model.StartResourceScanRequest;
import software.amazon.awssdk.services.accessanalyzer.model.StartResourceScanResponse;
import software.amazon.awssdk.services.accessanalyzer.model.TagResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.TagResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ThrottlingException;
import software.amazon.awssdk.services.accessanalyzer.model.UntagResourceRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UntagResourceResponse;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateArchiveRuleRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateArchiveRuleResponse;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateFindingsRequest;
import software.amazon.awssdk.services.accessanalyzer.model.UpdateFindingsResponse;
import software.amazon.awssdk.services.accessanalyzer.model.ValidationException;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesPublisher;
import software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsPublisher;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.CreateArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.DeleteAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.DeleteArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAnalyzedResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetAnalyzerRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.GetFindingRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAnalyzedResourcesRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListAnalyzersRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListArchiveRulesRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListFindingsRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.StartResourceScanRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UpdateArchiveRuleRequestMarshaller;
import software.amazon.awssdk.services.accessanalyzer.transform.UpdateFindingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AccessAnalyzerAsyncClient}.
 *
 * @see AccessAnalyzerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAccessAnalyzerAsyncClient implements AccessAnalyzerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAccessAnalyzerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAccessAnalyzerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an analyzer for your account.
     * </p>
     *
     * @param createAnalyzerRequest
     *        Creates an analyzer.
     * @return A Java Future containing the result of the CreateAnalyzer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ServiceQuotaExceededException Service quote met error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.CreateAnalyzer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAnalyzerResponse> createAnalyzer(CreateAnalyzerRequest createAnalyzerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAnalyzerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAnalyzerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAnalyzerRequest, CreateAnalyzerResponse>()
                            .withOperationName("CreateAnalyzer")
                            .withMarshaller(new CreateAnalyzerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createAnalyzerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an archive rule for the specified analyzer. Archive rules automatically archive findings that meet the
     * criteria you define when you create the rule.
     * </p>
     *
     * @param createArchiveRuleRequest
     *        Creates an archive rule.
     * @return A Java Future containing the result of the CreateArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ConflictException A conflict exception error.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ServiceQuotaExceededException Service quote met error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.CreateArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateArchiveRuleResponse> createArchiveRule(CreateArchiveRuleRequest createArchiveRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateArchiveRuleRequest, CreateArchiveRuleResponse>()
                            .withOperationName("CreateArchiveRule")
                            .withMarshaller(new CreateArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createArchiveRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified analyzer. When you delete an analyzer, Access Analyzer is disabled for the account in the
     * current or specific Region. All findings that were generated by the analyzer are deleted. You cannot undo this
     * action.
     * </p>
     *
     * @param deleteAnalyzerRequest
     *        Deletes an analyzer.
     * @return A Java Future containing the result of the DeleteAnalyzer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.DeleteAnalyzer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteAnalyzer" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAnalyzerResponse> deleteAnalyzer(DeleteAnalyzerRequest deleteAnalyzerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAnalyzerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAnalyzerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAnalyzerRequest, DeleteAnalyzerResponse>()
                            .withOperationName("DeleteAnalyzer")
                            .withMarshaller(new DeleteAnalyzerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteAnalyzerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified archive rule.
     * </p>
     *
     * @param deleteArchiveRuleRequest
     *        Deletes an archive rule.
     * @return A Java Future containing the result of the DeleteArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.DeleteArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/DeleteArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteArchiveRuleResponse> deleteArchiveRule(DeleteArchiveRuleRequest deleteArchiveRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteArchiveRuleRequest, DeleteArchiveRuleResponse>()
                            .withOperationName("DeleteArchiveRule")
                            .withMarshaller(new DeleteArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteArchiveRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a resource that was analyzed.
     * </p>
     *
     * @param getAnalyzedResourceRequest
     *        Retrieves an analyzed resource.
     * @return A Java Future containing the result of the GetAnalyzedResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetAnalyzedResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAnalyzedResourceResponse> getAnalyzedResource(
            GetAnalyzedResourceRequest getAnalyzedResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAnalyzedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAnalyzedResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAnalyzedResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAnalyzedResourceRequest, GetAnalyzedResourceResponse>()
                            .withOperationName("GetAnalyzedResource")
                            .withMarshaller(new GetAnalyzedResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAnalyzedResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified analyzer.
     * </p>
     *
     * @param getAnalyzerRequest
     *        Retrieves an analyzer.
     * @return A Java Future containing the result of the GetAnalyzer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetAnalyzer
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzer" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAnalyzerResponse> getAnalyzer(GetAnalyzerRequest getAnalyzerRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAnalyzerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAnalyzerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAnalyzerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAnalyzerRequest, GetAnalyzerResponse>()
                            .withOperationName("GetAnalyzer").withMarshaller(new GetAnalyzerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getAnalyzerRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about an archive rule.
     * </p>
     *
     * @param getArchiveRuleRequest
     *        Retrieves an archive rule.
     * @return A Java Future containing the result of the GetArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetArchiveRule" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetArchiveRuleResponse> getArchiveRule(GetArchiveRuleRequest getArchiveRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetArchiveRuleRequest, GetArchiveRuleResponse>()
                            .withOperationName("GetArchiveRule")
                            .withMarshaller(new GetArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getArchiveRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified finding.
     * </p>
     *
     * @param getFindingRequest
     *        Retrieves a finding.
     * @return A Java Future containing the result of the GetFinding operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.GetFinding
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetFinding" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetFindingResponse> getFinding(GetFindingRequest getFindingRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFindingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFindingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFindingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFindingRequest, GetFindingResponse>().withOperationName("GetFinding")
                            .withMarshaller(new GetFindingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getFindingRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     *
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return A Java Future containing the result of the ListAnalyzedResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzedResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnalyzedResourcesResponse> listAnalyzedResources(
            ListAnalyzedResourcesRequest listAnalyzedResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnalyzedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnalyzedResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnalyzedResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnalyzedResourcesRequest, ListAnalyzedResourcesResponse>()
                            .withOperationName("ListAnalyzedResources")
                            .withMarshaller(new ListAnalyzedResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAnalyzedResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of resources of the specified type that have been analyzed by the specified analyzer..
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnalyzedResources(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesPublisher publisher = client.listAnalyzedResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzedResourcesPublisher publisher = client.listAnalyzedResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnalyzedResources(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzedResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnalyzedResourcesRequest
     *        Retrieves a list of resources that have been analyzed.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzedResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzedResources"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnalyzedResourcesPublisher listAnalyzedResourcesPaginator(ListAnalyzedResourcesRequest listAnalyzedResourcesRequest) {
        return new ListAnalyzedResourcesPublisher(this, applyPaginatorUserAgent(listAnalyzedResourcesRequest));
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     *
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return A Java Future containing the result of the ListAnalyzers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnalyzersResponse> listAnalyzers(ListAnalyzersRequest listAnalyzersRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnalyzersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAnalyzersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnalyzersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnalyzersRequest, ListAnalyzersResponse>()
                            .withOperationName("ListAnalyzers")
                            .withMarshaller(new ListAnalyzersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listAnalyzersRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of analyzers.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnalyzers(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersPublisher publisher = client.listAnalyzersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListAnalyzersPublisher publisher = client.listAnalyzersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnalyzers(software.amazon.awssdk.services.accessanalyzer.model.ListAnalyzersRequest)} operation.</b>
     * </p>
     *
     * @param listAnalyzersRequest
     *        Retrieves a list of analyzers.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListAnalyzers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListAnalyzers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListAnalyzersPublisher listAnalyzersPaginator(ListAnalyzersRequest listAnalyzersRequest) {
        return new ListAnalyzersPublisher(this, applyPaginatorUserAgent(listAnalyzersRequest));
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     *
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return A Java Future containing the result of the ListArchiveRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListArchiveRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListArchiveRulesResponse> listArchiveRules(ListArchiveRulesRequest listArchiveRulesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListArchiveRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListArchiveRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListArchiveRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListArchiveRulesRequest, ListArchiveRulesResponse>()
                            .withOperationName("ListArchiveRules")
                            .withMarshaller(new ListArchiveRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listArchiveRulesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of archive rules created for the specified analyzer.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listArchiveRules(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesPublisher publisher = client.listArchiveRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListArchiveRulesPublisher publisher = client.listArchiveRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listArchiveRules(software.amazon.awssdk.services.accessanalyzer.model.ListArchiveRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listArchiveRulesRequest
     *        Retrieves a list of archive rules created for the specified analyzer.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListArchiveRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListArchiveRules"
     *      target="_top">AWS API Documentation</a>
     */
    public ListArchiveRulesPublisher listArchiveRulesPaginator(ListArchiveRulesRequest listArchiveRulesRequest) {
        return new ListArchiveRulesPublisher(this, applyPaginatorUserAgent(listArchiveRulesRequest));
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     *
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return A Java Future containing the result of the ListFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFindingsResponse> listFindings(ListFindingsRequest listFindingsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFindingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFindingsRequest, ListFindingsResponse>()
                            .withOperationName("ListFindings").withMarshaller(new ListFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listFindingsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of findings generated by the specified analyzer.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listFindings(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsPublisher publisher = client.listFindingsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.accessanalyzer.paginators.ListFindingsPublisher publisher = client.listFindingsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listFindings(software.amazon.awssdk.services.accessanalyzer.model.ListFindingsRequest)} operation.</b>
     * </p>
     *
     * @param listFindingsRequest
     *        Retrieves a list of findings generated by the specified analyzer.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindings" target="_top">AWS
     *      API Documentation</a>
     */
    public ListFindingsPublisher listFindingsPaginator(ListFindingsRequest listFindingsRequest) {
        return new ListFindingsPublisher(this, applyPaginatorUserAgent(listFindingsRequest));
    }

    /**
     * <p>
     * Retrieves a list of tags applied to the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Retrieves a list of tags applied to the specified resource.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Immediately starts a scan of the policies applied to the specified resource.
     * </p>
     *
     * @param startResourceScanRequest
     *        Starts a scan of the policies applied to the specified resource.
     * @return A Java Future containing the result of the StartResourceScan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.StartResourceScan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/StartResourceScan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartResourceScanResponse> startResourceScan(StartResourceScanRequest startResourceScanRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartResourceScanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartResourceScanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartResourceScanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartResourceScanRequest, StartResourceScanResponse>()
                            .withOperationName("StartResourceScan")
                            .withMarshaller(new StartResourceScanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startResourceScanRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a tag to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     *        Adds a tag to the specified resource.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     *        Removes a tag from the specified resource.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the criteria and values for the specified archive rule.
     * </p>
     *
     * @param updateArchiveRuleRequest
     *        Updates the specified archive rule.
     * @return A Java Future containing the result of the UpdateArchiveRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.UpdateArchiveRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateArchiveRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateArchiveRuleResponse> updateArchiveRule(UpdateArchiveRuleRequest updateArchiveRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateArchiveRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateArchiveRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateArchiveRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateArchiveRuleRequest, UpdateArchiveRuleResponse>()
                            .withOperationName("UpdateArchiveRule")
                            .withMarshaller(new UpdateArchiveRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateArchiveRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status for the specified findings.
     * </p>
     *
     * @param updateFindingsRequest
     *        Updates findings with the new values provided in the request.
     * @return A Java Future containing the result of the UpdateFindings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource could not be found.</li>
     *         <li>ValidationException Validation exception error.</li>
     *         <li>InternalServerException Internal server error.</li>
     *         <li>ThrottlingException Throttling limit exceeded error.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AccessAnalyzerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AccessAnalyzerAsyncClient.UpdateFindings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateFindings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateFindingsResponse> updateFindings(UpdateFindingsRequest updateFindingsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateFindingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateFindingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateFindingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateFindingsRequest, UpdateFindingsResponse>()
                            .withOperationName("UpdateFindings")
                            .withMarshaller(new UpdateFindingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateFindingsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AccessAnalyzerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private <T extends AccessAnalyzerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
