/**
 * <h1>Amazon Route53 Construct Library</h1>
 * <p>
 * To add a public hosted zone:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("fully.qualified.domain.com")
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add a private hosted zone, use <code>PrivateHostedZone</code>. Note that
 * <code>enableDnsHostnames</code> and <code>enableDnsSupport</code> must have been enabled for the
 * VPC you're configuring for private hosted zones.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * PrivateHostedZone zone = PrivateHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("fully.qualified.domain.com")
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Additional VPCs can be added with <code>zone.addVpc()</code>.
 * <p>
 * <h2>Adding Records</h2>
 * <p>
 * To add a TXT record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * TxtRecord.Builder.create(this, "TXTRecord")
 *         .zone(myZone)
 *         .recordName("_foo") // If the name ends with a ".", it will be used as-is;
 *         // if it ends with a "." followed by the zone name, a trailing "." will be added automatically;
 *         // otherwise, a ".", the zone name, and a trailing "." will be added automatically.
 *         // Defaults to zone root if not specified.
 *         .values(List.of("Bar!", "Baz?"))
 *         .ttl(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add a NS record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * NsRecord.Builder.create(this, "NSRecord")
 *         .zone(myZone)
 *         .recordName("foo")
 *         .values(List.of("ns-1.awsdns.co.uk.", "ns-2.awsdns.com."))
 *         .ttl(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add a DS record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * DsRecord.Builder.create(this, "DSRecord")
 *         .zone(myZone)
 *         .recordName("foo")
 *         .values(List.of("12345 3 1 123456789abcdef67890123456789abcdef67890"))
 *         .ttl(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add an A record to your zone:
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecord")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4", "5.6.7.8"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add an A record for an EC2 instance with an Elastic IP (EIP) to your zone:
 * <p>
 * <blockquote><pre>
 * Instance instance;
 * 
 * HostedZone myZone;
 * 
 * 
 * CfnEIP elasticIp = CfnEIP.Builder.create(this, "EIP")
 *         .domain("vpc")
 *         .instanceId(instance.getInstanceId())
 *         .build();
 * ARecord.Builder.create(this, "ARecord")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses(elasticIp.getRef()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * To add an AAAA record pointing to a CloudFront distribution:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudfront.*;
 * 
 * HostedZone myZone;
 * CloudFrontWebDistribution distribution;
 * 
 * AaaaRecord.Builder.create(this, "Alias")
 *         .zone(myZone)
 *         .target(RecordTarget.fromAlias(new CloudFrontTarget(distribution)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Constructs are available for A, AAAA, CAA, CNAME, MX, NS, SRV and TXT records.
 * <p>
 * Use the <code>CaaAmazonRecord</code> construct to easily restrict certificate authorities
 * allowed to issue certificates for a domain to Amazon only.
 * <p>
 * <h3>Working with existing record sets</h3>
 * <p>
 * Use the <code>deleteExisting</code> prop to delete an existing record set before deploying the new one.
 * This is useful if you want to minimize downtime and avoid "manual" actions while deploying a
 * stack with a record set that already exists. This is typically the case for record sets that
 * are not already "owned" by CloudFormation or "owned" by another stack or construct that is
 * going to be deleted (migration).
 * <p>
 * <blockquote><pre>
 * HostedZone myZone;
 * 
 * 
 * ARecord.Builder.create(this, "ARecord")
 *         .zone(myZone)
 *         .target(RecordTarget.fromIpAddresses("1.2.3.4", "5.6.7.8"))
 *         .deleteExisting(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Cross Account Zone Delegation</h3>
 * <p>
 * To add a NS record to a HostedZone in different account you can do the following:
 * <p>
 * In the account containing the parent hosted zone:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone parentZone = PublicHostedZone.Builder.create(this, "HostedZone")
 *         .zoneName("someexample.com")
 *         .crossAccountZoneDelegationPrincipal(new AccountPrincipal("12345678901"))
 *         .crossAccountZoneDelegationRoleName("MyDelegationRole")
 *         .build();
 * </pre></blockquote>
 * <p>
 * In the account containing the child zone to be delegated:
 * <p>
 * <blockquote><pre>
 * PublicHostedZone subZone = PublicHostedZone.Builder.create(this, "SubZone")
 *         .zoneName("sub.someexample.com")
 *         .build();
 * 
 * // import the delegation role by constructing the roleArn
 * String delegationRoleArn = Stack.of(this).formatArn(ArnComponents.builder()
 *         .region("") // IAM is global in each partition
 *         .service("iam")
 *         .account("parent-account-id")
 *         .resource("role")
 *         .resourceName("MyDelegationRole")
 *         .build());
 * IRole delegationRole = Role.fromRoleArn(this, "DelegationRole", delegationRoleArn);
 * 
 * // create the record
 * // create the record
 * CrossAccountZoneDelegationRecord.Builder.create(this, "delegate")
 *         .delegatedZone(subZone)
 *         .parentHostedZoneName("someexample.com") // or you can use parentHostedZoneId
 *         .delegationRole(delegationRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Imports</h2>
 * <p>
 * If you don't know the ID of the Hosted Zone to import, you can use the
 * <code>HostedZone.fromLookup</code>:
 * <p>
 * <blockquote><pre>
 * HostedZone.fromLookup(this, "MyZone", HostedZoneProviderProps.builder()
 *         .domainName("example.com")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <code>HostedZone.fromLookup</code> requires an environment to be configured. Check
 * out the <a href="https://docs.aws.amazon.com/cdk/latest/guide/environments.html">documentation</a> for more documentation and examples. CDK
 * automatically looks into your <code>~/.aws/config</code> file for the <code>[default]</code> profile.
 * If you want to specify a different account run <code>cdk deploy --profile [profile]</code>.
 * <p>
 * <blockquote><pre>
 * new MyDevStack(app, 'dev', {
 *   env: {
 *     account: process.env.CDK_DEFAULT_ACCOUNT,
 *     region: process.env.CDK_DEFAULT_REGION,
 *   },
 * });
 * </pre></blockquote>
 * <p>
 * If you know the ID and Name of a Hosted Zone, you can import it directly:
 * <p>
 * <blockquote><pre>
 * IHostedZone zone = HostedZone.fromHostedZoneAttributes(this, "MyZone", HostedZoneAttributes.builder()
 *         .zoneName("example.com")
 *         .hostedZoneId("ZOJJZC49E0EPZ")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Alternatively, use the <code>HostedZone.fromHostedZoneId</code> to import hosted zones if
 * you know the ID and the retrieval for the <code>zoneName</code> is undesirable.
 * <p>
 * <blockquote><pre>
 * IHostedZone zone = HostedZone.fromHostedZoneId(this, "MyZone", "ZOJJZC49E0EPZ");
 * </pre></blockquote>
 * <p>
 * You can import a Public Hosted Zone as well with the similar <code>PublicHostedZone.fromPublicHostedZoneId</code> and <code>PublicHostedZone.fromPublicHostedZoneAttributes</code> methods:
 * <p>
 * <blockquote><pre>
 * IPublicHostedZone zoneFromAttributes = PublicHostedZone.fromPublicHostedZoneAttributes(this, "MyZone", PublicHostedZoneAttributes.builder()
 *         .zoneName("example.com")
 *         .hostedZoneId("ZOJJZC49E0EPZ")
 *         .build());
 * 
 * // Does not know zoneName
 * IPublicHostedZone zoneFromId = PublicHostedZone.fromPublicHostedZoneId(this, "MyZone", "ZOJJZC49E0EPZ");
 * </pre></blockquote>
 * <p>
 * <h2>VPC Endpoint Service Private DNS</h2>
 * <p>
 * When you create a VPC endpoint service, AWS generates endpoint-specific DNS hostnames that consumers use to communicate with the service.
 * For example, vpce-1234-abcdev-us-east-1.vpce-svc-123345.us-east-1.vpce.amazonaws.com.
 * By default, your consumers access the service with that DNS name.
 * This can cause problems with HTTPS traffic because the DNS will not match the backend certificate:
 * <p>
 * <blockquote><pre>
 * curl: (60) SSL: no alternative certificate subject name matches target host name 'vpce-abcdefghijklmnopq-rstuvwx.vpce-svc-abcdefghijklmnopq.us-east-1.vpce.amazonaws.com'
 * </pre></blockquote>
 * <p>
 * Effectively, the endpoint appears untrustworthy. To mitigate this, clients have to create an alias for this DNS name in Route53.
 * <p>
 * Private DNS for an endpoint service lets you configure a private DNS name so consumers can
 * access the service using an existing DNS name without creating this Route53 DNS alias
 * This DNS name can also be guaranteed to match up with the backend certificate.
 * <p>
 * Before consumers can use the private DNS name, you must verify that you have control of the domain/subdomain.
 * <p>
 * Assuming your account has ownership of the particular domain/subdomain,
 * this construct sets up the private DNS configuration on the endpoint service,
 * creates all the necessary Route53 entries, and verifies domain ownership.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Stack;
 * import software.amazon.awscdk.services.ec2.Vpc;
 * import software.amazon.awscdk.services.ec2.VpcEndpointService;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.NetworkLoadBalancer;
 * import software.amazon.awscdk.services.route53.PublicHostedZone;
 * import software.amazon.awscdk.services.route53.VpcEndpointServiceDomainName;
 * 
 * Stack stack = new Stack();
 * Vpc vpc = new Vpc(stack, "VPC");
 * NetworkLoadBalancer nlb = NetworkLoadBalancer.Builder.create(stack, "NLB")
 *         .vpc(vpc)
 *         .build();
 * VpcEndpointService vpces = VpcEndpointService.Builder.create(stack, "VPCES")
 *         .vpcEndpointServiceLoadBalancers(List.of(nlb))
 *         .build();
 * // You must use a public hosted zone so domain ownership can be verified
 * PublicHostedZone zone = PublicHostedZone.Builder.create(stack, "PHZ")
 *         .zoneName("aws-cdk.dev")
 *         .build();
 * VpcEndpointServiceDomainName.Builder.create(stack, "EndpointDomain")
 *         .endpointService(vpces)
 *         .domainName("my-stuff.aws-cdk.dev")
 *         .publicHostedZone(zone)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.route53;
