/**
 * <h1>AWS Step Functions Construct Library</h1>
 * <p>
 * The <code>&#64;aws-cdk/aws-stepfunctions</code> package contains constructs for building
 * serverless workflows using objects. Use this in conjunction with the
 * <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> package, which contains classes used
 * to call other AWS services.
 * <p>
 * Defining a workflow looks like this (for the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/job-status-poller-sample.html">Step Functions Job Poller
 * example</a>):
 * <p>
 * <h2>Example</h2>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function submitLambda;
 * Function getStatusLambda;
 * 
 * 
 * LambdaInvoke submitJob = LambdaInvoke.Builder.create(this, "Submit Job")
 *         .lambdaFunction(submitLambda)
 *         // Lambda's result is in the attribute `Payload`
 *         .outputPath("$.Payload")
 *         .build();
 * 
 * Wait waitX = Wait.Builder.create(this, "Wait X Seconds")
 *         .time(WaitTime.secondsPath("$.waitSeconds"))
 *         .build();
 * 
 * LambdaInvoke getStatus = LambdaInvoke.Builder.create(this, "Get Job Status")
 *         .lambdaFunction(getStatusLambda)
 *         // Pass just the field named "guid" into the Lambda, put the
 *         // Lambda's result in a field called "status" in the response
 *         .inputPath("$.guid")
 *         .outputPath("$.Payload")
 *         .build();
 * 
 * Fail jobFailed = Fail.Builder.create(this, "Job Failed")
 *         .cause("AWS Batch Job Failed")
 *         .error("DescribeJob returned FAILED")
 *         .build();
 * 
 * LambdaInvoke finalStatus = LambdaInvoke.Builder.create(this, "Get Final Job Status")
 *         .lambdaFunction(getStatusLambda)
 *         // Use "guid" field as input
 *         .inputPath("$.guid")
 *         .outputPath("$.Payload")
 *         .build();
 * 
 * Chain definition = submitJob.next(waitX).next(getStatus).next(new Choice(this, "Job Complete?").when(Condition.stringEquals("$.status", "FAILED"), jobFailed).when(Condition.stringEquals("$.status", "SUCCEEDED"), finalStatus).otherwise(waitX));
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .timeout(Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can find more sample snippets and learn more about the service integrations
 * in the <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> package.
 * <p>
 * <h2>State Machine</h2>
 * <p>
 * A <code>stepfunctions.StateMachine</code> is a resource that takes a state machine
 * definition. The definition is specified by its start state, and encompasses
 * all states reachable from the start state:
 * <p>
 * <blockquote><pre>
 * Pass startState = new Pass(this, "StartState");
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(startState)
 *         .build();
 * </pre></blockquote>
 * <p>
 * State machines are made up of a sequence of <strong>Steps</strong>, which represent different actions
 * taken in sequence. Some of these steps represent <em>control flow</em> (like <code>Choice</code>, <code>Map</code> and <code>Wait</code>)
 * while others represent calls made against other AWS services (like <code>LambdaInvoke</code>).
 * The second category are called <code>Task</code>s and they can all be found in the module <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks-readme.html"><code>aws-stepfunctions-tasks</code></a>.
 * <p>
 * State machines execute using an IAM Role, which will automatically have all
 * permissions added that are required to make all state machine tasks execute
 * properly (for example, permissions to invoke any Lambda functions you add to
 * your workflow). A role will be created by default, but you can supply an
 * existing one as well.
 * <p>
 * <h2>State Machine Data</h2>
 * <p>
 * An Execution represents each time the State Machine is run. Every Execution has <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html">State Machine
 * Data</a>:
 * a JSON document containing keys and values that is fed into the state machine,
 * gets modified by individual steps as the state machine progresses, and finally
 * is produced as output.
 * <p>
 * By default, the entire Data object is passed into every state, and the return data of the step
 * becomes new the new Data object. This behavior can be modified by supplying values for <code>inputPath</code>,
 * <code>resultSelector</code>, <code>resultPath</code> and <code>outputPath</code>.
 * <p>
 * <h3>Manipulating state machine data using inputPath, resultSelector, resultPath and outputPath</h3>
 * <p>
 * These properties impact how each individual step interacts with the state machine data:
 * <p>
 * <ul>
 * <li><code>inputPath</code>: the part of the data object that gets passed to the step (<code>itemsPath</code> for <code>Map</code> states)</li>
 * <li><code>resultSelector</code>: the part of the step result that should be added to the state machine data</li>
 * <li><code>resultPath</code>: where in the state machine data the step result should be inserted</li>
 * <li><code>outputPath</code>: what part of the state machine data should be retained</li>
 * </ul>
 * <p>
 * Their values should be a string indicating a <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html">JSON path</a> into the State Machine Data object (like <code>"$.MyKey"</code>). If absent, the values are treated as if they were <code>"$"</code>, which means the entire object.
 * <p>
 * The following pseudocode shows how AWS Step Functions uses these parameters when executing a step:
 * <p>
 * <blockquote><pre>
 * // Schematically show how Step Functions evaluates functions.
 * // [] represents indexing into an object by a using JSON path.
 * 
 * input = state[inputPath]
 * 
 * result = invoke_step(select_parameters(input))
 * 
 * state[resultPath] = result[resultSelector]
 * 
 * state = state[outputPath]
 * </pre></blockquote>
 * <p>
 * Instead of a JSON path string, each of these paths can also have the special value <code>JsonPath.DISCARD</code>, which causes the corresponding indexing expression to return an empty object (<code>{}</code>). Effectively, that means there will be an empty input object, an empty result object, no effect on the state, or an empty state, respectively.
 * <p>
 * Some steps (mostly Tasks) have <em>Parameters</em>, which are selected differently. See the next section.
 * <p>
 * See the official documentation on <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html">input and output processing in Step Functions</a>.
 * <p>
 * <h3>Passing Parameters to Tasks</h3>
 * <p>
 * Tasks take parameters, whose values can be taken from the State Machine Data object. For example, your
 * workflow may want to start a CodeBuild with an environment variable that is taken from the State Machine data, or pass part of the State Machine Data into an AWS Lambda Function.
 * <p>
 * In the original JSON-based states language used by AWS Step Functions, you would
 * add <code>.$</code> to the end of a key to indicate that a value needs to be interpreted as
 * a JSON path. In the CDK API you do not change the names of any keys. Instead, you
 * pass special values. There are 3 types of task inputs to consider:
 * <p>
 * <ul>
 * <li>Tasks that accept a "payload" type of input (like AWS Lambda invocations, or posting messages to SNS topics or SQS queues), will take an object of type <code>TaskInput</code>, like <code>TaskInput.fromObject()</code> or <code>TaskInput.fromJsonPathAt()</code>.</li>
 * <li>When tasks expect individual string or number values to customize their behavior, you can also pass a value constructed by <code>JsonPath.stringAt()</code> or <code>JsonPath.numberAt()</code>.</li>
 * <li>When tasks expect strongly-typed resources and you want to vary the resource that is referenced based on a name from the State Machine Data, reference the resource as if it was external (using <code>JsonPath.stringAt()</code>). For example, for a Lambda function: <code>Function.fromFunctionName(this, 'ReferencedFunction', JsonPath.stringAt('$.MyFunctionName'))</code>.</li>
 * </ul>
 * <p>
 * For example, to pass the value that's in the data key of <code>OrderId</code> to a Lambda
 * function as you invoke it, use <code>JsonPath.stringAt('$.OrderId')</code>, like so:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function orderFn;
 * 
 * 
 * LambdaInvoke submitJob = LambdaInvoke.Builder.create(this, "InvokeOrderProcessor")
 *         .lambdaFunction(orderFn)
 *         .payload(TaskInput.fromObject(Map.of(
 *                 "OrderId", JsonPath.stringAt("$.OrderId"))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following methods are available:
 * <p>
 * | Method | Purpose |
 * |--------|---------|
 * | <code>JsonPath.stringAt('$.Field')</code> | reference a field, return the type as a <code>string</code>. |
 * | <code>JsonPath.listAt('$.Field')</code> | reference a field, return the type as a list of strings. |
 * | <code>JsonPath.numberAt('$.Field')</code> | reference a field, return the type as a number. Use this for functions that expect a number argument. |
 * | <code>JsonPath.objectAt('$.Field')</code> | reference a field, return the type as an <code>IResolvable</code>. Use this for functions that expect an object argument. |
 * | <code>JsonPath.entirePayload</code> | reference the entire data object (equivalent to a path of <code>$</code>). |
 * | <code>JsonPath.taskToken</code> | reference the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token">Task Token</a>, used for integration patterns that need to run for a long time. |
 * <p>
 * You can also call <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html">intrinsic functions</a> using the methods on <code>JsonPath</code>:
 * <p>
 * | Method | Purpose |
 * |--------|---------|
 * | <code>JsonPath.array(JsonPath.stringAt('$.Field'), ...)</code> | make an array from other elements. |
 * | <code>JsonPath.format('The value is {}.', JsonPath.stringAt('$.Value'))</code> | insert elements into a format string. |
 * | <code>JsonPath.stringToJson(JsonPath.stringAt('$.ObjStr'))</code> | parse a JSON string to an object |
 * | <code>JsonPath.jsonToString(JsonPath.objectAt('$.Obj'))</code> | stringify an object to a JSON string |
 * <p>
 * <h2>Amazon States Language</h2>
 * <p>
 * This library comes with a set of classes that model the <a href="https://states-language.net/spec.html">Amazon States
 * Language</a>. The following State classes
 * are supported:
 * <p>
 * <ul>
 * <li><a href="#task"><code>Task</code></a></li>
 * <li><a href="#pass"><code>Pass</code></a></li>
 * <li><a href="#wait"><code>Wait</code></a></li>
 * <li><a href="#choice"><code>Choice</code></a></li>
 * <li><a href="#parallel"><code>Parallel</code></a></li>
 * <li><a href="#succeed"><code>Succeed</code></a></li>
 * <li><a href="#fail"><code>Fail</code></a></li>
 * <li><a href="#map"><code>Map</code></a></li>
 * <li><a href="#custom-state"><code>Custom State</code></a></li>
 * </ul>
 * <p>
 * An arbitrary JSON object (specified at execution start) is passed from state to
 * state and transformed during the execution of the workflow. For more
 * information, see the States Language spec.
 * <p>
 * <h3>Task</h3>
 * <p>
 * A <code>Task</code> represents some work that needs to be done. Do not use the <code>Task</code> class directly.
 * <p>
 * Instead, use one of the classes in the <code>&#64;aws-cdk/aws-stepfunctions-tasks</code> module,
 * which provide a much more ergonomic way to integrate with various AWS services.
 * <p>
 * <h3>Pass</h3>
 * <p>
 * A <code>Pass</code> state passes its input to its output, without performing work.
 * Pass states are useful when constructing and debugging state machines.
 * <p>
 * The following example injects some fixed data into the state machine through
 * the <code>result</code> field. The <code>result</code> field will be added to the input and the result
 * will be passed as the state's output.
 * <p>
 * <blockquote><pre>
 * // Makes the current JSON state { ..., "subObject": { "hello": "world" } }
 * Pass pass = Pass.Builder.create(this, "Add Hello World")
 *         .result(Result.fromObject(Map.of("hello", "world")))
 *         .resultPath("$.subObject")
 *         .build();
 * 
 * // Set the next state
 * Pass nextState = new Pass(this, "NextState");
 * pass.next(nextState);
 * </pre></blockquote>
 * <p>
 * The <code>Pass</code> state also supports passing key-value pairs as input. Values can
 * be static, or selected from the input with a path.
 * <p>
 * The following example filters the <code>greeting</code> field from the state input
 * and also injects a field called <code>otherData</code>.
 * <p>
 * <blockquote><pre>
 * Pass pass = Pass.Builder.create(this, "Filter input and inject data")
 *         .parameters(Map.of( // input to the pass state
 *                 "input", JsonPath.stringAt("$.input.greeting"),
 *                 "otherData", "some-extra-stuff"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The object specified in <code>parameters</code> will be the input of the <code>Pass</code> state.
 * Since neither <code>Result</code> nor <code>ResultPath</code> are supplied, the <code>Pass</code> state copies
 * its input through to its output.
 * <p>
 * Learn more about the <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-pass-state.html">Pass state</a>
 * <p>
 * <h3>Wait</h3>
 * <p>
 * A <code>Wait</code> state waits for a given number of seconds, or until the current time
 * hits a particular time. The time to wait may be taken from the execution's JSON
 * state.
 * <p>
 * <blockquote><pre>
 * // Wait until it's the time mentioned in the the state object's "triggerTime"
 * // field.
 * Wait wait = Wait.Builder.create(this, "Wait For Trigger Time")
 *         .time(WaitTime.timestampPath("$.triggerTime"))
 *         .build();
 * 
 * // Set the next state
 * Pass startTheWork = new Pass(this, "StartTheWork");
 * wait.next(startTheWork);
 * </pre></blockquote>
 * <p>
 * <h3>Choice</h3>
 * <p>
 * A <code>Choice</code> state can take a different path through the workflow based on the
 * values in the execution's JSON state:
 * <p>
 * <blockquote><pre>
 * Choice choice = new Choice(this, "Did it work?");
 * 
 * // Add conditions with .when()
 * Pass successState = new Pass(this, "SuccessState");
 * Pass failureState = new Pass(this, "FailureState");
 * choice.when(Condition.stringEquals("$.status", "SUCCESS"), successState);
 * choice.when(Condition.numberGreaterThan("$.attempts", 5), failureState);
 * 
 * // Use .otherwise() to indicate what should be done if none of the conditions match
 * Pass tryAgainState = new Pass(this, "TryAgainState");
 * choice.otherwise(tryAgainState);
 * </pre></blockquote>
 * <p>
 * If you want to temporarily branch your workflow based on a condition, but have
 * all branches come together and continuing as one (similar to how an <code>if ... then ... else</code> works in a programming language), use the <code>.afterwards()</code> method:
 * <p>
 * <blockquote><pre>
 * Choice choice = new Choice(this, "What color is it?");
 * Pass handleBlueItem = new Pass(this, "HandleBlueItem");
 * Pass handleRedItem = new Pass(this, "HandleRedItem");
 * Pass handleOtherItemColor = new Pass(this, "HanldeOtherItemColor");
 * choice.when(Condition.stringEquals("$.color", "BLUE"), handleBlueItem);
 * choice.when(Condition.stringEquals("$.color", "RED"), handleRedItem);
 * choice.otherwise(handleOtherItemColor);
 * 
 * // Use .afterwards() to join all possible paths back together and continue
 * Pass shipTheItem = new Pass(this, "ShipTheItem");
 * choice.afterwards().next(shipTheItem);
 * </pre></blockquote>
 * <p>
 * If your <code>Choice</code> doesn't have an <code>otherwise()</code> and none of the conditions match
 * the JSON state, a <code>NoChoiceMatched</code> error will be thrown. Wrap the state machine
 * in a <code>Parallel</code> state if you want to catch and recover from this.
 * <p>
 * <h4>Available Conditions</h4>
 * <p>
 * see <a href="https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-choice-state.html#amazon-states-language-choice-state-rules">step function comparison operators</a>
 * <p>
 * <ul>
 * <li><code>Condition.isPresent</code> - matches if a json path is present</li>
 * <li><code>Condition.isNotPresent</code> - matches if a json path is not present</li>
 * <li><code>Condition.isString</code> - matches if a json path contains a string</li>
 * <li><code>Condition.isNotString</code> - matches if a json path is not a string</li>
 * <li><code>Condition.isNumeric</code> - matches if a json path is numeric</li>
 * <li><code>Condition.isNotNumeric</code> - matches if a json path is not numeric</li>
 * <li><code>Condition.isBoolean</code> - matches if a json path is boolean</li>
 * <li><code>Condition.isNotBoolean</code> - matches if a json path is not boolean</li>
 * <li><code>Condition.isTimestamp</code> - matches if a json path is a timestamp</li>
 * <li><code>Condition.isNotTimestamp</code> - matches if a json path is not a timestamp</li>
 * <li><code>Condition.isNotNull</code> - matches if a json path is not null</li>
 * <li><code>Condition.isNull</code> - matches if a json path is null</li>
 * <li><code>Condition.booleanEquals</code> - matches if a boolean field has a given value</li>
 * <li><code>Condition.booleanEqualsJsonPath</code> - matches if a boolean field equals a value in a given mapping path</li>
 * <li><code>Condition.stringEqualsJsonPath</code> - matches if a string field equals a given mapping path</li>
 * <li><code>Condition.stringEquals</code> - matches if a field equals a string value</li>
 * <li><code>Condition.stringLessThan</code> - matches if a string field sorts before a given value</li>
 * <li><code>Condition.stringLessThanJsonPath</code> - matches if a string field sorts before a value at given mapping path</li>
 * <li><code>Condition.stringLessThanEquals</code> - matches if a string field sorts equal to or before a given value</li>
 * <li><code>Condition.stringLessThanEqualsJsonPath</code> - matches if a string field sorts equal to or before a given mapping</li>
 * <li><code>Condition.stringGreaterThan</code> - matches if a string field sorts after a given value</li>
 * <li><code>Condition.stringGreaterThanJsonPath</code> - matches if a string field sorts after a value at a given mapping path</li>
 * <li><code>Condition.stringGreaterThanEqualsJsonPath</code> - matches if a string field sorts after or equal to value at a given mapping path</li>
 * <li><code>Condition.stringGreaterThanEquals</code> - matches if a string field sorts after or equal to a given value</li>
 * <li><code>Condition.numberEquals</code> - matches if a numeric field has the given value</li>
 * <li><code>Condition.numberEqualsJsonPath</code> - matches if a numeric field has the value in a given mapping path</li>
 * <li><code>Condition.numberLessThan</code> - matches if a numeric field is less than the given value</li>
 * <li><code>Condition.numberLessThanJsonPath</code> - matches if a numeric field is less than the value at the given mapping path</li>
 * <li><code>Condition.numberLessThanEquals</code> - matches if a numeric field is less than or equal to the given value</li>
 * <li><code>Condition.numberLessThanEqualsJsonPath</code> - matches if a numeric field is less than or equal to the numeric value at given mapping path</li>
 * <li><code>Condition.numberGreaterThan</code> - matches if a numeric field is greater than the given value</li>
 * <li><code>Condition.numberGreaterThanJsonPath</code> - matches if a numeric field is greater than the value at a given mapping path</li>
 * <li><code>Condition.numberGreaterThanEquals</code> - matches if a numeric field is greater than or equal to the given value</li>
 * <li><code>Condition.numberGreaterThanEqualsJsonPath</code> - matches if a numeric field is greater than or equal to the value at a given mapping path</li>
 * <li><code>Condition.timestampEquals</code> - matches if a timestamp field is the same time as the given timestamp</li>
 * <li><code>Condition.timestampEqualsJsonPath</code> - matches if a timestamp field is the same time as the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampLessThan</code> - matches if a timestamp field is before the given timestamp</li>
 * <li><code>Condition.timestampLessThanJsonPath</code> - matches if a timestamp field is before the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampLessThanEquals</code> - matches if a timestamp field is before or equal to the given timestamp</li>
 * <li><code>Condition.timestampLessThanEqualsJsonPath</code> - matches if a timestamp field is before or equal to the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThan</code> - matches if a timestamp field is after the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThanJsonPath</code> - matches if a timestamp field is after the timestamp at a given mapping path</li>
 * <li><code>Condition.timestampGreaterThanEquals</code> - matches if a timestamp field is after or equal to the given timestamp</li>
 * <li><code>Condition.timestampGreaterThanEqualsJsonPath</code> - matches if a timestamp field is after or equal to the timestamp at a given mapping path</li>
 * <li><code>Condition.stringMatches</code> - matches if a field matches a string pattern that can contain a wild card (<em>) e.g: log-</em>.txt or <em>LATEST</em>. No other characters other than "<em>" have any special meaning - * can be escaped: \</em></li>
 * </ul>
 * <p>
 * <h3>Parallel</h3>
 * <p>
 * A <code>Parallel</code> state executes one or more subworkflows in parallel. It can also
 * be used to catch and recover from errors in subworkflows.
 * <p>
 * <blockquote><pre>
 * Parallel parallel = new Parallel(this, "Do the work in parallel");
 * 
 * // Add branches to be executed in parallel
 * Pass shipItem = new Pass(this, "ShipItem");
 * Pass sendInvoice = new Pass(this, "SendInvoice");
 * Pass restock = new Pass(this, "Restock");
 * parallel.branch(shipItem);
 * parallel.branch(sendInvoice);
 * parallel.branch(restock);
 * 
 * // Retry the whole workflow if something goes wrong
 * parallel.addRetry(RetryProps.builder().maxAttempts(1).build());
 * 
 * // How to recover from errors
 * Pass sendFailureNotification = new Pass(this, "SendFailureNotification");
 * parallel.addCatch(sendFailureNotification);
 * 
 * // What to do in case everything succeeded
 * Pass closeOrder = new Pass(this, "CloseOrder");
 * parallel.next(closeOrder);
 * </pre></blockquote>
 * <p>
 * <h3>Succeed</h3>
 * <p>
 * Reaching a <code>Succeed</code> state terminates the state machine execution with a
 * successful status.
 * <p>
 * <blockquote><pre>
 * Succeed success = new Succeed(this, "We did it!");
 * </pre></blockquote>
 * <p>
 * <h3>Fail</h3>
 * <p>
 * Reaching a <code>Fail</code> state terminates the state machine execution with a
 * failure status. The fail state should report the reason for the failure.
 * Failures can be caught by encompassing <code>Parallel</code> states.
 * <p>
 * <blockquote><pre>
 * Fail success = Fail.Builder.create(this, "Fail")
 *         .error("WorkflowFailure")
 *         .cause("Something went wrong")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Map</h3>
 * <p>
 * A <code>Map</code> state can be used to run a set of steps for each element of an input array.
 * A <code>Map</code> state will execute the same steps for multiple entries of an array in the state input.
 * <p>
 * While the <code>Parallel</code> state executes multiple branches of steps using the same input, a <code>Map</code> state will
 * execute the same steps for multiple entries of an array in the state input.
 * <p>
 * <blockquote><pre>
 * Map map = Map.Builder.create(this, "Map State")
 *         .maxConcurrency(1)
 *         .itemsPath(JsonPath.stringAt("$.inputForMap"))
 *         .build();
 * map.iterator(new Pass(this, "Pass State"));
 * </pre></blockquote>
 * <p>
 * <h3>Custom State</h3>
 * <p>
 * It's possible that the high-level constructs for the states or <code>stepfunctions-tasks</code> do not have
 * the states or service integrations you are looking for. The primary reasons for this lack of
 * functionality are:
 * <p>
 * <ul>
 * <li>A <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-service-integrations.html">service integration</a> is available through Amazon States Langauge, but not available as construct
 * classes in the CDK.</li>
 * <li>The state or state properties are available through Step Functions, but are not configurable
 * through constructs</li>
 * </ul>
 * <p>
 * If a feature is not available, a <code>CustomState</code> can be used to supply any Amazon States Language
 * JSON-based object as the state definition.
 * <p>
 * <a href="https://docs.aws.amazon.com/step-functions/latest/dg/tutorial-code-snippet.html#tutorial-code-snippet-1">Code Snippets</a> are available and can be plugged in as the state definition.
 * <p>
 * Custom states can be chained together with any of the other states to create your state machine
 * definition. You will also need to provide any permissions that are required to the <code>role</code> that
 * the State Machine uses.
 * <p>
 * The following example uses the <code>DynamoDB</code> service integration to insert data into a DynamoDB table.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.dynamodb.*;
 * 
 * 
 * // create a table
 * Table table = Table.Builder.create(this, "montable")
 *         .partitionKey(Attribute.builder()
 *                 .name("id")
 *                 .type(AttributeType.STRING)
 *                 .build())
 *         .build();
 * 
 * Pass finalStatus = new Pass(this, "final step");
 * 
 * // States language JSON to put an item into DynamoDB
 * // snippet generated from https://docs.aws.amazon.com/step-functions/latest/dg/tutorial-code-snippet.html#tutorial-code-snippet-1
 * Map&lt;String, Object&gt; stateJson = Map.of(
 *         "Type", "Task",
 *         "Resource", "arn:aws:states:::dynamodb:putItem",
 *         "Parameters", Map.of(
 *                 "TableName", table.getTableName(),
 *                 "Item", Map.of(
 *                         "id", Map.of(
 *                                 "S", "MyEntry"))),
 *         "ResultPath", null);
 * 
 * // custom state which represents a task to insert data into DynamoDB
 * CustomState custom = CustomState.Builder.create(this, "my custom task")
 *         .stateJson(stateJson)
 *         .build();
 * 
 * Chain chain = Chain.start(custom).next(finalStatus);
 * 
 * StateMachine sm = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(chain)
 *         .timeout(Duration.seconds(30))
 *         .build();
 * 
 * // don't forget permissions. You need to assign them
 * table.grantWriteData(sm);
 * </pre></blockquote>
 * <p>
 * <h2>Task Chaining</h2>
 * <p>
 * To make defining work flows as convenient (and readable in a top-to-bottom way)
 * as writing regular programs, it is possible to chain most methods invocations.
 * In particular, the <code>.next()</code> method can be repeated. The result of a series of
 * <code>.next()</code> calls is called a <strong>Chain</strong>, and can be used when defining the jump
 * targets of <code>Choice.on</code> or <code>Parallel.branch</code>:
 * <p>
 * <blockquote><pre>
 * Pass step1 = new Pass(this, "Step1");
 * Pass step2 = new Pass(this, "Step2");
 * Pass step3 = new Pass(this, "Step3");
 * Pass step4 = new Pass(this, "Step4");
 * Pass step5 = new Pass(this, "Step5");
 * Pass step6 = new Pass(this, "Step6");
 * Pass step7 = new Pass(this, "Step7");
 * Pass step8 = new Pass(this, "Step8");
 * Pass step9 = new Pass(this, "Step9");
 * Pass step10 = new Pass(this, "Step10");
 * Choice choice = new Choice(this, "Choice");
 * Condition condition1 = Condition.stringEquals("$.status", "SUCCESS");
 * Parallel parallel = new Parallel(this, "Parallel");
 * Pass finish = new Pass(this, "Finish");
 * 
 * Chain definition = step1.next(step2).next(choice.when(condition1, step3.next(step4).next(step5)).otherwise(step6).afterwards()).next(parallel.branch(step7.next(step8)).branch(step9.next(step10))).next(finish);
 * 
 * StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you don't like the visual look of starting a chain directly off the first
 * step, you can use <code>Chain.start</code>:
 * <p>
 * <blockquote><pre>
 * Pass step1 = new Pass(this, "Step1");
 * Pass step2 = new Pass(this, "Step2");
 * Pass step3 = new Pass(this, "Step3");
 * 
 * Chain definition = Chain.start(step1).next(step2).next(step3);
 * </pre></blockquote>
 * <p>
 * <h2>State Machine Fragments</h2>
 * <p>
 * It is possible to define reusable (or abstracted) mini-state machines by
 * defining a construct that implements <code>IChainable</code>, which requires you to define
 * two fields:
 * <p>
 * <ul>
 * <li><code>startState: State</code>, representing the entry point into this state machine.</li>
 * <li><code>endStates: INextable[]</code>, representing the (one or more) states that outgoing
 * transitions will be added to if you chain onto the fragment.</li>
 * </ul>
 * <p>
 * Since states will be named after their construct IDs, you may need to prefix the
 * IDs of states if you plan to instantiate the same state machine fragment
 * multiples times (otherwise all states in every instantiation would have the same
 * name).
 * <p>
 * The class <code>StateMachineFragment</code> contains some helper functions (like
 * <code>prefixStates()</code>) to make it easier for you to do this. If you define your state
 * machine as a subclass of this, it will be convenient to use:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Stack;
 * import software.constructs.Construct;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * public class MyJobProps {
 *     private String jobFlavor;
 *     public String getJobFlavor() {
 *         return this.jobFlavor;
 *     }
 *     public MyJobProps jobFlavor(String jobFlavor) {
 *         this.jobFlavor = jobFlavor;
 *         return this;
 *     }
 * }
 * 
 * public class MyJob extends StateMachineFragment {
 *     public final State startState;
 *     public final INextable[] endStates;
 * 
 *     public MyJob(Construct parent, String id, MyJobProps props) {
 *         super(parent, id);
 * 
 *         Choice choice = new Choice(this, "Choice").when(Condition.stringEquals("$.branch", "left"), new Pass(this, "Left Branch")).when(Condition.stringEquals("$.branch", "right"), new Pass(this, "Right Branch"));
 * 
 *         // ...
 * 
 *         this.startState = choice;
 *         this.endStates = choice.afterwards().getEndStates();
 *     }
 * }
 * 
 * public class MyStack extends Stack {
 *     public MyStack(Construct scope, String id) {
 *         super(scope, id);
 *         // Do 3 different variants of MyJob in parallel
 *         Parallel parallel = new Parallel(this, "All jobs").branch(new MyJob(this, "Quick", new MyJobProps().jobFlavor("quick")).prefixStates()).branch(new MyJob(this, "Medium", new MyJobProps().jobFlavor("medium")).prefixStates()).branch(new MyJob(this, "Slow", new MyJobProps().jobFlavor("slow")).prefixStates());
 * 
 *         StateMachine.Builder.create(this, "MyStateMachine")
 *                 .definition(parallel)
 *                 .build();
 *     }
 * }
 * </pre></blockquote>
 * <p>
 * A few utility functions are available to parse state machine fragments.
 * <p>
 * <ul>
 * <li><code>State.findReachableStates</code>: Retrieve the list of states reachable from a given state.</li>
 * <li><code>State.findReachableEndStates</code>: Retrieve the list of end or terminal states reachable from a given state.</li>
 * </ul>
 * <p>
 * <h2>Activity</h2>
 * <p>
 * <strong>Activities</strong> represent work that is done on some non-Lambda worker pool. The
 * Step Functions workflow will submit work to this Activity, and a worker pool
 * that you run yourself, probably on EC2, will pull jobs from the Activity and
 * submit the results of individual jobs back.
 * <p>
 * You need the ARN to do so, so if you use Activities be sure to pass the Activity
 * ARN into your worker pool:
 * <p>
 * <blockquote><pre>
 * Activity activity = new Activity(this, "Activity");
 * 
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * // Read this CloudFormation Output from your application and use it to poll for work on
 * // the activity.
 * CfnOutput.Builder.create(this, "ActivityArn").value(activity.getActivityArn()).build();
 * </pre></blockquote>
 * <p>
 * <h3>Activity-Level Permissions</h3>
 * <p>
 * Granting IAM permissions to an activity can be achieved by calling the <code>grant(principal, actions)</code> API:
 * <p>
 * <blockquote><pre>
 * Activity activity = new Activity(this, "Activity");
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * 
 * activity.grant(role, "states:SendTaskSuccess");
 * </pre></blockquote>
 * <p>
 * This will grant the IAM principal the specified actions onto the activity.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * <code>Task</code> object expose various metrics on the execution of that particular task. For example,
 * to create an alarm on a particular task failing:
 * <p>
 * <blockquote><pre>
 * Task task;
 * 
 * Alarm.Builder.create(this, "TaskAlarm")
 *         .metric(task.metricFailed())
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * There are also metrics on the complete state machine:
 * <p>
 * <blockquote><pre>
 * StateMachine stateMachine;
 * 
 * Alarm.Builder.create(this, "StateMachineAlarm")
 *         .metric(stateMachine.metricFailed())
 *         .threshold(1)
 *         .evaluationPeriods(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * And there are metrics on the capacity of all state machines in your account:
 * <p>
 * <blockquote><pre>
 * Alarm.Builder.create(this, "ThrottledAlarm")
 *         .metric(StateTransitionMetric.metricThrottledEvents())
 *         .threshold(10)
 *         .evaluationPeriods(2)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Error names</h2>
 * <p>
 * Step Functions identifies errors in the Amazon States Language using case-sensitive strings, known as error names.
 * The Amazon States Language defines a set of built-in strings that name well-known errors, all beginning with the <code>States.</code> prefix.
 * <p>
 * <ul>
 * <li><code>States.ALL</code> - A wildcard that matches any known error name.</li>
 * <li><code>States.Runtime</code> - An execution failed due to some exception that could not be processed. Often these are caused by errors at runtime, such as attempting to apply InputPath or OutputPath on a null JSON payload. A <code>States.Runtime</code> error is not retriable, and will always cause the execution to fail. A retry or catch on <code>States.ALL</code> will NOT catch States.Runtime errors.</li>
 * <li><code>States.DataLimitExceeded</code> - A States.DataLimitExceeded exception will be thrown for the following:
 * <p>
 * <ul>
 * <li>When the output of a connector is larger than payload size quota.</li>
 * <li>When the output of a state is larger than payload size quota.</li>
 * <li>When, after Parameters processing, the input of a state is larger than the payload size quota.</li>
 * <li>See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/limits-overview.html">the AWS documentation</a> to learn more about AWS Step Functions Quotas.</li>
 * </ul></li>
 * <li><code>States.HeartbeatTimeout</code> - A Task state failed to send a heartbeat for a period longer than the HeartbeatSeconds value.</li>
 * <li><code>States.Timeout</code> - A Task state either ran longer than the TimeoutSeconds value, or failed to send a heartbeat for a period longer than the HeartbeatSeconds value.</li>
 * <li><code>States.TaskFailed</code>- A Task state failed during the execution. When used in a retry or catch, <code>States.TaskFailed</code> acts as a wildcard that matches any known error name except for <code>States.Timeout</code>.</li>
 * </ul>
 * <p>
 * <h2>Logging</h2>
 * <p>
 * Enable logging to CloudWatch by passing a logging configuration with a
 * destination LogGroup:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * LogGroup logGroup = new LogGroup(this, "MyLogGroup");
 * 
 * StateMachine.Builder.create(this, "MyStateMachine")
 *         .definition(Chain.start(new Pass(this, "Pass")))
 *         .logs(LogOptions.builder()
 *                 .destination(logGroup)
 *                 .level(LogLevel.ALL)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>X-Ray tracing</h2>
 * <p>
 * Enable X-Ray tracing for StateMachine:
 * <p>
 * <blockquote><pre>
 * StateMachine.Builder.create(this, "MyStateMachine")
 *         .definition(Chain.start(new Pass(this, "Pass")))
 *         .tracingEnabled(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/step-functions/latest/dg/concepts-xray-tracing.html">the AWS documentation</a>
 * to learn more about AWS Step Functions's X-Ray support.
 * <p>
 * <h2>State Machine Permission Grants</h2>
 * <p>
 * IAM roles, users, or groups which need to be able to work with a State Machine should be granted IAM permissions.
 * <p>
 * Any object that implements the <code>IGrantable</code> interface (has an associated principal) can be granted permissions by calling:
 * <p>
 * <ul>
 * <li><code>stateMachine.grantStartExecution(principal)</code> - grants the principal the ability to execute the state machine</li>
 * <li><code>stateMachine.grantRead(principal)</code> - grants the principal read access</li>
 * <li><code>stateMachine.grantTaskResponse(principal)</code> - grants the principal the ability to send task tokens to the state machine</li>
 * <li><code>stateMachine.grantExecution(principal, actions)</code> - grants the principal execution-level permissions for the IAM actions specified</li>
 * <li><code>stateMachine.grant(principal, actions)</code> - grants the principal state-machine-level permissions for the IAM actions specified</li>
 * </ul>
 * <p>
 * <h3>Start Execution Permission</h3>
 * <p>
 * Grant permission to start an execution of a state machine by calling the <code>grantStartExecution()</code> API.
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role permission to start execution of state machine
 * stateMachine.grantStartExecution(role);
 * </pre></blockquote>
 * <p>
 * The following permission is provided to a service principal by the <code>grantStartExecution()</code> API:
 * <p>
 * <ul>
 * <li><code>states:StartExecution</code> - to state machine</li>
 * </ul>
 * <p>
 * <h3>Read Permissions</h3>
 * <p>
 * Grant <code>read</code> access to a state machine by calling the <code>grantRead()</code> API.
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role read access to state machine
 * stateMachine.grantRead(role);
 * </pre></blockquote>
 * <p>
 * The following read permissions are provided to a service principal by the <code>grantRead()</code> API:
 * <p>
 * <ul>
 * <li><code>states:ListExecutions</code> - to state machine</li>
 * <li><code>states:ListStateMachines</code> - to state machine</li>
 * <li><code>states:DescribeExecution</code> - to executions</li>
 * <li><code>states:DescribeStateMachineForExecution</code> - to executions</li>
 * <li><code>states:GetExecutionHistory</code> - to executions</li>
 * <li><code>states:ListActivities</code> - to <code>*</code></li>
 * <li><code>states:DescribeStateMachine</code> - to <code>*</code></li>
 * <li><code>states:DescribeActivity</code> - to <code>*</code></li>
 * </ul>
 * <p>
 * <h3>Task Response Permissions</h3>
 * <p>
 * Grant permission to allow task responses to a state machine by calling the <code>grantTaskResponse()</code> API:
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role task response permissions to the state machine
 * stateMachine.grantTaskResponse(role);
 * </pre></blockquote>
 * <p>
 * The following read permissions are provided to a service principal by the <code>grantRead()</code> API:
 * <p>
 * <ul>
 * <li><code>states:SendTaskSuccess</code> - to state machine</li>
 * <li><code>states:SendTaskFailure</code> - to state machine</li>
 * <li><code>states:SendTaskHeartbeat</code> - to state machine</li>
 * </ul>
 * <p>
 * <h3>Execution-level Permissions</h3>
 * <p>
 * Grant execution-level permissions to a state machine by calling the <code>grantExecution()</code> API:
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("lambda.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * // Give role permission to get execution history of ALL executions for the state machine
 * stateMachine.grantExecution(role, "states:GetExecutionHistory");
 * </pre></blockquote>
 * <p>
 * <h3>Custom Permissions</h3>
 * <p>
 * You can add any set of permissions to a state machine by calling the <code>grant()</code> API.
 * <p>
 * <blockquote><pre>
 * IChainable definition;
 * User user = new User(this, "MyUser");
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definition(definition)
 *         .build();
 * 
 * //give user permission to send task success to the state machine
 * stateMachine.grant(user, "states:SendTaskSuccess");
 * </pre></blockquote>
 * <p>
 * <h2>Import</h2>
 * <p>
 * Any Step Functions state machine that has been created outside the stack can be imported
 * into your CDK stack.
 * <p>
 * State machines can be imported by their ARN via the <code>StateMachine.fromStateMachineArn()</code> API.
 * In addition, the StateMachine can be imported via the <code>StateMachine.fromStateMachineName()</code> method, as long as they are in the same account/region as the current construct.
 * <p>
 * <blockquote><pre>
 * App app = new App();
 * Stack stack = new Stack(app, "MyStack");
 * StateMachine.fromStateMachineArn(stack, "ViaArnImportedStateMachine", "arn:aws:states:us-east-1:123456789012:stateMachine:StateMachine2E01A3A5-N5TJppzoevKQ");
 * 
 * StateMachine.fromStateMachineName(stack, "ViaResourceNameImportedStateMachine", "StateMachine2E01A3A5-N5TJppzoevKQ");
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.stepfunctions;
