/**
 * <h1>Amazon Simple Email Service Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Email receiving</h2>
 * <p>
 * Create a receipt rule set with rules and actions (actions can be found in the
 * <code>&#64;aws-cdk/aws-ses-actions</code> package):
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.services.ses.actions.*;
 * 
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * Topic topic = new Topic(this, "Topic");
 * 
 * ReceiptRuleSet.Builder.create(this, "RuleSet")
 *         .rules(List.of(ReceiptRuleOptions.builder()
 *                 .recipients(List.of("hello&#64;aws.com"))
 *                 .actions(List.of(
 *                     AddHeader.Builder.create()
 *                             .name("X-Special-Header")
 *                             .value("aws")
 *                             .build(),
 *                     S3.Builder.create()
 *                             .bucket(bucket)
 *                             .objectKeyPrefix("emails/")
 *                             .topic(topic)
 *                             .build()))
 *                 .build(), ReceiptRuleOptions.builder()
 *                 .recipients(List.of("aws.com"))
 *                 .actions(List.of(
 *                     Sns.Builder.create()
 *                             .topic(topic)
 *                             .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, rules can be added to a rule set:
 * <p>
 * <blockquote><pre>
 * ReceiptRuleSet ruleSet = new ReceiptRuleSet(this, "RuleSet");
 * 
 * ReceiptRule awsRule = ruleSet.addRule("Aws", ReceiptRuleOptions.builder()
 *         .recipients(List.of("aws.com"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * And actions to rules:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ses.actions.*;
 * 
 * ReceiptRule awsRule;
 * Topic topic;
 * 
 * awsRule.addAction(Sns.Builder.create()
 *         .topic(topic)
 *         .build());
 * </pre></blockquote>
 * <p>
 * When using <code>addRule</code>, the new rule is added after the last added rule unless <code>after</code> is specified.
 * <p>
 * <h3>Drop spams</h3>
 * <p>
 * A rule to drop spam can be added by setting <code>dropSpam</code> to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * ReceiptRuleSet.Builder.create(this, "RuleSet")
 *         .dropSpam(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will add a rule at the top of the rule set with a Lambda action that stops processing messages that have at least one spam indicator. See <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html">Lambda Function Examples</a>.
 * <p>
 * <h3>Receipt filter</h3>
 * <p>
 * Create a receipt filter:
 * <p>
 * <blockquote><pre>
 * ReceiptFilter.Builder.create(this, "Filter")
 *         .ip("1.2.3.4/16")
 *         .build();
 * </pre></blockquote>
 * <p>
 * An allow list filter is also available:
 * <p>
 * <blockquote><pre>
 * AllowListReceiptFilter.Builder.create(this, "AllowList")
 *         .ips(List.of("10.0.0.0/16", "1.2.3.4/16"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will first create a block all filter and then create allow filters for the listed ip addresses.
 * <p>
 * <h2>Email sending</h2>
 * <p>
 * <h3>Dedicated IP pools</h3>
 * <p>
 * When you create a new Amazon SES account, your emails are sent from IP addresses that are shared with other
 * Amazon SES users. For <a href="https://aws.amazon.com/ses/pricing/">an additional monthly charge</a>, you can lease
 * dedicated IP addresses that are reserved for your exclusive use.
 * <p>
 * Use the <code>DedicatedIpPool</code> construct to create a pool of dedicated IP addresses:
 * <p>
 * <blockquote><pre>
 * new DedicatedIpPool(this, "Pool");
 * </pre></blockquote>
 * <p>
 * The pool can then be used in a configuration set.
 * <p>
 * <h3>Configuration sets</h3>
 * <p>
 * Configuration sets are groups of rules that you can apply to your verified identities. A verified identity is
 * a domain, subdomain, or email address you use to send email through Amazon SES. When you apply a configuration
 * set to an email, all of the rules in that configuration set are applied to the email.
 * <p>
 * Use the <code>ConfigurationSet</code> construct to create a configuration set:
 * <p>
 * <blockquote><pre>
 * IDedicatedIpPool myPool;
 * 
 * 
 * ConfigurationSet.Builder.create(this, "ConfigurationSet")
 *         .customTrackingRedirectDomain("track.cdk.dev")
 *         .suppressionReasons(SuppressionReasons.COMPLAINTS_ONLY)
 *         .tlsPolicy(ConfigurationSetTlsPolicy.REQUIRE)
 *         .dedicatedIpPool(myPool)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Email identity</h3>
 * <p>
 * In Amazon SES, a verified identity is a domain or email address that you use to send or receive email. Before you
 * can send an email using Amazon SES, you must create and verify each identity that you're going to use as a <code>From</code>,
 * <code>Source</code>, <code>Sender</code>, or <code>Return-Path</code> address. Verifying an identity with Amazon SES confirms that you own it and
 * helps prevent unauthorized use.
 * <p>
 * To verify an identity for a hosted zone, you create an <code>EmailIdentity</code>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * IPublicHostedZone myHostedZone;
 * 
 * 
 * EmailIdentity identity = EmailIdentity.Builder.create(stack, "Identity")
 *         .identity(Identity.publicHostedZone(myHostedZone))
 *         .mailFromDomain("mail.cdk.dev")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, <a href="https://docs.aws.amazon.com/ses/latest/dg/send-email-authentication-dkim-easy.html">Easy DKIM</a> with
 * a 2048-bit DKIM key is used.
 * <p>
 * You can instead configure DKIM authentication by using your own public-private key pair. This process is known
 * as <a href="https://docs.aws.amazon.com/ses/latest/dg/send-email-authentication-dkim-bring-your-own.html">Bring Your Own DKIM (BYODKIM)</a>:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * IPublicHostedZone myHostedZone;
 * 
 * 
 * EmailIdentity.Builder.create(stack, "Identity")
 *         .identity(Identity.publicHostedZone(myHostedZone))
 *         .dkimIdentity(DkimIdentity.byoDkim(Map.of(
 *                 "privateKey", SecretValue.secretsManager("dkim-private-key"),
 *                 "publicKey", "...base64-encoded-public-key...",
 *                 "selector", "selector")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When using <code>publicHostedZone()</code> for the identity, all necessary Amazon Route 53 records are created automatically:
 * <p>
 * <ul>
 * <li>CNAME records for Easy DKIM</li>
 * <li>TXT record for BYOD DKIM</li>
 * <li>MX and TXT records for the custom MAIL FROM</li>
 * </ul>
 * <p>
 * When working with <code>domain()</code>, records must be created manually:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * EmailIdentity identity = EmailIdentity.Builder.create(stack, "Identity")
 *         .identity(Identity.domain("cdk.dev"))
 *         .build();
 * 
 * for (Object record : identity.getDkimRecords()) {
 * }
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ses;
