/**
 * <h1>AWS CloudTrail Construct Library</h1>
 * <p>
 * <h2>Trail</h2>
 * <p>
 * AWS CloudTrail enables governance, compliance, and operational and risk auditing of your AWS account. Actions taken by
 * a user, role, or an AWS service are recorded as events in CloudTrail. Learn more at the <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-user-guide.html">CloudTrail
 * documentation</a>.
 * <p>
 * The <code>Trail</code> construct enables ongoing delivery of events as log files to an Amazon S3 bucket. Learn more about <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-create-and-update-a-trail.html">Creating
 * a Trail for Your AWS Account</a>.
 * The following code creates a simple CloudTrail for your account -
 * <p>
 * <blockquote><pre>
 * Trail trail = new Trail(this, "CloudTrail");
 * </pre></blockquote>
 * <p>
 * By default, this will create a new S3 Bucket that CloudTrail will write to, and choose a few other reasonable defaults
 * such as turning on multi-region and global service events.
 * The defaults for each property and how to override them are all documented on the <code>TrailProps</code> interface.
 * <p>
 * <h2>Log File Validation</h2>
 * <p>
 * In order to validate that the CloudTrail log file was not modified after CloudTrail delivered it, CloudTrail provides a
 * digital signature for each file. Learn more at <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-log-file-validation-intro.html">Validating CloudTrail Log File
 * Integrity</a>.
 * <p>
 * This is enabled on the <code>Trail</code> construct by default, but can be turned off by setting <code>enableFileValidation</code> to <code>false</code>.
 * <p>
 * <blockquote><pre>
 * Trail trail = Trail.Builder.create(this, "CloudTrail")
 *         .enableFileValidation(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Notifications</h2>
 * <p>
 * Amazon SNS notifications can be configured upon new log files containing Trail events are delivered to S3.
 * Learn more at <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/configure-sns-notifications-for-cloudtrail.html">Configuring Amazon SNS Notifications for
 * CloudTrail</a>.
 * The following code configures an SNS topic to be notified -
 * <p>
 * <blockquote><pre>
 * Topic topic = new Topic(this, "TrailTopic");
 * Trail trail = Trail.Builder.create(this, "CloudTrail")
 *         .snsTopic(topic)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Service Integrations</h2>
 * <p>
 * Besides sending trail events to S3, they can also be configured to notify other AWS services -
 * <p>
 * <h3>Amazon CloudWatch Logs</h3>
 * <p>
 * CloudTrail events can be delivered to a CloudWatch Logs LogGroup. By default, a new LogGroup is created with a
 * default retention setting. The following code enables sending CloudWatch logs but specifies a particular retention
 * period for the created Log Group.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * Trail trail = Trail.Builder.create(this, "CloudTrail")
 *         .sendToCloudWatchLogs(true)
 *         .cloudWatchLogsRetention(RetentionDays.FOUR_MONTHS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you would like to use a specific log group instead, this can be configured via <code>cloudwatchLogGroup</code>.
 * <p>
 * <h3>Amazon EventBridge</h3>
 * <p>
 * Amazon EventBridge rules can be configured to be triggered when CloudTrail events occur using the <code>Trail.onEvent()</code> API.
 * Using APIs available in <code>aws-events</code>, these events can be filtered to match to those that are of interest, either from
 * a specific service, account or time range. See <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#events-for-services-not-listed">Events delivered via
 * CloudTrail</a>
 * to learn more about the event structure for events from CloudTrail.
 * <p>
 * The following code filters events for S3 from a specific AWS account and triggers a lambda function.
 * <p>
 * <blockquote><pre>
 * Function myFunctionHandler = Function.Builder.create(this, "MyFunction")
 *         .code(Code.fromAsset("resource/myfunction"))
 *         .runtime(Runtime.NODEJS_14_X)
 *         .handler("index.handler")
 *         .build();
 * 
 * Rule eventRule = Trail.onEvent(this, "MyCloudWatchEvent", OnEventOptions.builder()
 *         .target(new LambdaFunction(myFunctionHandler))
 *         .build());
 * 
 * eventRule.addEventPattern(EventPattern.builder()
 *         .account(List.of("123456789012"))
 *         .source(List.of("aws.s3"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Multi-Region &amp; Global Service Events</h2>
 * <p>
 * By default, a <code>Trail</code> is configured to deliver log files from multiple regions to a single S3 bucket for a given
 * account. This creates shadow trails (replication of the trails) in all of the other regions. Learn more about <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-regional-and-global-services">How
 * CloudTrail Behaves Regionally</a>
 * and about the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudtrail-trail.html#cfn-cloudtrail-trail-ismultiregiontrail"><code>IsMultiRegion</code>
 * property</a>.
 * <p>
 * For most services, events are recorded in the region where the action occurred. For global services such as AWS IAM,
 * AWS STS, Amazon CloudFront, Route 53, etc., events are delivered to any trail that includes global services. Learn more
 * <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-global-service-events">About Global Service Events</a>.
 * <p>
 * Events for global services are turned on by default for <code>Trail</code> constructs in the CDK.
 * <p>
 * The following code disables multi-region trail delivery and trail delivery for global services for a specific <code>Trail</code> -
 * <p>
 * <blockquote><pre>
 * Trail trail = Trail.Builder.create(this, "CloudTrail")
 *         // ...
 *         .isMultiRegionTrail(false)
 *         .includeGlobalServiceEvents(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Events Types</h2>
 * <p>
 * <strong>Management events</strong> provide information about management operations that are performed on resources in your AWS
 * account. These are also known as control plane operations. Learn more about <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-events">Management
 * Events</a>.
 * <p>
 * By default, a <code>Trail</code> logs all management events. However, they can be configured to either be turned off, or to only
 * log 'Read' or 'Write' events.
 * <p>
 * The following code configures the <code>Trail</code> to only track management events that are of type 'Read'.
 * <p>
 * <blockquote><pre>
 * Trail trail = Trail.Builder.create(this, "CloudTrail")
 *         // ...
 *         .managementEvents(ReadWriteType.READ_ONLY)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Data events</strong> provide information about the resource operations performed on or in a resource. These are also known
 * as data plane operations. Learn more about <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-events">Data
 * Events</a>.
 * By default, no data events are logged for a <code>Trail</code>.
 * <p>
 * AWS CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions.
 * <p>
 * The <code>logAllS3DataEvents()</code> API configures the trail to log all S3 data events while the <code>addS3EventSelector()</code> API can
 * be used to configure logging of S3 data events for specific buckets and specific object prefix. The following code
 * configures logging of S3 data events for <code>fooBucket</code> and with object prefix <code>bar/</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * Bucket bucket;
 * 
 * 
 * Trail trail = new Trail(this, "MyAmazingCloudTrail");
 * 
 * // Adds an event selector to the bucket foo
 * trail.addS3EventSelector(List.of(S3EventSelector.builder()
 *         .bucket(bucket)
 *         .objectPrefix("bar/")
 *         .build()));
 * </pre></blockquote>
 * <p>
 * Similarly, the <code>logAllLambdaDataEvents()</code> configures the trail to log all Lambda data events while the
 * <code>addLambdaEventSelector()</code> API can be used to configure logging for specific Lambda functions. The following code
 * configures logging of Lambda data events for a specific Function.
 * <p>
 * <blockquote><pre>
 * Trail trail = new Trail(this, "MyAmazingCloudTrail");
 * Function amazingFunction = Function.Builder.create(this, "AnAmazingFunction")
 *         .runtime(Runtime.NODEJS_14_X)
 *         .handler("hello.handler")
 *         .code(Code.fromAsset("lambda"))
 *         .build();
 * 
 * // Add an event selector to log data events for the provided Lambda functions.
 * trail.addLambdaEventSelector(List.of(amazingFunction));
 * </pre></blockquote>
 * <p>
 * <h2>Organization Trail</h2>
 * <p>
 * It is possible to create a trail that will be applied to all accounts in an organization if the current account manages an organization.
 * To enable this, the property <code>isOrganizationTrail</code> must be set. If this property is set and the current account does not manage an organization, the stack will fail to deploy.
 * <p>
 * <blockquote><pre>
 * Trail.Builder.create(this, "OrganizationTrail")
 *         .isOrganizationTrail(true)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.cloudtrail;
