/**
 * <h1>Amazon OpenSearch Service Construct Library</h1>
 * <p>
 * Amazon OpenSearch Service is the successor to Amazon Elasticsearch Service.
 * <p>
 * See <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/aws-elasticsearch-readme.html#migrating-to-opensearch">Migrating to OpenSearch</a> for migration instructions from <code>&#64;aws-cdk/aws-elasticsearch</code> to this module, <code>&#64;aws-cdk/aws-opensearchservice</code>.
 * <p>
 * <h2>Quick start</h2>
 * <p>
 * Create a development cluster by simply specifying the version:
 * <p>
 * <blockquote><pre>
 * Domain devDomain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To perform version upgrades without replacing the entire domain, specify the <code>enableVersionUpgrade</code> property.
 * <p>
 * <blockquote><pre>
 * Domain devDomain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .enableVersionUpgrade(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Create a production grade cluster by also specifying things like capacity and az distribution
 * <p>
 * <blockquote><pre>
 * Domain prodDomain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .capacity(CapacityConfig.builder()
 *                 .masterNodes(5)
 *                 .dataNodes(20)
 *                 .build())
 *         .ebs(EbsOptions.builder()
 *                 .volumeSize(20)
 *                 .build())
 *         .zoneAwareness(ZoneAwarenessConfig.builder()
 *                 .availabilityZoneCount(3)
 *                 .build())
 *         .logging(LoggingOptions.builder()
 *                 .slowSearchLogEnabled(true)
 *                 .appLogEnabled(true)
 *                 .slowIndexLogEnabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * This creates an Amazon OpenSearch Service cluster and automatically sets up log groups for
 * logging the domain logs and slow search logs.
 * <p>
 * <h2>A note about SLR</h2>
 * <p>
 * Some cluster configurations (e.g VPC access) require the existence of the <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/slr.html"><code>AWSServiceRoleForAmazonElasticsearchService</code></a> Service-Linked Role.
 * <p>
 * When performing such operations via the AWS Console, this SLR is created automatically when needed. However, this is not the behavior when using CloudFormation. If an SLR is needed, but doesn't exist, you will encounter a failure message simlar to:
 * <p>
 * <blockquote><pre>
 * Before you can proceed, you must enable a service-linked role to give Amazon OpenSearch Service...
 * </pre></blockquote>
 * <p>
 * To resolve this, you need to <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/using-service-linked-roles.html#create-service-linked-role">create</a> the SLR. We recommend using the AWS CLI:
 * <p>
 * <blockquote><pre>
 * aws iam create-service-linked-role --aws-service-name es.amazonaws.com
 * </pre></blockquote>
 * <p>
 * You can also create it using the CDK, <strong>but note that only the first application deploying this will succeed</strong>:
 * <p>
 * <blockquote><pre>
 * CfnServiceLinkedRole slr = CfnServiceLinkedRole.Builder.create(this, "Service Linked Role")
 *         .awsServiceName("es.amazonaws.com")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Importing existing domains</h2>
 * <p>
 * <h3>Using a known domain endpoint</h3>
 * <p>
 * To import an existing domain into your CDK application, use the <code>Domain.fromDomainEndpoint</code> factory method.
 * This method accepts a domain endpoint of an already existing domain:
 * <p>
 * <blockquote><pre>
 * String domainEndpoint = "https://my-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com";
 * IDomain domain = Domain.fromDomainEndpoint(this, "ImportedDomain", domainEndpoint);
 * </pre></blockquote>
 * <p>
 * <h3>Using the output of another CloudFormation stack</h3>
 * <p>
 * To import an existing domain with the help of an exported value from another CloudFormation stack,
 * use the <code>Domain.fromDomainAttributes</code> factory method. This will accept tokens.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Object domainArn = Fn.importValue("another-cf-stack-export-domain-arn");
 * Object domainEndpoint = Fn.importValue("another-cf-stack-export-domain-endpoint");
 * Object domain = Domain.fromDomainAttributes(this, "ImportedDomain", Map.of(
 *         "domainArn", domainArn,
 *         "domainEndpoint", domainEndpoint));
 * </pre></blockquote>
 * <p>
 * <h2>Permissions</h2>
 * <p>
 * <h3>IAM</h3>
 * <p>
 * Helper methods also exist for managing access to the domain.
 * <p>
 * <blockquote><pre>
 * Function fn;
 * Domain domain;
 * 
 * 
 * // Grant write access to the app-search index
 * domain.grantIndexWrite("app-search", fn);
 * 
 * // Grant read access to the 'app-search/_search' path
 * domain.grantPathRead("app-search/_search", fn);
 * </pre></blockquote>
 * <p>
 * <h2>Encryption</h2>
 * <p>
 * The domain can also be created with encryption enabled:
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .ebs(EbsOptions.builder()
 *                 .volumeSize(100)
 *                 .volumeType(EbsDeviceVolumeType.GENERAL_PURPOSE_SSD)
 *                 .build())
 *         .nodeToNodeEncryption(true)
 *         .encryptionAtRest(EncryptionAtRestOptions.builder()
 *                 .enabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * This sets up the domain with node to node encryption and encryption at
 * rest. You can also choose to supply your own KMS key to use for encryption at
 * rest.
 * <p>
 * <h2>VPC Support</h2>
 * <p>
 * Domains can be placed inside a VPC, providing a secure communication between Amazon OpenSearch Service and other services within the VPC without the need for an internet gateway, NAT device, or VPN connection.
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/vpc.html">VPC Support for Amazon OpenSearch Service Domains</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * Vpc vpc = new Vpc(this, "Vpc");
 * DomainProps domainProps = DomainProps.builder()
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .vpc(vpc)
 *         // must be enabled since our VPC contains multiple private subnets.
 *         .zoneAwareness(ZoneAwarenessConfig.builder()
 *                 .enabled(true)
 *                 .build())
 *         .capacity(CapacityConfig.builder()
 *                 // must be an even number since the default az count is 2.
 *                 .dataNodes(2)
 *                 .build())
 *         .build();
 * new Domain(this, "Domain", domainProps);
 * </pre></blockquote>
 * <p>
 * In addition, you can use the <code>vpcSubnets</code> property to control which specific subnets will be used, and the <code>securityGroups</code> property to control
 * which security groups will be attached to the domain. By default, CDK will select all <em>private</em> subnets in the VPC, and create one dedicated security group.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * Helper methods exist to access common domain metrics for example:
 * <p>
 * <blockquote><pre>
 * Domain domain;
 * 
 * Metric freeStorageSpace = domain.metricFreeStorageSpace();
 * Metric masterSysMemoryUtilization = domain.metric("MasterSysMemoryUtilization");
 * </pre></blockquote>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Fine grained access control</h2>
 * <p>
 * The domain can also be created with a master user configured. The password can
 * be supplied or dynamically created if not supplied.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .enforceHttps(true)
 *         .nodeToNodeEncryption(true)
 *         .encryptionAtRest(EncryptionAtRestOptions.builder()
 *                 .enabled(true)
 *                 .build())
 *         .fineGrainedAccessControl(AdvancedSecurityOptions.builder()
 *                 .masterUserName("master-user")
 *                 .build())
 *         .build();
 * 
 * SecretValue masterUserPassword = domain.getMasterUserPassword();
 * </pre></blockquote>
 * <p>
 * <h2>Using unsigned basic auth</h2>
 * <p>
 * For convenience, the domain can be configured to allow unsigned HTTP requests
 * that use basic auth. Unless the domain is configured to be part of a VPC this
 * means anyone can access the domain using the configured master username and
 * password.
 * <p>
 * To enable unsigned basic auth access the domain is configured with an access
 * policy that allows anyonmous requests, HTTPS required, node to node encryption,
 * encryption at rest and fine grained access control.
 * <p>
 * If the above settings are not set they will be configured as part of enabling
 * unsigned basic auth. If they are set with conflicting values, an error will be
 * thrown.
 * <p>
 * If no master user is configured a default master user is created with the
 * username <code>admin</code>.
 * <p>
 * If no password is configured a default master user password is created and
 * stored in the AWS Secrets Manager as secret. The secret has the prefix
 * <code>&lt;domain id&gt;MasterUser</code>.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .useUnsignedBasicAuth(true)
 *         .build();
 * 
 * SecretValue masterUserPassword = domain.getMasterUserPassword();
 * </pre></blockquote>
 * <p>
 * <h2>Custom access policies</h2>
 * <p>
 * If the domain requires custom access control it can be configured either as a
 * constructor property, or later by means of a helper method.
 * <p>
 * For simple permissions the <code>accessPolicies</code> constructor may be sufficient:
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .accessPolicies(List.of(
 *             PolicyStatement.Builder.create()
 *                     .actions(List.of("es:*ESHttpPost", "es:ESHttpPut*"))
 *                     .effect(Effect.ALLOW)
 *                     .principals(List.of(new AccountPrincipal("123456789012")))
 *                     .resources(List.of("*"))
 *                     .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * For more complex use-cases, for example, to set the domain up to receive data from a
 * <a href="https://aws.amazon.com/premiumsupport/knowledge-center/kinesis-firehose-cross-account-streaming/">cross-account Kinesis Firehose</a> the <code>addAccessPolicies</code> helper method
 * allows for policies that include the explicit domain ARN.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .build();
 * domain.addAccessPolicies(
 * PolicyStatement.Builder.create()
 *         .actions(List.of("es:ESHttpPost", "es:ESHttpPut"))
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(new AccountPrincipal("123456789012")))
 *         .resources(List.of(domain.getDomainArn(), String.format("%s/*", domain.getDomainArn())))
 *         .build(),
 * PolicyStatement.Builder.create()
 *         .actions(List.of("es:ESHttpGet"))
 *         .effect(Effect.ALLOW)
 *         .principals(List.of(new AccountPrincipal("123456789012")))
 *         .resources(List.of(String.format("%s/_all/_settings", domain.getDomainArn()), String.format("%s/_cluster/stats", domain.getDomainArn()), String.format("%s/index-name*&#47;_mapping/type-name", domain.getDomainArn()), String.format("%s/roletest*&#47;_mapping/roletest", domain.getDomainArn()), String.format("%s/_nodes", domain.getDomainArn()), String.format("%s/_nodes/stats", domain.getDomainArn()), String.format("%s/_nodes/*&#47;stats", domain.getDomainArn()), String.format("%s/_stats", domain.getDomainArn()), String.format("%s/index-name*&#47;_stats", domain.getDomainArn()), String.format("%s/roletest*&#47;_stat", domain.getDomainArn())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Audit logs</h2>
 * <p>
 * Audit logs can be enabled for a domain, but only when fine grained access control is enabled.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .enforceHttps(true)
 *         .nodeToNodeEncryption(true)
 *         .encryptionAtRest(EncryptionAtRestOptions.builder()
 *                 .enabled(true)
 *                 .build())
 *         .fineGrainedAccessControl(AdvancedSecurityOptions.builder()
 *                 .masterUserName("master-user")
 *                 .build())
 *         .logging(LoggingOptions.builder()
 *                 .auditLogEnabled(true)
 *                 .slowSearchLogEnabled(true)
 *                 .appLogEnabled(true)
 *                 .slowIndexLogEnabled(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>UltraWarm</h2>
 * <p>
 * UltraWarm nodes can be enabled to provide a cost-effective way to store large amounts of read-only data.
 * <p>
 * <blockquote><pre>
 * Domain domain = Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .capacity(CapacityConfig.builder()
 *                 .masterNodes(2)
 *                 .warmNodes(2)
 *                 .warmInstanceType("ultrawarm1.medium.search")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Custom endpoint</h2>
 * <p>
 * Custom endpoints can be configured to reach the domain under a custom domain name.
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .customEndpoint(CustomEndpointOptions.builder()
 *                 .domainName("search.example.com")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * It is also possible to specify a custom certificate instead of the auto-generated one.
 * <p>
 * Additionally, an automatic CNAME-Record is created if a hosted zone is provided for the custom endpoint
 * <p>
 * <h2>Advanced options</h2>
 * <p>
 * <a href="https://docs.aws.amazon.com/opensearch-service/latest/developerguide/createupdatedomains.html#createdomain-configure-advanced-options">Advanced options</a> can used to configure additional options.
 * <p>
 * <blockquote><pre>
 * Domain.Builder.create(this, "Domain")
 *         .version(EngineVersion.OPENSEARCH_1_0)
 *         .advancedOptions(Map.of(
 *                 "rest.action.multi.allow_explicit_index", "false",
 *                 "indices.fielddata.cache.size", "25",
 *                 "indices.query.bool.max_clause_count", "2048"))
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.opensearchservice;
