/**
 * <h1>AWS CDK Custom Resources</h1>
 * <p>
 * <h2>Provider Framework</h2>
 * <p>
 * AWS CloudFormation <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-custom-resources.html">custom resources</a> are extension points to the provisioning
 * engine. When CloudFormation needs to create, update or delete a custom resource,
 * it sends a lifecycle event notification to a <strong>custom resource provider</strong>. The provider
 * handles the event (e.g. creates a resource) and sends back a response to CloudFormation.
 * <p>
 * The <code>&#64;aws-cdk/custom-resources.Provider</code> construct is a "mini-framework" for
 * implementing providers for AWS CloudFormation custom resources. The framework offers a high-level API which makes it easier to implement robust
 * and powerful custom resources. If you are looking to implement a custom resource provider, we recommend
 * you use this module unless you have good reasons not to. For an overview of different provider types you
 * could be using, see the <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib-readme.html#custom-resource-providers">Custom Resource Providers section in the core library documentation</a>.
 * <p>
 * <blockquote>
 * <p>
 * <strong>N.B.</strong>: if you use the provider framework in this module you will write AWS Lambda Functions that look a lot like, but aren't exactly the same as the Lambda Functions you would write if you wrote CloudFormation Custom Resources directly, without this framework.
 * <p>
 * Specifically, to report success or failure, have your Lambda Function exit in the right way: return data for success, or throw an
 * exception for failure. <em>Do not</em> post the success or failure of your custom resource to an HTTPS URL as the CloudFormation
 * documentation tells you to do.
 * <p>
 * </blockquote>
 * <p>
 * The framework has the following capabilities:
 * <p>
 * <ul>
 * <li>Handles responses to AWS CloudFormation and protects against blocked
 * deployments</li>
 * <li>Validates handler return values to help with correct handler implementation</li>
 * <li>Supports asynchronous handlers to enable operations that require a long waiting period for a resource, which can exceed the AWS Lambda timeout</li>
 * <li>Implements default behavior for physical resource IDs.</li>
 * </ul>
 * <p>
 * The following code shows how the <code>Provider</code> construct is used in conjunction
 * with a <code>CustomResource</code> and a user-provided AWS Lambda function which implements
 * the actual handler.
 * <p>
 * <blockquote><pre>
 * Function onEvent;
 * Function isComplete;
 * Role myRole;
 * 
 * 
 * Provider myProvider = Provider.Builder.create(this, "MyProvider")
 *         .onEventHandler(onEvent)
 *         .isCompleteHandler(isComplete) // optional async "waiter"
 *         .logRetention(RetentionDays.ONE_DAY) // default is INFINITE
 *         .role(myRole)
 *         .build();
 * 
 * CustomResource.Builder.create(this, "Resource1").serviceToken(myProvider.getServiceToken()).build();
 * CustomResource.Builder.create(this, "Resource2").serviceToken(myProvider.getServiceToken()).build();
 * </pre></blockquote>
 * <p>
 * Providers are implemented through AWS Lambda functions that are triggered by the
 * provider framework in response to lifecycle events.
 * <p>
 * At the minimum, users must define the <code>onEvent</code> handler, which is invoked by the
 * framework for all resource lifecycle events (<code>Create</code>, <code>Update</code> and <code>Delete</code>)
 * and returns a result which is then submitted to CloudFormation.
 * <p>
 * The following example is a skeleton for a Python implementation of <code>onEvent</code>:
 * <p>
 * <blockquote><pre>
 * def on_event(event, context):
 *   print(event)
 *   request_type = event['RequestType']
 *   if request_type == 'Create': return on_create(event)
 *   if request_type == 'Update': return on_update(event)
 *   if request_type == 'Delete': return on_delete(event)
 *   raise Exception("Invalid request type: %s" % request_type)
 * 
 * def on_create(event):
 *   props = event["ResourceProperties"]
 *   print("create new resource with props %s" % props)
 * 
 *   # add your create code here...
 *   physical_id = ...
 * 
 *   return { 'PhysicalResourceId': physical_id }
 * 
 * def on_update(event):
 *   physical_id = event["PhysicalResourceId"]
 *   props = event["ResourceProperties"]
 *   print("update resource %s with props %s" % (physical_id, props))
 *   # ...
 * 
 * def on_delete(event):
 *   physical_id = event["PhysicalResourceId"]
 *   print("delete resource %s" % physical_id)
 *   # ...
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * When writing your handlers, there are a couple of non-obvious corner cases you need to
 * pay attention to. See the <a href="#important-cases-to-handle">important cases to handle</a> section for more information.
 * <p>
 * </blockquote>
 * <p>
 * Users may also provide an additional handler called <code>isComplete</code>, for cases
 * where the lifecycle operation cannot be completed immediately. The
 * <code>isComplete</code> handler will be retried asynchronously after <code>onEvent</code> until it
 * returns <code>IsComplete: true</code>, or until the total provider timeout has expired.
 * <p>
 * The following example is a skeleton for a Python implementation of <code>isComplete</code>:
 * <p>
 * <blockquote><pre>
 * def is_complete(event, context):
 *   physical_id = event["PhysicalResourceId"]
 *   request_type = event["RequestType"]
 * 
 *   # check if resource is stable based on request_type
 *   is_ready = ...
 * 
 *   return { 'IsComplete': is_ready }
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Security Note</strong>: the Custom Resource Provider Framework will write the value of <code>ResponseURL</code>,
 * which is a pre-signed S3 URL used to report the success or failure of the Custom Resource execution
 * back to CloudFormation, in a readable form to the AWS Step Functions execution history.
 * <p>
 * Anybody who can list and read AWS StepFunction executions in your account will be able to write
 * a fake response to this URL and make your CloudFormation deployments fail.
 * <p>
 * Do not use this library if your threat model requires that you cannot trust actors who are able
 * to list StepFunction executions in your account.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Handling Lifecycle Events: onEvent</h3>
 * <p>
 * The user-defined <code>onEvent</code> AWS Lambda function is invoked whenever a resource
 * lifecycle event occurs. The function is expected to handle the event and return
 * a response to the framework that, at least, includes the physical resource ID.
 * <p>
 * If <code>onEvent</code> returns successfully, the framework will submit a "SUCCESS" response
 * to AWS CloudFormation for this resource operation.  If the provider is
 * <a href="#asynchronous-providers-iscomplete">asynchronous</a> (<code>isCompleteHandler</code> is
 * defined), the framework will only submit a response based on the result of
 * <code>isComplete</code>.
 * <p>
 * If <code>onEvent</code> throws an error, the framework will submit a "FAILED" response to
 * AWS CloudFormation.
 * <p>
 * The input event includes the following fields derived from the [Custom Resource
 * Provider Request]:
 * <p>
 * |Field|Type|Description
 * |-----|----|----------------
 * |<code>RequestType</code>|String|The type of lifecycle event: <code>Create</code>, <code>Update</code> or <code>Delete</code>.
 * |<code>LogicalResourceId</code>|String|The template developer-chosen name (logical ID) of the custom resource in the AWS CloudFormation template.
 * |<code>PhysicalResourceId</code>|String|This field will only be present for <code>Update</code> and <code>Delete</code> events and includes the value returned in <code>PhysicalResourceId</code> of the previous operation.
 * |<code>ResourceProperties</code>|JSON|This field contains the properties defined in the template for this custom resource.
 * |<code>OldResourceProperties</code>|JSON|This field will only be present for <code>Update</code> events and contains the resource properties that were declared previous to the update request.
 * |<code>ResourceType</code>|String|The resource type defined for this custom resource in the template. A provider may handle any number of custom resource types.
 * |<code>RequestId</code>|String|A unique ID for the request.
 * |<code>StackId</code>|String|The ARN that identifies the stack that contains the custom resource.
 * <p>
 * The return value from <code>onEvent</code> must be a JSON object with the following fields:
 * <p>
 * |Field|Type|Required|Description
 * |-----|----|--------|-----------
 * |<code>PhysicalResourceId</code>|String|No|The allocated/assigned physical ID of the resource. If omitted for <code>Create</code> events, the event's <code>RequestId</code> will be used. For <code>Update</code>, the current physical ID will be used. If a different value is returned, CloudFormation will follow with a subsequent <code>Delete</code> for the previous ID (resource replacement). For <code>Delete</code>, it will always return the current physical resource ID, and if the user returns a different one, an error will occur.
 * |<code>Data</code>|JSON|No|Resource attributes, which can later be retrieved through <code>Fn::GetAtt</code> on the custom resource object.
 * |<code>NoEcho</code>|Boolean|No|Whether to mask the output of the custom resource when retrieved by using the <code>Fn::GetAtt</code> function.
 * |<em>any</em>|<em>any</em>|No|Any other field included in the response will be passed through to <code>isComplete</code>. This can sometimes be useful to pass state between the handlers.
 * <p>
 * <h3>Asynchronous Providers: isComplete</h3>
 * <p>
 * It is not uncommon for the provisioning of resources to be an asynchronous
 * operation, which means that the operation does not immediately finish, and we
 * need to "wait" until the resource stabilizes.
 * <p>
 * The provider framework makes it easy to implement "waiters" by allowing users to
 * specify an additional AWS Lambda function in <code>isCompleteHandler</code>.
 * <p>
 * The framework will repeatedly invoke the handler every <code>queryInterval</code>. When
 * <code>isComplete</code> returns with <code>IsComplete: true</code>, the framework will submit a
 * "SUCCESS" response to AWS CloudFormation. If <code>totalTimeout</code> expires and the
 * operation has not yet completed, the framework will submit a "FAILED" response
 * with the message "Operation timed out".
 * <p>
 * If an error is thrown, the framework will submit a "FAILED" response to AWS
 * CloudFormation.
 * <p>
 * The input event to <code>isComplete</code> includes all request fields, combined with all
 * fields returned from <code>onEvent</code>. If <code>PhysicalResourceId</code> has not been explicitly
 * returned from <code>onEvent</code>, it's value will be calculated based on the heuristics
 * described above.
 * <p>
 * The return value must be a JSON object with the following fields:
 * <p>
 * |Field|Type|Required|Description
 * |-----|----|--------|-----------
 * |<code>IsComplete</code>|Boolean|Yes|Indicates if the operation has finished or not.
 * |<code>Data</code>|JSON|No|May only be sent if <code>IsComplete</code> is <code>true</code> and includes additional resource attributes. These attributes will be <strong>merged</strong> with the ones returned from <code>onEvent</code>
 * <p>
 * <h3>Physical Resource IDs</h3>
 * <p>
 * Every resource in CloudFormation has a physical resource ID. When a resource is
 * created, the <code>PhysicalResourceId</code> returned from the <code>Create</code> operation is stored
 * by AWS CloudFormation and assigned to the logical ID defined for this resource
 * in the template. If a <code>Create</code> operation returns without a <code>PhysicalResourceId</code>,
 * the framework will use <code>RequestId</code> as the default. This is sufficient for
 * various cases such as "pseudo-resources" which only query data.
 * <p>
 * For <code>Update</code> and <code>Delete</code> operations, the resource event will always include the
 * current <code>PhysicalResourceId</code> of the resource.
 * <p>
 * When an <code>Update</code> operation occurs, the default behavior is to return the current
 * physical resource ID. if the <code>onEvent</code> returns a <code>PhysicalResourceId</code> which is
 * different from the current one, AWS CloudFormation will treat this as a
 * <strong>resource replacement</strong>, and it will issue a subsequent <code>Delete</code> operation for
 * the old resource.
 * <p>
 * As a rule of thumb, if your custom resource supports configuring a physical name
 * (e.g. you can specify a <code>BucketName</code> when you define an <code>AWS::S3::Bucket</code>), you
 * must return this name in <code>PhysicalResourceId</code> and make sure to handle
 * replacement properly. The <code>S3File</code> example demonstrates this
 * through the <code>objectKey</code> property.
 * <p>
 * <h3>When there are errors</h3>
 * <p>
 * As mentioned above, if any of the user handlers fail (i.e. throws an exception)
 * or times out (due to their AWS Lambda timing out), the framework will trap these
 * errors and submit a "FAILED" response to AWS CloudFormation, along with the error
 * message.
 * <p>
 * Since errors can occur in multiple places in the provider (framework, <code>onEvent</code>,
 * <code>isComplete</code>), it is important to know that there could situations where a
 * resource operation fails even though the operation technically succeeded (i.e.
 * isComplete throws an error).
 * <p>
 * When AWS CloudFormation receives a "FAILED" response, it will attempt to roll
 * back the stack to it's last state. This has different meanings for different
 * lifecycle events:
 * <p>
 * <ul>
 * <li>If a <code>Create</code> event fails, the resource provider framework will automatically
 * ignore the subsequent <code>Delete</code> operation issued by AWS CloudFormation. The
 * framework currently does not support customizing this behavior (see
 * https://github.com/aws/aws-cdk/issues/5524).</li>
 * <li>If an <code>Update</code> event fails, CloudFormation will issue an additional <code>Update</code>
 * with the previous properties.</li>
 * <li>If a <code>Delete</code> event fails, CloudFormation will abandon this resource.</li>
 * </ul>
 * <p>
 * <h3>Important cases to handle</h3>
 * <p>
 * You should keep the following list in mind when writing custom resources to
 * make sure your custom resource behaves correctly in all cases:
 * <p>
 * <ul>
 * <li>During <code>Create</code>:
 * <p>
 * <ul>
 * <li>If the create fails, the <em>provider framework</em> will make sure you
 * don't get a subsequent <code>Delete</code> event. If your create involves multiple distinct
 * operations, it is your responsibility to catch and rethrow and clean up
 * any partial updates that have already been performed. Make sure your
 * API call timeouts and Lambda timeouts allow for this.</li>
 * </ul></li>
 * <li>During <code>Update</code>:
 * <p>
 * <ul>
 * <li>If the update fails, you will get a subsequent <code>Update</code> event
 * to roll back to the previous state (with <code>ResourceProperties</code> and
 * <code>OldResourceProperties</code> reversed).</li>
 * <li>If you return a different <code>PhysicalResourceId</code>, you will subsequently
 * receive a <code>Delete</code> event to clean up the previous state of the resource.</li>
 * </ul></li>
 * <li>During <code>Delete</code>:
 * <p>
 * <ul>
 * <li>If the behavior of your custom resource is tied to another AWS resource
 * (for example, it exists to clean the contents of a stateful resource), keep
 * in mind that your custom resource may be deleted independently of the other
 * resource and you must confirm that it is appropriate to perform the action.</li>
 * <li>(only if you are <em>not</em> using the provider framework) a <code>Delete</code> event
 * may be caused by a failed <code>Create</code>. You must be able to handle the case
 * where the resource you are trying to delete hasn't even been created yet.</li>
 * </ul></li>
 * <li>If you update the code of your custom resource and change the format of the
 * resource properties, be aware that there may still be already-deployed
 * instances of your custom resource out there, and you may still receive
 * the <em>old</em> property format in <code>ResourceProperties</code> (during <code>Delete</code> and
 * rollback <code>Updates</code>) or in <code>OldResourceProperties</code> (during rollforward
 * <code>Update</code>). You must continue to handle all possible sets of properties
 * your custom resource could have ever been created with in the past.</li>
 * </ul>
 * <p>
 * <h3>Provider Framework Execution Policy</h3>
 * <p>
 * Similarly to any AWS Lambda function, if the user-defined handlers require
 * access to AWS resources, you will have to define these permissions
 * by calling "grant" methods such as <code>myBucket.grantRead(myHandler)</code>), using <code>myHandler.addToRolePolicy</code>
 * or specifying an <code>initialPolicy</code> when defining the function.
 * <p>
 * Bear in mind that in most cases, a single provider will be used for multiple
 * resource instances. This means that the execution policy of the provider must
 * have the appropriate privileges.
 * <p>
 * The following example grants the <code>onEvent</code> handler <code>s3:GetObject*</code> permissions
 * to all buckets:
 * <p>
 * <blockquote><pre>
 * Function.Builder.create(this, "OnEventHandler")
 *         .runtime(Runtime.NODEJS_14_X)
 *         .handler("index.handler")
 *         .code(Code.fromInline("my code"))
 *         .initialPolicy(List.of(
 *             PolicyStatement.Builder.create().actions(List.of("s3:GetObject*")).resources(List.of("*")).build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Timeouts</h3>
 * <p>
 * Users are responsible to define the timeouts for the AWS Lambda functions for
 * user-defined handlers. It is recommended not to exceed a <strong>14 minutes</strong> timeout,
 * since all framework functions are configured to time out after 15 minutes, which
 * is the maximal AWS Lambda timeout.
 * <p>
 * If your operation takes over <strong>14 minutes</strong>, the recommended approach is to
 * implement an <a href="#asynchronous-providers-iscomplete">asynchronous provider</a>, and
 * then configure the timeouts for the asynchronous retries through the
 * <code>queryInterval</code> and the <code>totalTimeout</code> options.
 * <p>
 * <h3>Provider Framework Examples</h3>
 * <p>
 * This module includes a few examples for custom resource implementations:
 * <p>
 * <h4>S3File</h4>
 * <p>
 * Provisions an object in an S3 bucket with textual contents. See the source code
 * for the
 * <a href="https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/custom-resources/test/provider-framework/integration-test-fixtures/s3-file.ts">construct</a> and
 * <a href="https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/custom-resources/test/provider-framework/integration-test-fixtures/s3-file-handler/index.ts">handler</a>.
 * <p>
 * The following example will create the file <code>folder/file1.txt</code> inside <code>myBucket</code>
 * with the contents <code>hello!</code>.
 * <p>
 * <blockquote><pre>
 * // This example exists only for TypeScript
 * 
 * declare const myBucket: s3.Bucket;
 * new cr.S3File(this, 'MyFile', {
 *   bucket: myBucket,
 *   objectKey: 'folder/file1.txt', // optional
 *   content: 'hello!',
 *   public: true, // optional
 * });
 * </pre></blockquote>
 * <p>
 * This sample demonstrates the following concepts:
 * <p>
 * <ul>
 * <li>Synchronous implementation (<code>isComplete</code> is not defined)</li>
 * <li>Automatically generates the physical name if <code>objectKey</code> is not defined</li>
 * <li>Handles physical name changes</li>
 * <li>Returns resource attributes</li>
 * <li>Handles deletions</li>
 * <li>Implemented in TypeScript</li>
 * </ul>
 * <p>
 * <h4>S3Assert</h4>
 * <p>
 * Checks that the textual contents of an S3 object matches a certain value. The check will be retried for 5 minutes as long as the object is not found or the value is different. See the source code for the <a href="test/provider-framework/integration-test-fixtures/s3-assert.ts">construct</a> and <a href="test/provider-framework/integration-test-fixtures/s3-assert-handler/index.py">handler</a>.
 * <p>
 * The following example defines an <code>S3Assert</code> resource which waits until
 * <code>myfile.txt</code> in <code>myBucket</code> exists and includes the contents <code>foo bar</code>:
 * <p>
 * <blockquote><pre>
 * // This example exists only for TypeScript
 * 
 * declare const myBucket: s3.Bucket;
 * new cr.S3Assert(this, 'AssertMyFile', {
 *   bucket: myBucket,
 *   objectKey: 'myfile.txt',
 *   expectedContent: 'foo bar',
 * });
 * </pre></blockquote>
 * <p>
 * This sample demonstrates the following concepts:
 * <p>
 * <ul>
 * <li>Asynchronous implementation</li>
 * <li>Non-intrinsic physical IDs</li>
 * <li>Implemented in Python</li>
 * </ul>
 * <p>
 * <h3>Customizing Provider Function name</h3>
 * <p>
 * In multi-account environments or when the custom resource may be re-utilized across several
 * stacks it may be useful to manually set a name for the Provider Function Lambda and therefore
 * have a predefined service token ARN.
 * <p>
 * <blockquote><pre>
 * Function onEvent;
 * Function isComplete;
 * Role myRole;
 * 
 * Provider myProvider = Provider.Builder.create(this, "MyProvider")
 *         .onEventHandler(onEvent)
 *         .isCompleteHandler(isComplete)
 *         .logRetention(RetentionDays.ONE_DAY)
 *         .role(myRole)
 *         .providerFunctionName("the-lambda-name")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Custom Resources for AWS APIs</h2>
 * <p>
 * Sometimes a single API call can fill the gap in the CloudFormation coverage. In
 * this case you can use the <code>AwsCustomResource</code> construct. This construct creates
 * a custom resource that can be customized to make specific API calls for the
 * <code>CREATE</code>, <code>UPDATE</code> and <code>DELETE</code> events. Additionally, data returned by the API
 * call can be extracted and used in other constructs/resources (creating a real
 * CloudFormation dependency using <code>Fn::GetAtt</code> under the hood).
 * <p>
 * The physical id of the custom resource can be specified or derived from the data
 * returned by the API call.
 * <p>
 * The <code>AwsCustomResource</code> uses the AWS SDK for JavaScript. Services, actions and
 * parameters can be found in the <a href="https://docs.aws.amazon.com/AWSJavaScriptSDK/latest/index.html">API documentation</a>.
 * <p>
 * Path to data must be specified using a dot notation, e.g. to get the string value
 * of the <code>Title</code> attribute for the first item returned by <code>dynamodb.query</code> it should
 * be <code>Items.0.Title.S</code>.
 * <p>
 * To make sure that the newest API calls are available the latest AWS SDK v2 is installed
 * in the Lambda function implementing the custom resource. The installation takes around 60
 * seconds. If you prefer to optimize for speed, you can disable the installation by setting
 * the <code>installLatestAwsSdk</code> prop to <code>false</code>.
 * <p>
 * <h3>Custom Resource Execution Policy</h3>
 * <p>
 * You must provide the <code>policy</code> property defining the IAM Policy that will be applied to the API calls.
 * The library provides two factory methods to quickly configure this:
 * <p>
 * <ul>
 * <li><strong><code>AwsCustomResourcePolicy.fromSdkCalls</code></strong> - Use this to auto-generate IAM
 * Policy statements based on the configured SDK calls. Keep two things in mind
 * when using this policy:
 * <p>
 * <ul>
 * <li>This policy variant assumes the IAM policy name has the same name as the API
 * call. This is true in 99% of cases, but there are exceptions (for example,
 * S3's <code>PutBucketLifecycleConfiguration</code> requires
 * <code>s3:PutLifecycleConfiguration</code> permissions, Lambda's <code>Invoke</code> requires
 * <code>lambda:InvokeFunction</code> permissions). Use <code>fromStatements</code> if you want to
 * do a call that requires different IAM action names.</li>
 * <li>You will have to either provide specific ARNs, or explicitly use
 * <code>AwsCustomResourcePolicy.ANY_RESOURCE</code> to allow access to any resource.</li>
 * </ul></li>
 * <li><strong><code>AwsCustomResourcePolicy.fromStatements</code></strong> - Use this to specify your own
 * custom statements.</li>
 * </ul>
 * <p>
 * The custom resource also implements <code>iam.IGrantable</code>, making it possible to use the <code>grantXxx()</code> methods.
 * <p>
 * As this custom resource uses a singleton Lambda function, it's important to note
 * that the function's role will eventually accumulate the permissions/grants from all
 * resources.
 * <p>
 * Chained API calls can be achieved by creating dependencies:
 * <p>
 * <blockquote><pre>
 * AwsCustomResource awsCustom1 = AwsCustomResource.Builder.create(this, "API1")
 *         .onCreate(AwsSdkCall.builder()
 *                 .service("...")
 *                 .action("...")
 *                 .physicalResourceId(PhysicalResourceId.of("..."))
 *                 .build())
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * 
 * AwsCustomResource awsCustom2 = AwsCustomResource.Builder.create(this, "API2")
 *         .onCreate(AwsSdkCall.builder()
 *                 .service("...")
 *                 .action("...")
 *                 .parameters(Map.of(
 *                         "text", awsCustom1.getResponseField("Items.0.text")))
 *                 .physicalResourceId(PhysicalResourceId.of("..."))
 *                 .build())
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Physical Resource Id Parameter</h3>
 * <p>
 * Some AWS APIs may require passing the physical resource id in as a parameter for doing updates and deletes. You can pass it by using <code>PhysicalResourceIdReference</code>.
 * <p>
 * <blockquote><pre>
 * AwsCustomResource awsCustom = AwsCustomResource.Builder.create(this, "aws-custom")
 *         .onCreate(AwsSdkCall.builder()
 *                 .service("...")
 *                 .action("...")
 *                 .parameters(Map.of(
 *                         "text", "..."))
 *                 .physicalResourceId(PhysicalResourceId.of("..."))
 *                 .build())
 *         .onUpdate(AwsSdkCall.builder()
 *                 .service("...")
 *                 .action("...")
 *                 .parameters(Map.of(
 *                         "text", "...",
 *                         "resourceId", new PhysicalResourceIdReference()))
 *                 .build())
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Handling Custom Resource Errors</h3>
 * <p>
 * Every error produced by the API call is treated as is and will cause a "FAILED" response to be submitted to CloudFormation.
 * You can ignore some errors by specifying the <code>ignoreErrorCodesMatching</code> property, which accepts a regular expression that is
 * tested against the <code>code</code> property of the response. If matched, a "SUCCESS" response is submitted.
 * Note that in such a case, the call response data and the <code>Data</code> key submitted to CloudFormation would both be an empty JSON object.
 * Since a successful resource provisioning might or might not produce outputs, this presents us with some limitations:
 * <p>
 * <ul>
 * <li><code>PhysicalResourceId.fromResponse</code> - Since the call response data might be empty, we cannot use it to extract the physical id.</li>
 * <li><code>getResponseField</code> and <code>getResponseFieldReference</code> - Since the <code>Data</code> key is empty, the resource will not have any attributes, and therefore, invoking these functions will result in an error.</li>
 * </ul>
 * <p>
 * In both the cases, you will get a synth time error if you attempt to use it in conjunction with <code>ignoreErrorCodesMatching</code>.
 * <p>
 * <h3>Customizing the Lambda function implementing the custom resource</h3>
 * <p>
 * Use the <code>role</code>, <code>timeout</code>, <code>logRetention</code> and <code>functionName</code> properties to customize
 * the Lambda function implementing the custom resource:
 * <p>
 * <blockquote><pre>
 * Role myRole;
 * 
 * AwsCustomResource.Builder.create(this, "Customized")
 *         .role(myRole) // must be assumable by the `lambda.amazonaws.com` service principal
 *         .timeout(Duration.minutes(10)) // defaults to 2 minutes
 *         .logRetention(RetentionDays.ONE_WEEK) // defaults to never delete logs
 *         .functionName("my-custom-name") // defaults to a CloudFormation generated name
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Restricting the output of the Custom Resource</h3>
 * <p>
 * CloudFormation imposes a hard limit of 4096 bytes for custom resources response
 * objects. If your API call returns an object that exceeds this limit, you can restrict
 * the data returned by the custom resource to specific paths in the API response:
 * <p>
 * <blockquote><pre>
 * AwsCustomResource.Builder.create(this, "ListObjects")
 *         .onCreate(AwsSdkCall.builder()
 *                 .service("s3")
 *                 .action("listObjectsV2")
 *                 .parameters(Map.of(
 *                         "Bucket", "my-bucket"))
 *                 .physicalResourceId(PhysicalResourceId.of("id"))
 *                 .outputPaths(List.of("Contents.0.Key", "Contents.1.Key"))
 *                 .build())
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that even if you restrict the output of your custom resource you can still use any
 * path in <code>PhysicalResourceId.fromResponse()</code>.
 * <p>
 * <h3>Custom Resource Examples</h3>
 * <p>
 * <h4>Get the latest version of a secure SSM parameter</h4>
 * <p>
 * <blockquote><pre>
 * AwsCustomResource getParameter = AwsCustomResource.Builder.create(this, "GetParameter")
 *         .onUpdate(AwsSdkCall.builder() // will also be called for a CREATE event
 *                 .service("SSM")
 *                 .action("getParameter")
 *                 .parameters(Map.of(
 *                         "Name", "my-parameter",
 *                         "WithDecryption", true))
 *                 .physicalResourceId(PhysicalResourceId.of(Date.now().toString())).build())
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * 
 * // Use the value in another construct with
 * getParameter.getResponseField("Parameter.Value");
 * </pre></blockquote>
 * <p>
 * <h4>Associate a PrivateHostedZone with VPC shared from another account</h4>
 * <p>
 * <blockquote><pre>
 * AwsCustomResource getParameter = AwsCustomResource.Builder.create(this, "AssociateVPCWithHostedZone")
 *         .onCreate(AwsSdkCall.builder()
 *                 .assumedRoleArn("arn:aws:iam::OTHERACCOUNT:role/CrossAccount/ManageHostedZoneConnections")
 *                 .service("Route53")
 *                 .action("associateVPCWithHostedZone")
 *                 .parameters(Map.of(
 *                         "HostedZoneId", "hz-123",
 *                         "VPC", Map.of(
 *                                 "VPCId", "vpc-123",
 *                                 "VPCRegion", "region-for-vpc")))
 *                 .physicalResourceId(PhysicalResourceId.of("${vpcStack.SharedVpc.VpcId}-${vpcStack.Region}-${PrivateHostedZone.HostedZoneId}"))
 *                 .build())
 *         //Will ignore any resource and use the assumedRoleArn as resource and 'sts:AssumeRole' for service:action
 *         .policy(AwsCustomResourcePolicy.fromSdkCalls(SdkCallsPolicyOptions.builder()
 *                 .resources(AwsCustomResourcePolicy.ANY_RESOURCE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <hr>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 */
package software.amazon.awscdk.customresources;
