/**
 * <h1>Amazon Relational Database Service Construct Library</h1>
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.rds.*;
 * </pre></blockquote>
 * <p>
 * <h2>Starting a clustered database</h2>
 * <p>
 * To set up a clustered database (like Aurora), define a <code>DatabaseCluster</code>. You must
 * always launch a database in a VPC. Use the <code>vpcSubnets</code> attribute to control whether
 * your instances will be launched privately or publicly:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.auroraMysql(AuroraMysqlClusterEngineProps.builder().version(AuroraMysqlEngineVersion.VER_2_08_1).build()))
 *         .credentials(Credentials.fromGeneratedSecret("clusteradmin")) // Optional - will default to 'admin' username and generated password
 *         .instanceProps(InstanceProps.builder()
 *                 // optional , defaults to t3.medium
 *                 .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.SMALL))
 *                 .vpcSubnets(SubnetSelection.builder()
 *                         .subnetType(SubnetType.PRIVATE)
 *                         .build())
 *                 .vpc(vpc)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If there isn't a constant for the exact version you want to use,
 * all of the <code>Version</code> classes have a static <code>of</code> method that can be used to create an arbitrary version.
 * <p>
 * <blockquote><pre>
 * AuroraMysqlEngineVersion customEngineVersion = AuroraMysqlEngineVersion.of("5.7.mysql_aurora.2.08.1");
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager with auto-generated description.
 * <p>
 * Your cluster will be empty by default. To add a default database upon construction, specify the
 * <code>defaultDatabaseName</code> attribute.
 * <p>
 * Use <code>DatabaseClusterFromSnapshot</code> to create a cluster from a snapshot:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseClusterFromSnapshot.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.aurora(AuroraClusterEngineProps.builder().version(AuroraEngineVersion.VER_1_22_2).build()))
 *         .instanceProps(InstanceProps.builder()
 *                 .vpc(vpc)
 *                 .build())
 *         .snapshotIdentifier("mySnapshot")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Starting an instance database</h2>
 * <p>
 * To set up a instance database, define a <code>DatabaseInstance</code>. You must
 * always launch a database in a VPC. Use the <code>vpcSubnets</code> attribute to control whether
 * your instances will be launched privately or publicly:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder().version(OracleEngineVersion.VER_19_0_0_0_2020_04_R1).build()))
 *         // optional, defaults to m5.large
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE3, InstanceSize.SMALL))
 *         .credentials(Credentials.fromGeneratedSecret("syscdk")) // Optional - will default to 'admin' username and generated password
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If there isn't a constant for the exact engine version you want to use,
 * all of the <code>Version</code> classes have a static <code>of</code> method that can be used to create an arbitrary version.
 * <p>
 * <blockquote><pre>
 * OracleEngineVersion customEngineVersion = OracleEngineVersion.of("19.0.0.0.ru-2020-04.rur-2020-04.r1", "19");
 * </pre></blockquote>
 * <p>
 * By default, the master password will be generated and stored in AWS Secrets Manager.
 * <p>
 * To use the storage auto scaling option of RDS you can specify the maximum allocated storage.
 * This is the upper limit to which RDS can automatically scale the storage. More info can be found
 * <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PIOPS.StorageTypes.html#USER_PIOPS.Autoscaling">here</a>
 * Example for max storage configuration:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_12_3).build()))
 *         // optional, defaults to m5.large
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.SMALL))
 *         .vpc(vpc)
 *         .maxAllocatedStorage(200)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>DatabaseInstanceFromSnapshot</code> and <code>DatabaseInstanceReadReplica</code> to create an instance from snapshot or
 * a source database respectively:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance sourceInstance;
 * 
 * DatabaseInstanceFromSnapshot.Builder.create(this, "Instance")
 *         .snapshotIdentifier("my-snapshot")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_12_3).build()))
 *         // optional, defaults to m5.large
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .build();
 * DatabaseInstanceReadReplica.Builder.create(this, "ReadReplica")
 *         .sourceDatabaseInstance(sourceInstance)
 *         .instanceType(InstanceType.of(InstanceClass.BURSTABLE2, InstanceSize.LARGE))
 *         .vpc(vpc)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Automatic backups of read replica instances are only supported for MySQL and MariaDB. By default,
 * automatic backups are disabled for read replicas and can only be enabled (using <code>backupRetention</code>)
 * if also enabled on the source instance.
 * <p>
 * Creating a "production" Oracle database instance with option and parameter groups:
 * <p>
 * <a href="test/integ.instance.lit.ts">example of setting up a production oracle instance</a>
 * <p>
 * <h2>Setting Public Accessibility</h2>
 * <p>
 * You can set public accessibility for the database instance or cluster using the <code>publiclyAccessible</code> property.
 * If you specify <code>true</code>, it creates an instance with a publicly resolvable DNS name, which resolves to a public IP address.
 * If you specify <code>false</code>, it creates an internal instance with a DNS name that resolves to a private IP address.
 * The default value depends on <code>vpcSubnets</code>.
 * It will be <code>true</code> if <code>vpcSubnets</code> is <code>subnetType: SubnetType.PUBLIC</code>, <code>false</code> otherwise.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * // Setting public accessibility for DB instance
 * // Setting public accessibility for DB instance
 * DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder()
 *                 .version(MysqlEngineVersion.VER_8_0_19)
 *                 .build()))
 *         .vpc(vpc)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .subnetType(SubnetType.PRIVATE)
 *                 .build())
 *         .publiclyAccessible(true)
 *         .build();
 * 
 * // Setting public accessibility for DB cluster
 * // Setting public accessibility for DB cluster
 * DatabaseCluster.Builder.create(this, "DatabaseCluster")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .instanceProps(InstanceProps.builder()
 *                 .vpc(vpc)
 *                 .vpcSubnets(SubnetSelection.builder()
 *                         .subnetType(SubnetType.PRIVATE)
 *                         .build())
 *                 .publiclyAccessible(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Instance events</h2>
 * <p>
 * To define Amazon CloudWatch event rules for database instances, use the <code>onEvent</code>
 * method:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * Function fn;
 * 
 * Rule rule = instance.onEvent("InstanceEvent", OnEventOptions.builder().target(new LambdaFunction(fn)).build());
 * </pre></blockquote>
 * <p>
 * <h2>Login credentials</h2>
 * <p>
 * By default, database instances and clusters will have <code>admin</code> user with an auto-generated password.
 * An alternative username (and password) may be specified for the admin user instead of the default.
 * <p>
 * The following examples use a <code>DatabaseInstance</code>, but the same usage is applicable to <code>DatabaseCluster</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * IInstanceEngine engine = DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_12_3).build());
 * DatabaseInstance.Builder.create(this, "InstanceWithUsername")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromGeneratedSecret("postgres"))
 *         .build();
 * 
 * DatabaseInstance.Builder.create(this, "InstanceWithUsernameAndPassword")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromPassword("postgres", SecretValue.ssmSecure("/dbPassword", "1")))
 *         .build();
 * 
 * ISecret mySecret = Secret.fromSecretName(this, "DBSecret", "myDBLoginInfo");
 * DatabaseInstance.Builder.create(this, "InstanceWithSecretLogin")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromSecret(mySecret))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Secrets generated by <code>fromGeneratedSecret()</code> can be customized:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * IInstanceEngine engine = DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder().version(PostgresEngineVersion.VER_12_3).build());
 * Key myKey = new Key(this, "MyKey");
 * 
 * DatabaseInstance.Builder.create(this, "InstanceWithCustomizedSecret")
 *         .engine(engine)
 *         .vpc(vpc)
 *         .credentials(Credentials.fromGeneratedSecret("postgres", CredentialsBaseOptions.builder()
 *                 .secretName("my-cool-name")
 *                 .encryptionKey(myKey)
 *                 .excludeCharacters("!&amp;*^#&#64;()")
 *                 .replicaRegions(List.of(ReplicaRegion.builder().region("eu-west-1").build(), ReplicaRegion.builder().region("eu-west-2").build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Connecting</h2>
 * <p>
 * To control who can access the cluster or instance, use the <code>.connections</code> attribute. RDS databases have
 * a default port, so you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * cluster.connections.allowFromAnyIpv4(Port.allTraffic(), "Open to the world");
 * </pre></blockquote>
 * <p>
 * The endpoints to access your database cluster will be available as the <code>.clusterEndpoint</code> and <code>.readerEndpoint</code>
 * attributes:
 * <p>
 * <blockquote><pre>
 * DatabaseCluster cluster;
 * 
 * String writeAddress = cluster.getClusterEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * For an instance database:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * 
 * String address = instance.getInstanceEndpoint().getSocketAddress();
 * </pre></blockquote>
 * <p>
 * <h2>Rotating credentials</h2>
 * <p>
 * When the master password is generated and stored in AWS Secrets Manager, it can be rotated automatically:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * DatabaseInstance instance;
 * 
 * instance.addRotationSingleUser(RotationSingleUserOptions.builder()
 *         .automaticallyAfter(Duration.days(7)) // defaults to 30 days
 *         .excludeCharacters("!&#64;#$%^&amp;*")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <a href="test/integ.cluster-rotation.lit.ts">example of setting up master password rotation for a cluster</a>
 * <p>
 * The multi user rotation scheme is also available:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * DatabaseSecret myImportedSecret;
 * 
 * instance.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder()
 *         .secret(myImportedSecret)
 *         .build());
 * </pre></blockquote>
 * <p>
 * It's also possible to create user credentials together with the instance/cluster and add rotation:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * 
 * DatabaseSecret myUserSecret = DatabaseSecret.Builder.create(this, "MyUserSecret")
 *         .username("myuser")
 *         .secretName("my-user-secret") // optional, defaults to a CloudFormation-generated name
 *         .masterSecret(instance.getSecret())
 *         .excludeCharacters("{}[]()'\"/\\")
 *         .build();
 * ISecret myUserSecretAttached = myUserSecret.attach(instance); // Adds DB connections information in the secret
 * 
 * instance.addRotationMultiUser("MyUser", RotationMultiUserOptions.builder() // Add rotation using the multi user scheme
 *         .secret(myUserSecretAttached).build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: This user must be created manually in the database using the master credentials.
 * The rotation will start as soon as this user exists.
 * <p>
 * Access to the Secrets Manager API is required for the secret rotation. This can be achieved either with
 * internet connectivity (through NAT) or with a VPC interface endpoint. By default, the rotation Lambda function
 * is deployed in the same subnets as the instance/cluster. If access to the Secrets Manager API is not possible from
 * those subnets or using the default API endpoint, use the <code>vpcSubnets</code> and/or <code>endpoint</code> options:
 * <p>
 * <blockquote><pre>
 * DatabaseInstance instance;
 * InterfaceVpcEndpoint myEndpoint;
 * 
 * 
 * instance.addRotationSingleUser(RotationSingleUserOptions.builder()
 *         .vpcSubnets(SubnetSelection.builder().subnetType(SubnetType.PRIVATE_WITH_NAT).build()) // Place rotation Lambda in private subnets
 *         .endpoint(myEndpoint)
 *         .build());
 * </pre></blockquote>
 * <p>
 * See also <a href="https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-secretsmanager/README.md">&#64;aws-cdk/aws-secretsmanager</a> for credentials rotation of existing clusters/instances.
 * <p>
 * <h2>IAM Authentication</h2>
 * <p>
 * You can also authenticate to a database instance using AWS Identity and Access Management (IAM) database authentication;
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.html</a> for more information
 * and a list of supported versions and limitations.
 * <p>
 * The following example shows enabling IAM authentication for a database instance and granting connection access to an IAM role.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_19).build()))
 *         .vpc(vpc)
 *         .iamAuthentication(true)
 *         .build();
 * Role role = Role.Builder.create(this, "DBRole").assumedBy(new AccountPrincipal(this.account)).build();
 * instance.grantConnect(role);
 * </pre></blockquote>
 * <p>
 * The following example shows granting connection access for RDS Proxy to an IAM role.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .instanceProps(InstanceProps.builder().vpc(vpc).build())
 *         .build();
 * 
 * DatabaseProxy proxy = DatabaseProxy.Builder.create(this, "Proxy")
 *         .proxyTarget(ProxyTarget.fromCluster(cluster))
 *         .secrets(List.of(cluster.getSecret()))
 *         .vpc(vpc)
 *         .build();
 * 
 * Role role = Role.Builder.create(this, "DBProxyRole").assumedBy(new AccountPrincipal(this.account)).build();
 * proxy.grantConnect(role, "admin");
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: In addition to the setup above, a database user will need to be created to support IAM auth.
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/UsingWithRDS.IAMDBAuth.DBAccounts.html</a> for setup instructions.
 * <p>
 * <h2>Kerberos Authentication</h2>
 * <p>
 * You can also authenticate using Kerberos to a database instance using AWS Managed Microsoft AD for authentication;
 * See <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html</a> for more information
 * and a list of supported versions and limitations.
 * <p>
 * The following example shows enabling domain support for a database instance and creating an IAM role to access
 * Directory Services.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * Role role = Role.Builder.create(this, "RDSDirectoryServicesRole")
 *         .assumedBy(new ServicePrincipal("rds.amazonaws.com"))
 *         .managedPolicies(List.of(ManagedPolicy.fromAwsManagedPolicyName("service-role/AmazonRDSDirectoryServiceAccess")))
 *         .build();
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.mysql(MySqlInstanceEngineProps.builder().version(MysqlEngineVersion.VER_8_0_19).build()))
 *         .vpc(vpc)
 *         .domain("d-????????") // The ID of the domain for the instance to join.
 *         .domainRole(role)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: In addition to the setup above, you need to make sure that the database instance has network connectivity
 * to the domain controllers. This includes enabling cross-VPC traffic if in a different VPC and setting up the
 * appropriate security groups/network ACL to allow traffic between the database instance and domain controllers.
 * Once configured, see <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html">https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/kerberos-authentication.html</a> for details
 * on configuring users for each available database engine.
 * <p>
 * <h2>Metrics</h2>
 * <p>
 * Database instances and clusters both expose metrics (<code>cloudwatch.Metric</code>):
 * <p>
 * <blockquote><pre>
 * // The number of database connections in use (average over 5 minutes)
 * DatabaseInstance instance;
 * 
 * // Average CPU utilization over 5 minutes
 * DatabaseCluster cluster;
 * 
 * Metric dbConnections = instance.metricDatabaseConnections();
 * Metric cpuUtilization = cluster.metricCPUUtilization();
 * 
 * // The average amount of time taken per disk I/O operation (average over 1 minute)
 * Metric readLatency = instance.metric("ReadLatency", MetricOptions.builder().statistic("Average").period(Duration.seconds(60)).build());
 * </pre></blockquote>
 * <p>
 * <h2>Enabling S3 integration</h2>
 * <p>
 * Data in S3 buckets can be imported to and exported from certain database engines using SQL queries. To enable this
 * functionality, set the <code>s3ImportBuckets</code> and <code>s3ExportBuckets</code> properties for import and export respectively. When
 * configured, the CDK automatically creates and configures IAM roles as required.
 * Additionally, the <code>s3ImportRole</code> and <code>s3ExportRole</code> properties can be used to set this role directly.
 * <p>
 * You can read more about loading data to (or from) S3 here:
 * <p>
 * <ul>
 * <li>Aurora MySQL - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html">import</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html">export</a>.</li>
 * <li>Aurora PostgreSQL - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html#USER_PostgreSQL.S3Import">import</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html">export</a>.</li>
 * <li>Microsoft SQL Server - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/SQLServer.Procedural.Importing.html">import and export</a></li>
 * <li>PostgreSQL - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/PostgreSQL.Procedural.Importing.html">import</a>
 * and <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/postgresql-s3-export.html">export</a></li>
 * <li>Oracle - <a href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html">import and export</a></li>
 * </ul>
 * <p>
 * The following snippet sets up a database cluster with different S3 buckets where the data is imported and exported -
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * 
 * Vpc vpc;
 * 
 * Bucket importBucket = new Bucket(this, "importbucket");
 * Bucket exportBucket = new Bucket(this, "exportbucket");
 * DatabaseCluster.Builder.create(this, "dbcluster")
 *         .engine(DatabaseClusterEngine.AURORA)
 *         .instanceProps(InstanceProps.builder()
 *                 .vpc(vpc)
 *                 .build())
 *         .s3ImportBuckets(List.of(importBucket))
 *         .s3ExportBuckets(List.of(exportBucket))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Creating a Database Proxy</h2>
 * <p>
 * Amazon RDS Proxy sits between your application and your relational database to efficiently manage
 * connections to the database and improve scalability of the application. Learn more about at <a href="https://aws.amazon.com/rds/proxy/">Amazon RDS Proxy</a>
 * <p>
 * The following code configures an RDS Proxy for a <code>DatabaseInstance</code>.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * Secret[] secrets;
 * DatabaseInstance dbInstance;
 * 
 * 
 * DatabaseProxy proxy = dbInstance.addProxy("proxy", DatabaseProxyOptions.builder()
 *         .borrowTimeout(Duration.seconds(30))
 *         .maxConnectionsPercent(50)
 *         .secrets(secrets)
 *         .vpc(vpc)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Exporting Logs</h2>
 * <p>
 * You can publish database logs to Amazon CloudWatch Logs. With CloudWatch Logs, you can perform real-time analysis of the log data,
 * store the data in highly durable storage, and manage the data with the CloudWatch Logs Agent. This is available for both database
 * instances and clusters; the types of logs available depend on the database type and engine being used.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * Role myLogsPublishingRole;
 * Vpc vpc;
 * 
 * 
 * // Exporting logs from a cluster
 * DatabaseCluster cluster = DatabaseCluster.Builder.create(this, "Database")
 *         .engine(DatabaseClusterEngine.aurora(AuroraClusterEngineProps.builder()
 *                 .version(AuroraEngineVersion.VER_1_17_9)
 *                 .build()))
 *         .instanceProps(InstanceProps.builder()
 *                 .vpc(vpc)
 *                 .build())
 *         .cloudwatchLogsExports(List.of("error", "general", "slowquery", "audit")) // Export all available MySQL-based logs
 *         .cloudwatchLogsRetention(RetentionDays.THREE_MONTHS) // Optional - default is to never expire logs
 *         .cloudwatchLogsRetentionRole(myLogsPublishingRole)
 *         .build();
 * 
 * // Exporting logs from an instance
 * DatabaseInstance instance = DatabaseInstance.Builder.create(this, "Instance")
 *         .engine(DatabaseInstanceEngine.postgres(PostgresInstanceEngineProps.builder()
 *                 .version(PostgresEngineVersion.VER_12_3)
 *                 .build()))
 *         .vpc(vpc)
 *         .cloudwatchLogsExports(List.of("postgresql"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Option Groups</h2>
 * <p>
 * Some DB engines offer additional features that make it easier to manage data and databases, and to provide additional security for your database.
 * Amazon RDS uses option groups to enable and configure these features. An option group can specify features, called options,
 * that are available for a particular Amazon RDS DB instance.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * SecurityGroup securityGroup;
 * 
 * 
 * OptionGroup.Builder.create(this, "Options")
 *         .engine(DatabaseInstanceEngine.oracleSe2(OracleSe2InstanceEngineProps.builder()
 *                 .version(OracleEngineVersion.VER_19)
 *                 .build()))
 *         .configurations(List.of(OptionConfiguration.builder()
 *                 .name("OEM")
 *                 .port(5500)
 *                 .vpc(vpc)
 *                 .securityGroups(List.of(securityGroup))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Serverless</h2>
 * <p>
 * <a href="https://aws.amazon.com/rds/aurora/serverless/">Amazon Aurora Serverless</a> is an on-demand, auto-scaling configuration for Amazon
 * Aurora. The database will automatically start up, shut down, and scale capacity
 * up or down based on your application's needs. It enables you to run your database
 * in the cloud without managing any database instances.
 * <p>
 * The following example initializes an Aurora Serverless PostgreSql cluster.
 * Aurora Serverless clusters can specify scaling properties which will be used to
 * automatically scale the database cluster seamlessly based on the workload.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * ServerlessCluster cluster = ServerlessCluster.Builder.create(this, "AnotherCluster")
 *         .engine(DatabaseClusterEngine.AURORA_POSTGRESQL)
 *         .parameterGroup(ParameterGroup.fromParameterGroupName(this, "ParameterGroup", "default.aurora-postgresql10"))
 *         .vpc(vpc)
 *         .scaling(ServerlessScalingOptions.builder()
 *                 .autoPause(Duration.minutes(10)) // default is to pause after 5 minutes of idle time
 *                 .minCapacity(AuroraCapacityUnit.ACU_8) // default is 2 Aurora capacity units (ACUs)
 *                 .maxCapacity(AuroraCapacityUnit.ACU_32)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Aurora Serverless Clusters do not support the following features:
 * <p>
 * <ul>
 * <li>Loading data from an Amazon S3 bucket</li>
 * <li>Saving data to an Amazon S3 bucket</li>
 * <li>Invoking an AWS Lambda function with an Aurora MySQL native function</li>
 * <li>Aurora replicas</li>
 * <li>Backtracking</li>
 * <li>Multi-master clusters</li>
 * <li>Database cloning</li>
 * <li>IAM database cloning</li>
 * <li>IAM database authentication</li>
 * <li>Restoring a snapshot from MySQL DB instance</li>
 * <li>Performance Insights</li>
 * <li>RDS Proxy</li>
 * </ul>
 * <p>
 * Read more about the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html#aurora-serverless.limitations">limitations of Aurora Serverless</a>
 * <p>
 * Learn more about using Amazon Aurora Serverless by reading the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/aurora-serverless.html">documentation</a>
 * <p>
 * Use <code>ServerlessClusterFromSnapshot</code> to create a serverless cluster from a snapshot:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * ServerlessClusterFromSnapshot.Builder.create(this, "Cluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc)
 *         .snapshotIdentifier("mySnapshot")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Data API</h3>
 * <p>
 * You can access your Aurora Serverless DB cluster using the built-in Data API. The Data API doesn't require a persistent connection to the DB cluster. Instead, it provides a secure HTTP endpoint and integration with AWS SDKs.
 * <p>
 * The following example shows granting Data API access to a Lamba function.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * Code code;
 * 
 * 
 * ServerlessCluster cluster = ServerlessCluster.Builder.create(this, "AnotherCluster")
 *         .engine(DatabaseClusterEngine.AURORA_MYSQL)
 *         .vpc(vpc)
 *         .enableDataApi(true)
 *         .build();
 * Function fn = Function.Builder.create(this, "MyFunction")
 *         .runtime(Runtime.NODEJS_12_X)
 *         .handler("index.handler")
 *         .code(code)
 *         .environment(Map.of(
 *                 "CLUSTER_ARN", cluster.getClusterArn(),
 *                 "SECRET_ARN", cluster.getSecret().getSecretArn()))
 *         .build();
 * cluster.grantDataApiAccess(fn);
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: To invoke the Data API, the resource will need to read the secret associated with the cluster.
 * <p>
 * To learn more about using the Data API, see the <a href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html">documentation</a>.
 */
package software.amazon.awscdk.services.rds;
