/**
 * <h1>Amazon ECR Construct Library</h1>
 * <p>
 * This package contains constructs for working with Amazon Elastic Container Registry.
 * <p>
 * <h2>Repositories</h2>
 * <p>
 * Define a repository by creating a new instance of <code>Repository</code>. A repository
 * holds multiple verions of a single container image.
 * <p>
 * <blockquote><pre>
 * Repository repository = new Repository(this, "Repository");
 * </pre></blockquote>
 * <p>
 * <h2>Image scanning</h2>
 * <p>
 * Amazon ECR image scanning helps in identifying software vulnerabilities in your container images. You can manually scan container images stored in Amazon ECR, or you can configure your repositories to scan images when you push them to a repository. To create a new repository to scan on push, simply enable <code>imageScanOnPush</code> in the properties
 * <p>
 * <blockquote><pre>
 * Repository repository = Repository.Builder.create(this, "Repo")
 *         .imageScanOnPush(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * To create an <code>onImageScanCompleted</code> event rule and trigger the event target
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * SomeTarget target;
 * 
 * 
 * repository.onImageScanCompleted("ImageScanComplete").addTarget(target);
 * </pre></blockquote>
 * <p>
 * <h3>Authorization Token</h3>
 * <p>
 * Besides the Amazon ECR APIs, ECR also allows the Docker CLI or a language-specific Docker library to push and pull
 * images from an ECR repository. However, the Docker CLI does not support native IAM authentication methods and
 * additional steps must be taken so that Amazon ECR can authenticate and authorize Docker push and pull requests.
 * More information can be found at at <a href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/Registries.html#registry_auth">Registry Authentication</a>.
 * <p>
 * A Docker authorization token can be obtained using the <code>GetAuthorizationToken</code> ECR API. The following code snippets
 * grants an IAM user access to call this API.
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "User");
 * AuthorizationToken.grantRead(user);
 * </pre></blockquote>
 * <p>
 * If you access images in the <a href="https://gallery.ecr.aws/">Public ECR Gallery</a> as well, it is recommended you authenticate to the registry to benefit from
 * higher rate and bandwidth limits.
 * <p>
 * <blockquote>
 * <p>
 * See <code>Pricing</code> in https://aws.amazon.com/blogs/aws/amazon-ecr-public-a-new-public-container-registry/ and <a href="https://docs.aws.amazon.com/AmazonECR/latest/public/public-service-quotas.html">Service quotas</a>.
 * <p>
 * </blockquote>
 * <p>
 * The following code snippet grants an IAM user access to retrieve an authorization token for the public gallery.
 * <p>
 * <blockquote><pre>
 * User user = new User(this, "User");
 * PublicGalleryAuthorizationToken.grantRead(user);
 * </pre></blockquote>
 * <p>
 * This user can then proceed to login to the registry using one of the <a href="https://docs.aws.amazon.com/AmazonECR/latest/public/public-registries.html#public-registry-auth">authentication methods</a>.
 * <p>
 * <h3>Image tag immutability</h3>
 * <p>
 * You can set tag immutability on images in our repository using the <code>imageTagMutability</code> construct prop.
 * <p>
 * <blockquote><pre>
 * Repository.Builder.create(this, "Repo").imageTagMutability(TagMutability.IMMUTABLE).build();
 * </pre></blockquote>
 * <p>
 * <h2>Automatically clean up repositories</h2>
 * <p>
 * You can set life cycle rules to automatically clean up old images from your
 * repository. The first life cycle rule that matches an image will be applied
 * against that image. For example, the following deletes images older than
 * 30 days, while keeping all images tagged with prod (note that the order
 * is important here):
 * <p>
 * <blockquote><pre>
 * Repository repository;
 * 
 * repository.addLifecycleRule(LifecycleRule.builder().tagPrefixList(List.of("prod")).maxImageCount(9999).build());
 * repository.addLifecycleRule(LifecycleRule.builder().maxImageAge(Duration.days(30)).build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ecr;
