/**
 * <h1>AWS App Mesh Construct Library</h1>
 * <p>
 * AWS App Mesh is a service mesh based on the <a href="https://www.envoyproxy.io/">Envoy</a> proxy that makes it easy to monitor and control microservices. App Mesh standardizes how your microservices communicate, giving you end-to-end visibility and helping to ensure high-availability for your applications.
 * <p>
 * App Mesh gives you consistent visibility and network traffic controls for every microservice in an application.
 * <p>
 * App Mesh supports microservice applications that use service discovery naming for their components. To use App Mesh, you must have an existing application running on AWS Fargate, Amazon ECS, Amazon EKS, Kubernetes on AWS, or Amazon EC2.
 * <p>
 * For further information on <strong>AWS App Mesh</strong>, visit the <a href="https://docs.aws.amazon.com/app-mesh/index.html">AWS App Mesh Documentation</a>.
 * <p>
 * <h2>Create the App and Stack</h2>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * App app = new App();
 * Stack stack = new Stack(app, "stack");
 * </pre></blockquote>
 * <p>
 * <h2>Creating the Mesh</h2>
 * <p>
 * A service mesh is a logical boundary for network traffic between the services that reside within it.
 * <p>
 * After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.
 * <p>
 * The following example creates the <code>AppMesh</code> service mesh with the default egress filter of <code>DROP_ALL</code>. See <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html">the AWS CloudFormation <code>EgressFilter</code> resource</a> for more info on egress filters.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh = Mesh.Builder.create(this, "AppMesh")
 *         .meshName("myAwsMesh")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The mesh can instead be created with the <code>ALLOW_ALL</code> egress filter by providing the <code>egressFilter</code> property.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh = Mesh.Builder.create(this, "AppMesh")
 *         .meshName("myAwsMesh")
 *         .egressFilter(MeshFilterType.ALLOW_ALL)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding VirtualRouters</h2>
 * <p>
 * A <em>mesh</em> uses  <em>virtual routers</em> as logical units to route requests to <em>virtual nodes</em>.
 * <p>
 * Virtual routers handle traffic for one or more virtual services within your mesh.
 * After you create a virtual router, you can create and associate routes to your virtual router that direct incoming requests to different virtual nodes.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * 
 * VirtualRouter router = mesh.addVirtualRouter("router", VirtualRouterBaseProps.builder()
 *         .listeners(List.of(VirtualRouterListener.http(8080)))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Note that creating the router using the <code>addVirtualRouter()</code> method places it in the same stack as the mesh
 * (which might be different from the current stack).
 * The router can also be created using the <code>VirtualRouter</code> constructor (passing in the mesh) instead of calling the <code>addVirtualRouter()</code> method.
 * This is particularly useful when splitting your resources between many stacks: for example, defining the mesh itself as part of an infrastructure stack, but defining the other resources, such as routers, in the application stack:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Stack infraStack;
 * Stack appStack;
 * 
 * 
 * Mesh mesh = Mesh.Builder.create(infraStack, "AppMesh")
 *         .meshName("myAwsMesh")
 *         .egressFilter(MeshFilterType.ALLOW_ALL)
 *         .build();
 * 
 * // the VirtualRouter will belong to 'appStack',
 * // even though the Mesh belongs to 'infraStack'
 * VirtualRouter router = VirtualRouter.Builder.create(appStack, "router")
 *         .mesh(mesh) // notice that mesh is a required property when creating a router with the 'new' statement
 *         .listeners(List.of(VirtualRouterListener.http(8081)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The same is true for other <code>add*()</code> methods in the App Mesh construct library.
 * <p>
 * The <code>VirtualRouterListener</code> class lets you define protocol-specific listeners.
 * The <code>http()</code>, <code>http2()</code>, <code>grpc()</code> and <code>tcp()</code> methods create listeners for the named protocols.
 * They accept a single parameter that defines the port to on which requests will be matched.
 * The port parameter defaults to 8080 if omitted.
 * <p>
 * <h2>Adding a VirtualService</h2>
 * <p>
 * A <em>virtual service</em> is an abstraction of a real service that is provided by a virtual node directly, or indirectly by means of a virtual router. Dependent services call your virtual service by its <code>virtualServiceName</code>, and those requests are routed to the virtual node or virtual router specified as the provider for the virtual service.
 * <p>
 * We recommend that you use the service discovery name of the real service that you're targeting (such as <code>my-service.default.svc.cluster.local</code>).
 * <p>
 * When creating a virtual service:
 * <p>
 * <ul>
 * <li>If you want the virtual service to spread traffic across multiple virtual nodes, specify a virtual router.</li>
 * <li>If you want the virtual service to reach a virtual node directly, without a virtual router, specify a virtual node.</li>
 * </ul>
 * <p>
 * Adding a virtual router as the provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * 
 * 
 * VirtualService.Builder.create(this, "virtual-service")
 *         .virtualServiceName("my-service.default.svc.cluster.local") // optional
 *         .virtualServiceProvider(VirtualServiceProvider.virtualRouter(router))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Adding a virtual node as the provider:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualNode node;
 * 
 * 
 * VirtualService.Builder.create(this, "virtual-service")
 *         .virtualServiceName("my-service.default.svc.cluster.local") // optional
 *         .virtualServiceProvider(VirtualServiceProvider.virtualNode(node))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding a VirtualNode</h2>
 * <p>
 * A <em>virtual node</em> acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment.
 * <p>
 * When you create a virtual node, accept inbound traffic by specifying a <em>listener</em>. Outbound traffic that your virtual node expects to send should be specified as a <em>back end</em>.
 * <p>
 * The response metadata for your new virtual node contains the Amazon Resource Name (ARN) that is associated with the virtual node. Set this value (either the full ARN or the truncated resource name) as the <code>APPMESH_VIRTUAL_NODE_NAME</code> environment variable for your task group's Envoy proxy container in your task definition or pod spec. For example, the value could be <code>mesh/default/virtualNode/simpleapp</code>. This is then mapped to the <code>node.id</code> and <code>node.cluster</code> Envoy parameters.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note</strong>
 * If you require your Envoy stats or tracing to use a different name, you can override the <code>node.cluster</code> value that is set by <code>APPMESH_VIRTUAL_NODE_NAME</code> with the <code>APPMESH_VIRTUAL_NODE_CLUSTER</code> environment variable.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * Vpc vpc = new Vpc(this, "vpc");
 * PrivateDnsNamespace namespace = PrivateDnsNamespace.Builder.create(this, "test-namespace")
 *         .vpc(vpc)
 *         .name("domain.local")
 *         .build();
 * Service service = namespace.createService("Svc");
 * VirtualNode node = mesh.addVirtualNode("virtual-node", VirtualNodeBaseProps.builder()
 *         .serviceDiscovery(ServiceDiscovery.cloudMap(service))
 *         .listeners(List.of(VirtualNodeListener.http(HttpVirtualNodeListenerOptions.builder()
 *                 .port(8081)
 *                 .healthCheck(HealthCheck.http(HttpHealthCheckOptions.builder()
 *                         .healthyThreshold(3)
 *                         .interval(Duration.seconds(5)) // minimum
 *                         .path("/health-check-path")
 *                         .timeout(Duration.seconds(2)) // minimum
 *                         .unhealthyThreshold(2)
 *                         .build()))
 *                 .build())))
 *         .accessLog(AccessLog.fromFilePath("/dev/stdout"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Create a <code>VirtualNode</code> with the constructor and add tags.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * Service service;
 * 
 * 
 * VirtualNode node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(ServiceDiscovery.cloudMap(service))
 *         .listeners(List.of(VirtualNodeListener.http(HttpVirtualNodeListenerOptions.builder()
 *                 .port(8080)
 *                 .healthCheck(HealthCheck.http(HttpHealthCheckOptions.builder()
 *                         .healthyThreshold(3)
 *                         .interval(Duration.seconds(5))
 *                         .path("/ping")
 *                         .timeout(Duration.seconds(2))
 *                         .unhealthyThreshold(2)
 *                         .build()))
 *                 .timeout(HttpTimeout.builder()
 *                         .idle(Duration.seconds(5))
 *                         .build())
 *                 .build())))
 *         .backendDefaults(BackendDefaults.builder()
 *                 .tlsClientPolicy(TlsClientPolicy.builder()
 *                         .validation(TlsValidation.builder()
 *                                 .trust(TlsValidationTrust.file("/keys/local_cert_chain.pem"))
 *                                 .build())
 *                         .build())
 *                 .build())
 *         .accessLog(AccessLog.fromFilePath("/dev/stdout"))
 *         .build();
 * 
 * Tags.of(node).add("Environment", "Dev");
 * </pre></blockquote>
 * <p>
 * Create a <code>VirtualNode</code> with the constructor and add backend virtual service.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * VirtualRouter router;
 * Service service;
 * 
 * 
 * VirtualNode node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(ServiceDiscovery.cloudMap(service))
 *         .listeners(List.of(VirtualNodeListener.http(HttpVirtualNodeListenerOptions.builder()
 *                 .port(8080)
 *                 .healthCheck(HealthCheck.http(HttpHealthCheckOptions.builder()
 *                         .healthyThreshold(3)
 *                         .interval(Duration.seconds(5))
 *                         .path("/ping")
 *                         .timeout(Duration.seconds(2))
 *                         .unhealthyThreshold(2)
 *                         .build()))
 *                 .timeout(HttpTimeout.builder()
 *                         .idle(Duration.seconds(5))
 *                         .build())
 *                 .build())))
 *         .accessLog(AccessLog.fromFilePath("/dev/stdout"))
 *         .build();
 * 
 * VirtualService virtualService = VirtualService.Builder.create(this, "service-1")
 *         .virtualServiceProvider(VirtualServiceProvider.virtualRouter(router))
 *         .virtualServiceName("service1.domain.local")
 *         .build();
 * 
 * node.addBackend(Backend.virtualService(virtualService));
 * </pre></blockquote>
 * <p>
 * The <code>listeners</code> property can be left blank and added later with the <code>node.addListener()</code> method. The <code>serviceDiscovery</code> property must be specified when specifying a listener.
 * <p>
 * The <code>backends</code> property can be added with <code>node.addBackend()</code>. In the example, we define a virtual service and add it to the virtual node to allow egress traffic to other nodes.
 * <p>
 * The <code>backendDefaults</code> property is added to the node while creating the virtual node. These are the virtual node's default settings for all backends.
 * <p>
 * The <code>VirtualNode.addBackend()</code> method is especially useful if you want to create a circular traffic flow by having a Virtual Service as a backend whose provider is that same Virtual Node:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * 
 * 
 * VirtualNode node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(ServiceDiscovery.dns("node"))
 *         .build();
 * 
 * VirtualService virtualService = VirtualService.Builder.create(this, "service-1")
 *         .virtualServiceProvider(VirtualServiceProvider.virtualNode(node))
 *         .virtualServiceName("service1.domain.local")
 *         .build();
 * 
 * node.addBackend(Backend.virtualService(virtualService));
 * </pre></blockquote>
 * <p>
 * <h3>Adding TLS to a listener</h3>
 * <p>
 * The <code>tls</code> property specifies TLS configuration when creating a listener for a virtual node or a virtual gateway.
 * Provide the TLS certificate to the proxy in one of the following ways:
 * <p>
 * <ul>
 * <li>A certificate from AWS Certificate Manager (ACM).</li>
 * <li>A customer-provided certificate (specify a <code>certificateChain</code> path file and a <code>privateKey</code> file path).</li>
 * <li>A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its <code>secretName</code>).</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * // A Virtual Node with listener TLS from an ACM provided certificate
 * Certificate cert;
 * Mesh mesh;
 * 
 * 
 * VirtualNode node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         .serviceDiscovery(ServiceDiscovery.dns("node"))
 *         .listeners(List.of(VirtualNodeListener.grpc(GrpcVirtualNodeListenerOptions.builder()
 *                 .port(80)
 *                 .tls(ListenerTlsOptions.builder()
 *                         .mode(TlsMode.STRICT)
 *                         .certificate(TlsCertificate.acm(cert))
 *                         .build())
 *                 .build())))
 *         .build();
 * 
 * // A Virtual Gateway with listener TLS from a customer provided file certificate
 * VirtualGateway gateway = VirtualGateway.Builder.create(this, "gateway")
 *         .mesh(mesh)
 *         .listeners(List.of(VirtualGatewayListener.grpc(GrpcGatewayListenerOptions.builder()
 *                 .port(8080)
 *                 .tls(ListenerTlsOptions.builder()
 *                         .mode(TlsMode.STRICT)
 *                         .certificate(TlsCertificate.file("path/to/certChain", "path/to/privateKey"))
 *                         .build())
 *                 .build())))
 *         .virtualGatewayName("gateway")
 *         .build();
 * 
 * // A Virtual Gateway with listener TLS from a SDS provided certificate
 * VirtualGateway gateway2 = VirtualGateway.Builder.create(this, "gateway2")
 *         .mesh(mesh)
 *         .listeners(List.of(VirtualGatewayListener.http2(Http2GatewayListenerOptions.builder()
 *                 .port(8080)
 *                 .tls(ListenerTlsOptions.builder()
 *                         .mode(TlsMode.STRICT)
 *                         .certificate(TlsCertificate.sds("secrete_certificate"))
 *                         .build())
 *                 .build())))
 *         .virtualGatewayName("gateway2")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Adding mutual TLS authentication</h3>
 * <p>
 * Mutual TLS authentication is an optional component of TLS that offers two-way peer authentication.
 * To enable mutual TLS authentication, add the <code>mutualTlsCertificate</code> property to TLS client policy and/or the <code>mutualTlsValidation</code> property to your TLS listener.
 * <p>
 * <code>tls.mutualTlsValidation</code> and <code>tlsClientPolicy.mutualTlsCertificate</code> can be sourced from either:
 * <p>
 * <ul>
 * <li>A customer-provided certificate (specify a <code>certificateChain</code> path file and a <code>privateKey</code> file path).</li>
 * <li>A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its <code>secretName</code>).</li>
 * </ul>
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note</strong>
 * Currently, a certificate from AWS Certificate Manager (ACM) cannot be used for mutual TLS authentication.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * 
 * 
 * VirtualNode node1 = VirtualNode.Builder.create(this, "node1")
 *         .mesh(mesh)
 *         .serviceDiscovery(ServiceDiscovery.dns("node"))
 *         .listeners(List.of(VirtualNodeListener.grpc(GrpcVirtualNodeListenerOptions.builder()
 *                 .port(80)
 *                 .tls(ListenerTlsOptions.builder()
 *                         .mode(TlsMode.STRICT)
 *                         .certificate(TlsCertificate.file("path/to/certChain", "path/to/privateKey"))
 *                         // Validate a file client certificates to enable mutual TLS authentication when a client provides a certificate.
 *                         .mutualTlsValidation(MutualTlsValidation.builder()
 *                                 .trust(TlsValidationTrust.file("path-to-certificate"))
 *                                 .build())
 *                         .build())
 *                 .build())))
 *         .build();
 * 
 * String certificateAuthorityArn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012";
 * VirtualNode node2 = VirtualNode.Builder.create(this, "node2")
 *         .mesh(mesh)
 *         .serviceDiscovery(ServiceDiscovery.dns("node2"))
 *         .backendDefaults(BackendDefaults.builder()
 *                 .tlsClientPolicy(TlsClientPolicy.builder()
 *                         .ports(List.of(8080, 8081))
 *                         .validation(TlsValidation.builder()
 *                                 .subjectAlternativeNames(SubjectAlternativeNames.matchingExactly("mesh-endpoint.apps.local"))
 *                                 .trust(TlsValidationTrust.acm(List.of(CertificateAuthority.fromCertificateAuthorityArn(this, "certificate", certificateAuthorityArn))))
 *                                 .build())
 *                         // Provide a SDS client certificate when a server requests it and enable mutual TLS authentication.
 *                         .mutualTlsCertificate(TlsCertificate.sds("secret_certificate"))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Adding outlier detection to a Virtual Node listener</h3>
 * <p>
 * The <code>outlierDetection</code> property adds outlier detection to a Virtual Node listener. The properties
 * <code>baseEjectionDuration</code>, <code>interval</code>, <code>maxEjectionPercent</code>, and <code>maxServerErrors</code> are required.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * // Cloud Map service discovery is currently required for host ejection by outlier detection
 * Vpc vpc = new Vpc(this, "vpc");
 * PrivateDnsNamespace namespace = PrivateDnsNamespace.Builder.create(this, "test-namespace")
 *         .vpc(vpc)
 *         .name("domain.local")
 *         .build();
 * Service service = namespace.createService("Svc");
 * VirtualNode node = mesh.addVirtualNode("virtual-node", VirtualNodeBaseProps.builder()
 *         .serviceDiscovery(ServiceDiscovery.cloudMap(service))
 *         .listeners(List.of(VirtualNodeListener.http(HttpVirtualNodeListenerOptions.builder()
 *                 .outlierDetection(OutlierDetection.builder()
 *                         .baseEjectionDuration(Duration.seconds(10))
 *                         .interval(Duration.seconds(30))
 *                         .maxEjectionPercent(50)
 *                         .maxServerErrors(5)
 *                         .build())
 *                 .build())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Adding a connection pool to a listener</h3>
 * <p>
 * The <code>connectionPool</code> property can be added to a Virtual Node listener or Virtual Gateway listener to add a request connection pool. Each listener protocol type has its own connection pool properties.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * // A Virtual Node with a gRPC listener with a connection pool set
 * Mesh mesh;
 * 
 * VirtualNode node = VirtualNode.Builder.create(this, "node")
 *         .mesh(mesh)
 *         // DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
 *         // LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
 *         // whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
 *         // By default, the response type is assumed to be LOAD_BALANCER
 *         .serviceDiscovery(ServiceDiscovery.dns("node", DnsResponseType.ENDPOINTS))
 *         .listeners(List.of(VirtualNodeListener.http(HttpVirtualNodeListenerOptions.builder()
 *                 .port(80)
 *                 .connectionPool(HttpConnectionPool.builder()
 *                         .maxConnections(100)
 *                         .maxPendingRequests(10)
 *                         .build())
 *                 .build())))
 *         .build();
 * 
 * // A Virtual Gateway with a gRPC listener with a connection pool set
 * VirtualGateway gateway = VirtualGateway.Builder.create(this, "gateway")
 *         .mesh(mesh)
 *         .listeners(List.of(VirtualGatewayListener.grpc(GrpcGatewayListenerOptions.builder()
 *                 .port(8080)
 *                 .connectionPool(GrpcConnectionPool.builder()
 *                         .maxRequests(10)
 *                         .build())
 *                 .build())))
 *         .virtualGatewayName("gateway")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Adding a Route</h2>
 * <p>
 * A <em>route</em> matches requests with an associated virtual router and distributes traffic to its associated virtual nodes.
 * The route distributes matching requests to one or more target virtual nodes with relative weighting.
 * <p>
 * The <code>RouteSpec</code> class lets you define protocol-specific route specifications.
 * The <code>tcp()</code>, <code>http()</code>, <code>http2()</code>, and <code>grpc()</code> methods create a specification for the named protocols.
 * <p>
 * For HTTP-based routes, the match field can match on path (prefix, exact, or regex), HTTP method, scheme,
 * HTTP headers, and query parameters. By default, HTTP-based routes match all requests.
 * <p>
 * For gRPC-based routes, the match field can  match on service name, method name, and metadata.
 * When specifying the method name, the service name must also be specified.
 * <p>
 * For example, here's how to add an HTTP route that matches based on a prefix of the URL path:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-http", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.http(HttpRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder()
 *                         .virtualNode(node)
 *                         .build()))
 *                 .match(HttpRouteMatch.builder()
 *                         // Path that is passed to this method must start with '/'.
 *                         .path(HttpRoutePathMatch.startsWith("/path-to-app"))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add an HTTP2 route that matches based on exact path, method, scheme, headers, and query parameters:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-http2", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.http2(HttpRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder()
 *                         .virtualNode(node)
 *                         .build()))
 *                 .match(HttpRouteMatch.builder()
 *                         .path(HttpRoutePathMatch.exactly("/exact"))
 *                         .method(HttpRouteMethod.POST)
 *                         .protocol(HttpRouteProtocol.HTTPS)
 *                         .headers(List.of(HeaderMatch.valueIs("Content-Type", "application/json"), HeaderMatch.valueIsNot("Content-Type", "application/json")))
 *                         .queryParameters(List.of(QueryParameterMatch.valueIs("query-field", "value")))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add a single route with two targets and split traffic 50/50:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-http", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.http(HttpRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder()
 *                         .virtualNode(node)
 *                         .weight(50)
 *                         .build(), WeightedTarget.builder()
 *                         .virtualNode(node)
 *                         .weight(50)
 *                         .build()))
 *                 .match(HttpRouteMatch.builder()
 *                         .path(HttpRoutePathMatch.startsWith("/path-to-app"))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add an http2 route with retries:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-http2-retry", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.http2(HttpRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder().virtualNode(node).build()))
 *                 .retryPolicy(HttpRetryPolicy.builder()
 *                         // Retry if the connection failed
 *                         .tcpRetryEvents(List.of(TcpRetryEvent.CONNECTION_ERROR))
 *                         // Retry if HTTP responds with a gateway error (502, 503, 504)
 *                         .httpRetryEvents(List.of(HttpRetryEvent.GATEWAY_ERROR))
 *                         // Retry five times
 *                         .retryAttempts(5)
 *                         // Use a 1 second timeout per retry
 *                         .retryTimeout(Duration.seconds(1))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add a gRPC route with retries:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-grpc-retry", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.grpc(GrpcRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder().virtualNode(node).build()))
 *                 .match(GrpcRouteMatch.builder().serviceName("servicename").build())
 *                 .retryPolicy(GrpcRetryPolicy.builder()
 *                         .tcpRetryEvents(List.of(TcpRetryEvent.CONNECTION_ERROR))
 *                         .httpRetryEvents(List.of(HttpRetryEvent.GATEWAY_ERROR))
 *                         // Retry if gRPC responds that the request was cancelled, a resource
 *                         // was exhausted, or if the service is unavailable
 *                         .grpcRetryEvents(List.of(GrpcRetryEvent.CANCELLED, GrpcRetryEvent.RESOURCE_EXHAUSTED, GrpcRetryEvent.UNAVAILABLE))
 *                         .retryAttempts(5)
 *                         .retryTimeout(Duration.seconds(1))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add an gRPC route that matches based on method name and metadata:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-grpc-retry", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.grpc(GrpcRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder().virtualNode(node).build()))
 *                 .match(GrpcRouteMatch.builder()
 *                         // When method name is specified, service name must be also specified.
 *                         .methodName("methodname")
 *                         .serviceName("servicename")
 *                         .metadata(List.of(HeaderMatch.valueStartsWith("Content-Type", "application/"), HeaderMatch.valueDoesNotStartWith("Content-Type", "text/")))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * Add a gRPC route with timeout:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualRouter router;
 * VirtualNode node;
 * 
 * 
 * router.addRoute("route-http", RouteBaseProps.builder()
 *         .routeSpec(RouteSpec.grpc(GrpcRouteSpecOptions.builder()
 *                 .weightedTargets(List.of(WeightedTarget.builder()
 *                         .virtualNode(node)
 *                         .build()))
 *                 .match(GrpcRouteMatch.builder()
 *                         .serviceName("my-service.default.svc.cluster.local")
 *                         .build())
 *                 .timeout(GrpcTimeout.builder()
 *                         .idle(Duration.seconds(2))
 *                         .perRequest(Duration.seconds(1))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Adding a Virtual Gateway</h2>
 * <p>
 * A <em>virtual gateway</em> allows resources outside your mesh to communicate with resources inside your mesh.
 * The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance.
 * Unlike a virtual node, which represents Envoy running with an application, a virtual gateway represents Envoy deployed by itself.
 * <p>
 * A virtual gateway is similar to a virtual node in that it has a listener that accepts traffic for a particular port and protocol (HTTP, HTTP2, gRPC).
 * Traffic received by the virtual gateway is directed to other services in your mesh
 * using rules defined in gateway routes which can be added to your virtual gateway.
 * <p>
 * Create a virtual gateway with the constructor:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * 
 * String certificateAuthorityArn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012";
 * 
 * VirtualGateway gateway = VirtualGateway.Builder.create(this, "gateway")
 *         .mesh(mesh)
 *         .listeners(List.of(VirtualGatewayListener.http(HttpGatewayListenerOptions.builder()
 *                 .port(443)
 *                 .healthCheck(HealthCheck.http(HttpHealthCheckOptions.builder()
 *                         .interval(Duration.seconds(10))
 *                         .build()))
 *                 .build())))
 *         .backendDefaults(BackendDefaults.builder()
 *                 .tlsClientPolicy(TlsClientPolicy.builder()
 *                         .ports(List.of(8080, 8081))
 *                         .validation(TlsValidation.builder()
 *                                 .trust(TlsValidationTrust.acm(List.of(CertificateAuthority.fromCertificateAuthorityArn(this, "certificate", certificateAuthorityArn))))
 *                                 .build())
 *                         .build())
 *                 .build())
 *         .accessLog(AccessLog.fromFilePath("/dev/stdout"))
 *         .virtualGatewayName("virtualGateway")
 *         .build();
 * </pre></blockquote>
 * <p>
 * Add a virtual gateway directly to the mesh:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * 
 * 
 * VirtualGateway gateway = mesh.addVirtualGateway("gateway", VirtualGatewayBaseProps.builder()
 *         .accessLog(AccessLog.fromFilePath("/dev/stdout"))
 *         .virtualGatewayName("virtualGateway")
 *         .listeners(List.of(VirtualGatewayListener.http(HttpGatewayListenerOptions.builder()
 *                 .port(443)
 *                 .healthCheck(HealthCheck.http(HttpHealthCheckOptions.builder()
 *                         .interval(Duration.seconds(10))
 *                         .build()))
 *                 .build())))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>listeners</code> field defaults to an HTTP Listener on port 8080 if omitted.
 * A gateway route can be added using the <code>gateway.addGatewayRoute()</code> method.
 * <p>
 * The <code>backendDefaults</code> property, provided when creating the virtual gateway, specifies the virtual gateway's default settings for all backends.
 * <p>
 * <h2>Adding a Gateway Route</h2>
 * <p>
 * A <em>gateway route</em> is attached to a virtual gateway and routes matching traffic to an existing virtual service.
 * <p>
 * For HTTP-based gateway routes, the <code>match</code> field can be used to match on
 * path (prefix, exact, or regex), HTTP method, host name, HTTP headers, and query parameters.
 * By default, HTTP-based gateway routes match all requests.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualGateway gateway;
 * VirtualService virtualService;
 * 
 * 
 * gateway.addGatewayRoute("gateway-route-http", GatewayRouteBaseProps.builder()
 *         .routeSpec(GatewayRouteSpec.http(HttpGatewayRouteSpecOptions.builder()
 *                 .routeTarget(virtualService)
 *                 .match(HttpGatewayRouteMatch.builder()
 *                         .path(HttpGatewayRoutePathMatch.regex("regex"))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For gRPC-based gateway routes, the <code>match</code> field can be used to match on service name, host name, and metadata.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualGateway gateway;
 * VirtualService virtualService;
 * 
 * 
 * gateway.addGatewayRoute("gateway-route-grpc", GatewayRouteBaseProps.builder()
 *         .routeSpec(GatewayRouteSpec.grpc(GrpcGatewayRouteSpecOptions.builder()
 *                 .routeTarget(virtualService)
 *                 .match(GrpcGatewayRouteMatch.builder()
 *                         .hostname(GatewayRouteHostnameMatch.endsWith(".example.com"))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For HTTP based gateway routes, App Mesh automatically rewrites the matched prefix path in Gateway Route to “/”.
 * This automatic rewrite configuration can be overwritten in following ways:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualGateway gateway;
 * VirtualService virtualService;
 * 
 * 
 * gateway.addGatewayRoute("gateway-route-http", GatewayRouteBaseProps.builder()
 *         .routeSpec(GatewayRouteSpec.http(HttpGatewayRouteSpecOptions.builder()
 *                 .routeTarget(virtualService)
 *                 .match(HttpGatewayRouteMatch.builder()
 *                         // This disables the default rewrite to '/', and retains original path.
 *                         .path(HttpGatewayRoutePathMatch.startsWith("/path-to-app/", ""))
 *                         .build())
 *                 .build()))
 *         .build());
 * 
 * gateway.addGatewayRoute("gateway-route-http-1", GatewayRouteBaseProps.builder()
 *         .routeSpec(GatewayRouteSpec.http(HttpGatewayRouteSpecOptions.builder()
 *                 .routeTarget(virtualService)
 *                 .match(HttpGatewayRouteMatch.builder()
 *                         // If the request full path is '/path-to-app/xxxxx', this rewrites the path to '/rewrittenUri/xxxxx'.
 *                         // Please note both `prefixPathMatch` and `rewriteTo` must start and end with the `/` character.
 *                         .path(HttpGatewayRoutePathMatch.startsWith("/path-to-app/", "/rewrittenUri/"))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If matching other path (exact or regex), only specific rewrite path can be specified.
 * Unlike <code>startsWith()</code> method above, no default rewrite is performed.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualGateway gateway;
 * VirtualService virtualService;
 * 
 * 
 * gateway.addGatewayRoute("gateway-route-http-2", GatewayRouteBaseProps.builder()
 *         .routeSpec(GatewayRouteSpec.http(HttpGatewayRouteSpecOptions.builder()
 *                 .routeTarget(virtualService)
 *                 .match(HttpGatewayRouteMatch.builder()
 *                         // This rewrites the path from '/test' to '/rewrittenPath'.
 *                         .path(HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath"))
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For HTTP/gRPC based routes, App Mesh automatically rewrites
 * the original request received at the Virtual Gateway to the destination Virtual Service name.
 * This default host name rewrite can be configured by specifying the rewrite rule as one of the <code>match</code> property:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * VirtualGateway gateway;
 * VirtualService virtualService;
 * 
 * 
 * gateway.addGatewayRoute("gateway-route-grpc", GatewayRouteBaseProps.builder()
 *         .routeSpec(GatewayRouteSpec.grpc(GrpcGatewayRouteSpecOptions.builder()
 *                 .routeTarget(virtualService)
 *                 .match(GrpcGatewayRouteMatch.builder()
 *                         .hostname(GatewayRouteHostnameMatch.exactly("example.com"))
 *                         // This disables the default rewrite to virtual service name and retain original request.
 *                         .rewriteRequestHostname(false)
 *                         .build())
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Importing Resources</h2>
 * <p>
 * Each App Mesh resource class comes with two static methods, <code>from&lt;Resource&gt;Arn</code> and <code>from&lt;Resource&gt;Attributes</code> (where <code>&lt;Resource&gt;</code> is replaced with the resource name, such as <code>VirtualNode</code>) for importing a reference to an existing App Mesh resource.
 * These imported resources can be used with other resources in your mesh as if they were defined directly in your CDK application.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * String arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh/virtualNode/testNode";
 * VirtualNode.fromVirtualNodeArn(this, "importedVirtualNode", arn);
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * String virtualNodeName = "my-virtual-node";
 * VirtualNode.fromVirtualNodeAttributes(this, "imported-virtual-node", VirtualNodeAttributes.builder()
 *         .mesh(Mesh.fromMeshName(this, "Mesh", "testMesh"))
 *         .virtualNodeName(virtualNodeName)
 *         .build());
 * </pre></blockquote>
 * <p>
 * To import a mesh, again there are two static methods, <code>fromMeshArn</code> and <code>fromMeshName</code>.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * String arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh";
 * Mesh.fromMeshArn(this, "imported-mesh", arn);
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh.fromMeshName(this, "imported-mesh", "abc");
 * </pre></blockquote>
 * <p>
 * <h2>IAM Grants</h2>
 * <p>
 * <code>VirtualNode</code> and <code>VirtualGateway</code> provide <code>grantStreamAggregatedResources</code> methods that grant identities that are running
 * Envoy access to stream generated config from App Mesh.
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * Mesh mesh;
 * 
 * VirtualGateway gateway = VirtualGateway.Builder.create(this, "testGateway").mesh(mesh).build();
 * User envoyUser = new User(this, "envoyUser");
 * 
 * /**
 *  * This will grant `grantStreamAggregatedResources` ONLY for this gateway.
 *  *&#47;
 * gateway.grantStreamAggregatedResources(envoyUser);
 * </pre></blockquote>
 * <p>
 * <h2>Adding Resources to shared meshes</h2>
 * <p>
 * A shared mesh allows resources created by different accounts to communicate with each other in the same mesh:
 * <p>
 * <blockquote><pre>
 * // Example automatically generated from non-compiling source. May contain errors.
 * // This is the ARN for the mesh from different AWS IAM account ID.
 * // Ensure mesh is properly shared with your account. For more details, see: https://github.com/aws/aws-cdk/issues/15404
 * String arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh";
 * IMesh sharedMesh = Mesh.fromMeshArn(this, "imported-mesh", arn);
 * 
 * // This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
 * // This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
 * VirtualNode.Builder.create(this, "test-node")
 *         .mesh(sharedMesh)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.appmesh;
