package pl.decerto.hyperon.persistence.proxy;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Objects;

import pl.decerto.hyperon.persistence.model.value.CollectionProperty;
import pl.decerto.hyperon.persistence.model.value.EntityProperty;
import pl.decerto.hyperon.persistence.model.value.Property;

/**
 * @author przemek hertel
 */
public class HyperonEntityList<T extends HyperonEntity> implements HyperonList<T> {

	private final CollectionProperty coll;

	private final ModelFactory<T> factory;

	private final Class<T> adapterClass;

	public HyperonEntityList(Property coll, Class<T> adapterClass) {
		this.coll = coll.asCollection();
		this.adapterClass = adapterClass;
		this.factory = new ModelFactoryImpl<>(adapterClass);
	}

	@Override
	public CollectionProperty unproxy() {
		return coll;
	}

	@Override
	public int size() {
		return coll.size();
	}

	@Override
	public boolean isEmpty() {
		return size() == 0;
	}

	@Override
	public boolean contains(Object o) {
		if (!adapterClass.isAssignableFrom(o.getClass())) {
			return false;
		}
		return coll.getList().contains(adapterClass.cast(o).unproxy());
	}

	@Override
	public Iterator<T> iterator() {
		return new Itr();
	}

	@Override
	public Object[] toArray() {

		int size = coll.size();
		Object[] array = new Object[size];

		for (int i = 0; i < size; i++) {
			Property p = coll.at(i);
			array[i] = factory.create(p);
		}

		return array;
	}

	@Override
	@SuppressWarnings("unchecked")
	public <E> E[] toArray(E[] a) {

		if (a.length < coll.size()) {
			return (E[]) toArray();
		}

		int commonSize = Math.min(coll.size(), a.length);

		for (int i = 0; i < commonSize; i++) {
			Property p = coll.at(i);
			a[i] = (E) factory.create(p);
		}

		if (a.length > coll.size()) {
			a[coll.size()] = null;
		}

		return a;
	}

	@Override
	public boolean add(T t) {
		coll.add(t.unproxy());
		return true;
	}

	@Override
	public void add(int index, T t) {
		EntityProperty e = t.unproxy();
		coll.asCollection().add(index, e);
	}

	@Override
	public boolean remove(Object o) {

		if (o instanceof HyperonEntity) {
			EntityProperty p = ((HyperonEntity) o).unproxy();

			for (Iterator<Property> it = coll.iterator(); it.hasNext(); ) {
				Property e = it.next();

				if (Objects.equals(e, p)) {
					it.remove();
					return true;
				}
			}
		}

		return false;
	}

	@Override
	public boolean containsAll(Collection<?> c) {
		for (Object e : c) {
			if (!contains(e)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public boolean addAll(Collection<? extends T> c) {
		for (T e : c) {
			add(e);
		}
		return true;
	}

	@Override
	public boolean addAll(int index, Collection<? extends T> c) {
		int i = index;
		for (T e : c) {
			add(i++, e);
		}
		return false;
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		Objects.requireNonNull(c);
		boolean modified = false;
		Iterator<T> it = iterator();
		while (it.hasNext()) {
			if (c.contains(it.next())) {
				it.remove();
				modified = true;
			}
		}
		return modified;
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		Objects.requireNonNull(c);
		boolean modified = false;
		Iterator<T> it = iterator();
		while (it.hasNext()) {
			if (!c.contains(it.next())) {
				it.remove();
				modified = true;
			}
		}
		return modified;
	}

	@Override
	public void clear() {
		coll.clear();
	}

	@Override
	public T get(int index) {
		Property p = coll.at(index);
		return p != null ? factory.create(p) : null;
	}

	@Override
	public T set(int index, T t) {

		// remember previous element at this index
		Property prev = coll.at(index);

		// set new element
		coll.set(index, t.unproxy());

		// return previous or null
		return prev != null ? factory.create(prev) : null;
	}

	@Override
	public T remove(int index) {
		Property prev = coll.at(index).remove();
		return prev != null ? factory.create(prev) : null;
	}

	@Override
	public int indexOf(Object o) {

		if (o != null && adapterClass.isAssignableFrom(o.getClass())) {

			final EntityProperty ep = adapterClass.cast(o).unproxy();
			final int size = size();

			for (int i = 0; i < size; i++) {
				Property p = coll.at(i);
				if (p != null && p.equals(ep)) {
					return i;
				}
			}
		}

		return -1;
	}

	@Override
	public int lastIndexOf(Object o) {

		if (o != null && adapterClass.isAssignableFrom(o.getClass())) {
			final EntityProperty ep = adapterClass.cast(o).unproxy();

			for (int i = size() - 1; i >= 0; i--) {
				if (coll.at(i).equals(ep)) {
					return i;
				}
			}
		}

		return -1;

	}

	@Override
	public ListIterator<T> listIterator() {
		throw new UnsupportedOperationException();
	}

	@Override
	public ListIterator<T> listIterator(int index) {
		throw new UnsupportedOperationException();
	}

	@Override
	public List<T> subList(int fromIndex, int toIndex) {
		throw new UnsupportedOperationException();
	}

	private class Itr implements Iterator<T> {

		private final Iterator<Property> it;

		private Property curr;

		private Itr() {
			this.it = coll.iterator();
		}

		@Override
		public boolean hasNext() {
			return it.hasNext();
		}

		@Override
		public T next() {
			curr = it.next();
			return factory.create(curr);
		}

		@Override
		public void remove() {

			// remove from underlying collection
			it.remove();
		}
	}
}
