package pl.decerto.hyperon.persistence.model.value;

import java.util.Objects;

import pl.decerto.hyperon.persistence.model.def.TypeDef;

/**
 * This class represents reference (pointer) to real {@code target} {@link Property}. Created reference property acts as simple wrapper of {@code target}.
 * All methods call on reference will actually invoke methods on target. Reference and target property are of the same type - {@link TypeDef}.
 * What is also important, that reference property is equals to target property in the sense of {@code equals and hashCode} implementation.
 *
 * @author przemek hertel
 * @see TypeDef
 * @see Property
 */
public class RefProperty extends Property {

	private final Property target;

	/**
	 * Creates reference property, which points to {@code target} {@link Property}.
	 * New created {@link RefProperty} will be of the same type, as passed {@code target}.
	 *
	 * @param target property, to which created reference property points
	 */
	public RefProperty(Property target) {
		super(target.getType());
		this.target = target;
	}

	/**
	 * @return target's id
	 */
	@Override
	public long getId() {
		return target.getId();
	}

	/**
	 * Creates new instance of {@link RefProperty} with same target instance.
	 *
	 * @param resetIds this flag doesn't apply here
	 * @return new reference property instance
	 */
	@Override
	public Property deepcopy(boolean resetIds) {
		return deepcopyInternal(resetIds);
	}

	/**
	 * Creates new instance of {@link RefProperty} with same target instance.
	 *
	 * @return new reference property instance
	 */
	@Override
	protected Property deepcopyInternal() {
		return new RefProperty(target);
	}

	/**
	 * Works same as {@link #deepcopyInternal()}
	 * @param resetIds this flag doesn't apply here
	 * @return new instance of {@link RefProperty} with same target instance.
	 */
	@Override
	protected Property deepcopyInternal(boolean resetIds) {
		return new RefProperty(target);
	}

	/**
	 * Adds to {@code sb} formatted String, specific for {@link RefProperty}, which looks like:
	 * <pre>{@code
	 *  * [REF] cover1  (Cover#1003  @25658)
	 *  - [REF] cover2  (Cover#1004  @95658)
	 * }</pre>
	 *
	 * @param sb StringBuilder containing data to print
	 * @param level level used to indentation
	 * @param prefix value to used as prefix
	 * @param suffix value to used as suffix
	 * @see #toString()
	 */
	@Override
	public void print(StringBuilder sb, int level, String prefix, String suffix) {
		write(sb, level, dot(), "[REF] " + prefix, "  (", type, "#", target.getId(), "  @", target.addr(), ")");
	}

	/**
	 * Removes child from target. Remove mechanism my vary for different {@link Property} subclasses.
	 *
	 * @param child to be removed
	 * @return removed child if was found and removed, otherwise null
	 * @see ValueProperty#removeChild(Property)
	 * @see CollectionProperty#removeChild(Property)
	 * @see EntityProperty#removeChild(Property)
	 */
	@Override
	protected Property removeChild(Property child) {
		return target.removeChild(child);
	}

	/**
	 * Gets direct child from target. Finding child my vary for different {@link Property} subclasses.
	 *
	 * @param name of child property
	 * @return found child
	 * @see EntityProperty#getChild(String)
	 */
	@Override
	protected Property getChild(String name) {
		return target.getChild(name);
	}

	/**
	 * {@inheritDoc}
	 * <br>Here it means that {@code this} reference will be removed from property tree, and new value will be used instead.
	 * @param value new reference
	 */
	@Override
	public void set(Object value) {

		// use local variables (container and name), since [this] property will be removed and cleared
		Property container = getContainer();
		String name = getName();

		if (container.isCollection()) {

			// remove [this] and replace with new [value]
			container.asCollection().replace(this, (Property) value);

		} else if (container.isEntity()) {

			// explicitly remove [this] ref
			remove();

			// set new [value] on parent entity
			container.set(name, value);
		}
	}

	/**
	 * {@inheritDoc}
	 * @return {@code true}
	 */
	@Override
	public boolean isRef() {
		return true;
	}

	/**
	 * {@inheritDoc}
	 * @return target property
	 */
	@Override
	public Property getRefTarget() {
		return target;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getRefCount() {
		return target.getRefCount();
	}

	/**
	 * Find child property in the target.
	 *
	 * @param id child id
	 * @return property within target
	 */
	@Override
	public Property find(long id) {
		return target.find(id);
	}

	/**
	 * Checks if target property has element of given {@code path}.
	 *
	 * @param path of element to be checked
	 * @return {@code true} if target has element, or false otherwise
	 */
	@Override
	public boolean has(String path) {
		return target.has(path);
	}

	/**
	 * @return target as {@link EntityProperty}
	 */
	@Override
	public EntityProperty asEntity() {
		return target.asEntity();
	}

	/**
	 * @return {@link ElementType#REF} element type
	 */
	@Override
	public ElementType getElementType() {
		return ElementType.REF;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof RefProperty)) {
			return false;
		}

		RefProperty that = (RefProperty) o;
		return target == that.target;
	}

	@Override
	public int hashCode() {
		return 31 + Objects.hashCode(target);
	}

	/**
	 * Custom String format:
	 * <pre>{@code
	 *   RefProperty[cart.policies, target=TypeCode#1234 @123456789]
	 * }</pre>
	 *
	 * @return formatted String
	 */
	@Override
	public String toString() {
		return "RefProperty[" + getPath() + ", target=" + target.getTypeCode() +
			"#" + target.getId() + " @" + target.addr() + ']';
	}

}
