package pl.decerto.hyperon.persistence.model.value;

import java.math.BigDecimal;
import java.util.Date;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.smartparam.engine.core.type.ValueHolder;
import org.smartparam.engine.util.Separator;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceIllegalGetException;
import pl.decerto.hyperon.persistence.exception.HyperonPersistenceRemoveException;
import pl.decerto.hyperon.persistence.exception.HyperonPersistenceUsageException;
import pl.decerto.hyperon.persistence.helper.BundleWalker;
import pl.decerto.hyperon.persistence.helper.PropertyVisitor;
import pl.decerto.hyperon.persistence.model.def.EntityType;
import pl.decerto.hyperon.persistence.model.def.TypeDef;
import pl.decerto.hyperon.runtime.helper.StrUtil;
import pl.decerto.hyperon.runtime.helper.TypeConverter;

/**
 * This is main class representing property managed by Hyperon Persistence Engine. It exposes all necessary API for interaction.
 *
 * <p>
 *     Most basic interactions are setting property - {@link #set(String, Object)} and retrieving - {@link #get(String)} it based on certain path.
 *     For more, examine API here and all subclasses.
 * </p>
 * <span>Allowed subclasses are:</span>
 * <ul>
 *     <li>{@link ValueProperty} - represents property for <i>simple(Java) types</i>, like String, int, etc.</li>
 *     <li>{@link EntityProperty} - represents complex property with <i>fields</i></li>
 *     <li>{@link CollectionProperty} - represents collection property, that can store many properties as a container</li>
 *     <li>{@link RefProperty} - represents reference property, that points to other property</li>
 * </ul>
 * <p>
 *     Some methods are only available for certain property types. For example, it is not possible to call {@link #at(int)} on {@link RefProperty},
 *     but only on {@link CollectionProperty}.
 *     So be aware of property type and it's supported functionality.
 * </p>
 *
 * @see CollectionProperty
 * @see ValueProperty
 * @see RefProperty
 * @see EntityProperty
 * @implSpec Some methods define default behaviour with either default values or throw exception. Make sure to provide proper implementation
 * for each subclass.
 * @author przemek hertel
 */
public abstract class Property implements Handle {

	private static final String SEP = ".";

	protected static TypeConverter converter = new TypeConverter();

	/**
	 * Type of this bundle property/element.
	 */
	protected final TypeDef type;

	/**
	 * Name of this property.
	 */
	protected String name;

	/**
	 * Owning entity.
	 */
	protected EntityProperty parent;

	/**
	 * Owning entity identifier.
	 */
	protected long ownerId;

	/**
	 * Containing property.
	 */
	protected Property container;

	/**
	 * Bundle containing this property.
	 */
	protected Bundle bundle;

	/**
	 * Number of nodes referencing this property.
	 */
	protected int refCount;

	/**
	 * Persistence state of this property
	 */
	protected EntityState state = EntityState.TRANSIENT;

	/**
	 * Creates {@code Property} with provided {@link TypeDef}.
	 *
	 * @param type type definition
	 * @see TypeDef
	 */
	protected Property(TypeDef type) {
		this.type = type;
	}

	/**
	 * Creates {@code Property} with provided {@link EntityType}.
	 *
	 * @param entityType entity type defined for collections or entity objects
	 * @see EntityProperty
	 * @see CollectionProperty
	 * @see EntityType
	 */
	protected Property(EntityType entityType) {
		this(TypeDef.buildCompoundType(entityType));
	}

	/**
	 * Creates {@code Property} with provided {@code simpleType}. Simple types must have support with
	 * {@link TypeConverter} or there will be problem later on using this property.
	 *
	 * @param simpleType supported simple type by {@link TypeConverter}
	 */
	protected Property(String simpleType) {
		this(TypeDef.buildSimpleType(simpleType));
	}

	/**
	 * Gets name or parent name, if name is not setup.
	 *
	 * @return null if name or parent name is not setup
	 * @see #getOwnerPropertyName()
	 */
	public String getName() {
		return name != null ? name : getParentName();
	}

	/**
	 * Gets parent name or null if not setup.
	 *
	 * @return null if parent name is not setup
	 */
	public String getParentName() {
		return parent != null ? parent.getName() : null;
	}

	/**
	 * Gets name or parent name, if name is not setup.
	 *
	 * @return null if name or parent name is not setup
	 * @see #getName()
	 */
	public String getOwnerPropertyName() {
		return getName();
	}

	/**
	 * Sets property name.
	 *
	 * @param name of property
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Gets owning(parent) entity.
	 *
	 * @return parent (might be null)
	 */
	public EntityProperty getParent() {
		return parent;
	}

	/**
	 * Sets parent(owner).
	 *
	 * @param parent of property
	 */
	public void setParent(EntityProperty parent) {
		this.parent = parent;
	}

	/**
	 * Returns full path to this {@code property} with . separator between elements names.<br>
	 * Example:
	 * <pre>{@code
	 * - cart.policies
	 * - cart.policies.variants.covers.premium
	 * }</pre>
	 *
	 * @return path or empty string "" if path does not exist
	 * @see #getCanonicalPath()
	 */
	public String getPath() {
		return getPath(new StringBuilder()).toString();
	}

	/**
	 * Basic implementation of path creation.
	 * First parent's name (recursive) is fetched, and then child name is added at the end of path.
	 *
	 * @param sb {@code StringBuilder} for recursive path creation
	 * @return empty {@code StringBuilder}, if property has no name and parent, path otherwise
	 * @see #getPath()
	 */
	protected StringBuilder getPath(StringBuilder sb) {
		StringBuilder path = Objects.nonNull(getParent()) ? getParent().getPath(sb) : sb;

		if (path.length() > 0) {
			path.append('.');
		}

		if (Objects.nonNull(name)) {
			path.append(name);
		}

		return path;
	}

	/**
	 * Returns canonical path for this element.
	 * It will return exact position of this property within parent-child relation.
	 * For collections properties, name of property is also constructed with "array[index] like notation" - [0], [1], etc.<br>
	 *
	 * Examples:
	 * <pre>{@code
	 * - cart.policies[1]
	 * - cart.policies[0].variants[2].covers[3].premium
	 * }</pre>
	 *
	 * @return canonical path or null, if path is empty
	 * @see #getPath()
	 */
	public String getCanonicalPath() {
		String canonicalPath = getCanonicalPath(new StringBuilder()).toString();
		return StringUtils.isEmpty(canonicalPath) ? null : canonicalPath;
	}

	/**
	 * Basic implementation of canonical path creation.
	 *
	 * @param sb {@code StringBuilder} for recursive path creation
	 * @return empty {@code StringBuilder}, if property has no name and parent, path otherwise
	 * @see #getCanonicalPath()
	 */
	protected StringBuilder getCanonicalPath(StringBuilder sb) {
		// canonical path to parent
		StringBuilder path = Objects.nonNull(getParent()) ? getParent().getCanonicalPath(sb) : sb;

		// token for this node has a form: 'name' or 'name[ix]'
		String token = name;
		if (Objects.nonNull(container) && container.isCollection()) {
			int index = container.asCollection().getList().indexOf(this);
			token += "[" + index + "]";
		}

		if (path.length() > 0) {
			path.append('.');
		}

		if (Objects.nonNull(token)) {
			path.append(token);
		}

		return path;
	}

	/**
	 * Gets owner (parent) id.
	 *
	 * @return owner's id or 0 if there is no owner
	 */
	public long getOwnerId() {
		return ownerId;
	}

	/**
	 * Sets owner (parent) id.
	 *
	 * @param ownerId owner id
	 */
	public void setOwnerId(long ownerId) {
		this.ownerId = ownerId;
	}

	/**
	 * @return container for property
	 */
	public Property getContainer() {
		return container;
	}

	/**
	 * Sets container for property.
	 *
	 * @param container for property
	 */
	public void setContainer(Property container) {
		this.container = container;
	}

	/**
	 * This method is similar to {@link #setOwnerId(long)}, but id is fetched from parent.
	 * If parent is not setup, then {@code ownerId} will not be changed.
	 */
	public void setupOwnerId() {
		if (parent != null) {
			setOwnerId(parent.getId());
		}
	}

	/**
	 * Gets type of this bundle property/element.
	 *
	 * @return type associated to this property. Can't be null
	 */
	public TypeDef getType() {
		return type;
	}

	/**
	 * Gets type's code from associated definition.
	 *
	 * @return type code, shouldn't be null
	 */
	public String getTypeCode() {
		return type != null ? type.getCode() : null;
	}

	/**
	 * Gets bundle root.
	 *
	 * @return bundle root, shouldn't be null
	 * @see Bundle
	 */
	public Bundle getBundle() {
		return bundle;
	}

	/**
	 * Same as {@link #getBundle()}
	 *
	 * @return bundle root, shouldn't be null
	 * @see Bundle
	 */
	public Bundle bundle() {
		return bundle;
	}

	/**
	 * - creates deepcopy of this property
	 * - resets all IDs and ownerIDs
	 * @return property created from deepcopy
	 */
	public Property deepcopy() {
		return deepcopy(true);
	}

	/**
	 * Reset ids.
	 * @implSpec Each subclass should define, what should be reseted
	 */
	public void resetIds() {
		ownerId = 0;
	}

	/**
	 * Sets bundle for property.
	 *
	 * @param bundle root for parameter
	 */
	protected void setBundle(Bundle bundle) {
		this.bundle = bundle;
	}

	//TODO: maybe there should be multiple write methods for like up to 5 arguments, and remove this one
	protected void write(StringBuilder sb, int level, Object... args) {
		sb.append(indent(level));
		for (Object arg : args) {
			sb.append(arg);
		}
		sb.append(Separator.DEFAULT.getValue());
	}

	private String indent(int n) {
		return StringUtils.repeat(' ', n * 2);
	}

	/**
	 * Increment reference count.
	 */
	protected void incRefCount() {
		refCount++;
	}

	/**
	 * Decrement reference count.
	 */
	protected void decRefCount() {
		refCount--;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof Property)) {
			return false;
		}

		Property other = (Property) o;
		return Objects.equals(getName(), other.getName()) && Objects.equals(type, other.type) && ownerId == other.ownerId;
	}

	@Override
	public int hashCode() {
		return Objects.hash(ownerId, getName(), type);
	}


	/*
	 *  ==================================  abstract methods   ====================================
	 */

	/**
	 * @return id of property
	 */
	public abstract long getId();

	/**
	 * Returns deep copy of this property's subtree.
	 *
	 * @param resetIds whether to reset ids on copy
	 * @return property created from {@code deepcopy}
	 * @see #deepcopy()
	 */
	public abstract Property deepcopy(boolean resetIds);

	/**
	 * Prints subtree to string builder.
	 *
	 * @param sb StringBuilder containing data to print
	 * @param level level used to indentation
	 * @param prefix value to used as prefix
	 * @param suffix value to used as suffix
	 */
	public abstract void print(StringBuilder sb, int level, String prefix, String suffix);

	/**
	 * Removes children by reference {@code child}.
	 *
	 * @param child to be removed
	 * @return removed child
	 * @implSpec subclasses should control what happens, if child doesn't exist within property.
	 */
	protected abstract Property removeChild(Property child);

	/**
	 * This method is no longer used by Hyperon Peristence Engine.
	 * Please use {@link #deepcopyInternal(boolean)}
	 *
	 * @return full copy of property and its children
	 * @deprecated Will be removed in the future.
	 * @see #deepcopyInternal(boolean)
	 */
	@Deprecated
	protected abstract Property deepcopyInternal();

	/**
	 * This method must be implemented by subclasses.
	 * It should create new instances of whole subtree with copied simple values.
	 * {@code resetIds} flag allows to control, if id's of properties during coping should be set to 0 - {@code true}, or not {@code false}.
	 *
	 * @param resetIds control flag for resetting ids of properties
	 * @return new instance with copied elements
	 */
	protected abstract Property deepcopyInternal(boolean resetIds);



	/*
	 *  ===============================  basic API implementation   ===============================
	 */

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property get(String path) {

		if (StrUtil.isEmpty(path)) {
			return this;
		}

		if (isSingleToken(path)) {
			return getChild(path);
		}

		String propertyName = getFirstToken(path);
		String subPath = skipFirstToken(path);

		return getChild(propertyName).get(subPath);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ValueHolder getHolder(String path) {
		return get(path).getHolder();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getString(String path) {
		return get(path).getString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getInteger(String path) {
		return get(path).getInteger();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public BigDecimal getDecimal(String path) {
		return get(path).getDecimal();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Long getLong(String path) {
		return get(path).getLong();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean getBoolean(String path) {
		return get(path).getBoolean();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Date getDate(String path) {
		return get(path).getDate();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Date getDatetime(String path) {
		return get(path).getDatetime();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public double getNumber(String path) {
		return get(path).getNumber();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int intValue(String path) {
		return get(path).intValue();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean booleanValue(String path) {
		return get(path).booleanValue();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property set(String path, Object value) {
		get(path).set(value);
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property add(String path, Property... properties) {
		return get(path).add(properties);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property remove(String path) {
		return get(path).remove();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String print() {
		StringBuilder sb = new StringBuilder(128);
		print(sb, 1, name, name);
		return sb.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getRefCount() {
		return refCount;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isValue() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEntity() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isCollection() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isRoot() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CollectionProperty asCollection() {
		return (CollectionProperty) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public RefProperty asRef() {
		return (RefProperty) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EntityProperty asEntity() {
		return (EntityProperty) this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ValueProperty asValue() {
		return (ValueProperty) this;
	}

	/*
	 *  ===========================  implementation in proper subclass  ===========================
	 */

	/**
	 * Gets child of this property by given {@code name}.
	 *
	 * @param name of child property
	 * @return child property if found
	 * @implSpec by default it will throw {@link HyperonPersistenceUsageException}. Subclasses might provide different behaviour.
	 * @see #at(int)
	 * @see #get(String)
	 */
	protected Property getChild(String name) {
		throw new HyperonPersistenceUsageException("Getting child [" + name + "] not allowed for this property", this);
	}

	/**
	 * {@inheritDoc}
	 * @implSpec by default it will throw {@link HyperonPersistenceUsageException}. Subclasses might provide different behaviour.
	 */
	@Override
	public Property create() {
		throw new HyperonPersistenceUsageException("Creating prototype not allowed for this property", this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property remove() {

		// maybe this property is already removed (detached)
		if (getContainer() == null) {
			return this;
		}

		// check if remove is safe
		checkReferences();

		// remove this node from its container
		getContainer().removeChild(this);

		// deep-remove this node
		if (bundle != null) {
			traverse(new PropertyVisitor() {
				@Override public void visit(Property p, String path, ElementType type) {
					throw new UnsupportedOperationException();
				}

				@Override public void visit(Property p, ElementType type) {
					// remove from tracking set
					bundle.identitySet().remove(p);

					// update ref counter if needed
					if (p.isRef()) {
						p.getRefTarget().decRefCount();
					}
				}
			}, false);
		}

		// clear node information
		setParent(null);
		setContainer(null);
		setName(null);
		setOwnerId(0);
		setBundle(null);

		return this;
	}

	/**
	 * Checks whether {@code this} and given {@code prop} exist in the same tree.
	 * @return true if there is property within {@code this}, false otherwise
	 */
	protected boolean alreadyInTree(final Property prop) {

		// 1. use fast identity set if available

		if (bundle != null) {
			return bundle.identitySet().contains(prop);
		}

		// 2. detached entity, search subtree

		final MutableBoolean result = new MutableBoolean(false);

		root().traverse(new PropertyVisitor() {
			@Override public void visit(Property p, String path, ElementType type) {
				throw new UnsupportedOperationException();
			}

			@Override public void visit(Property p, ElementType type) {
				if (p == prop) {
					result.setTrue();
				}
			}
		}, false);

		return result.booleanValue();
	}

	/**
	 * Find root of {@code this} property.
	 *
	 * @return root property
	 */
	protected Property root() {
		Property root = this;
		while (root.getParent() != null) {
			root = root.getParent();
		}
		return root;
	}

	private void checkReferences() {

		// each property inside this node
		final IdentityHashMap<Property, RefCnt> refMap = new IdentityHashMap<>(4);

		// count internal references to each property inside this node
		traverse(new PropertyVisitor() {
			@Override public void visit(Property p, String path, ElementType type) {
				throw new UnsupportedOperationException();
			}

			@Override public void visit(Property p, ElementType type) {
				if (p.isRef()) {
					refCnt(refMap, p.getRefTarget()).internalRefCnt++;

				} else if (p.getRefCount() > 0) {
					refCnt(refMap, p).totalRefCnt = p.getRefCount();
				}
			}
		}, false);

		// if any prop is externally referenced - forbid removal
		for (RefCnt cnt : refMap.values()) {
			if (cnt.totalRefCnt > cnt.internalRefCnt) {
				throw new HyperonPersistenceRemoveException(this, cnt.prop);
			}
		}

	}

	private RefCnt refCnt(IdentityHashMap<Property, RefCnt> refMap, Property p) {
		return refMap.computeIfAbsent(p, RefCnt::new);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void traverse(PropertyVisitor visitor) {
		BundleWalker.walk(this, visitor);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void traverse(PropertyVisitor visitor, boolean usePathsInVisitor) {
		BundleWalker.walk(this, visitor, usePathsInVisitor);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public String getString() {
		throw new HyperonPersistenceIllegalGetException("String", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public Integer getInteger() {
		throw new HyperonPersistenceIllegalGetException("Integer", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public BigDecimal getDecimal() {
		throw new HyperonPersistenceIllegalGetException("Decimal", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public boolean booleanValue() {
		throw new HyperonPersistenceIllegalGetException("boolean", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public Boolean getBoolean() {
		throw new HyperonPersistenceIllegalGetException("Boolean", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public Date getDate() {
		throw new HyperonPersistenceIllegalGetException("Date", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public Date getDatetime() {
		throw new HyperonPersistenceIllegalGetException("Datetime", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public ValueHolder getHolder() {
		throw new HyperonPersistenceIllegalGetException("Holder", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public Long getLong() {
		throw new HyperonPersistenceIllegalGetException("Long", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public double getNumber() {
		throw new HyperonPersistenceIllegalGetException("Number", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public int intValue() {
		throw new HyperonPersistenceIllegalGetException("int", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}.
	 */
	@Override
	public void set(Object value) {
		throw new HyperonPersistenceUsageException("Setting value not allowed for non-terminal property", this);
	}

	/**
	 * {@inheritDoc}.
	 *  @implSpec The default implementation will throw {@link HyperonPersistenceUsageException}. This is only supported by collection property type.
	 */
	@Override
	public Iterator<Property> iterator() {
		throw new HyperonPersistenceUsageException("Iterating not allowed for non-collection property", this);
	}

	/**
	 * {@inheritDoc}
	 *  @implSpec By default, this method will throw exception. Subclass should implement proper behaviour.
	 */
	@Override
	public Property add(Property... properties) {
		throw new HyperonPersistenceUsageException("Adding elements is allowed only for collection property", this);
	}

	@Override
	public int size() {
		throw new HyperonPersistenceUsageException("method size() is not allowed for this type", this);
	}

	@Override
	public Property at(int index) {
		throw new HyperonPersistenceUsageException("method at() is allowed only for collection property, not for this", this);
	}

	@Override
	public Property clear() {
		throw new HyperonPersistenceUsageException("method clear() is allowed only for collection property, not for this", this);
	}

	/**
	 * {@inheritDoc}
	 * @param path of entity property
	 * @return entity property if found or exception will be thrown
	 * @throws HyperonPersistenceUsageException if element is not found or found but it is not instance of {@link EntityProperty}
	 */
	@Override
	public EntityProperty getEntity(String path) {
		Property prop = get(path);

		if (prop != null && prop.isRef()) {
			prop = prop.getRefTarget();
		}

		if (prop instanceof EntityProperty) {
			return (EntityProperty) prop;
		}

		throw new HyperonPersistenceUsageException("Result property is not entity: ", prop);
	}

	/**
	 * {@inheritDoc}
	 * @param path of collection property
	 * @return collection property if found or exception will be thrown
	 * @throws HyperonPersistenceUsageException if element is not found or found but it is not instance of {@link CollectionProperty}
	 */
	@Override
	public CollectionProperty getCollection(String path) {
		Property prop = get(path);

		if (prop instanceof CollectionProperty) {
			return (CollectionProperty) prop;
		}

		throw new HyperonPersistenceUsageException("Result property is not collection: ", prop);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property getRefTarget() {
		throw new HyperonPersistenceUsageException("This is not ref element: ", this);
	}

	/*
	 *  ===================================  helper methods  ======================================
	 */

	protected final String getFirstToken(String path) {
		return StringUtils.substringBefore(path, SEP);
	}

	protected final String skipFirstToken(String path) {
		return StringUtils.substringAfter(path, SEP);
	}

	protected final boolean isSingleToken(String path) {
		return !path.contains(SEP);
	}

	protected final String dot() {
		return state == EntityState.PERSISTENT ? "* " : "- ";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EntityState getState() {
		return state;
	}

	/**
	 * {@inheritDoc}
	 * @see EntityState#PERSISTENT
	 */
	@Override
	public boolean isPersistent() {
		return getState() == EntityState.PERSISTENT;
	}

	/**
	 * {@inheritDoc}
	 * @see EntityState#TRANSIENT
	 */
	@Override
	public boolean isTransient() {
		return getState() == EntityState.TRANSIENT;
	}

	/**
	 * Sets state of property.
	 *
	 * @param state new state of property
	 */
	protected void setState(EntityState state) {
		this.state = state;
	}

	protected int addr() {
		return System.identityHashCode(this) & 0x0000ffff;
	}

	private final class RefCnt {

		final Property prop;
		int internalRefCnt;
		int totalRefCnt;

		private RefCnt(Property prop) {
			this.prop = prop;
		}
	}

}
