package pl.decerto.hyperon.persistence.model.value;

import java.math.BigDecimal;
import java.util.Date;

import org.smartparam.engine.core.type.ValueHolder;

import pl.decerto.hyperon.persistence.helper.PropertyVisitor;

/**
 * This interface should be only used within Hyperon Peristence Engine as grouping facade of all possible actions on it's subclasses.
 * <p>
 * What's important is that collection property type allows iteration thanks to ({@link Iterable}).
 * </p>
 *
 * @author przemek hertel
 * @apiNote It might be removed in the future, or broke apart into smaller interfaces, which group common behaviour, like collection methods, references, etc.
 */
public interface Handle extends Iterable<Property> {

	/**
	 * Gets property under given {@code path}. There are few path resolving possibilities:
	 * <pre>{@code
	 *	1. Direct access to child property (element, reference, value or collection)
	 *
	 *		property.get("child") - direct child property
	 *		property.get("children") - direct children collection property
	 *		property.get("child.name") - indirect child's name property as value
	 *
	 *	2. Access to collection element at available index
	 *
	 *		property.get("children.id") - is not possible, if "children" is collection property
	 *		property.get("children[0].id") - proper access to child, that exists within children collection
	 *
	 *  3. Chain of properties
	 *
	 *  	property.get("supplier.car.model.productionYear")
	 * }</pre>
	 * If path is empty or null, property should return itself.
	 *
	 * @param path should exists within this property
	 * @return property if path was properly resolved and property exists, null otherwise
	 * @implSpec property elements should be separated by <i>.</i> (dot char).
	 * @see #set(String, Object)
	 */
	Property get(String path);

	/**
	 * Sets new {@code value} under given {@code path}, which must exists. All rules of path design, are the same as described in {@link #get(String)}.<br>
	 * Example: <pre>{@code
	 *  Property agent = property.get("agent").set("age", 31).set("name", "steph");
	 * }</pre>
	 *	This method is useful for setting up multiple properties of one parent property, in the sense of <i>Builder Pattern</i>.
	 *
	 * @param path should exists within this property
	 * @param value new value for path.
	 * @implSpec new value might override previous value. It depends on exact implementation.
	 * @return same property instance.
	 * @see #get(String)
	 */
	Property set(String path, Object value);

	/**
	 * Sets new {@code value} of this property.
	 *
	 * @param value new value
	 * @implSpec specific behaviour may vary for each implementation.
	 */
	void set(Object value);

	/**
	 * Find child property by id.
	 *
	 * @param id child id
	 * @return child if found, null otherwise
	 * @implSpec default implementation will return null.
	 */
	default Property find(long id) {
		return null;
	}

	/**
	 * Find entity property based on given {@code path} within this property subtree.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path of entity property
	 * @return found property
	 * @implSpec might return null or throw exception. That depends on implementation.
	 * @see #get(String)
	 * @see EntityProperty
	 */
	Property getEntity(String path);

	/**
	 * Find collection property based on given {@code path} within this property subtree.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path of collection property
	 * @return found property
	 * @implSpec might return null or throw exception. That depends on implementation.
	 * @see CollectionProperty
	 */
	CollectionProperty getCollection(String path);

	/**
	 * Gets associated element type, based on property implementation.
	 * @implSpec each implementation must be mapped to {@link ElementType}.
	 * @see #get(String)
	 * @return property type
	 */
	ElementType getElementType();

	/**
	 * Is property an entity.
	 *
	 * @return true if property is entity type, false otherwise
	 * @implSpec The default implementation returns {@code false}. Adequate subclasses should provide custom implementation.
	 * @see EntityProperty
	 */
	default boolean isEntity() {
		return false;
	}

	/**
	 * Is property a value type.
	 *
	 * @return true if property is value type, false otherwise
	 * @implSpec The default implementation returns {@code false}. Adequate subclasses should provide custom implementation.
	 * @see ValueProperty
	 */
	default boolean isValue() {
		return false;
	}

	/**
	 * Is property a collection type.
	 *
	 * @return true if property is collection type, false otherwise
	 * @implSpec The default implementation returns {@code false}. Adequate subclasses should provide custom implementation.
	 * @see CollectionProperty
	 */
	default boolean isCollection() {
		return false;
	}

	/**
	 * Is property a root element.
	 *
	 * @return true if property is root element, false otherwise
	 * @implSpec The default implementation returns {@code false}.Adequate subclasses should provide custom implementation.
	 * @see Bundle
	 */
	default boolean isRoot() {
		return false;
	}

	/**
	 * @return property as {@link ValueProperty}
	 */
	ValueProperty asValue();

	/**
	 * @return property as {@link EntityProperty}
	 */
	EntityProperty asEntity();

	/**
	 * @return property as {@link CollectionProperty}
	 */
	CollectionProperty asCollection();

	/**
	 * @return property as {@link RefProperty}
	 */
	RefProperty asRef();

	/**
	 * This method walks on property subtree and invoke some action with {@link PropertyVisitor}. Think of it as <i>Visitor Pattern</i>.
	 *
	 * @param visitor property
	 * @see PropertyVisitor
	 */
	void traverse(PropertyVisitor visitor);

	/**
	 * This method walks on property subtree and invoke some action with {@link PropertyVisitor}. Think of it as <i>Visitor Pattern</i>.
	 * Control flag {@code usePathsInVisitor} allows to switch between new and old way of resolving subtree. The outcome should be the same to the API user.
	 * If set to {@code true}, the old way will be used. If set to {@code false}, the new mechanism will be used.
	 *
	 * @param visitor property
	 * @param usePathsInVisitor control flag if it should use paths to walk over subtree
	 * @see PropertyVisitor
	 * @deprecated This method will be removed in the future. New implementation is not using paths to resolve subtree.
	 */
	@Deprecated
	void traverse(PropertyVisitor visitor, boolean usePathsInVisitor);

	/**
	 * Creates whole sub tree of this property as formatted String.
	 *
	 * @return formatted String of subtree
	 */
	String print();

	/*
	 * collection api
	 */

	/**
	 * Adds one or multiple properties to collection property, that exists under given {@code path}.
	 * The order of properties should be kept as they are provided. If there are properties added to collection property, which resides under this path,
	 * they should be added at the end.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to collection property
	 * @param properties one or multiple properties that will be added
	 * @return property under {@code path}
	 * @see #get(String)
	 * @implSpec each subclass should handle itself how it responds, if there is no element under path or is not collection property
	 */
	Property add(String path, Property... properties);

	/**
	 * Adds one or multiple properties to collection property.
	 * The order of properties should be kept as they are provided. If there are properties added to collection property, which resides under this path,
	 * they should be added at the end.
	 *
	 * @param properties one or multiple properties that will be added
	 * @return {@code this} property
	 * @implSpec each subclass should handle itself how it responds
	 */
	Property add(Property... properties);

	/**
	 * Removes property from given {@code path}.
	 * Property will first be detached from whole subtree, like collections and references. If that is possible, then property will be removed.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path points to property, that will be removed
	 * @return removed property
	 * @see #get(String)
	 */
	Property remove(String path);

	/**
	 * Removes itself.
	 * Property will first be detached from whole subtree, like collections and references. If that is possible, then property will be removed.
	 *
	 * @return removed property
	 */
	Property remove();

	/**
	 * Clear all children of {@code this} property.
	 *
	 * @return itself
	 */
	Property clear();

	/**
	 * @return number of properties
	 */
	int size();

	/**
	 * Gets property at given {@code index} withing {@code this} property.
	 * Index starts with 0.
	 *
	 * @param index of element within property
	 * @return found property of index
	 * @implSpec should throw exception or return null, if {@code index > size} of elements within property
	 */
	Property at(int index);

	/**
	 * @return new property
	 * @implSpec each subclass should implement custom behaviour
	 */
	Property create();

	/**
	 * Checks if this property has element of given {@code path}.
	 *
	 * @param path of element to be checked
	 * @return {@code true}, if property has element with {@code path}, {@code false} otherwise
	 * @implSpec By default, it returns {@code false}
	 */
	default boolean has(String path) {
		return false;
	}

	/*
	 * ref api
	 */

	/**
	 * Is property a reference type.
	 *
	 * @return true if property is reference type, {@code false} otherwise
	 * @implSpec The default implementation returns {@code false}.
	 */
	default boolean isRef() {
		return false;
	}

	/**
	 * Gets property of reference's target.
	 *
	 * @return reference's target. If property is reference type, it shouldn't return {@code null}
	 */
	Property getRefTarget();

	/**
	 * Number of nodes referencing target's property.
	 *
	 * @return number of references
	 */
	int getRefCount();

	/*
	 * state api
	 */

	/**
	 * Is property in persistent state.
	 *
	 * @return true if property is in persistent state, false otherwise
	 */
	boolean isPersistent();

	/**
	 * Is property in transient state.
	 *
	 * @return true if property is in transient state, false otherwise
	 */
	boolean isTransient();

	/**
	 * Gets state of property.
	 * There are two possible states of entity:
	 * <ul>
	 *     <li>{@link EntityState#TRANSIENT} - if property was just created and not yet marked for persistence</li>
	 *     <li>{@link EntityState#PERSISTENT} - if property was marked for persistence by {@link pl.decerto.hyperon.persistence.service.BundleService}</li>
	 * </ul>
	 * For more specific description, please read {@link pl.decerto.hyperon.persistence.helper.PersistenceMarker} docs.
	 *
	 * @return state of property
	 * @see pl.decerto.hyperon.persistence.helper.PersistenceMarker
	 * @see pl.decerto.hyperon.persistence.service.BundleService
	 */
	EntityState getState();

	/*
	 * getting value
	 */

	/**
	 * Gets value of property wrapped into {@link ValueHolder} from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property wrapped into {@link ValueHolder}
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	ValueHolder getHolder(String path);

	/**
	 * @return value of property wrapped into {@link ValueHolder}
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	ValueHolder getHolder();

	/**
	 * Gets value as String from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as String
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	String getString(String path);

	/**
	 * @return value of property as String
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	String getString();

	/**
	 * Gets value as Integer from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as Integer
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Integer getInteger(String path);

	/**
	 * @return value of property as Integer
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Integer getInteger();

	/**
	 * Gets value as BigDecimal from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as BigDecimal
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	BigDecimal getDecimal(String path);

	/**
	 * @return value of property as BigDecimal
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	BigDecimal getDecimal();

	/**
	 * Gets value as Long from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as Long
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Long getLong(String path);

	/**
	 * @return value of property as Long
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Long getLong();

	/**
	 * Gets value as Boolean from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as Boolean
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Boolean getBoolean(String path);

	/**
	 * @return value of property as Boolean
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Boolean getBoolean();

	/**
	 * Gets value as Date from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as Date
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Date getDate(String path);

	/**
	 * @return value of property as Date
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Date getDate();

	/**
	 * Gets value as Date with time from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as Date
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Date getDatetime(String path);

	/**
	 * @return value of property as Date with time
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	Date getDatetime();

	/**
	 * Gets value as double from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as double
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	double getNumber(String path);

	/**
	 * @return value of property as double
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	double getNumber();

	/**
	 * Gets value as int from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as int
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	int intValue(String path);

	/**
	 * @return value of property as int
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	int intValue();

	/**
	 * Gets value as boolean from given {@code path}.
	 * All rules of path design, are the same as described in {@link #get(String)}.<br>
	 *
	 * @param path to value
	 * @return value of property as boolean
	 * @see #get(String)
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	boolean booleanValue(String path);

	/**
	 * @return value of property as boolean
	 * @implSpec If not supported on subclasses, it should throw custom runtime exception.
	 */
	boolean booleanValue();

}
