package pl.decerto.hyperon.persistence.model.value;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceException;
import pl.decerto.hyperon.persistence.exception.HyperonPersistenceUsageException;
import pl.decerto.hyperon.persistence.helper.BundleHelper;
import pl.decerto.hyperon.persistence.helper.RefLink;
import pl.decerto.hyperon.persistence.model.def.EntityType;
import pl.decerto.hyperon.persistence.model.def.PropertyDef;
import pl.decerto.hyperon.runtime.helper.StrUtil;

/**
 * This class represents property of entity (complex) type with fields. Each filed is stored as:
 * <pre>{@code
 *  * key -> field name, which is case sensitive
 *  * value -> another Property - subtype of {@link Property}
 * }</pre>
 * Basic API methods:
 * <ul>
 *  <li>Add property with field name - {@link #add(String, Property)}</li>
 *  <li>Get property with field name - {@link #getProp(String)}</li>
 * </ul>
 *
 * @author przemek hertel
 * @see EntityType
 */
public class EntityProperty extends Property {

	protected long id;

	/**
	 * Using LinkedHashMap is faster than HashMap and TreeMap in dominant operation - iteration through values.
	 */
	protected final Map<String, Property> fields = new LinkedHashMap<>();

	/**
	 * Creates entity (complex) property based on given {@code entityType}.
	 *
	 * @param entityType complex entity types
	 * @see EntityType
	 */
	public EntityProperty(EntityType entityType) {
		super(entityType);
	}

	/**
	 * Creates entity (complex) property with id, based on given {@code entityType}.
	 *
	 * @param id for property
	 * @param entityType complex entity types
	 * @see EntityType
	 */
	public EntityProperty(long id, EntityType entityType) {
		this(entityType);
		this.id = id;
	}

	/**
	 * Adds given {@code property} under provided {@code name}. Name is case sensitive.
	 * If there is already some property under this name, then new {@code prop} will override previous property.
	 * If provided property is {@link CollectionProperty}, then all it's children will also have mapped parent to this.
	 *
	 * @param name unique name
	 * @param prop property to store under name
	 * @return same {@link EntityProperty} instance
	 * @see #set(Object)
	 */
	public EntityProperty add(String name, Property prop) {

		// prepare property
		prop.setName(name);
		prop.setParent(this);
		prop.setOwnerId(this.id);
		prop.setContainer(this);
		prop.setBundle(bundle);

		if (prop instanceof CollectionProperty) {
			CollectionProperty coll = (CollectionProperty) prop;

			final int size = coll.size();
			for (int i = 0; i < size; i++) {
				Property element = coll.at(i);
				element.setParent(this);
				element.setOwnerId(this.id);
				element.setName(name);
				element.setBundle(bundle);
			}
		}

		// remove current child if any
		Property curr = fields.get(name);
		if (curr != null) {
			curr.remove();
		}

		// add property to internal map
		fields.put(name, prop);

		// scan property
		if (bundle != null) {
			prop.traverse(new IdentityScanner(bundle), false);
		}

		return this;
	}

	/**
	 * {@inheritDoc}
	 * <br>
	 * children fields will also have their bundle setup.
	 */
	@Override
	public void setBundle(Bundle bundle) {
		super.setBundle(bundle);

		// cascade bundle ref to whole subtree
		for (Property child : fields.values()) {
			child.setBundle(bundle);
		}
	}

	/**
	 * @return property fields, where key is name of a field, and value is associated property
	 */
	public Map<String, Property> getFields() {
		return fields;
	}

	/**
	 * Gets property based on given {@code name}.
	 *
	 * @param name of field
	 * @return property if found or null
	 * @see #getValue(String)
	 */
	public Property getProp(String name) {
		return fields.get(name);
	}

	/**
	 * Gets {@link ValueProperty} based on given {@code name}.
	 * It may throw cast exception, if property of name is not type of {@link ValueProperty}.
	 *
	 * @param name of field
	 * @return property if found or null
	 * @see #getProp(String)
	 */
	public ValueProperty getValue(String name) {
		return (ValueProperty) fields.get(name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public long getId() {
		return id;
	}

	/**
	 * Sets id.
	 * @param id for property
	 */
	public void setId(long id) {
		this.id = id;
	}

	/**
	 * @return {@link ElementType#ENTITY} element type
	 */
	@Override
	public ElementType getElementType() {
		return ElementType.ENTITY;
	}

	/**
	 * @return {@code true}
	 */
	@Override
	public boolean isEntity() {
		return true;
	}

	/*
	 *  ===========================  abstract methods implementation  =============================
	 */

	/**
	 * {@inheritDoc}
	 * <br>It copies nested properties and reconstructs references to this property.
	 *
	 * @param resetIds control flag, if true then property and all elements will have their id set to 0, if false, id will be copied
	 * @return new instance of {@link EntityProperty}
	 */
	@Override
	public Property deepcopy(boolean resetIds) {

		// prepare copy
		EntityProperty copy = deepcopyInternal(resetIds);

		// reconstruct refs on copy
		copyRefs(copy);

		return copy;
	}

	protected void copyRefs(EntityProperty copy) {

		// collect refs and their targets
		List<RefLink> refLinks = BundleHelper.createRefLinks(this);

		// reconstruct refs on copy
		for (RefLink link : refLinks) {

			// ref's and target's canonical positions
			String targetPath = link.getTargetCanonicalPath();
			String refPath = link.getRefCanonicalPathWithoutIndex();

			String prefix = this.getPath();

			if (!StrUtil.isBlank(prefix)) {

				targetPath = removePrefix(targetPath, prefix);
				refPath = removePrefix(refPath, prefix);
			}

			// target in copy tree
			Property target = copy.get(targetPath);

			// add ref to target in proper position
			Property ref = copy.get(refPath);
			if (ref.isCollection()) {
				ref.asCollection().add(link.getRefIndex(), target);

			} else {
				ref.set(target);
			}
		}
	}

	private String removePrefix(String canonicalPath, String prefix) {

		if (canonicalPath.startsWith(prefix)) {
			canonicalPath = canonicalPath.substring(prefix.length());

			if (canonicalPath.startsWith("[")) {
				int rbracket = canonicalPath.indexOf(']');
				canonicalPath = canonicalPath.substring(rbracket + 1);
			}

			canonicalPath = canonicalPath.substring(1);
		}

		return canonicalPath;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected EntityProperty deepcopyInternal() {
		return deepcopyInternal(false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected EntityProperty deepcopyInternal(boolean resetIds) {
		EntityProperty copy = new EntityProperty(resetIds ? 0 : id, type.getCompoundType());
		copy.setName(name);
		copy.setOwnerId(resetIds ? 0 : ownerId);
		copy.setState(state);

		deepcopyInternalFields(copy, resetIds);

		return copy;
	}

	/**
	 * Copies all fields from this property and adds them to {@code copy} of entity property.
	 *
	 * @param copy entity property with fields to copy from
	 * @param resetIds control flag, if copied properties should also have mapped ids
	 */
	protected void deepcopyInternalFields(EntityProperty copy, boolean resetIds) {
		for (Map.Entry<String, Property> e : fields.entrySet()) {
			String name = e.getKey();
			Property child = e.getValue();

			if (!child.isRef()) {
				copy.add(name, child.deepcopyInternal(resetIds));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 * Also it will reset ids of all children properties stored in {@code fields}.
	 */
	@Override
	public void resetIds() {

		// rest this entity
		setId(0);
		setOwnerId(0);

		// reset children
		for (Property p : fields.values()) {
			p.resetIds();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void print(StringBuilder sb, int level, String prefix, String suffix) {
		String refc = getRefCount() > 0 ? " refc=" + getRefCount() : "";
		write(sb, level, dot(), prefix, "  (", type, "#", id, "  @", addr(), ")", refc);

		for (Map.Entry<String, Property> e : fields.entrySet()) {
			Property value = e.getValue();
			value.print(sb, level + 1, e.getKey(), suffix + "." + e.getKey());
		}
	}

	/**
	 * {@inheritDoc}
	 * <br> Given {@code child} will be searched within {@code fields} of this property.
	 * @return removed child or null if child was not found
	 */
	@Override
	protected Property removeChild(Property child) {
		for (Iterator<Property> it = fields.values().iterator(); it.hasNext(); ) {
			Property p = it.next();
			if (p == child) {
				it.remove();
				return p;
			}
		}
		return null;
	}

	/**
	 * Creates formatted String.
	 * Example:
	 * <pre>{@code
	 * 	 EntityProperty[entityName#123, owner=44, values={field1=test}]
	 * }</pre>
	 *
	 * @return formatted String
	 */
	public String printValues() {
		StringBuilder sb = new StringBuilder();

		for (Map.Entry<String, Property> e : getFields().entrySet()) {
			if (e.getValue().isValue()) {
				sb.append(e.getKey()).append('=').append(e.getValue().getString()).append(' ');
			}
		}

		return "EntityProperty[" + getName() + "#" + id + ", owner=" + ownerId + ", values={" + sb.toString() + "}]";
	}


	/*
	 *  ==================================  API implementation   ==================================
	 */

	/**
	 * {@inheritDoc}.
	 * @param token may contain index (if collection element)
	 */
	@Override
	protected Property getChild(String token) {

		// property name (without possible index)
		String name = token;

		// element index (if token points to n-th element in collection)
		int index = -1;

		if (hasBrackets(name)) {
			int lbracket = name.lastIndexOf('[');
			index = Integer.parseInt(name.substring(lbracket + 1, name.length() - 1));
			name = name.substring(0, lbracket);
		}

		// name is property name without index
		Property child = getProp(name);

		if (child == null) {
			PropertyDef childDef = getType().getCompoundType().getProp(name);

			if (childDef != null) {
				child = childDef.createProperty();
				add(name, child);

			} else {
				throw new HyperonPersistenceException(String.format("Unknown child name: '%s' for element: '%s'", token, getPath()));
			}

		}

		if (index >= 0) {
			return child.at(index);
		}

		return child;
	}

	private boolean hasBrackets(String token) {
		return token.charAt(token.length() - 1) == ']';
	}

	/**
	 * @return number of fields within this property
	 */
	@Override
	public int size() {
		return fields.size();
	}

	/**
	 * @return new instance of {@link EntityProperty} based on {@code this} compound type definition.
	 */
	@Override
	public Property create() {
		return new EntityProperty(type.getCompoundType());
	}

	/**
	 * {@inheritDoc}
	 * <br>This can check either direct field or nested children.
	 *
	 * @param path of element to be checked
	 * @return true if found, false otherwise
	 */
	@Override
	public boolean has(String path) {

		if (isSingleToken(path)) {
			return fields.containsKey(path);
		}

		String name = getFirstToken(path);
		String subpath = skipFirstToken(path);

		return has(name) && get(name).has(subpath);
	}

	/**
	 * {@inheritDoc}.
	 * <br>It changes references and collection(container) properties with this new value or adds a new one.
	 *
	 * @param value must be sub type of {@link Property} or exception will be thrown.
	 * @throws HyperonPersistenceUsageException if value is not sub type of {@link Property}
	 */
	@Override
	public void set(Object value) {

		if (value instanceof Property) {

			Property prop = (Property) value;

			// unproxy if value is already a reference
			if (prop.isRef()) {
				prop = prop.getRefTarget();
			}

			if (container != null) {

				// use ref proxy if prop is already used in bundle
				if (alreadyInTree(prop)) {
					prop = new RefProperty(prop);
				}

				if (container.isEntity()) {

					// add this prop to parent (overwrite if needed)
					container.asEntity().add(name, prop);
				} else if (container.isCollection()) {

					// replace this element with new prop
					container.asCollection().replace(this, prop);
				}
			}

		} else {

			// value is not instance of Property
			super.set(value);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Property find(long id) {

		// self check
		if (getId() == id) {
			return this;
		}

		// check all children (recursivelly)
		for (Property child : fields.values()) {
			Property r = child.find(id);
			if (r != null) {
				return r;
			}
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EntityState getState() {
		return state;
	}

	/**
	 * Marks entity property to {@link EntityState#PERSISTENT}.
	 */
	public void mark() {
		state = EntityState.PERSISTENT;
	}

	/*
	 *  ===================================  ordinary methods   ===================================
	 */

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof EntityProperty)) {
			return false;
		}

		EntityProperty that = (EntityProperty) o;
		return super.equals(that) && id == that.id && Objects.equals(fields, that.fields);
	}

	@Override
	public int hashCode() {
		return 31 * super.hashCode() + Objects.hashCode(id);
	}

	/**
	 * Formatted String as follows:
	 * <pre>{@code
	 *  EntityProperty[name#23, @432423432, owner=101, fields=field1, refc=2]
	 * }</pre>
	 * @return formatted String
	 */
	@Override
	public String toString() {
		return "EntityProperty[" + getName() + "#" + id +
			", @" + addr() +
			", owner=" + ownerId + ", fields=" + fields.keySet() +
			", refc=" + getRefCount() + "]";
	}

}
