/*
 * Copyright 2002-2005 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.core.style;

import org.springframework.util.ClassUtils;
import org.springframework.util.ObjectUtils;

/**
 * Spring's default <code>toString()</code> styler. Underneath the hood,
 * uses the reflective visitor pattern to nicely encapsulate styling
 * algorithms for each type of styled object.
 *
 * This class is used by ToStringBuilder to style <code>toString()</code>
 * output in a consistent manner.
 *
 * @author Keith Donald
 * @author Juergen Hoeller
 * @since 1.2.2
 */
public class DefaultToStringStyler implements ToStringStyler {

	private final ValueStyler valueStyler;


	public DefaultToStringStyler(ValueStyler valueStyler) {
		this.valueStyler = valueStyler;
	}

	protected final ValueStyler getValueStyler() {
		return valueStyler;
	}


	public void styleStart(StringBuffer buffer, Object obj) {
		if (!obj.getClass().isArray()) {
			buffer.append('[').append(ClassUtils.getShortName(obj.getClass()));
			styleIdentityHashCode(buffer, obj);
		}
		else {
			buffer.append('[');
			styleIdentityHashCode(buffer, obj);
			buffer.append(' ');
			styleValue(buffer, obj);
		}
	}

	private void styleIdentityHashCode(StringBuffer buffer, Object obj) {
		buffer.append('@');
		buffer.append(ObjectUtils.getIdentityHexString(obj));
	}

	public void styleEnd(StringBuffer buffer, Object o) {
		buffer.append(']');
	}

	public void styleField(StringBuffer buffer, String fieldName, Object value) {
		styleFieldStart(buffer, fieldName);
		styleValue(buffer, value);
		styleFieldEnd(buffer, fieldName);
	}

	protected void styleFieldStart(StringBuffer buffer, String fieldName) {
		buffer.append(' ').append(fieldName).append(" = ");
	}

	protected void styleFieldEnd(StringBuffer buffer, String fieldName) {
	}

	public void styleValue(StringBuffer buffer, Object value) {
		buffer.append(this.valueStyler.style(value));
	}

	public void styleFieldSeparator(StringBuffer buffer) {
		buffer.append(',');
	}

}
