/*
 * Copyright 2002-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.web.util;

import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.springframework.core.NestedExceptionUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

/**
 * Utility methods to assist with identifying and logging exceptions that
 * indicate the server response connection is lost, for example because the
 * client has gone away. This class helps to identify such exceptions and
 * minimize logging to a single line at DEBUG level, while making the full
 * error stacktrace at TRACE level.
 *
 * @author Rossen Stoyanchev
 * @author Yanming Zhou
 * @since 6.1
 */
public class DisconnectedClientHelper {

	private static final Set<String> EXCEPTION_PHRASES =
			Set.of("broken pipe", "connection reset by peer");

	private static final Set<String> EXCEPTION_TYPE_NAMES =
			Set.of("AbortedException", "ClientAbortException",
					"EOFException", "EofException", "AsyncRequestNotUsableException");

	private static final Set<Class<?>> EXCLUDED_EXCEPTION_TYPES = new LinkedHashSet<>(4);

	static {
		addExcludedExceptionType("org.springframework.web.client.RestClientException");
		addExcludedExceptionType("org.springframework.web.reactive.function.client.WebClientException");
		addExcludedExceptionType("org.springframework.dao.DataAccessException");
		addExcludedExceptionType("org.springframework.messaging.MessagingException");
	}


	private final Log logger;


	public DisconnectedClientHelper(String logCategory) {
		Assert.notNull(logCategory, "'logCategory' is required");
		this.logger = LogFactory.getLog(logCategory);
	}


	/**
	 * Check via  {@link #isClientDisconnectedException} if the exception
	 * indicates the remote client disconnected, and if so log a single line
	 * message when DEBUG is on, and a full stacktrace when TRACE is on for
	 * the configured logger.
 	 */
	public boolean checkAndLogClientDisconnectedException(Throwable ex) {
		if (isClientDisconnectedException(ex)) {
			if (logger.isTraceEnabled()) {
				logger.trace("Looks like the client has gone away", ex);
			}
			else if (logger.isDebugEnabled()) {
				logger.debug("Looks like the client has gone away: " + ex +
						" (For a full stack trace, set the log category '" + logger + "' to TRACE level.)");
			}
			return true;
		}
		return false;
	}

	/**
	 * Whether the given exception indicates the client has gone away.
	 * <p>Known cases covered:
	 * <ul>
	 * <li>ClientAbortException or EOFException for Tomcat
	 * <li>EofException for Jetty
	 * <li>IOException "Broken pipe" or "connection reset by peer"
	 * </ul>
	 */
	public static boolean isClientDisconnectedException(@Nullable Throwable ex) {
		if (ex == null) {
			return false;
		}

		Throwable currentEx = ex;
		Throwable lastEx = null;
		while (currentEx != null && currentEx != lastEx) {
			// Ignore onward connection issues to other servers (500 error)
			for (Class<?> exceptionType : EXCLUDED_EXCEPTION_TYPES) {
				if (exceptionType.isInstance(currentEx)) {
					return false;
				}
			}
			if (EXCEPTION_TYPE_NAMES.contains(currentEx.getClass().getSimpleName())) {
				return true;
			}
			lastEx = currentEx;
			currentEx = currentEx.getCause();
		}

		String message = NestedExceptionUtils.getMostSpecificCause(ex).getMessage();
		if (message != null) {
			String text = message.toLowerCase(Locale.ROOT);
			for (String phrase : EXCEPTION_PHRASES) {
				if (text.contains(phrase)) {
					return true;
				}
			}
		}

		return false;
	}

	private static void addExcludedExceptionType(String type) {
		try {
			ClassLoader classLoader = DisconnectedClientHelper.class.getClassLoader();
			EXCLUDED_EXCEPTION_TYPES.add(ClassUtils.forName(type, classLoader));
		}
		catch (ClassNotFoundException ex) {
			// ignore
		}
	}

}
