/*
 * Copyright 2020-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.data.rest.webmvc;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.repository.support.Repositories;
import org.springframework.data.repository.support.RepositoryInvokerFactory;
import org.springframework.data.rest.core.config.RepositoryRestConfiguration;
import org.springframework.data.rest.core.mapping.RepositoryResourceMappings;
import org.springframework.data.rest.webmvc.alps.AlpsController;
import org.springframework.data.rest.webmvc.json.JsonSchema;
import org.springframework.data.rest.webmvc.json.PersistentEntityToJsonSchemaConverter;
import org.springframework.data.rest.webmvc.support.RepositoryEntityLinks;

/**
 * Configuration class registering required {@link org.springframework.stereotype.Component components} that declare
 * request mappings as {@link Bean beans}.
 *
 * @author Christoph Strobl
 * @since 3.4
 */
@Configuration(proxyBeanMethods = false)
public class RestControllerConfiguration {

	private final RepositoryRestConfiguration restConfiguration;
	private final RepositoryResourceMappings resourceMappings;
	private final Repositories repositories;
	private final RepositoryEntityLinks entityLinks;

	RestControllerConfiguration(RepositoryRestConfiguration restConfiguration,
			RepositoryResourceMappings resourceMappings, Repositories repositories, RepositoryEntityLinks entityLinks) {

		this.restConfiguration = restConfiguration;
		this.resourceMappings = resourceMappings;
		this.repositories = repositories;
		this.entityLinks = entityLinks;
	}

	/**
	 * The controller for the root resource exposing links to the repository resources.
	 *
	 * @param entityLinks the accessor to links pointing to controllers backing an entity type. Must not be
	 *          {@literal null}.
	 * @return never {@literal null}.
	 */
	@Bean
	RepositoryController repositoryController() {
		return new RepositoryController(repositories, entityLinks, resourceMappings);
	}

	/**
	 * The root controller for entities reachable via {@code /{repository}}.
	 *
	 * @param entityLinks the accessor to links pointing to controllers backing an entity type. Must not be *
	 *          {@literal null}.
	 * @param headersPreparer must not be {@literal null}.
	 * @return never {@literal null}.
	 */
	@Bean
	RepositoryEntityController repositoryEntityController(HttpHeadersPreparer headersPreparer) {
		return new RepositoryEntityController(restConfiguration, entityLinks, headersPreparer);
	}

	/**
	 * The controller to access referenced properties via {@code /{repository}/{id}/{property}}.
	 *
	 * @param repositoryInvokerFactory must not be {@literal null}.
	 * @return never {@literal null}.
	 */
	@Bean
	RepositoryPropertyReferenceController repositoryPropertyReferenceController(
			RepositoryInvokerFactory repositoryInvokerFactory) {
		return new RepositoryPropertyReferenceController(repositories, repositoryInvokerFactory);
	}

	/**
	 * The controller that performs lookups and executes searches.
	 *
	 * @param entityLinks he accessor to links pointing to controllers backing an entity type. Must not be *
	 *          {@literal null}.
	 * @param headersPreparer must not be {@literal null}.
	 * @return never {@literal null}.
	 */
	@Bean
	RepositorySearchController repositorySearchController(HttpHeadersPreparer headersPreparer) {
		return new RepositorySearchController(entityLinks, resourceMappings, headersPreparer);
	}

	/**
	 * The controller that exposes the JSON schema via {@code /repository/schema}.
	 *
	 * @param jsonSchemaConverter the converter to create the {@link JsonSchema}. Must not be {@literal null}.
	 * @return never {@literal null}.
	 */
	@Bean
	RepositorySchemaController repositorySchemaController(
			PersistentEntityToJsonSchemaConverter jsonSchemaConverter) {
		return new RepositorySchemaController(jsonSchemaConverter);
	}

	/**
	 * The controller that exposes semantic documentation in the <a href="http://alps.io/">ALPS</a> (Application Level
	 * Profile Semantics) format.
	 *
	 * @return never {@literal null}.
	 */
	@Bean
	AlpsController alpsController() {
		return new AlpsController(restConfiguration);
	}

	/**
	 * Profile-based controller exposing multiple forms of metadata via {@code /profile}.
	 *
	 * @return never {@literal null}.
	 */
	@Bean
	ProfileController profileController() {
		return new ProfileController(restConfiguration, resourceMappings, repositories);
	}
}
