/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.core;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import org.aopalliance.aop.Advice;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.cglib.proxy.MethodInterceptor;
import org.springframework.cglib.proxy.MethodProxy;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisCommand;
import org.springframework.lang.Nullable;
import org.springframework.transaction.support.ResourceHolder;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;

public abstract class RedisConnectionUtils {
    private static final Log log = LogFactory.getLog(RedisConnectionUtils.class);

    public static RedisConnection bindConnection(RedisConnectionFactory factory) {
        return RedisConnectionUtils.bindConnection(factory, false);
    }

    public static RedisConnection bindConnection(RedisConnectionFactory factory, boolean transactionSupport) {
        return RedisConnectionUtils.doGetConnection(factory, true, true, transactionSupport);
    }

    public static RedisConnection getConnection(RedisConnectionFactory factory) {
        return RedisConnectionUtils.getConnection(factory, false);
    }

    public static RedisConnection getConnection(RedisConnectionFactory factory, boolean transactionSupport) {
        return RedisConnectionUtils.doGetConnection(factory, true, false, transactionSupport);
    }

    public static RedisConnection doGetConnection(RedisConnectionFactory factory, boolean allowCreate, boolean bind, boolean transactionSupport) {
        Assert.notNull((Object)factory, (String)"No RedisConnectionFactory specified");
        RedisConnectionHolder connHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource((Object)factory);
        if (connHolder != null) {
            if (transactionSupport) {
                RedisConnectionUtils.potentiallyRegisterTransactionSynchronisation(connHolder, factory);
            }
            return connHolder.getConnection();
        }
        if (!allowCreate) {
            throw new IllegalArgumentException("No connection found and allowCreate = false");
        }
        if (log.isDebugEnabled()) {
            log.debug((Object)"Opening RedisConnection");
        }
        RedisConnection conn = factory.getConnection();
        if (bind) {
            RedisConnection connectionToBind = conn;
            if (transactionSupport && RedisConnectionUtils.isActualNonReadonlyTransactionActive()) {
                connectionToBind = RedisConnectionUtils.createConnectionProxy(conn, factory);
            }
            connHolder = new RedisConnectionHolder(connectionToBind);
            TransactionSynchronizationManager.bindResource((Object)factory, (Object)connHolder);
            if (transactionSupport) {
                RedisConnectionUtils.potentiallyRegisterTransactionSynchronisation(connHolder, factory);
            }
            return connHolder.getConnection();
        }
        return conn;
    }

    private static void potentiallyRegisterTransactionSynchronisation(RedisConnectionHolder connHolder, RedisConnectionFactory factory) {
        if (RedisConnectionUtils.isActualNonReadonlyTransactionActive() && !connHolder.isTransactionSyncronisationActive()) {
            connHolder.setTransactionSyncronisationActive(true);
            RedisConnection conn = connHolder.getConnection();
            conn.multi();
            TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new RedisTransactionSynchronizer(connHolder, conn, factory));
        }
    }

    private static boolean isActualNonReadonlyTransactionActive() {
        return TransactionSynchronizationManager.isActualTransactionActive() && !TransactionSynchronizationManager.isCurrentTransactionReadOnly();
    }

    private static RedisConnection createConnectionProxy(RedisConnection connection, RedisConnectionFactory factory) {
        ProxyFactory proxyFactory = new ProxyFactory((Object)connection);
        proxyFactory.addAdvice((Advice)new ConnectionSplittingInterceptor(factory));
        return (RedisConnection)RedisConnection.class.cast(proxyFactory.getProxy());
    }

    @Deprecated
    public static void releaseConnection(@Nullable RedisConnection conn, RedisConnectionFactory factory) {
        RedisConnectionUtils.releaseConnection(conn, factory, false);
    }

    public static void releaseConnection(@Nullable RedisConnection conn, RedisConnectionFactory factory, boolean transactionSupport) {
        if (conn == null) {
            return;
        }
        RedisConnectionHolder connHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource((Object)factory);
        if (connHolder != null && connHolder.isTransactionSyncronisationActive()) {
            if (log.isDebugEnabled()) {
                log.debug((Object)"Redis Connection will be closed when transaction finished.");
            }
            return;
        }
        if (RedisConnectionUtils.isConnectionTransactional(conn, factory)) {
            if (transactionSupport && TransactionSynchronizationManager.isCurrentTransactionReadOnly()) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)"Unbinding Redis Connection.");
                }
                RedisConnectionUtils.unbindConnection(factory);
            } else if (log.isDebugEnabled()) {
                log.debug((Object)"Leaving bound Redis Connection attached.");
            }
        } else {
            RedisConnectionUtils.doCloseConnection(conn);
        }
    }

    public static void unbindConnection(RedisConnectionFactory factory) {
        RedisConnectionHolder connHolder = (RedisConnectionHolder)TransactionSynchronizationManager.unbindResourceIfPossible((Object)factory);
        if (connHolder == null) {
            return;
        }
        if (connHolder.isTransactionSyncronisationActive()) {
            if (log.isDebugEnabled()) {
                log.debug((Object)"Redis Connection will be closed when outer transaction finished.");
            }
        } else {
            RedisConnectionUtils.doCloseConnection(connHolder.getConnection());
        }
    }

    public static boolean isConnectionTransactional(RedisConnection conn, RedisConnectionFactory connFactory) {
        Assert.notNull((Object)connFactory, (String)"No RedisConnectionFactory specified");
        RedisConnectionHolder connHolder = (RedisConnectionHolder)TransactionSynchronizationManager.getResource((Object)connFactory);
        return connHolder != null && conn == connHolder.getConnection();
    }

    private static void doCloseConnection(RedisConnection connection) {
        if (log.isDebugEnabled()) {
            log.debug((Object)"Closing Redis Connection.");
        }
        try {
            connection.close();
        }
        catch (DataAccessException ex) {
            log.debug((Object)"Could not close Redis Connection", (Throwable)ex);
        }
        catch (Throwable ex) {
            log.debug((Object)"Unexpected exception on closing Redis Connection", ex);
        }
    }

    private static class RedisConnectionHolder
    implements ResourceHolder {
        private boolean unbound;
        private final RedisConnection conn;
        private boolean transactionSyncronisationActive;

        public RedisConnectionHolder(RedisConnection conn) {
            this.conn = conn;
        }

        public boolean isVoid() {
            return this.unbound;
        }

        public RedisConnection getConnection() {
            return this.conn;
        }

        public void reset() {
        }

        public void unbound() {
            this.unbound = true;
        }

        public boolean isTransactionSyncronisationActive() {
            return this.transactionSyncronisationActive;
        }

        public void setTransactionSyncronisationActive(boolean transactionSyncronisationActive) {
            this.transactionSyncronisationActive = transactionSyncronisationActive;
        }
    }

    static class ConnectionSplittingInterceptor
    implements org.aopalliance.intercept.MethodInterceptor,
    MethodInterceptor {
        private final RedisConnectionFactory factory;

        public ConnectionSplittingInterceptor(RedisConnectionFactory factory) {
            this.factory = factory;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Object intercept(Object obj, Method method, Object[] args, MethodProxy proxy) throws Throwable {
            RedisCommand commandToExecute = RedisCommand.failsafeCommandLookup(method.getName());
            if (this.isPotentiallyThreadBoundCommand(commandToExecute)) {
                if (log.isDebugEnabled()) {
                    log.debug((Object)String.format("Invoke '%s' on bound connection", method.getName()));
                }
                return this.invoke(method, obj, args);
            }
            if (log.isDebugEnabled()) {
                log.debug((Object)String.format("Invoke '%s' on unbound connection", method.getName()));
            }
            RedisConnection connection = this.factory.getConnection();
            try {
                Object object = this.invoke(method, connection, args);
                return object;
            }
            finally {
                if (!connection.isClosed()) {
                    RedisConnectionUtils.doCloseConnection(connection);
                }
            }
        }

        private Object invoke(Method method, Object target, Object[] args) throws Throwable {
            try {
                return method.invoke(target, args);
            }
            catch (InvocationTargetException e) {
                throw e.getCause();
            }
        }

        public Object invoke(MethodInvocation invocation) throws Throwable {
            return this.intercept(invocation.getThis(), invocation.getMethod(), invocation.getArguments(), null);
        }

        private boolean isPotentiallyThreadBoundCommand(RedisCommand command) {
            return RedisCommand.UNKNOWN.equals((Object)command) || !command.isReadonly();
        }
    }

    private static class RedisTransactionSynchronizer
    implements TransactionSynchronization {
        private final RedisConnectionHolder connHolder;
        private final RedisConnection connection;
        private final RedisConnectionFactory factory;

        RedisTransactionSynchronizer(RedisConnectionHolder connHolder, RedisConnection connection, RedisConnectionFactory factory) {
            this.connHolder = connHolder;
            this.connection = connection;
            this.factory = factory;
        }

        /*
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        public void afterCompletion(int status) {
            try {
                switch (status) {
                    case 0: {
                        this.connection.exec();
                        return;
                    }
                    default: {
                        this.connection.discard();
                        return;
                    }
                }
            }
            finally {
                if (log.isDebugEnabled()) {
                    log.debug((Object)("Closing bound connection after transaction completed with " + status));
                }
                this.connHolder.setTransactionSyncronisationActive(false);
                RedisConnectionUtils.doCloseConnection(this.connection);
                TransactionSynchronizationManager.unbindResource((Object)this.factory);
            }
        }
    }
}

