/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.cloud.kubernetes.leader;

import io.fabric8.kubernetes.api.model.ConfigMap;
import io.fabric8.kubernetes.api.model.ConfigMapBuilder;
import io.fabric8.kubernetes.api.model.ConfigMapFluent;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.PodResource;
import io.fabric8.kubernetes.client.dsl.Replaceable;
import io.fabric8.kubernetes.client.dsl.Resource;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.kubernetes.leader.Leader;
import org.springframework.cloud.kubernetes.leader.LeaderContext;
import org.springframework.cloud.kubernetes.leader.LeaderProperties;
import org.springframework.cloud.kubernetes.leader.PodReadinessWatcher;
import org.springframework.integration.leader.Candidate;
import org.springframework.integration.leader.Context;
import org.springframework.integration.leader.event.LeaderEventPublisher;

public class LeadershipController {
    private static final String PROVIDER_KEY = "provider";
    private static final String PROVIDER = "spring-cloud-kubernetes";
    private static final String KIND_KEY = "kind";
    private static final String KIND = "leaders";
    private static final Logger LOGGER = LoggerFactory.getLogger(LeadershipController.class);
    private final Candidate candidate;
    private final LeaderProperties leaderProperties;
    private final LeaderEventPublisher leaderEventPublisher;
    private final KubernetesClient kubernetesClient;
    private Leader localLeader;
    private PodReadinessWatcher leaderReadinessWatcher;

    public LeadershipController(Candidate candidate, LeaderProperties leaderProperties, LeaderEventPublisher leaderEventPublisher, KubernetesClient kubernetesClient) {
        this.candidate = candidate;
        this.leaderProperties = leaderProperties;
        this.leaderEventPublisher = leaderEventPublisher;
        this.kubernetesClient = kubernetesClient;
    }

    public Optional<Leader> getLocalLeader() {
        return Optional.ofNullable(this.localLeader);
    }

    public synchronized void update() {
        LOGGER.debug("Checking leader state");
        ConfigMap configMap = this.getConfigMap();
        Leader leader = this.extractLeader(configMap);
        if (leader != null && this.isPodReady(leader.getId())) {
            this.handleLeaderChange(leader);
            return;
        }
        if (leader != null && leader.isCandidate(this.candidate)) {
            this.revoke(configMap);
        } else {
            this.acquire(configMap);
        }
    }

    public synchronized void revoke() {
        ConfigMap configMap = this.getConfigMap();
        Leader leader = this.extractLeader(configMap);
        if (leader != null && leader.isCandidate(this.candidate)) {
            this.revoke(configMap);
        }
    }

    private void revoke(ConfigMap configMap) {
        LOGGER.debug("Trying to revoke leadership for '{}'", (Object)this.candidate);
        try {
            String leaderKey = this.getLeaderKey();
            this.removeConfigMapEntry(configMap, leaderKey);
            this.handleLeaderChange(null);
        }
        catch (KubernetesClientException e) {
            LOGGER.warn("Failure when revoking leadership for '{}': {}", (Object)this.candidate, (Object)e.getMessage());
        }
    }

    private void acquire(ConfigMap configMap) {
        LOGGER.debug("Trying to acquire leadership for '{}'", (Object)this.candidate);
        if (!this.isPodReady(this.candidate.getId())) {
            LOGGER.debug("Pod of '{}' is not ready at the moment, cannot acquire leadership", (Object)this.candidate);
            return;
        }
        try {
            Map<String, String> data = this.getLeaderData(this.candidate);
            if (configMap == null) {
                this.createConfigMap(data);
            } else {
                this.updateConfigMapEntry(configMap, data);
            }
            Leader newLeader = new Leader(this.candidate.getRole(), this.candidate.getId());
            this.handleLeaderChange(newLeader);
        }
        catch (KubernetesClientException e) {
            LOGGER.warn("Failure when acquiring leadership for '{}': {}", (Object)this.candidate, (Object)e.getMessage());
            this.notifyOnFailedToAcquire();
        }
    }

    private void handleLeaderChange(Leader newLeader) {
        if (Objects.equals(this.localLeader, newLeader)) {
            LOGGER.debug("Leader is still '{}'", (Object)this.localLeader);
            return;
        }
        Leader oldLeader = this.localLeader;
        this.localLeader = newLeader;
        if (oldLeader != null && oldLeader.isCandidate(this.candidate)) {
            this.notifyOnRevoked();
        } else if (newLeader != null && newLeader.isCandidate(this.candidate)) {
            this.notifyOnGranted();
        }
        this.restartLeaderReadinessWatcher();
        LOGGER.debug("New leader is '{}'", (Object)this.localLeader);
    }

    private void notifyOnGranted() {
        LOGGER.debug("Leadership has been granted for '{}'", (Object)this.candidate);
        LeaderContext context = new LeaderContext(this.candidate, this);
        this.leaderEventPublisher.publishOnGranted((Object)this, (Context)context, this.candidate.getRole());
        try {
            this.candidate.onGranted((Context)context);
        }
        catch (InterruptedException e) {
            LOGGER.warn(e.getMessage());
            Thread.currentThread().interrupt();
        }
    }

    private void notifyOnRevoked() {
        LOGGER.debug("Leadership has been revoked for '{}'", (Object)this.candidate);
        LeaderContext context = new LeaderContext(this.candidate, this);
        this.leaderEventPublisher.publishOnRevoked((Object)this, (Context)context, this.candidate.getRole());
        this.candidate.onRevoked((Context)context);
    }

    private void notifyOnFailedToAcquire() {
        if (this.leaderProperties.isPublishFailedEvents()) {
            LeaderContext context = new LeaderContext(this.candidate, this);
            this.leaderEventPublisher.publishOnFailedToAcquire((Object)this, (Context)context, this.candidate.getRole());
        }
    }

    private void restartLeaderReadinessWatcher() {
        if (this.leaderReadinessWatcher != null) {
            this.leaderReadinessWatcher.stop();
            this.leaderReadinessWatcher = null;
        }
        if (this.localLeader != null && !this.localLeader.isCandidate(this.candidate)) {
            this.leaderReadinessWatcher = new PodReadinessWatcher(this.localLeader.getId(), this.kubernetesClient, this);
            this.leaderReadinessWatcher.start();
        }
    }

    private String getLeaderKey() {
        return this.leaderProperties.getLeaderIdPrefix() + this.candidate.getRole();
    }

    private Map<String, String> getLeaderData(Candidate candidate) {
        String leaderKey = this.getLeaderKey();
        return Collections.singletonMap(leaderKey, candidate.getId());
    }

    private Leader extractLeader(ConfigMap configMap) {
        String leaderKey;
        if (configMap == null || configMap.getData() == null) {
            return null;
        }
        Map data = configMap.getData();
        String leaderId = (String)data.get(leaderKey = this.getLeaderKey());
        if (leaderId == null) {
            return null;
        }
        return new Leader(this.candidate.getRole(), leaderId);
    }

    private boolean isPodReady(String name) {
        return ((PodResource)this.kubernetesClient.pods().withName(name)).isReady();
    }

    private ConfigMap getConfigMap() {
        return (ConfigMap)((Resource)((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.leaderProperties.getNamespace(this.kubernetesClient.getNamespace()))).withName(this.leaderProperties.getConfigMapName())).get();
    }

    private void createConfigMap(Map<String, String> data) {
        LOGGER.debug("Creating new config map with data: {}", data);
        ConfigMap newConfigMap = ((ConfigMapBuilder)((ConfigMapBuilder)((ConfigMapFluent.MetadataNested)((ConfigMapFluent.MetadataNested)((ConfigMapFluent.MetadataNested)new ConfigMapBuilder().withNewMetadata().withName(this.leaderProperties.getConfigMapName())).addToLabels(PROVIDER_KEY, PROVIDER)).addToLabels(KIND_KEY, KIND)).endMetadata()).addToData(data)).build();
        ((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.leaderProperties.getNamespace(this.kubernetesClient.getNamespace()))).create((Object[])new ConfigMap[]{newConfigMap});
    }

    private void updateConfigMapEntry(ConfigMap configMap, Map<String, String> newData) {
        LOGGER.debug("Adding new data to config map: {}", newData);
        ConfigMap newConfigMap = ((ConfigMapBuilder)new ConfigMapBuilder(configMap).addToData(newData)).build();
        this.updateConfigMap(configMap, newConfigMap);
    }

    private void removeConfigMapEntry(ConfigMap configMap, String key) {
        LOGGER.debug("Removing config map entry '{}'", (Object)key);
        ConfigMap newConfigMap = ((ConfigMapBuilder)new ConfigMapBuilder(configMap).removeFromData(key)).build();
        this.updateConfigMap(configMap, newConfigMap);
    }

    private void updateConfigMap(ConfigMap oldConfigMap, ConfigMap newConfigMap) {
        ((Replaceable)((Resource)((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.leaderProperties.getNamespace(this.kubernetesClient.getNamespace()))).withName(this.leaderProperties.getConfigMapName())).lockResourceVersion(oldConfigMap.getMetadata().getResourceVersion())).replace((Object)newConfigMap);
    }
}

