package shaded.shaded.org.apache.maven.model.building;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import shaded.shaded.org.apache.maven.model.Model;
import shaded.shaded.org.apache.maven.model.composition.DefaultDependencyManagementImporter;
import shaded.shaded.org.apache.maven.model.composition.DependencyManagementImporter;
import shaded.shaded.org.apache.maven.model.inheritance.DefaultInheritanceAssembler;
import shaded.shaded.org.apache.maven.model.inheritance.InheritanceAssembler;
import shaded.shaded.org.apache.maven.model.interpolation.ModelInterpolator;
import shaded.shaded.org.apache.maven.model.interpolation.StringSearchModelInterpolator;
import shaded.shaded.org.apache.maven.model.io.DefaultModelReader;
import shaded.shaded.org.apache.maven.model.io.ModelReader;
import shaded.shaded.org.apache.maven.model.locator.DefaultModelLocator;
import shaded.shaded.org.apache.maven.model.locator.ModelLocator;
import shaded.shaded.org.apache.maven.model.management.DefaultDependencyManagementInjector;
import shaded.shaded.org.apache.maven.model.management.DefaultPluginManagementInjector;
import shaded.shaded.org.apache.maven.model.management.DependencyManagementInjector;
import shaded.shaded.org.apache.maven.model.management.PluginManagementInjector;
import shaded.shaded.org.apache.maven.model.normalization.DefaultModelNormalizer;
import shaded.shaded.org.apache.maven.model.normalization.ModelNormalizer;
import shaded.shaded.org.apache.maven.model.path.DefaultModelPathTranslator;
import shaded.shaded.org.apache.maven.model.path.DefaultModelUrlNormalizer;
import shaded.shaded.org.apache.maven.model.path.DefaultPathTranslator;
import shaded.shaded.org.apache.maven.model.path.DefaultUrlNormalizer;
import shaded.shaded.org.apache.maven.model.path.ModelPathTranslator;
import shaded.shaded.org.apache.maven.model.path.ModelUrlNormalizer;
import shaded.shaded.org.apache.maven.model.path.PathTranslator;
import shaded.shaded.org.apache.maven.model.path.UrlNormalizer;
import shaded.shaded.org.apache.maven.model.plugin.DefaultPluginConfigurationExpander;
import shaded.shaded.org.apache.maven.model.plugin.DefaultReportConfigurationExpander;
import shaded.shaded.org.apache.maven.model.plugin.DefaultReportingConverter;
import shaded.shaded.org.apache.maven.model.plugin.LifecycleBindingsInjector;
import shaded.shaded.org.apache.maven.model.plugin.PluginConfigurationExpander;
import shaded.shaded.org.apache.maven.model.plugin.ReportConfigurationExpander;
import shaded.shaded.org.apache.maven.model.plugin.ReportingConverter;
import shaded.shaded.org.apache.maven.model.profile.DefaultProfileInjector;
import shaded.shaded.org.apache.maven.model.profile.DefaultProfileSelector;
import shaded.shaded.org.apache.maven.model.profile.ProfileInjector;
import shaded.shaded.org.apache.maven.model.profile.ProfileSelector;
import shaded.shaded.org.apache.maven.model.profile.activation.FileProfileActivator;
import shaded.shaded.org.apache.maven.model.profile.activation.JdkVersionProfileActivator;
import shaded.shaded.org.apache.maven.model.profile.activation.OperatingSystemProfileActivator;
import shaded.shaded.org.apache.maven.model.profile.activation.ProfileActivator;
import shaded.shaded.org.apache.maven.model.profile.activation.PropertyProfileActivator;
import shaded.shaded.org.apache.maven.model.superpom.DefaultSuperPomProvider;
import shaded.shaded.org.apache.maven.model.superpom.SuperPomProvider;
import shaded.shaded.org.apache.maven.model.validation.DefaultModelValidator;
import shaded.shaded.org.apache.maven.model.validation.ModelValidator;

/**
 * A factory to create model builder instances when no dependency injection is available. <em>Note:</em> This class is
 * only meant as a utility for developers that want to employ the model builder outside of the Maven build system, Maven
 * plugins should always acquire model builder instances via dependency injection. Developers might want to subclass
 * this factory to provide custom implementations for some of the components used by the model builder.
 * 
 * @author Benjamin Bentmann
 */
public class DefaultModelBuilderFactory
{

    protected ModelProcessor newModelProcessor()
    {
        DefaultModelProcessor processor = new DefaultModelProcessor();
        processor.setModelLocator( newModelLocator() );
        processor.setModelReader( newModelReader() );
        return processor;
    }

    protected ModelLocator newModelLocator()
    {
        return new DefaultModelLocator();
    }

    protected ModelReader newModelReader()
    {
        return new DefaultModelReader();
    }

    protected ProfileSelector newProfileSelector()
    {
        DefaultProfileSelector profileSelector = new DefaultProfileSelector();

        for ( ProfileActivator activator : newProfileActivators() )
        {
            profileSelector.addProfileActivator( activator );
        }

        return profileSelector;
    }

    protected ProfileActivator[] newProfileActivators()
    {
        return new ProfileActivator[] { new JdkVersionProfileActivator(), new OperatingSystemProfileActivator(),
            new PropertyProfileActivator(), new FileProfileActivator().setPathTranslator( newPathTranslator() ) };
    }

    protected UrlNormalizer newUrlNormalizer()
    {
        return new DefaultUrlNormalizer();
    }

    protected PathTranslator newPathTranslator()
    {
        return new DefaultPathTranslator();
    }

    protected ModelInterpolator newModelInterpolator()
    {
        UrlNormalizer urlNormalizer = newUrlNormalizer();
        PathTranslator pathTranslator = newPathTranslator();
        return new StringSearchModelInterpolator().setPathTranslator( pathTranslator ).setUrlNormalizer( urlNormalizer );
    }

    protected ModelValidator newModelValidator()
    {
        return new DefaultModelValidator();
    }

    protected ModelNormalizer newModelNormalizer()
    {
        return new DefaultModelNormalizer();
    }

    protected ModelPathTranslator newModelPathTranslator()
    {
        return new DefaultModelPathTranslator().setPathTranslator( newPathTranslator() );
    }

    protected ModelUrlNormalizer newModelUrlNormalizer()
    {
        return new DefaultModelUrlNormalizer().setUrlNormalizer( newUrlNormalizer() );
    }

    protected InheritanceAssembler newInheritanceAssembler()
    {
        return new DefaultInheritanceAssembler();
    }

    protected ProfileInjector newProfileInjector()
    {
        return new DefaultProfileInjector();
    }

    protected SuperPomProvider newSuperPomProvider()
    {
        return new DefaultSuperPomProvider().setModelProcessor( newModelProcessor() );
    }

    protected DependencyManagementImporter newDependencyManagementImporter()
    {
        return new DefaultDependencyManagementImporter();
    }

    protected DependencyManagementInjector newDependencyManagementInjector()
    {
        return new DefaultDependencyManagementInjector();
    }

    protected LifecycleBindingsInjector newLifecycleBindingsInjector()
    {
        return new StubLifecycleBindingsInjector();
    }

    protected PluginManagementInjector newPluginManagementInjector()
    {
        return new DefaultPluginManagementInjector();
    }

    protected PluginConfigurationExpander newPluginConfigurationExpander()
    {
        return new DefaultPluginConfigurationExpander();
    }

    protected ReportConfigurationExpander newReportConfigurationExpander()
    {
        return new DefaultReportConfigurationExpander();
    }

    protected ReportingConverter newReportingConverter()
    {
        return new DefaultReportingConverter();
    }

    /**
     * Creates a new model builder instance.
     * 
     * @return The new model builder instance, never {@code null}.
     */
    public DefaultModelBuilder newInstance()
    {
        DefaultModelBuilder modelBuilder = new DefaultModelBuilder();

        modelBuilder.setModelProcessor( newModelProcessor() );
        modelBuilder.setModelValidator( newModelValidator() );
        modelBuilder.setModelNormalizer( newModelNormalizer() );
        modelBuilder.setModelPathTranslator( newModelPathTranslator() );
        modelBuilder.setModelUrlNormalizer( newModelUrlNormalizer() );
        modelBuilder.setModelInterpolator( newModelInterpolator() );
        modelBuilder.setInheritanceAssembler( newInheritanceAssembler() );
        modelBuilder.setProfileInjector( newProfileInjector() );
        modelBuilder.setProfileSelector( newProfileSelector() );
        modelBuilder.setSuperPomProvider( newSuperPomProvider() );
        modelBuilder.setDependencyManagementImporter( newDependencyManagementImporter() );
        modelBuilder.setDependencyManagementInjector( newDependencyManagementInjector() );
        modelBuilder.setLifecycleBindingsInjector( newLifecycleBindingsInjector() );
        modelBuilder.setPluginManagementInjector( newPluginManagementInjector() );
        modelBuilder.setPluginConfigurationExpander( newPluginConfigurationExpander() );
        modelBuilder.setReportConfigurationExpander( newReportConfigurationExpander() );
        modelBuilder.setReportingConverter( newReportingConverter() );

        return modelBuilder;
    }

    private static class StubLifecycleBindingsInjector
        implements LifecycleBindingsInjector
    {

        public void injectLifecycleBindings( Model model, ModelBuildingRequest request, ModelProblemCollector problems )
        {
        }

    }

}
