/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.jdbc;

import com.zaxxer.hikari.HikariConfigMXBean;
import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.pool.HikariPool;
import java.lang.reflect.Field;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import javax.sql.DataSource;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.jdbc.DataSourceUnwrapper;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.Lifecycle;
import org.springframework.core.log.LogMessage;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

public class HikariCheckpointRestoreLifecycle
implements Lifecycle {
    private static final Log logger = LogFactory.getLog(HikariCheckpointRestoreLifecycle.class);
    private static final Field CLOSE_CONNECTION_EXECUTOR;
    private final Function<HikariPool, Boolean> hasOpenConnections;
    private final HikariDataSource dataSource;
    private final ConfigurableApplicationContext applicationContext;

    public HikariCheckpointRestoreLifecycle(DataSource dataSource, ConfigurableApplicationContext applicationContext) {
        this.dataSource = DataSourceUnwrapper.unwrap(dataSource, HikariConfigMXBean.class, HikariDataSource.class);
        this.applicationContext = applicationContext;
        this.hasOpenConnections = pool -> {
            ThreadPoolExecutor closeConnectionExecutor = (ThreadPoolExecutor)ReflectionUtils.getField((Field)CLOSE_CONNECTION_EXECUTOR, (Object)pool);
            Assert.state((closeConnectionExecutor != null ? 1 : 0) != 0, (String)"'closeConnectionExecutor' was null");
            return closeConnectionExecutor.getActiveCount() > 0;
        };
    }

    public void start() {
        if (this.dataSource == null || this.dataSource.isRunning()) {
            return;
        }
        Assert.state((!this.dataSource.isClosed() ? 1 : 0) != 0, (String)"DataSource has been closed and cannot be restarted");
        if (this.dataSource.isAllowPoolSuspension()) {
            logger.info((Object)"Resuming Hikari pool");
            this.dataSource.getHikariPoolMXBean().resumePool();
        }
    }

    public void stop() {
        if (this.dataSource == null || !this.dataSource.isRunning()) {
            return;
        }
        if (this.dataSource.isAllowPoolSuspension()) {
            logger.info((Object)"Suspending Hikari pool");
            this.dataSource.getHikariPoolMXBean().suspendPool();
        } else if (this.applicationContext != null && !this.applicationContext.isClosed()) {
            logger.warn((Object)(String.valueOf(this.dataSource) + " is not configured to allow pool suspension. This will cause problems when the application is checkpointed. Please configure allow-pool-suspension to fix this!"));
        }
        this.closeConnections(Duration.ofMillis(this.dataSource.getConnectionTimeout() + 250L));
    }

    private void closeConnections(Duration shutdownTimeout) {
        logger.info((Object)"Evicting Hikari connections");
        this.dataSource.getHikariPoolMXBean().softEvictConnections();
        logger.debug((Object)LogMessage.format((String)"Waiting %d seconds for Hikari connections to be closed", (Object)shutdownTimeout.toSeconds()));
        CompletableFuture<Void> allConnectionsClosed = CompletableFuture.runAsync(this::waitForConnectionsToClose);
        try {
            allConnectionsClosed.get(shutdownTimeout.toMillis(), TimeUnit.MILLISECONDS);
            logger.debug((Object)"Hikari connections closed");
        }
        catch (InterruptedException ex) {
            logger.warn((Object)"Interrupted while waiting for connections to be closed", (Throwable)ex);
            Thread.currentThread().interrupt();
        }
        catch (TimeoutException ex) {
            logger.warn((Object)LogMessage.format((String)"Hikari connections could not be closed within %s", (Object)shutdownTimeout), (Throwable)ex);
        }
        catch (ExecutionException ex) {
            throw new RuntimeException("Failed to close Hikari connections", ex);
        }
    }

    private void waitForConnectionsToClose() {
        while (this.hasOpenConnections.apply((HikariPool)this.dataSource.getHikariPoolMXBean()).booleanValue()) {
            try {
                TimeUnit.MILLISECONDS.sleep(50L);
            }
            catch (InterruptedException ex) {
                logger.error((Object)"Interrupted while waiting for datasource connections to be closed", (Throwable)ex);
                Thread.currentThread().interrupt();
            }
        }
    }

    public boolean isRunning() {
        return this.dataSource != null && this.dataSource.isRunning();
    }

    static {
        Field closeConnectionExecutor = ReflectionUtils.findField(HikariPool.class, (String)"closeConnectionExecutor");
        Assert.state((closeConnectionExecutor != null ? 1 : 0) != 0, (String)"Unable to locate closeConnectionExecutor for HikariPool");
        Assert.state((boolean)ThreadPoolExecutor.class.isAssignableFrom(closeConnectionExecutor.getType()), () -> "Expected ThreadPoolExecutor for closeConnectionExecutor but found %s".formatted(closeConnectionExecutor.getType()));
        ReflectionUtils.makeAccessible((Field)closeConnectionExecutor);
        CLOSE_CONNECTION_EXECUTOR = closeConnectionExecutor;
    }
}

