/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.jarmode.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.FileTime;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.jspecify.annotations.Nullable;
import org.springframework.boot.jarmode.tools.Command;
import org.springframework.boot.jarmode.tools.Context;
import org.springframework.boot.jarmode.tools.IndexedJarStructure;
import org.springframework.boot.jarmode.tools.JarStructure;
import org.springframework.boot.jarmode.tools.Layers;
import org.springframework.boot.loader.jarmode.JarModeErrorException;
import org.springframework.util.Assert;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

class ExtractCommand
extends Command {
    static final Command.Option LAUNCHER_OPTION = Command.Option.flag("launcher", "Whether to extract the Spring Boot launcher");
    static final Command.Option LAYERS_OPTION = Command.Option.of("layers", "string list", "Layers to extract", true);
    static final Command.Option DESTINATION_OPTION = Command.Option.of("destination", "string", "Directory to extract files to. Defaults to a directory named after the uber JAR (without the file extension)");
    static final Command.Option FORCE_OPTION = Command.Option.flag("force", "Whether to ignore non-empty directories, extract anyway");
    private static final Command.Option LIBRARIES_DIRECTORY_OPTION = Command.Option.of("libraries", "string", "Name of the libraries directory. Only applicable when not using --launcher. Defaults to lib/");
    private static final Command.Option APPLICATION_FILENAME_OPTION = Command.Option.of("application-filename", "string", "Name of the application JAR file. Only applicable when not using --launcher. Defaults to the uber JAR filename");
    private final Context context;
    private final @Nullable Layers layers;

    ExtractCommand(Context context) {
        this(context, null);
    }

    ExtractCommand(Context context, @Nullable Layers layers) {
        super("extract", "Extract the contents from the jar", Command.Options.of(LAUNCHER_OPTION, LAYERS_OPTION, DESTINATION_OPTION, LIBRARIES_DIRECTORY_OPTION, APPLICATION_FILENAME_OPTION, FORCE_OPTION), Command.Parameters.none());
        this.context = context;
        this.layers = layers;
    }

    @Override
    void run(PrintStream out, Map<Command.Option, @Nullable String> options, List<String> parameters) {
        try {
            this.checkJarCompatibility();
            File destination = this.getDestination(options);
            ExtractCommand.checkDirectoryIsEmpty(options, destination);
            FileResolver fileResolver = this.getFileResolver(destination, options);
            fileResolver.createDirectories();
            if (options.containsKey(LAUNCHER_OPTION)) {
                this.extractArchive(fileResolver);
            } else {
                JarStructure jarStructure = this.getJarStructure();
                this.extractLibraries(fileResolver, jarStructure, options);
                this.createApplication(jarStructure, fileResolver, options);
            }
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private static void checkDirectoryIsEmpty(Map<Command.Option, @Nullable String> options, File destination) {
        if (options.containsKey(FORCE_OPTION)) {
            return;
        }
        if (!destination.exists()) {
            return;
        }
        if (!destination.isDirectory()) {
            throw new JarModeErrorException(String.valueOf(destination.getAbsoluteFile()) + " already exists and is not a directory");
        }
        File[] files = destination.listFiles();
        if (files != null && files.length > 0) {
            throw new JarModeErrorException(String.valueOf(destination.getAbsoluteFile()) + " already exists and is not empty");
        }
    }

    private void checkJarCompatibility() throws IOException {
        File file = this.context.getArchiveFile();
        try (ZipInputStream stream = new ZipInputStream(new FileInputStream(file));){
            ZipEntry entry = stream.getNextEntry();
            if (entry == null) {
                throw new JarModeErrorException("File '%s' is not compatible; ensure jar file is valid and launch script is not enabled".formatted(file));
            }
        }
    }

    private void extractLibraries(FileResolver fileResolver, JarStructure jarStructure, Map<Command.Option, @Nullable String> options) throws IOException {
        String librariesDirectory = ExtractCommand.getLibrariesDirectory(options);
        this.extractArchive(fileResolver, jarEntry -> {
            JarStructure.Entry entry = jarStructure.resolve(jarEntry);
            if (entry != null && entry.type() == JarStructure.Entry.Type.LIBRARY) {
                return librariesDirectory + entry.location();
            }
            return null;
        });
    }

    private static String getLibrariesDirectory(Map<Command.Option, @Nullable String> options) {
        String libraryDirectory = options.get(LIBRARIES_DIRECTORY_OPTION);
        if (libraryDirectory != null) {
            if (libraryDirectory.endsWith("/")) {
                return libraryDirectory;
            }
            return libraryDirectory + "/";
        }
        return "lib/";
    }

    private FileResolver getFileResolver(File destination, Map<Command.Option, @Nullable String> options) {
        String applicationFilename = this.getApplicationFilename(options);
        if (!options.containsKey(LAYERS_OPTION)) {
            return new NoLayersFileResolver(destination, applicationFilename);
        }
        Layers layers = this.getLayers();
        Set layersToExtract = StringUtils.commaDelimitedListToSet((String)options.get(LAYERS_OPTION));
        return new LayersFileResolver(destination, layers, layersToExtract, applicationFilename);
    }

    private File getDestination(Map<Command.Option, @Nullable String> options) {
        String value = options.get(DESTINATION_OPTION);
        if (value != null) {
            File destination = new File(value);
            if (destination.isAbsolute()) {
                return destination;
            }
            return new File(this.context.getWorkingDir(), destination.getPath());
        }
        return new File(this.context.getWorkingDir(), ExtractCommand.stripExtension(this.context.getArchiveFile().getName()));
    }

    private static String stripExtension(String name) {
        if (name.toLowerCase(Locale.ROOT).endsWith(".jar") || name.toLowerCase(Locale.ROOT).endsWith(".war")) {
            return name.substring(0, name.length() - 4);
        }
        return name;
    }

    private JarStructure getJarStructure() {
        IndexedJarStructure jarStructure = IndexedJarStructure.get(this.context.getArchiveFile());
        Assert.state((jarStructure != null ? 1 : 0) != 0, (String)"Couldn't read classpath index");
        return jarStructure;
    }

    private void extractArchive(FileResolver fileResolver) throws IOException {
        this.extractArchive(fileResolver, ZipEntry::getName);
    }

    private void extractArchive(FileResolver fileResolver, EntryNameTransformer entryNameTransformer) throws IOException {
        ExtractCommand.withJarEntries(this.context.getArchiveFile(), (stream, jarEntry) -> {
            if (jarEntry.isDirectory()) {
                return;
            }
            String name = entryNameTransformer.getName(jarEntry);
            if (name == null) {
                return;
            }
            File file = fileResolver.resolve(jarEntry, name);
            if (file != null) {
                ExtractCommand.extractEntry(stream, jarEntry, file);
            }
        });
    }

    private Layers getLayers() {
        return this.layers != null ? this.layers : Layers.get(this.context);
    }

    private void createApplication(JarStructure jarStructure, FileResolver fileResolver, Map<Command.Option, @Nullable String> options) throws IOException {
        File file = fileResolver.resolveApplication();
        if (file == null) {
            return;
        }
        String librariesDirectory = ExtractCommand.getLibrariesDirectory(options);
        Manifest manifest = jarStructure.createLauncherManifest(library -> librariesDirectory + library);
        ExtractCommand.mkdirs(file.getParentFile());
        try (JarOutputStream output = new JarOutputStream((OutputStream)new FileOutputStream(file), manifest);){
            EnumSet<JarStructure.Entry.Type> allowedTypes = EnumSet.of(JarStructure.Entry.Type.APPLICATION_CLASS_OR_RESOURCE, JarStructure.Entry.Type.META_INF);
            HashSet writtenEntries = new HashSet();
            ExtractCommand.withJarEntries(this.context.getArchiveFile(), (stream, jarEntry) -> {
                JarStructure.Entry entry = jarStructure.resolve(jarEntry);
                if (entry != null && allowedTypes.contains((Object)entry.type()) && StringUtils.hasLength((String)entry.location())) {
                    JarEntry newJarEntry = ExtractCommand.createJarEntry(entry.location(), jarEntry);
                    if (writtenEntries.add(newJarEntry.getName())) {
                        output.putNextEntry(newJarEntry);
                        StreamUtils.copy((InputStream)stream, (OutputStream)output);
                        output.closeEntry();
                    } else if (!newJarEntry.isDirectory()) {
                        throw new IllegalStateException("Duplicate jar entry '%s' from original location '%s'".formatted(newJarEntry.getName(), entry.originalLocation()));
                    }
                }
            });
        }
    }

    private String getApplicationFilename(Map<Command.Option, @Nullable String> options) {
        String value = options.get(APPLICATION_FILENAME_OPTION);
        if (value != null) {
            return value;
        }
        return this.context.getArchiveFile().getName();
    }

    private static void extractEntry(InputStream stream, JarEntry entry, File file) throws IOException {
        ExtractCommand.mkdirs(file.getParentFile());
        try (FileOutputStream out = new FileOutputStream(file);){
            StreamUtils.copy((InputStream)stream, (OutputStream)out);
        }
        try {
            Files.getFileAttributeView(file.toPath(), BasicFileAttributeView.class, new LinkOption[0]).setTimes(ExtractCommand.getLastModifiedTime(entry), ExtractCommand.getLastAccessTime(entry), ExtractCommand.getCreationTime(entry));
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static @Nullable FileTime getCreationTime(JarEntry entry) {
        return entry.getCreationTime() != null ? entry.getCreationTime() : entry.getLastModifiedTime();
    }

    private static @Nullable FileTime getLastAccessTime(JarEntry entry) {
        return entry.getLastAccessTime() != null ? entry.getLastAccessTime() : ExtractCommand.getLastModifiedTime(entry);
    }

    private static @Nullable FileTime getLastModifiedTime(JarEntry entry) {
        return entry.getLastModifiedTime() != null ? entry.getLastModifiedTime() : entry.getCreationTime();
    }

    private static void mkdirs(File file) throws IOException {
        if (!file.exists() && !file.mkdirs()) {
            throw new IOException("Unable to create directory " + String.valueOf(file));
        }
    }

    private static JarEntry createJarEntry(String location, JarEntry originalEntry) {
        FileTime creationTime;
        FileTime lastAccessTime;
        JarEntry jarEntry = new JarEntry(location);
        FileTime lastModifiedTime = ExtractCommand.getLastModifiedTime(originalEntry);
        if (lastModifiedTime != null) {
            jarEntry.setLastModifiedTime(lastModifiedTime);
        }
        if ((lastAccessTime = ExtractCommand.getLastAccessTime(originalEntry)) != null) {
            jarEntry.setLastAccessTime(lastAccessTime);
        }
        if ((creationTime = ExtractCommand.getCreationTime(originalEntry)) != null) {
            jarEntry.setCreationTime(creationTime);
        }
        return jarEntry;
    }

    private static void withJarEntries(File file, ThrowingConsumer callback) throws IOException {
        try (JarFile jarFile = new JarFile(file);){
            Enumeration<JarEntry> entries = jarFile.entries();
            while (entries.hasMoreElements()) {
                JarEntry entry = entries.nextElement();
                if (!StringUtils.hasLength((String)entry.getName())) continue;
                InputStream stream = jarFile.getInputStream(entry);
                try {
                    callback.accept(stream, entry);
                }
                finally {
                    if (stream == null) continue;
                    stream.close();
                }
            }
        }
    }

    private static File assertFileIsContainedInDirectory(File directory, File file, String name) throws IOException {
        String canonicalOutputPath = directory.getCanonicalPath() + File.separator;
        String canonicalEntryPath = file.getCanonicalPath();
        Assert.state((boolean)canonicalEntryPath.startsWith(canonicalOutputPath), () -> "Entry '%s' would be written to '%s'. This is outside the output location of '%s'. Verify the contents of your archive.".formatted(name, canonicalEntryPath, canonicalOutputPath));
        return file;
    }

    private static interface FileResolver {
        public void createDirectories() throws IOException;

        default public @Nullable File resolve(JarEntry entry, String newName) throws IOException {
            return this.resolve(entry.getName(), newName);
        }

        public @Nullable File resolve(String var1, String var2) throws IOException;

        public @Nullable File resolveApplication() throws IOException;
    }

    @FunctionalInterface
    private static interface EntryNameTransformer {
        public @Nullable String getName(JarEntry var1);
    }

    private static final class NoLayersFileResolver
    implements FileResolver {
        private final File directory;
        private final String applicationFilename;

        private NoLayersFileResolver(File directory, String applicationFilename) {
            this.directory = directory;
            this.applicationFilename = applicationFilename;
        }

        @Override
        public void createDirectories() {
        }

        @Override
        public File resolve(String originalName, String newName) throws IOException {
            return ExtractCommand.assertFileIsContainedInDirectory(this.directory, new File(this.directory, newName), newName);
        }

        @Override
        public File resolveApplication() throws IOException {
            return this.resolve(this.applicationFilename, this.applicationFilename);
        }
    }

    private static final class LayersFileResolver
    implements FileResolver {
        private final Layers layers;
        private final Set<String> layersToExtract;
        private final File directory;
        private final String applicationFilename;

        LayersFileResolver(File directory, Layers layers, Set<String> layersToExtract, String applicationFilename) {
            this.layers = layers;
            this.layersToExtract = layersToExtract;
            this.directory = directory;
            this.applicationFilename = applicationFilename;
        }

        @Override
        public void createDirectories() throws IOException {
            for (String layer : this.layers) {
                if (!this.shouldExtractLayer(layer)) continue;
                ExtractCommand.mkdirs(this.getLayerDirectory(layer));
            }
        }

        @Override
        public @Nullable File resolve(String originalName, String newName) throws IOException {
            String layer = this.layers.getLayer(originalName);
            if (this.shouldExtractLayer(layer)) {
                File directory = this.getLayerDirectory(layer);
                return ExtractCommand.assertFileIsContainedInDirectory(directory, new File(directory, newName), newName);
            }
            return null;
        }

        @Override
        public @Nullable File resolveApplication() throws IOException {
            String layer = this.layers.getApplicationLayerName();
            if (this.shouldExtractLayer(layer)) {
                File directory = this.getLayerDirectory(layer);
                return ExtractCommand.assertFileIsContainedInDirectory(directory, new File(directory, this.applicationFilename), this.applicationFilename);
            }
            return null;
        }

        private File getLayerDirectory(String layer) {
            return new File(this.directory, layer);
        }

        private boolean shouldExtractLayer(String layer) {
            return this.layersToExtract.isEmpty() || this.layersToExtract.contains(layer);
        }
    }

    @FunctionalInterface
    private static interface ThrowingConsumer {
        public void accept(InputStream var1, JarEntry var2) throws IOException;
    }
}

