/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.devtools.tunnel.client;

import java.io.Closeable;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.devtools.tunnel.client.TunnelConnection;
import org.springframework.boot.devtools.tunnel.payload.HttpTunnelPayload;
import org.springframework.boot.devtools.tunnel.payload.HttpTunnelPayloadForwarder;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.HttpStatus;
import org.springframework.http.client.ClientHttpRequest;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.util.Assert;

public class HttpTunnelConnection
implements TunnelConnection {
    private static Log logger = LogFactory.getLog(HttpTunnelConnection.class);
    private final URI uri;
    private final ClientHttpRequestFactory requestFactory;
    private final Executor executor;

    public HttpTunnelConnection(String url, ClientHttpRequestFactory requestFactory) {
        this(url, requestFactory, null);
    }

    protected HttpTunnelConnection(String url, ClientHttpRequestFactory requestFactory, Executor executor) {
        Assert.hasLength((String)url, (String)"URL must not be empty");
        Assert.notNull((Object)requestFactory, (String)"RequestFactory must not be null");
        try {
            this.uri = new URL(url).toURI();
        }
        catch (URISyntaxException ex) {
            throw new IllegalArgumentException("Malformed URL '" + url + "'");
        }
        catch (MalformedURLException ex) {
            throw new IllegalArgumentException("Malformed URL '" + url + "'");
        }
        this.requestFactory = requestFactory;
        this.executor = executor == null ? Executors.newCachedThreadPool(new TunnelThreadFactory()) : executor;
    }

    @Override
    public TunnelChannel open(WritableByteChannel incomingChannel, Closeable closeable) throws Exception {
        logger.trace((Object)("Opening HTTP tunnel to " + this.uri));
        return new TunnelChannel(incomingChannel, closeable);
    }

    protected final ClientHttpRequest createRequest(boolean hasPayload) throws IOException {
        HttpMethod method = hasPayload ? HttpMethod.POST : HttpMethod.GET;
        return this.requestFactory.createRequest(this.uri, method);
    }

    private static class TunnelThreadFactory
    implements ThreadFactory {
        private TunnelThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            Thread thread = new Thread(runnable, "HTTP Tunnel Connection");
            thread.setDaemon(true);
            return thread;
        }
    }

    protected class TunnelChannel
    implements WritableByteChannel {
        private final HttpTunnelPayloadForwarder forwarder;
        private final Closeable closeable;
        private boolean open = true;
        private AtomicLong requestSeq = new AtomicLong();

        public TunnelChannel(WritableByteChannel incomingChannel, Closeable closeable) {
            this.forwarder = new HttpTunnelPayloadForwarder(incomingChannel);
            this.closeable = closeable;
            this.openNewConnection(null);
        }

        @Override
        public boolean isOpen() {
            return this.open;
        }

        @Override
        public void close() throws IOException {
            if (this.open) {
                this.open = false;
                this.closeable.close();
            }
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            int size = src.remaining();
            if (size > 0) {
                this.openNewConnection(new HttpTunnelPayload(this.requestSeq.incrementAndGet(), src));
            }
            return size;
        }

        private synchronized void openNewConnection(final HttpTunnelPayload payload) {
            HttpTunnelConnection.this.executor.execute(new Runnable(){

                @Override
                public void run() {
                    try {
                        TunnelChannel.this.sendAndReceive(payload);
                    }
                    catch (IOException ex) {
                        logger.trace((Object)"Unexpected connection error", (Throwable)ex);
                        this.closeQuitely();
                    }
                }

                private void closeQuitely() {
                    try {
                        TunnelChannel.this.close();
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
            });
        }

        private void sendAndReceive(HttpTunnelPayload payload) throws IOException {
            ClientHttpRequest request = HttpTunnelConnection.this.createRequest(payload != null);
            if (payload != null) {
                payload.logIncoming();
                payload.assignTo((HttpOutputMessage)request);
            }
            this.handleResponse(request.execute());
        }

        private void handleResponse(ClientHttpResponse response) throws IOException {
            HttpTunnelPayload payload;
            if (response.getStatusCode() == HttpStatus.GONE) {
                this.close();
                return;
            }
            if (response.getStatusCode() == HttpStatus.OK && (payload = HttpTunnelPayload.get((HttpInputMessage)response)) != null) {
                this.forwarder.forward(payload);
            }
            if (response.getStatusCode() != HttpStatus.TOO_MANY_REQUESTS) {
                this.openNewConnection(null);
            }
        }
    }
}

