/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.es;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ListMultimap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang.math.RandomUtils;
import org.sonar.api.Startable;
import org.sonar.api.config.Configuration;
import org.sonar.api.utils.System2;
import org.sonar.api.utils.log.Logger;
import org.sonar.api.utils.log.Loggers;
import org.sonar.api.utils.log.Profiler;
import org.sonar.core.util.stream.MoreCollectors;
import org.sonar.db.DbClient;
import org.sonar.db.DbSession;
import org.sonar.db.es.EsQueueDto;
import org.sonar.server.es.IndexType;
import org.sonar.server.es.IndexingResult;
import org.sonar.server.es.ResilientIndexer;

public class RecoveryIndexer
implements Startable {
    private static final Logger LOGGER = Loggers.get(RecoveryIndexer.class);
    private static final String LOG_PREFIX = "Elasticsearch recovery - ";
    private static final String PROPERTY_INITIAL_DELAY = "sonar.search.recovery.initialDelayInMs";
    private static final String PROPERTY_DELAY = "sonar.search.recovery.delayInMs";
    private static final String PROPERTY_MIN_AGE = "sonar.search.recovery.minAgeInMs";
    private static final String PROPERTY_LOOP_LIMIT = "sonar.search.recovery.loopLimit";
    private static final long DEFAULT_DELAY_IN_MS = 300000L;
    private static final long DEFAULT_MIN_AGE_IN_MS = 300000L;
    private static final int DEFAULT_LOOP_LIMIT = 10000;
    private static final double CIRCUIT_BREAKER_IN_PERCENT = 0.7;
    private final ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setPriority(1).setNameFormat("RecoveryIndexer-%d").build());
    private final System2 system2;
    private final Configuration config;
    private final DbClient dbClient;
    private final Map<IndexType, ResilientIndexer> indexersByType;
    private final long minAgeInMs;
    private final long loopLimit;

    public RecoveryIndexer(System2 system2, Configuration config, DbClient dbClient, ResilientIndexer ... indexers) {
        this.system2 = system2;
        this.config = config;
        this.dbClient = dbClient;
        this.indexersByType = new HashMap<IndexType, ResilientIndexer>();
        Arrays.stream(indexers).forEach(i -> i.getIndexTypes().forEach(indexType -> this.indexersByType.put((IndexType)indexType, (ResilientIndexer)i)));
        this.minAgeInMs = this.getSetting(PROPERTY_MIN_AGE, 300000L);
        this.loopLimit = this.getSetting(PROPERTY_LOOP_LIMIT, 10000L);
    }

    public void start() {
        long delayInMs = this.getSetting(PROPERTY_DELAY, 300000L);
        long initialDelayInMs = this.getSetting(PROPERTY_INITIAL_DELAY, RandomUtils.nextInt((int)(1 + (int)(delayInMs / 2L))));
        this.executorService.scheduleAtFixedRate(this::recover, initialDelayInMs, delayInMs, TimeUnit.MILLISECONDS);
    }

    public void stop() {
        try {
            this.executorService.shutdown();
            this.executorService.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            LOGGER.error("Elasticsearch recovery - Unable to stop recovery indexer in timely fashion", (Throwable)e);
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    @VisibleForTesting
    void recover() {
        try (DbSession dbSession = this.dbClient.openSession(false);){
            Profiler profiler = Profiler.create((Logger)LOGGER).start();
            long beforeDate = this.system2.now() - this.minAgeInMs;
            IndexingResult result = new IndexingResult();
            Collection items = this.dbClient.esQueueDao().selectForRecovery(dbSession, beforeDate, this.loopLimit);
            while (!items.isEmpty()) {
                IndexingResult loopResult = new IndexingResult();
                RecoveryIndexer.groupItemsByType(items).asMap().forEach((type, typeItems) -> loopResult.add(this.doIndex(dbSession, (IndexType)type, (Collection<EsQueueDto>)typeItems)));
                result.add(loopResult);
                if (loopResult.getSuccessRatio() <= 0.7) {
                    LOGGER.error("Elasticsearch recovery - too many failures [{}/{} documents], waiting for next run", (Object)loopResult.getFailures(), (Object)loopResult.getTotal());
                    break;
                }
                if (loopResult.getTotal() == 0L) break;
                items = this.dbClient.esQueueDao().selectForRecovery(dbSession, beforeDate, this.loopLimit);
            }
            if (result.getTotal() > 0L) {
                profiler.stopInfo(LOG_PREFIX + String.format("%d documents processed [%d failures]", result.getTotal(), result.getFailures()));
            }
        }
        catch (Throwable t) {
            LOGGER.error("Elasticsearch recovery - fail to recover documents", t);
        }
    }

    private IndexingResult doIndex(DbSession dbSession, IndexType type, Collection<EsQueueDto> typeItems) {
        LOGGER.trace("Elasticsearch recovery - processing {} {}", (Object)typeItems.size(), (Object)type);
        ResilientIndexer indexer = this.indexersByType.get(type);
        if (indexer == null) {
            LOGGER.error("Elasticsearch recovery - ignore {} items with unsupported type {}", (Object)typeItems.size(), (Object)type);
            return new IndexingResult();
        }
        return indexer.index(dbSession, typeItems);
    }

    private static ListMultimap<IndexType, EsQueueDto> groupItemsByType(Collection<EsQueueDto> items) {
        return (ListMultimap)items.stream().collect(MoreCollectors.index(i -> IndexType.parse(i.getDocType())));
    }

    private long getSetting(String key, long defaultValue) {
        long val = this.config.getLong(key).orElse(defaultValue);
        LOGGER.debug("Elasticsearch recovery - {}={}", (Object)key, (Object)val);
        return val;
    }
}

